use {
    anchor_lang::{
        prelude::*,
        solana_program::system_program
    },
    crate::state::*,
    std::mem::size_of
};

#[derive(Accounts)]
#[instruction(
    value: Pubkey,
    bump: u8,
)]
pub struct PushElement<'info> {
    #[account(
        mut, 
        seeds = [
            SEED_STACK,
            stack.owner.key().as_ref(), 
            stack.namespace.as_ref()
        ],
        bump = stack.bump, 
        has_one = owner,
    )]
    pub stack: Account<'info, Stack>,

    #[account(mut)]
    pub owner: Signer<'info>,

    #[account(mut)]
    pub payer: Signer<'info>,

    #[account(
        init,
        seeds = [
            SEED_ELEMENT,
            stack.key().as_ref(),
            stack.count.to_be_bytes().as_ref(),
        ],
        bump = bump,
        payer = payer,
        space = 8 + size_of::<Element>(),
    )]
    pub element: Account<'info, Element>,
    
    #[account(address = system_program::ID)]
    pub system_program: Program<'info, System>,
}

pub fn handler(
    ctx: Context<PushElement>,
    value: Pubkey,
    bump: u8,
) -> ProgramResult {
    // Get accounts.
    let element = &mut ctx.accounts.element;
    let stack = &mut ctx.accounts.stack;
    
    // Initialize element account.
    element.position = stack.count;
    element.value = value;
    element.bump = bump;

    // Increment stack counter.
    stack.count += 1;
    
    return Ok(());
}
