import assert from "assert";
import { BN, Provider, setProvider } from "@project-serum/anchor";
import { Keypair } from "@solana/web3.js";
import { airdrop, PDA, signAndSubmit } from "@faktorfi/utils";
import { StackProgram } from ".";

describe("Stack Program", () => {
  // Configure the stackProgram to use the local cluster.
  const provider = Provider.env();
  setProvider(provider);
  const stackProgram = new StackProgram(provider);

  // Shared test data.
  const addressA = Keypair.generate().publicKey;
  const addressB = Keypair.generate().publicKey;
  const namespace = Keypair.generate().publicKey;
  const owner = Keypair.generate();
  const signer = Keypair.generate();
  let stackPDA: PDA;

  before(async () => {
    await airdrop(1, owner.publicKey, provider.connection);
    await airdrop(1, signer.publicKey, provider.connection);
  });

  it("creates an stack", async () => {
    // Generate instruction and accounts.
    const ix = await stackProgram.instruction.createStack({
      owner: owner.publicKey,
      payer: owner.publicKey,
      namespace: namespace,
    });

    // Sign and submit transaction.
    await signAndSubmit(provider.connection, [ix], owner);

    // Validate index account state.
    stackPDA = await stackProgram.account.stack.pda(owner.publicKey, namespace);
    const stackData = await stackProgram.account.stack.data(stackPDA.address);
    assert.ok(stackData.owner.toString() === owner.publicKey.toString());
    assert.ok(stackData.namespace.toString() === namespace.toString());
    assert.ok(stackData.count.toNumber() === 0);
    assert.ok(stackData.bump === stackPDA.bump);
  });

  it("creates a element at position 0", async () => {
    // Generate instruction.
    const name = "foo";
    const ix = await stackProgram.instruction.pushElement({
      stack: stackPDA.address,
      owner: owner.publicKey,
      value: addressA,
    });

    // Sign and submit transaction.
    await signAndSubmit(provider.connection, [ix], owner);

    // Validate index account data.
    const stackData = await stackProgram.account.stack.data(stackPDA.address);
    assert.ok(stackData.owner.toString() === owner.publicKey.toString());
    assert.ok(stackData.namespace.toString() === namespace.toString());
    assert.ok(stackData.count.toNumber() === 1);
    assert.ok(stackData.bump === stackPDA.bump);

    // Validate pointer account data.
    const ZERO = new BN(0);
    const elementPDA = await stackProgram.account.element.pda(
      stackPDA.address,
      ZERO
    );
    const elementData = await stackProgram.account.element.data(
      elementPDA.address
    );
    assert.ok(elementData.position.eq(ZERO));
    assert.ok(elementData.value.toString() === addressA.toString());
    assert.ok(elementData.bump === elementPDA.bump);
  });

  it("creates a element at position 1", async () => {
    // Generate instruction.
    const ix = await stackProgram.instruction.pushElement({
      stack: stackPDA.address,
      owner: owner.publicKey,
      value: addressB,
    });

    // Sign and submit transaction.
    await signAndSubmit(provider.connection, [ix], owner);

    // Validate index account data.
    const stackData = await stackProgram.account.stack.data(stackPDA.address);
    assert.ok(stackData.owner.toString() === owner.publicKey.toString());
    assert.ok(stackData.namespace.toString() === namespace.toString());
    assert.ok(stackData.count.toNumber() === 2);
    assert.ok(stackData.bump === stackPDA.bump);

    // Validate pointer account data.
    const ONE = new BN(1);
    const elementPDA = await stackProgram.account.element.pda(
      stackPDA.address,
      ONE
    );
    const elementData = await stackProgram.account.element.data(
      elementPDA.address
    );
    assert.ok(elementData.position.eq(ONE));
    assert.ok(elementData.value.toString() === addressB.toString());
    assert.ok(elementData.bump === elementPDA.bump);
  });

  it("pops the element at position 1", async () => {
    // Generate instruction.
    const ix = await stackProgram.instruction.popElement({
      stack: stackPDA.address,
    });

    // Sign and submit transaction.
    await signAndSubmit(provider.connection, [ix], owner);

    // Validate pointer data.
    const ONE = new BN(1);
    const elementPDA = await stackProgram.account.element.pda(
      stackPDA.address,
      ONE
    );
    await assert.rejects(
      stackProgram.account.element.data(elementPDA.address),
      `Error: Account does not exist ${elementPDA.address}`
    );

    // Validate index data.
    const stackData = await stackProgram.account.stack.data(stackPDA.address);
    assert.ok(stackData.owner.toString() === owner.publicKey.toString());
    assert.ok(stackData.namespace.toString() === namespace.toString());
    assert.ok(stackData.count.toNumber() === 1);
    assert.ok(stackData.bump === stackPDA.bump);
  });

  it("pops the element at position 0", async () => {
    // Generate instruction.
    const ix = await stackProgram.instruction.popElement({
      stack: stackPDA.address,
    });

    // Sign and submit transaction.
    await signAndSubmit(provider.connection, [ix], owner);

    // Validate pointer data.
    const ZERO = new BN(0);
    const elementPDA = await stackProgram.account.element.pda(
      stackPDA.address,
      ZERO
    );
    await assert.rejects(
      stackProgram.account.element.data(elementPDA.address),
      `Error: Account does not exist ${elementPDA.address}`
    );

    // Validate index data.
    const stackData = await stackProgram.account.stack.data(stackPDA.address);
    assert.ok(stackData.owner.toString() === owner.publicKey.toString());
    assert.ok(stackData.namespace.toString() === namespace.toString());
    assert.ok(stackData.count.toNumber() === 0);
    assert.ok(stackData.bump === stackPDA.bump);
  });

  it("deletes a stack", async () => {
    // Generate instruction.
    const ix = await stackProgram.instruction.deleteStack({
      stack: stackPDA.address,
    });

    // Sign and submit transaction.
    await signAndSubmit(provider.connection, [ix], owner);

    // Validate pointer data.
    await assert.rejects(
      stackProgram.account.stack.data(stackPDA.address),
      `Error: Account does not exist ${stackPDA.address}`
    );
  });
});
