use borsh::{BorshDeserialize, BorshSerialize};

#[derive(Clone, Copy, PartialEq, Eq, PartialOrd, Ord, BorshSerialize, BorshDeserialize)]
pub enum StorageOutcome {
    /// A value has been added to storage
    Inserted([u8; 32]),
    /// The existing value has been replaced with the new value
    Updated([u8; 32]),
    /// The value has been removed
    Zeroed([u8; 32]),
    /// The value does not exist
    NonExistentValue([u8; 32]),
    /// The value already exists in storage
    ValueExists([u8; 32]),
    /// The capacity of the storage has been reached
    CapacityExceeded,
    /// Storage filled with zero bytes array
    Vacant,
    /// Storage filled with non-zero bytes array
    Occupied,
}

#[cfg(feature = "hex_debug")]
use core::fmt;
#[cfg(feature = "hex_debug")]
impl fmt::Debug for StorageOutcome {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        write!(
            f,
            "StorageOutcome::{}",
            match self {
                StorageOutcome::Inserted(value) =>
                    format!("{}({})", "Inserted", hex::encode(value)),
                StorageOutcome::Updated(value) => format!("{}({})", "Updated", hex::encode(value)),
                StorageOutcome::Zeroed(value) => format!("{}({})", "Zeroed", hex::encode(value)),
                StorageOutcome::NonExistentValue(value) =>
                    format!("{}({})", "NonExistentValue", hex::encode(value)),
                StorageOutcome::ValueExists(value) =>
                    format!("{}({})", "ValueExists", hex::encode(value)),
                StorageOutcome::CapacityExceeded => format!("{}", "CapacityExceeded"),
                StorageOutcome::Vacant => format!("{}", "Vacant"),
                StorageOutcome::Occupied => format!("{}", "Occupied"),
            }
        )
    }
}

#[cfg(feature = "satoshis_ghost")]
use core::fmt;
#[cfg(feature = "satoshis_ghost")]
impl fmt::Debug for StorageOutcome {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        write!(
            f,
            "StorageOutcome::{}",
            match self {
                StorageOutcome::Inserted(value) =>
                    format!("{}({})", "Inserted", bs58::encode(value).into_string()),
                StorageOutcome::Updated(value) =>
                    format!("{}({})", "Updated", bs58::encode(value).into_string()),
                StorageOutcome::Zeroed(value) =>
                    format!("{}({})", "Zeroed", bs58::encode(value).into_string()),
                StorageOutcome::NonExistentValue(value) => format!(
                    "{}({})",
                    "NonExistentValue",
                    bs58::encode(value).into_string()
                ),
                StorageOutcome::ValueExists(value) =>
                    format!("{}({})", "ValueExists", bs58::encode(value).into_string()),
                StorageOutcome::CapacityExceeded => format!("{}", "CapacityExceeded"),
                StorageOutcome::Vacant => format!("{}", "Vacant"),
                StorageOutcome::Occupied => format!("{}", "Occupied"),
            }
        )
    }
}
