mod memdb;
mod outcomes;
mod stack_usize;
mod utilities;

pub use memdb::*;
pub use outcomes::*;
pub use stack_usize::*;
pub use utilities::*;

#[cfg(test)]

mod storage_tests {

    #[test]
    fn test_stack_usize() {
        use crate::{init_storage_stack, StackMemDB, StackUsize};

        let stack_memdb = StackMemDB::<{ init_storage_stack(&StackUsize::U4) }>::new();

        assert!(stack_memdb.storage_len() == init_storage_stack(&StackUsize::U4));
    }

    #[test]
    fn test_insert() {
        use crate::{init_storage_stack, StackMemDB, StackUsize, StorageOutcome};
        use wasmium_random::WasmiumRandom;

        let mut stack_memdb = StackMemDB::<{ init_storage_stack(&StackUsize::U4) }>::new();

        assert!(stack_memdb.storage_len() == init_storage_stack(&StackUsize::U4));

        let alice = WasmiumRandom::secure_numeric32();
        let bob = WasmiumRandom::secure_numeric32();
        let eve = WasmiumRandom::secure_numeric32();
        let john = WasmiumRandom::secure_numeric32();
        let jane = WasmiumRandom::secure_numeric32();

        assert_eq!(stack_memdb.insert(alice), StorageOutcome::Inserted(alice));
        assert_eq!(stack_memdb.insert(bob), StorageOutcome::Inserted(bob));
        assert_eq!(stack_memdb.insert(eve), StorageOutcome::Inserted(eve));
        assert_eq!(stack_memdb.insert(john), StorageOutcome::Inserted(john));
        assert_eq!(stack_memdb.insert(jane), StorageOutcome::CapacityExceeded);
    }

    #[test]
    fn test_update() {
        use crate::{init_storage_stack, StackMemDB, StackUsize, StorageOutcome};
        use wasmium_random::WasmiumRandom;

        let mut stack_memdb = StackMemDB::<{ init_storage_stack(&StackUsize::U4) }>::new();

        assert!(stack_memdb.storage_len() == init_storage_stack(&StackUsize::U4));

        let alice = WasmiumRandom::secure_numeric32();
        let bob = WasmiumRandom::secure_numeric32();
        let eve = WasmiumRandom::secure_numeric32();
        let john = WasmiumRandom::secure_numeric32();
        let jane = WasmiumRandom::secure_numeric32();
        let doe = WasmiumRandom::secure_numeric32();

        assert_eq!(stack_memdb.insert(alice), StorageOutcome::Inserted(alice));
        assert_eq!(stack_memdb.insert(bob), StorageOutcome::Inserted(bob));
        assert_eq!(stack_memdb.insert(eve), StorageOutcome::Inserted(eve));
        assert_eq!(stack_memdb.insert(john), StorageOutcome::Inserted(john));
        assert_eq!(stack_memdb.insert(jane), StorageOutcome::CapacityExceeded);
        assert_eq!(
            stack_memdb.update(alice, doe),
            StorageOutcome::Updated(alice)
        );
    }

    #[test]
    fn test_remove() {
        use crate::{init_storage_stack, StackMemDB, StackUsize, StorageOutcome};
        use wasmium_random::WasmiumRandom;

        let mut stack_memdb = StackMemDB::<{ init_storage_stack(&StackUsize::U3) }>::new();

        assert!(stack_memdb.storage_len() == init_storage_stack(&StackUsize::U3));

        let alice = WasmiumRandom::secure_numeric32();
        let bob = WasmiumRandom::secure_numeric32();
        let eve = WasmiumRandom::secure_numeric32();

        stack_memdb.insert(alice);
        stack_memdb.insert(bob);
        stack_memdb.insert(eve);
        assert_eq!(stack_memdb.remove(alice), StorageOutcome::Zeroed(alice));

        assert_eq!(
            Some(0_usize),
            match stack_memdb.find_and_index([0_u8; 32]) {
                (Some(index), _) => Some(index),
                _ => None,
            }
        )
    }

    #[test]
    fn test_find() {
        use crate::{init_storage_stack, StackMemDB, StackUsize};
        use wasmium_random::WasmiumRandom;

        let mut stack_memdb = StackMemDB::<{ init_storage_stack(&StackUsize::U4) }>::new();

        assert!(stack_memdb.storage_len() == init_storage_stack(&StackUsize::U4));

        let alice = WasmiumRandom::secure_numeric32();
        let bob = WasmiumRandom::secure_numeric32();
        let eve = WasmiumRandom::secure_numeric32();
        let john = WasmiumRandom::secure_numeric32();
        let jane = WasmiumRandom::secure_numeric32();

        stack_memdb.insert(alice);
        stack_memdb.insert(bob);
        stack_memdb.insert(eve);
        stack_memdb.insert(john);
        stack_memdb.insert(jane);

        assert_eq!(stack_memdb.find(alice), Some(alice));
        assert_eq!(stack_memdb.find(bob), Some(bob));
        assert_eq!(stack_memdb.find(eve), Some(eve));
        assert_eq!(stack_memdb.find(john), Some(john));
        assert_eq!(stack_memdb.find(jane), None);
    }

    #[test]
    fn test_borsh() {
        use crate::{init_storage_stack, StackMemDB, StackUsize};
        use borsh::{BorshDeserialize, BorshSerialize};
        use wasmium_random::WasmiumRandom;

        let mut stack_memdb = StackMemDB::<{ init_storage_stack(&StackUsize::U4) }>::new();

        assert!(stack_memdb.storage_len() == init_storage_stack(&StackUsize::U4));

        let alice = WasmiumRandom::secure_numeric32();
        let bob = WasmiumRandom::secure_numeric32();
        let eve = WasmiumRandom::secure_numeric32();
        let john = WasmiumRandom::secure_numeric32();
        let jane = WasmiumRandom::secure_numeric32();

        stack_memdb.insert(alice);
        stack_memdb.insert(bob);
        stack_memdb.insert(eve);
        stack_memdb.insert(john);
        stack_memdb.insert(jane);

        let ser = stack_memdb.try_to_vec().unwrap();

        let deser_stack_memdb =
            StackMemDB::<{ init_storage_stack(&StackUsize::U4) }>::try_from_slice(&ser).unwrap();

        assert_eq!(stack_memdb, deser_stack_memdb)
    }
}
