use crate::json_ld::Direction;
use langtag::{
	LanguageTag,
	LanguageTagBuf
};

/// Language string.
/// 
/// A language string is a string tagged with language and reading direction information.
/// 
/// A valid language string is associated to either a language tag or a direction, or both.
#[derive(Clone, PartialEq, Eq, Hash, Debug)]
pub struct LangString {
	/// Actual content of the string.
	data: String,
	language: Option<LanguageTagBuf>,
	direction: Option<Direction>
}

/// Raised when something tried to build a language string without language tag or direction.
#[derive(Clone, Copy, Debug)]
pub struct InvalidLangString;

impl LangString {
	/// Create a new language string.
	pub fn new(str: String, language: Option<LanguageTagBuf>, direction: Option<Direction>) -> Result<LangString, String> {
		if language.is_some() || direction.is_some() {
			Ok(LangString {
				data: str,
				language: language,
				direction: direction
			})
		} else {
			Err(str)
		}
	}

	/// Reference to the undrlying string.
	pub fn as_str(&self) -> &str {
		self.data.as_str()
	}

	/// Gets the associated language tag, if any.
	pub fn language(&self) -> Option<LanguageTag> {
		match &self.language {
			Some(tag) => Some(tag.as_ref()),
			None => None
		}
	}

	/// Sets the associated language tag.
	/// 
	/// If `None` is given, the direction must be set,
	/// otherwise this function will fail with an [`InvalidLangString`] error.
	pub fn set_language(&mut self, language: Option<LanguageTagBuf>) -> Result<(), InvalidLangString> {
		if self.direction.is_some() || language.is_some() {
			self.language = language;
			Ok(())
		} else {
			Err(InvalidLangString)
		}
	}

	/// Gets the associated direction, if any.
	pub fn direction(&self) -> Option<Direction> {
		self.direction
	}

	/// Sets the associated direction.
	/// 
	/// If `None` is given, a language tag must be set,
	/// otherwise this function will fail with an [`InvalidLangString`] error.
	pub fn set_direction(&mut self, direction: Option<Direction>) -> Result<(), InvalidLangString> {
		if direction.is_some() || self.language.is_some() {
			self.direction = direction;
			Ok(())
		} else {
			Err(InvalidLangString)
		}
	}

	/// Set both the language tag and direction.
	/// 
	/// If both `language` and `direction` are `None`,
	/// this function will fail with an [`InvalidLangString`] error.
	pub fn set(&mut self, language: Option<LanguageTagBuf>, direction: Option<Direction>) -> Result<(), InvalidLangString> {
		if direction.is_some() || language.is_some() {
			self.language = language;
			self.direction = direction;
			Ok(())
		} else {
			Err(InvalidLangString)
		}
	}
}
