extern crate rand;
extern crate rand_distr;

extern crate sr_rcd;

use rand::prelude::*;
use rand_distr::StandardNormal;

use std::time::Instant;

use sr_rcd::{
    Point,
    QMSpace,
    rcd,
    RMP,
    Sensor
};

fn make_random_sr_problem(qmsp: &impl QMSpace, rad: f64, n: usize, err_dev: f64) -> (Vec<Sensor>, Point) {
    let mut rng = thread_rng();
    let s = Point::new_random(&mut rng, qmsp.dim(), rad);
    let t0 = rng.gen_range((-16.0)..=16.0);
    ((0..n).map(|_| {
        let p = Point::new_random(&mut rng, qmsp.dim(), rad);
        let err = err_dev * rng.sample::<f64, _>(StandardNormal);
        let t = t0 + qmsp.tau(&s, &p) + err;
        Sensor::make(&p, t)
    }).collect::<Vec<Sensor>>(), s)
}

fn main() {
    let space = RMP::new(2, 3.14, 1e-4, &Point::new(vec![0.2, -0.15]));

    let (sensorium, s) = make_random_sr_problem(&space, 50.0, 32, 0.0);

    let start = Instant::now();
    match rcd(&sensorium, &space, &Point::new_default(space.dim()), 100.0, 0.1) {
        Ok(z) => {
            println!("Time: {:.3} sec", start.elapsed().as_secs_f64());
            println!("RCD-approximated source: {:.4?}", &z);
            println!("True source: {:.4?}", &s);
            println!("Quasi-metric errors: {:.4}, {:.4}", space.tau(&z, &s), space.tau(&s, &z));
        },
        Err(s) => println!("ERROR: {}", s)
    }
}
