use crate::point::Point;
use crate::qmspace::QMSpace;

/// Index of coverand whose anchor is a grid point.
struct GPIndex {
    cs: Vec<usize>
}

/// Normed space R^m_p with constant wind.
pub struct RMP {
    dim: usize,
    power: f64,
    tau_precision: f64,
    wind: Point,

    unit_rc_anchors: Vec<Point>
}

impl GPIndex {
    fn new_default(dim: usize) -> Self {
        Self{cs: vec![0usize; dim]}
    }

    fn inc(&mut self, imax: usize) -> bool {
        let dim = self.cs.len();
        let mut carry = true;
        let mut i: usize = 0;
        while carry && (i < dim) {
            self.cs[i] += 1;
            if self.cs[i] > imax {
                self.cs[i] = 0;
                i += 1;
            } else {
                carry = false;
            }
        }
        i == dim
    }    
}    

impl RMP {
    fn norm(&self, p: &Point) -> f64 {
        p.cs.iter().map(|&c| c.abs().powf(self.power)).sum::<f64>().powf(1.0 / self.power)
    }

    fn make_unit_rc_anchors(&self) -> Vec<Point> {
        let mut c_index = GPIndex::new_default(self.dim);
        let wind_norm = self.norm(& self.wind);
        let n = (2.0 * (1.0 + wind_norm) / (1.0 - wind_norm) * (self.dim as f64).powf(1.0 / self.power)) as usize;
        let step = 2.0 / (n as f64) * (1.0 + wind_norm);
        let mut anchors: Vec<Point> = Vec::with_capacity((n+1).pow(self.dim as u32));
    
        loop {
            let anchor = Point::new(
                c_index.cs.iter().map(|&i| -1.0 - wind_norm + step * (i as f64)).collect::<Vec<f64>>()
            );
            if self.norm(&anchor) <= 1.5 * (1.0 + wind_norm) {
                anchors.push(anchor);
            }    
            if c_index.inc(n) {
                break;
            }
        }
    
        anchors
    }

    pub fn new(dim: usize, power: f64, tau_precision: f64, wind: &Point) -> Self {
        let slf = Self{dim, power, tau_precision, wind: wind.clone(), unit_rc_anchors: vec![]}; // incomplete
        Self{dim, power, tau_precision, wind: wind.clone(), unit_rc_anchors: slf.make_unit_rc_anchors()}
    }
   
}

impl QMSpace for RMP {
    fn dim(&self) -> usize {
        self.dim
    }

    fn tau(&self, p_from: &Point, p_to: &Point) -> f64 {
        let mut t: f64 = 0.0;
        loop {
            let tn = self.norm(& p_from.sub(p_to).add(& self.wind.mult(t)));
            if (t - tn).abs() < self.tau_precision {
                break tn;
            } else {
                t = tn;
            }
        }
    }

    fn make_rc_anchors(&self, anchor: &Point, radius: f64) -> Vec<Point> {
        self.unit_rc_anchors.iter().map(|a| a.mult(radius).add(&anchor)).collect::<Vec<Point>>()
    }

}
