extern crate rayon;

use rayon::prelude::*;

use crate::{
    point::Point,
    qmspace::QMSpace
};

pub struct Sensor {
    p: Point,
    t: f64
}

impl Sensor {
    pub fn make(p: &Point, t: f64) -> Self {
        Self{p: p.clone(), t}
    }
}

/// Measures "spread" of *backward moments*
/// when the sound should be emitted by this point in order to reach each sensor at its moment.
/// At the source point this spread is 0.
fn defect(x: &Point, sensorium: &Vec<Sensor>, qmsp: &impl QMSpace) -> f64 {
    let mut tau_min = f64::MAX;
    let mut tau_max = f64::MIN;

    sensorium.iter().for_each(|r| {
        let tau = r.t - qmsp.tau(x, & r.p);
        tau_min = tau_min.min(tau);
        tau_max = tau_max.max(tau);
    });

    tau_max - tau_min
}

/// Refining-Cover-by-Defect approximation.
pub fn rcd(sensorium: &Vec<Sensor>, qmsp: &impl QMSpace, anchor0: &Point, radius0: f64, precision: f64) -> Result<Point, String> {
    // Step 0
    let mut k: usize = 0;
    let mut r: f64 = radius0;
    let mut cover_anchors = vec![anchor0.clone()];
    let mut z: Point;
    loop {
        // Step 1
        let rc_anchors = cover_anchors.par_iter().map(|a| qmsp.make_rc_anchors(a, r).into_par_iter()).flatten().collect::<Vec<Point>>();
        cover_anchors = rc_anchors.into_par_iter().filter(|a| defect(a, sensorium, qmsp) <= r).collect::<Vec<Point>>();
        if cover_anchors.len() == 0 {
            break Err(format!("cover became empty"));
        }

        // Step 2
        z = cover_anchors[cover_anchors.len() >> 1].clone();

        // Step 3
        k += 1;
        r *= 0.5;
        
        let d = r + cover_anchors.par_iter().fold(|| 0f64, |max, a| max.max(qmsp.tau(&z, &a)).max(qmsp.tau(&a, &z))).reduce(|| 0f64, |a, b| a.max(b));
        println!("Iteration {}: {} coverands, d = {:.4}", k, cover_anchors.len(), d);
        if d < precision {
            break Ok(z);
        }
    }
}
