use crate::point::Point;

/// Quasi-metric space.
///
/// Let X be a non-empty set
/// and `tau` (τ) be a nonnegative function defined on X×X
/// that satisfies the following axioms:
///
/// 1) τ(x, y) = 0 if and only if x = y,
/// 2) for any x, y, z in X: τ(x, y) ≤ τ(x, z) + τ(z, y).
///
/// Then (X, τ) is called a *quasi-metric space*.
///
/// If τ satisfies the symmetry axiom: for any x, y in X τ(x, y) = τ(y, x),
/// then (X, τ) is a usual metric space.
pub trait QMSpace: Sync {
    /// How many coordinates a point of the space has.
    fn dim(&self) -> usize;

    /// Time required for sound wave emitted by `p_from` to propagate to `p_to`.
    fn tau(&self, p_from: &Point, p_to: &Point) -> f64;

    /// Anchors of coverands of `radius`/2 that cover the coverand with anchor `anchor` of `radius`.
    fn make_rc_anchors(&self, anchor: &Point, radius: f64) -> Vec<Point>;
}
