#[macro_use]
mod macros;
mod insert;
mod update;
mod select;
mod delete;


use sqlx::database::HasArguments;
use sqlx::query::{Query, QueryAs};
use sqlx::{Database, FromRow};
use std::any::TypeId;
use std::fmt::Display;


static mut TABLE_PREFIX: String = String::new();
//表名
pub struct TableName {
    name: String,
}
impl Display for TableName{
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        unsafe{
            write!(f,"{}{}",TABLE_PREFIX,self.name)
        }
    }
}
impl TableName {
    /// 设置表前缀
    pub fn set_prefix(str: String) {
        unsafe {
            TABLE_PREFIX = str;
        }
    }
    /// 新建表名
    pub fn new(name: &str) -> Self {
        Self {
            name: name.to_string(),
        }
    }
    /// 得到完整表名
    pub fn full_name(&self) -> String {
        unsafe { return format!("{}{}", TABLE_PREFIX, self.name) }
    }
}
#[derive(PartialEq)]
pub enum DbType
{
    Mysql,
    Sqlite,
}
impl DbType
 {
    pub fn type_new<DB:sqlx::Database>() 
    -> Self 
    {
        if TypeId::of::<DB>() == TypeId::of::<sqlx::MySql>() {
            return DbType::Mysql;
        }
        // if TypeId::of::<T>() == TypeId::of::<sqlx::pool::Pool<sqlx::Sqlite>>() {
        //     return DbType::Sqlite;
        // }
        unimplemented!()
    }
    pub fn mark(&self, pos: usize) -> String {
        match self {
            DbType::Mysql => "?".to_string(),
            DbType::Sqlite => {
                format!("${}", pos)
            }
        }
    }
}


//得到表名
pub trait ModelTableName {
    fn table_name() -> TableName;
}
//得到表字段和字段值绑定
pub trait ModelTableField<DB>
where
DB:Database
{
    fn table_pk() -> TableFields;
    fn table_column() -> TableFields;
    fn query_sqlx_bind<'t>(
        &'t self,
        table_field_val: &FieldItem,
        res: Query<'t,DB,<DB as HasArguments<'t>>::Arguments>,
    ) -> Query<'t,DB,<DB as HasArguments<'t>>::Arguments>
    ;
    fn query_as_sqlx_bind<'t, M>(
        &'t self,
        table_field_val: &FieldItem,
        res:QueryAs<'t,DB, M,<DB as HasArguments<'t>>::Arguments>,
    ) -> QueryAs<'t,DB, M,<DB as HasArguments<'t>>::Arguments>
    where
        for<'r> M: FromRow<'r, DB::Row> + Send + Unpin;
}

//字段
#[derive(Clone,PartialEq)]
pub struct FieldItem {
    pub name: String
}
impl Display for FieldItem{
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        write!(f,"{}",self.name)
    }
}
impl FieldItem {
    pub fn new(name: &str) -> Self {
        FieldItem {
            name: name.to_string()
        }
    }
}

//表字段
pub struct TableFields(Vec<FieldItem>);
impl Display for TableFields{
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        let fileds=self.0.iter().map(|e|{
            format!("{}",e)
        }).collect::<Vec<String>>().join(",");
        write!(f,"{}",fileds)
    }
}
impl TableFields {
    pub fn new(fields: Vec<FieldItem>) -> Self {
        TableFields(fields)
    }
    pub fn marge(&mut self, field: Vec<FieldItem>) {
        for val in field.iter() {
            if !self.0.iter().any(|e| e.name == val.name) {
                self.0.push(val.to_owned())
            }
        }
    }
    pub fn intersect(&mut self, field: Vec<FieldItem>) {
        self.0 = self
            .0
            .iter()
            .filter_map(|e| {
                if field.contains(&e) {
                    Some(e.to_owned())
                } else {
                    None
                }
            })
            .collect();
    }
    pub fn del(&mut self, name: &str) {
        self.0 = self
            .0
            .iter()
            .filter_map(|e| {
                if name == &e.name {
                    None
                } else {
                    Some(e.to_owned())
                }
            })
            .collect();
    }
    pub fn to_vec(&self) -> Vec<String> {
        let field = self.0.iter();
        field.map(|e| e.name.clone()).collect::<Vec<String>>()
    }
}


pub use insert::*;
pub use update::*;
pub use select::*;
pub use delete::*;


