use std::pin::Pin;

use futures::stream::Stream;
use futures::stream::TryCollect;
use futures::Future;
use futures::{future, TryFutureExt, TryStreamExt};
use sqlx::database::HasArguments;
use sqlx::query::Query;
use sqlx::{Database, Encode, Executor, FromRow, IntoArguments, Type};

/// Type alias for methods returning a single element. The future resolves to and
/// `Result<T, sqlx::Error>`.
pub type CrudFut<'e, T> = Pin<Box<dyn Future<Output = Result<T, sqlx::Error>> + 'e>>;

/// Type alias for a [`Stream`] returning items of type `Result<T, sqlxError>`.
pub type CrudStream<'e, T> =
    Pin<Box<dyn Stream<Item = Result<T, sqlx::Error>> + std::marker::Send + 'e>>;

/// Type alias for a [`TryCollect`] future that resolves to `Result<Vec<T>, sqlx::Error>`.
pub type TryCollectFut<'e, T> = TryCollect<CrudStream<'e, T>, Vec<T>>;

/// Database schema information about a struct implementing sqlx [FromRow].
/// [Schema] defines methods for accessing the derived database schema
/// and query information.
///
/// This trait is implemented by the [SqlxCrud] derive macro.
///
/// # Example
///
/// ```rust
/// use sqlx::FromRow;
/// use sqlx_crud::SqlxCrud;
///
/// #[derive(FromRow, SqlxCrud)]
/// pub struct User {
///     user_id: i32,
///     name: String,
/// }
/// ```
///
/// [FromRow]: https://docs.rs/sqlx/latest/sqlx/trait.FromRow.html
pub trait Schema {
    /// Type of the table primary key column.
    type Id: Send;

    /// Database name of the table. Used by the query generation code and
    /// available for introspection. This is generated by taking the plural
    /// _snake_case_ of the struct's name. See: [Inflector to_table_case].
    ///
    /// ```rust
    /// use sqlx::FromRow;
    /// use sqlx_crud::SqlxCrud;
    ///
    /// #[derive(FromRow, SqlxCrud)]
    /// struct GoogleIdToken {
    ///     id: i32,
    ///     audience: String,
    /// }
    ///
    /// assert_eq!("google_id_tokens", GoogleIdToken::table_name());
    /// ```
    ///
    /// [Inflector to_table_case]: https://docs.rs/Inflector/latest/inflector/cases/tablecase/fn.to_table_case.html
    fn table_name() -> &'static str;

    /// Returns the id of the current instance.
    fn id(&self) -> Self::Id;

    /// Returns the column name of the primary key.
    fn id_column() -> &'static str;

    /// Returns an array of column names.
    fn columns() -> &'static [&'static str];

    /// Returns the SQL string for a SELECT query against the table.
    ///
    /// # Example
    ///
    /// ```rust
    /// # use sqlx_crud::doctest::User;
    /// use sqlx_crud::Schema;
    ///
    /// assert_eq!("SELECT user_id, name FROM users", User::select_sql());
    /// ```
    fn select_sql() -> &'static str;

    /// Returns the SQL string for a SELECT query against the table with a
    /// WHERE clause for the primary key.
    ///
    /// # Example
    ///
    /// ```rust
    /// # use sqlx_crud::doctest::User;
    /// use sqlx_crud::Schema;
    ///
    /// assert_eq!(
    ///     "SELECT user_id, name FROM users WHERE user_id = ? LIMIT 1",
    ///     User::select_by_id_sql()
    /// );
    /// ```
    fn select_by_id_sql() -> &'static str;

    /// Returns the SQL for inserting a new record in to the database. It does not
    /// handle database assigned IDs.
    ///
    /// # Example
    ///
    /// ```rust
    /// # use sqlx_crud::doctest::User;
    /// use sqlx_crud::Schema;
    ///
    /// assert_eq!("INSERT INTO users VALUES (?, ?)", User::insert_sql());
    /// ```
    fn insert_sql() -> &'static str;

    /// Returns the SQL for updating an existing record in the database.
    ///
    /// # Example
    ///
    /// ```rust
    /// # use sqlx_crud::doctest::User;
    /// use sqlx_crud::Schema;
    ///
    /// assert_eq!("UPDATE users SET name = ? WHERE user_id = ?", User::update_by_id_sql());
    /// ```
    fn update_by_id_sql() -> &'static str;

    /// Returns the SQL for deleting an existing record by ID from the database.
    ///
    /// # Example
    ///
    /// ```rust
    /// # use sqlx_crud::doctest::User;
    /// use sqlx_crud::Schema;
    ///
    /// assert_eq!("DELETE FROM users WHERE user_id = ?", User::delete_by_id_sql());
    /// ```
    fn delete_by_id_sql() -> &'static str;
}

/// Common Create, Read, Update, and Delete behaviors. This trait requires that
/// [Schema] and [FromRow] are implemented for Self.
///
/// This trait is implemented by the [SqlxCrud] derive macro. Implementors
/// define how to assign query insert and update bindings.
///
/// [FromRow]: https://docs.rs/sqlx/latest/sqlx/trait.FromRow.html
/// [Schema]: trait.Schema.html
/// [SqlxCrud]: ../derive.SqlxCrud.html
pub trait Crud<'e, E>
where
    Self: 'e
        + Sized
        + Send
        + Unpin
        + for<'r> FromRow<'r, <E::Database as Database>::Row>
        + Schema,
    <Self as Schema>::Id:
        Encode<'e, <E as Executor<'e>>::Database> + Type<<E as Executor<'e>>::Database>,
    E: Executor<'e> + 'e,
    <E::Database as HasArguments<'e>>::Arguments: IntoArguments<'e, <E as Executor<'e>>::Database>,
{
    /// Given a query returns a new query with parameters suitable for an
    /// INSERT bound to it. The [SqlxCrud] implementation will bind the
    /// primary key, and each additional field to the query.
    ///
    /// # Example
    ///
    /// Typically you would just use `<Self as Crud>::insert()` but if you
    /// wanted to modify the query you could use something like:
    ///
    /// ```rust
    /// # use sqlx_crud::doctest::{setup, User};
    /// # tokio_test::block_on(async {
    /// # let pool = setup().await;
    /// use sqlx_crud::{Crud, Schema};
    ///
    /// let user = User { user_id: 1, name: "Test".to_string() };
    /// let query = sqlx::query(User::insert_sql());
    /// let query = user.insert_binds(query);
    /// query.execute(&pool).await?;
    ///
    /// # Ok::<(), sqlx::Error>(())
    /// # });
    /// ```
    ///
    /// This would bind `user_id`, `name` to the query in that order.
    ///
    /// [SqlxCrud]: ../derive.SqlxCrud.html
    fn insert_binds(
        &'e self,
        query: Query<'e, E::Database, <E::Database as HasArguments<'e>>::Arguments>,
    ) -> Query<'e, E::Database, <E::Database as HasArguments<'e>>::Arguments>;

    /// Given a query returns a new query with parameters suitable for an
    /// UPDATE bound to it. The [SqlxCrud] implementation will bind every
    /// column except for the primary key followed by the primary key to
    /// the query.
    ///
    /// # Example
    ///
    /// Typically you would just use `<Self as Crud>::update()` but if you
    /// wanted to modify the query you could use something like:
    ///
    /// ```rust
    /// # use sqlx_crud::doctest::{setup, User};
    /// # tokio_test::block_on(async {
    /// # let pool = setup().await;
    /// use sqlx_crud::{Crud, Schema};
    ///
    /// let user = User { user_id: 1, name: "Test".to_string() };
    /// let query = sqlx::query(User::update_by_id_sql());
    /// let query = user.update_binds(query);
    /// query.execute(&pool).await?;
    ///
    /// # Ok::<(), sqlx::Error>(())
    /// # });
    /// ```
    ///
    /// This would bind `name`, `user_id` to the query in that order.
    ///
    /// [SqlxCrud]: ../derive.SqlxCrud.html
    fn update_binds(
        &'e self,
        query: Query<'e, E::Database, <E::Database as HasArguments<'e>>::Arguments>,
    ) -> Query<'e, E::Database, <E::Database as HasArguments<'e>>::Arguments>;

    /// Returns a future that resolves to an insert or `sqlx::Error` of the
    /// current instance.
    ///
    /// # Example
    ///
    /// ```rust
    /// # use sqlx_crud::doctest::{setup, User};
    /// # tokio_test::block_on(async {
    /// # let pool = setup().await;
    /// use sqlx_crud::{Crud, Schema};
    ///
    /// let user = User { user_id: 1, name: "test".to_string() };
    /// user.create(&pool).await?;
    ///
    /// # Ok::<(), sqlx::Error>(())
    /// # });
    /// ```
    fn create(&'e self, pool: E) -> CrudFut<'e, ()> {
        Box::pin(async move {
            let query = sqlx::query(<Self as Schema>::insert_sql());
            let query = self.insert_binds(query);
            query.execute(pool).await?;

            Ok(())
        })
    }

    /// Queries all records from the table and returns a future that returns
    /// to a [try_collect] stream, which resolves to a `Vec<Self>` or a
    /// `sqlx::Error` on error.
    ///
    /// # Example
    ///
    /// ```rust
    /// # use sqlx_crud::doctest::{setup, User};
    /// # tokio_test::block_on(async {
    /// # let pool = setup().await;
    /// use sqlx_crud::Crud;
    ///
    /// let all_users: Vec<User> = User::all(&pool).await?;
    ///
    /// # Ok::<(), sqlx::Error>(())
    /// # });
    /// ```
    /// 
    /// [try_collect]: https://docs.rs/futures/latest/futures/stream/trait.TryStreamExt.html#method.try_collect
    fn all(pool: E) -> TryCollectFut<'e, Self> {
        let stream =
            sqlx::query_as::<E::Database, Self>(<Self as Schema>::select_sql()).fetch(pool);
        stream.try_collect()
    }

    #[doc(hidden)]
    fn paged(_pool: E) -> TryCollectFut<'e, Self> {
        unimplemented!()
    }

    /// Looks up a row by ID and returns a future that resolves an
    /// `Option<Self>`. Returns `None` if and a record with the corresponding ID
    /// cannot be found and `Some` if it exists.
    ///
    /// # Example
    ///
    /// ```rust
    /// # use sqlx_crud::doctest::{setup, User};
    /// # tokio_test::block_on(async {
    /// # let pool = setup().await;
    /// use sqlx_crud::Crud;
    ///
    /// let user: Option<User> = User::by_id(&pool, 1).await?;
    ///
    /// # Ok::<(), sqlx::Error>(())
    /// # });
    /// ```
    fn by_id(pool: E, id: <Self as Schema>::Id) -> CrudFut<'e, Option<Self>> {
        Box::pin(
            sqlx::query_as::<E::Database, Self>(<Self as Schema>::select_by_id_sql())
                .bind(id)
                .fetch_optional(pool)
        )
    }

    /// Updates the database with the current instance state and returns a
    /// future that resolves an `()` on success and `sqlx::Error` on error.
    ///
    /// # Example
    ///
    /// ```rust
    /// # use sqlx_crud::doctest::{setup, User};
    /// # tokio_test::block_on(async {
    /// # let pool = setup().await;
    /// use sqlx_crud::Crud;
    ///
    /// if let Some(mut user) = User::by_id(&pool, 1).await? {
    ///     user.name = "Harry".to_string();
    ///     user.update(&pool).await?;
    /// }
    ///
    /// # Ok::<(), sqlx::Error>(())
    /// # });
    /// ```
    fn update(&'e self, pool: E) -> CrudFut<'e, ()> {
        Box::pin(async move {
            let query = sqlx::query(<Self as Schema>::update_by_id_sql());
            let query = self.update_binds(query);
            query.execute(pool).await?;

            Ok(())
        })
    }

    /// Deletes a record from the database by ID and returns a future that
    /// resolves to `()` on success or `sqlx::Error` on failure.
    ///
    /// # Example
    ///
    /// ```rust
    /// # use sqlx_crud::doctest::{setup, User};
    /// # tokio_test::block_on(async {
    /// # let pool = setup().await;
    /// use sqlx_crud::Crud;
    ///
    /// if let Some(user) = User::by_id(&pool, 1).await? {
    ///     user.delete(&pool).await?;
    /// }
    /// assert!(User::by_id(&pool, 1).await?.is_none());
    ///
    /// # Ok::<(), sqlx::Error>(())
    /// # });
    /// ```
    fn delete(self, pool: E) -> CrudFut<'e, ()> {
        let query = sqlx::query(<Self as Schema>::delete_by_id_sql()).bind(self.id());
        Box::pin(
            query
                .execute(pool)
                .and_then(|_| future::ok(())),
        )
    }
}
