/*
MIT License

Copyright (c) 2022 Molese

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all
copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
SOFTWARE.
*/

#![allow(missing_debug_implementations)]
#![deny(
    missing_copy_implementations,
    trivial_casts,
    trivial_numeric_casts,
    unstable_features,
    unused_import_braces,
    unused_qualifications
)]

//! MoCkInG SpOnGeBoB SqUaRePaNtS TeXt gEnErAtOr.

use rand::{prelude::ThreadRng, Rng};
use std::{env, error::Error, fs::File, io::Read};

/// The necessary configurations for initializing spongmock.
///
pub struct Config {
    /// The content where the mock will be conducted, can be raw data or file along with path.
    pub content: String,
    /// The type of the content, between raw data or file along with path.
    pub content_type: String,
    /// Whether to randomize the mock behavior or not.
    pub randomize: String,
}

impl Config {
    /// Initializes a new Config.
    ///
    /// Returns error if incorrect option is passed.
    pub fn new(mut args: env::Args) -> Result<Config, &'static str> {
        if args.len() > 4 {
            return Err("More than 3 arguments passed");
        }

        args.next();

        let (content, content_type, randomize): (String, String, String) = (
            match args.next() {
                Some(arg) => arg,
                None => return Err("Content not specified; try <path/to/file>/<'raw data'>"),
            },
            match args.next() {
                Some(arg) => arg,
                None => return Err("Content type not specified; try --<file/raw>"),
            },
            match args.next() {
                Some(arg) => arg,
                None => return Err("Mock behavior not specified; try --<randomize/no-randomize>"),
            },
        );

        /* var names are same as in the struct, so e don't need explicit key:val initialization style. */
        Ok(Config {
            content,
            content_type,
            randomize,
        })
    }
}

/// Returns a random boolean value.
pub fn rand_bool() -> bool {
    let mut rng: ThreadRng = rand::thread_rng();
    rng.gen()
}

fn mock_refstr(s: &str, randomize: bool) -> String {
    let mut res: String = String::new();

    for (i, c) in s.chars().enumerate() {
        if randomize {
            if rand_bool() {
                res.push(c.to_ascii_uppercase());
            } else {
                res.push(c.to_ascii_lowercase());
            }
        } else {
            if i % 2 == 0 {
                res.push(c.to_ascii_uppercase());
            } else {
                res.push(c.to_ascii_lowercase());
            }
        }
    }
    res
}

/// Returns mock string.
/// If randomize is true, the mock behavior will be randomized.
/// If randomize is false, the mock behavior will be standard; e.g: AbcDeF.
pub fn mock_any<T: ToString>(s: T, randomize: bool) -> String {
    mock_refstr(&s.to_string(), randomize)
}

/// Runs the library logic.
/// Expects a reference to [`Config`](struct@Config).
///
/// # Errors
///
/// Returns error on the following situations:
///
/// - `file` doesn't exist.
/// - `file` could not be read (unauthorized read permission).
///
/// - Invalid content type.
/// - Invalid argument.
pub fn run(config: Config) -> Result<(), Box<dyn Error>> {
    let mut data: String = String::new();

    if &config.content_type.to_ascii_lowercase() == "--file" {
        let mut file: File = File::open(&config.content)?;
        file.read_to_string(&mut data)?;
    } else if &config.content_type.to_ascii_lowercase() == "--raw" {
        data = config.content.to_string();
    } else {
        Err("Invalid content type")?
    }

    let result: String = mock_any(
        data,
        if &config.randomize.to_lowercase() == "--randomize" {
            true
        } else if &config.randomize.to_lowercase() == "--no-randomize" {
            false
        } else {
            Err("Invalid argument")?
        },
    );

    println!("{}", result);

    /* `()` is a unit type.
     * We do, however, care about the errors that might occur, and that's why the result type
     * exists with a dynamic error return type. */
    Ok(())
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_mock_not_random_string() {
        assert_eq!(mock_any(String::from("ABCDEFG"), false), "AbCdEfG");
    }

    #[test]
    fn test_mock_not_random_ref_string() {
        assert_eq!(mock_any(&"ABCDEFG", false), "AbCdEfG");
    }

    #[test]
    fn test_mock_random() {
        assert_ne!(mock_any("ABCDEFG", true), "AbCdEfG");
    }
}
