//! View the fully-resolved SPF tree for a given domain.
//!
//! Optionally, only view the domains, or only IP addresses.
use anyhow::{bail, Result};
use jacklog::{debug, instrument, trace};
use lazy_static::lazy_static;
use regex::Regex;
use serde::Serialize;
use std::{
    fmt::Debug,
    str::{self, FromStr},
};
use structopt::StructOpt;
use trust_dns_resolver::Resolver;

lazy_static! {
    static ref SPF_REGEX: Regex = Regex::new("include:(.+)").unwrap();
    static ref EXISTS_REGEX: Regex = Regex::new("exists:(.+)").unwrap();
    static ref IP4_REGEX: Regex = Regex::new("ip4:(.+)").unwrap();
    static ref IP6_REGEX: Regex = Regex::new("ip6:(.+)").unwrap();
}

#[derive(Debug, Serialize)]
#[serde(rename_all = "snake_case", untagged)]
enum Value {
    Ip4(String),
    Ip6(String),
    Exists(String),
    Domain { name: String, children: Vec<Value> },
}

#[derive(Debug)]
enum Format {
    Text,
    Json,
    Yaml,
    Toml,
}

impl Default for Format {
    fn default() -> Self {
        Self::Text
    }
}

impl FromStr for Format {
    type Err = anyhow::Error;

    fn from_str(s: &str) -> Result<Self, Self::Err> {
        Ok(match s {
            "text" => Self::Text,
            "JSON" | "json" | "j" => Self::Json,
            "YML" | "YAML" | "yml" | "yaml" | "y" => Self::Yaml,
            "TOML" | "toml" | "t" => Self::Toml,
            _ => bail!("{} is not a recognized format", s),
        })
    }
}

/// Recursively look up the SPF record chain for a domain.
#[derive(Debug, Default, StructOpt)]
#[structopt(setting = structopt::clap::AppSettings::ColoredHelp)]
struct Cli {
    /// How verbose logs should be. Defaults to WARN; pass multiple times to
    /// increase verbosity.
    #[structopt(short, long, parse(from_occurrences))]
    verbose: usize,

    /// Number of spaces to use for indentation, per level.
    #[structopt(long, default_value = "4")]
    indentation: usize,

    /// Only print domains, not the final, resolved IPs. Only honored when using
    /// text output format.
    #[structopt(short, long)]
    domains_only: bool,

    /// Only print IP addresses, not the domains. Only honored when using text
    /// output format.
    #[structopt(short, long)]
    ips_only: bool,

    /// Specify an output format.
    #[structopt(short, long, default_value = "text")]
    format: Format,

    /// The domain to do a recursive lookup on.
    domain: String,
}

#[instrument]
fn main() -> Result<()> {
    // Parse the CLI args.
    let opts = Cli::from_args();

    // Configure logging.
    jacklog::from_level(opts.verbose + 2, Some(&[env!("CARGO_BIN_NAME")]))?;
    debug!(?opts, "logging configured");

    let mut resolver = Resolver::from_system_conf()?;

    let records = lookup(&mut resolver, &opts.domain)?;
    debug!("{:?}", records);

    match opts.format {
        Format::Json => {
            println!("{}", serde_json::to_string_pretty(&records)?);
        }
        Format::Toml => {
            println!("{}", toml::to_string_pretty(&records)?);
        }
        Format::Yaml => {
            println!("{}", serde_yaml::to_string(&records)?);
        }
        Format::Text => {
            if opts.domains_only {
                print_domains_as_tree(&records, 0, opts.indentation);
                return Ok(());
            }

            if opts.ips_only {
                print_ips(&records);
                return Ok(());
            }

            print_all_as_tree(&records, 0, opts.indentation);
        }
    }

    Ok(())
}

#[instrument]
fn print_all_as_tree(val: &Value, depth: usize, steps: usize) {
    match val {
        Value::Ip4(ip) | Value::Ip6(ip) | Value::Exists(ip) => {
            println!("{}{}", " ".repeat(depth), ip);
        }
        Value::Domain { name, children } => {
            println!("{}{}", " ".repeat(depth), name);
            for child in children {
                print_all_as_tree(child, depth + steps, steps);
            }
        }
    };
}

#[instrument]
fn print_domains_as_tree(val: &Value, depth: usize, steps: usize) {
    if let Value::Domain { name, children } = val {
        println!("{} {}", " ".repeat(depth), name);
        for child in children {
            print_domains_as_tree(child, depth + steps, steps);
        }
    }
}

#[instrument]
fn print_ips(val: &Value) {
    match val {
        Value::Ip4(ip) | Value::Ip6(ip) | Value::Exists(ip) => println!("{}", ip),
        Value::Domain { children, .. } => {
            for child in children {
                print_ips(child);
            }
        }
    };
}

/// Do a recursive lookup on a domain.
#[instrument(skip(resolver))]
fn lookup<T: AsRef<str> + Debug>(resolver: &mut Resolver, domain: T) -> Result<Value> {
    debug!("looking up domain");

    let res = resolver.txt_lookup(domain.as_ref())?;
    trace!(?res, "looked up domain");

    let mut children = vec![];
    for line in txt_query(resolver, domain.as_ref())? {
        trace!(?line);

        for piece in line.split_whitespace() {
            trace!(?piece);

            if let Some(m) = IP4_REGEX.captures(piece) {
                children.push(Value::Ip4(m.get(1).unwrap().as_str().into()));
                continue;
            }

            if let Some(m) = IP6_REGEX.captures(piece) {
                children.push(Value::Ip6(m.get(1).unwrap().as_str().into()));
                continue;
            }

            if let Some(m) = EXISTS_REGEX.captures(piece) {
                children.push(Value::Exists(m.get(1).unwrap().as_str().into()));
                continue;
            }

            let domain = if let Some(m) = SPF_REGEX.captures(piece) {
                m.get(1).unwrap()
            } else {
                trace!(?piece, "not a valid segment with a domain");
                continue;
            };

            children.push(lookup(resolver, domain.as_str())?);
        }
    }

    let out = Value::Domain {
        name: domain.as_ref().to_string(),
        children,
    };

    Ok(out)
}

/// Make a TXT query and parse the results as UTF8 strings.
///
/// # Errors
///
/// Returns an error if the lookup fails or the contents are not valid UTF8.
#[instrument(skip(resolver))]
fn txt_query<T: AsRef<str> + Debug>(resolver: &mut Resolver, domain: T) -> Result<Vec<String>> {
    debug!("looking up domain");

    // Get the raw response.
    let res = resolver.txt_lookup(domain.as_ref())?;
    trace!(?res, "looked up domain");

    Ok(res
        .iter()
        .flat_map(|record| -> Result<String> {
            // Iterate through each record (only some will be SPF records).
            trace!(?record);

            record
                // Get the data of the record as bytes.
                .txt_data()
                .iter()
                .map(|line| {
                    trace!(?line);

                    // Convert the bytes to a UTF8 string.
                    let line = str::from_utf8(line)?;
                    trace!(?line);

                    // If this all worked, return the contents.
                    Ok(line.to_string())
                })
                .collect::<Result<String>>()
        })
        .collect::<Vec<String>>())
}
