mod common;

pub use common::*;
use spf_milter::*;
use std::{
    ffi::OsString,
    io,
    net::{Ipv4Addr, SocketAddr},
    path::{Path, PathBuf},
    process::ExitStatus,
};
use tokio::{fs, sync::mpsc};
use viaspf::lookup::LookupError;

#[tokio::test]
async fn config_reload() {
    let config_file = config_file_name();

    let opts = configure_logging(CliOptions::builder())
        .config_file(&config_file)
        .build();

    let lookup = MockLookup::builder()
        .lookup_txt(|name| match name.as_str() {
            "example.org." => Ok(vec!["v=spf1 a -all".into()]),
            _ => Err(LookupError::NoRecords),
        })
        .lookup_a(|name| match name.as_str() {
            "example.org." => Ok(vec![Ipv4Addr::new(123, 123, 123, 123)]),
            _ => Err(LookupError::NoRecords),
        })
        .build();

    // Write initial configuration with `Received-SPF` header.
    fs::write(
        &config_file,
        "
        socket = inet:127.0.0.1:0
        verify_helo = no
        header = Received-SPF
        ",
    )
    .await
    .unwrap();

    let config = Config::read_with_lookup(opts, lookup).await.unwrap();

    let milter = SpfMilter::spawn(config).await.unwrap();

    let addr = milter.addr();
    let reload_tx = milter.reload_handle();
    let exit_code = run_miltertest_with_script(file!(), move |fname| {
        Box::pin(test_script(fname, addr, reload_tx))
    })
    .await
    .unwrap();

    milter.shutdown().await.unwrap();

    assert!(exit_code.success());
}

async fn test_script(
    file_name: OsString,
    addr: SocketAddr,
    reload_config: mpsc::Sender<()>,
) -> io::Result<ExitStatus> {
    let config_file = config_file_name();

    let exit_code = exec_miltertest_with_vars(&file_name, addr, ["received_spf"]).await?;
    if !exit_code.success() {
        return Ok(exit_code);
    }

    // Write updated configuration with `Authentication-Results` header. Also
    // try changing `socket`, which needs a restart to become effective, but
    // does not prevent reloading.
    fs::write(
        &config_file,
        "
        socket = inet:127.0.0.1:3333
        verify_helo = no
        header = Authentication-Results
        ",
    )
    .await?;

    reload_config.send(()).await.unwrap();

    let exit_code = exec_miltertest(&file_name, addr).await?;
    if !exit_code.success() {
        return Ok(exit_code);
    }

    // Write configuration with the initial `Received-SPF` header, but since the
    // configuration is now invalid it won’t get reloaded.
    fs::write(
        &config_file,
        "
        socket = inet:127.0.0.1:3333
        verify_helo = no
        header = Received-SPF
        invalid_key = invalid_value
        ",
    )
    .await?;

    reload_config.send(()).await.unwrap();

    exec_miltertest(&file_name, addr).await
}

fn config_file_name() -> PathBuf {
    // Place configuration file next to the `spf-milter` executable used to run
    // this integration test.
    let file_path = to_config_file_name(file!());
    Path::new(env!("CARGO_BIN_EXE_spf-milter"))
        .parent()
        .unwrap()
        .join(file_path.file_name().unwrap())
}
