use async_trait::async_trait;
use spf_milter::{CliOptionsBuilder, Config, LogDestination, LogLevel, Socket};
use std::{
    env,
    ffi::{OsStr, OsString},
    future::Future,
    io, iter,
    net::{IpAddr, Ipv4Addr, Ipv6Addr, SocketAddr},
    path::PathBuf,
    pin::Pin,
    process::ExitStatus,
};
use tokio::{
    net::TcpListener,
    process::Command,
    sync::{mpsc, oneshot},
    task::JoinHandle,
};
use viaspf::lookup::{Lookup, LookupError, LookupResult, Name};

pub fn configure_logging(opts: CliOptionsBuilder) -> CliOptionsBuilder {
    // Set this environment variable to test debug logging to syslog.
    if env::var("SPF_MILTER_TEST_LOG").is_ok() {
        opts.log_destination(LogDestination::Syslog)
            .log_level(LogLevel::Debug)
    } else {
        opts.log_destination(LogDestination::Stderr)
    }
}

#[derive(Default)]
pub struct MockLookupBuilder {
    lookup_a: Option<Box<dyn Fn(&Name) -> LookupResult<Vec<Ipv4Addr>> + Send + Sync + 'static>>,
    lookup_aaaa: Option<Box<dyn Fn(&Name) -> LookupResult<Vec<Ipv6Addr>> + Send + Sync + 'static>>,
    lookup_mx: Option<Box<dyn Fn(&Name) -> LookupResult<Vec<Name>> + Send + Sync + 'static>>,
    lookup_txt: Option<Box<dyn Fn(&Name) -> LookupResult<Vec<String>> + Send + Sync + 'static>>,
    lookup_ptr: Option<Box<dyn Fn(IpAddr) -> LookupResult<Vec<Name>> + Send + Sync + 'static>>,
}

impl MockLookupBuilder {
    pub fn lookup_a(
        mut self,
        value: impl Fn(&Name) -> LookupResult<Vec<Ipv4Addr>> + Send + Sync + 'static,
    ) -> Self {
        self.lookup_a = Some(Box::new(value));
        self
    }

    pub fn lookup_aaaa(
        mut self,
        value: impl Fn(&Name) -> LookupResult<Vec<Ipv6Addr>> + Send + Sync + 'static,
    ) -> Self {
        self.lookup_aaaa = Some(Box::new(value));
        self
    }

    pub fn lookup_mx(
        mut self,
        value: impl Fn(&Name) -> LookupResult<Vec<Name>> + Send + Sync + 'static,
    ) -> Self {
        self.lookup_mx = Some(Box::new(value));
        self
    }

    pub fn lookup_txt(
        mut self,
        value: impl Fn(&Name) -> LookupResult<Vec<String>> + Send + Sync + 'static,
    ) -> Self {
        self.lookup_txt = Some(Box::new(value));
        self
    }

    pub fn lookup_ptr(
        mut self,
        value: impl Fn(IpAddr) -> LookupResult<Vec<Name>> + Send + Sync + 'static,
    ) -> Self {
        self.lookup_ptr = Some(Box::new(value));
        self
    }

    pub fn build(self) -> MockLookup {
        MockLookup {
            lookup_a: self.lookup_a,
            lookup_aaaa: self.lookup_aaaa,
            lookup_mx: self.lookup_mx,
            lookup_txt: self.lookup_txt,
            lookup_ptr: self.lookup_ptr,
        }
    }
}

#[derive(Default)]
pub struct MockLookup {
    lookup_a: Option<Box<dyn Fn(&Name) -> LookupResult<Vec<Ipv4Addr>> + Send + Sync + 'static>>,
    lookup_aaaa: Option<Box<dyn Fn(&Name) -> LookupResult<Vec<Ipv6Addr>> + Send + Sync + 'static>>,
    lookup_mx: Option<Box<dyn Fn(&Name) -> LookupResult<Vec<Name>> + Send + Sync + 'static>>,
    lookup_txt: Option<Box<dyn Fn(&Name) -> LookupResult<Vec<String>> + Send + Sync + 'static>>,
    lookup_ptr: Option<Box<dyn Fn(IpAddr) -> LookupResult<Vec<Name>> + Send + Sync + 'static>>,
}

impl MockLookup {
    pub fn builder() -> MockLookupBuilder {
        Default::default()
    }
}

#[async_trait]
impl Lookup for MockLookup {
    async fn lookup_a(&self, name: &Name) -> LookupResult<Vec<Ipv4Addr>> {
        self.lookup_a
            .as_ref()
            .map_or(Err(LookupError::NoRecords), |f| f(name))
    }

    async fn lookup_aaaa(&self, name: &Name) -> LookupResult<Vec<Ipv6Addr>> {
        self.lookup_aaaa
            .as_ref()
            .map_or(Err(LookupError::NoRecords), |f| f(name))
    }

    async fn lookup_mx(&self, name: &Name) -> LookupResult<Vec<Name>> {
        self.lookup_mx
            .as_ref()
            .map_or(Err(LookupError::NoRecords), |f| f(name))
    }

    async fn lookup_txt(&self, name: &Name) -> LookupResult<Vec<String>> {
        self.lookup_txt
            .as_ref()
            .map_or(Err(LookupError::NoRecords), |f| f(name))
    }

    async fn lookup_ptr(&self, ip: IpAddr) -> LookupResult<Vec<Name>> {
        self.lookup_ptr
            .as_ref()
            .map_or(Err(LookupError::NoRecords), |f| f(ip))
    }
}

pub fn to_config_file_name(file_name: &str) -> PathBuf {
    let mut path = PathBuf::from(file_name);
    path.set_extension("conf");
    path
}

pub struct SpfMilter {
    milter_handle: JoinHandle<io::Result<()>>,
    reload: mpsc::Sender<()>,
    shutdown: oneshot::Sender<()>,
    addr: SocketAddr,
}

impl SpfMilter {
    pub async fn spawn(config: Config) -> io::Result<Self> {
        let listener = match config.socket() {
            Socket::Inet(socket) => TcpListener::bind(socket).await?,
            Socket::Unix(_) => unimplemented!(),
        };

        let addr = listener.local_addr()?;

        let (reload_tx, reload_rx) = mpsc::channel(1);
        let (shutdown_tx, shutdown_rx) = oneshot::channel();

        let milter = tokio::spawn(spf_milter::run(listener, config, reload_rx, shutdown_rx));

        Ok(Self {
            milter_handle: milter,
            reload: reload_tx,
            shutdown: shutdown_tx,
            addr,
        })
    }

    pub fn addr(&self) -> SocketAddr {
        self.addr
    }

    pub fn reload_handle(&self) -> mpsc::Sender<()> {
        self.reload.clone()
    }

    pub async fn shutdown(self) -> io::Result<()> {
        let _ = self.shutdown.send(());

        self.milter_handle.await?
    }
}

/// Executes the miltertest test for the file `test_file_name`.
pub async fn run_miltertest(test_file_name: &str, addr: SocketAddr) -> io::Result<ExitStatus> {
    run_miltertest_with_script(test_file_name, move |file_name| {
        Box::pin(exec_miltertest(file_name, addr))
    })
    .await
}

/// Executes the given milter test script closure.
///
/// The test script closure gets passed the file name of the miltertest test for
/// `test_file_name`. The closure should execute this test with `miltertest` and
/// return the test’s exit code.
pub async fn run_miltertest_with_script<F>(
    test_file_name: &str,
    test_script: F,
) -> io::Result<ExitStatus>
where
    F: FnOnce(OsString) -> Pin<Box<dyn Future<Output = io::Result<ExitStatus>> + Send>>
        + Send + Sync + 'static,
{
    let file_name = to_miltertest_file_name(test_file_name);

    test_script(file_name).await
}

fn to_miltertest_file_name(file_name: &str) -> OsString {
    let mut path = PathBuf::from(file_name);
    path.set_extension("lua");
    path.into_os_string()
}

const MILTERTEST_PROGRAM: &str = "/usr/bin/miltertest";

pub async fn exec_miltertest(
    file_name: impl AsRef<OsStr>,
    addr: SocketAddr,
) -> io::Result<ExitStatus> {
    exec_miltertest_with_vars(file_name, addr, iter::empty::<OsString>()).await
}

pub async fn exec_miltertest_with_vars<I, S>(
    file_name: impl AsRef<OsStr>,
    addr: SocketAddr,
    global_vars: I,
) -> io::Result<ExitStatus>
where
    I: IntoIterator<Item = S>,
    S: AsRef<OsStr>,
{
    let mut miltertest = Command::new(MILTERTEST_PROGRAM);

    let port = addr.port();
    miltertest.arg("-D").arg(format!("port={}", port));

    for var in global_vars {
        miltertest.arg("-D").arg(var);
    }

    let mut miltertest = miltertest
        // .arg("-vvv")
        .arg("-s")
        .arg(file_name)
        .spawn()?;

    miltertest.wait().await
}
