//! This module defines few models found in web specs.

use std::fmt::Display;

use actix_http::StatusCode;
use once_cell::sync::Lazy;
use thiserror::Error;

use crate::Directive;

/// http response parameters that can be suggested by a directive.
pub struct HttpResponseParams {
    pub status_code: StatusCode,
    pub message: Option<String>,
}

/// Metadata associated with an interaction directive in a web spec
#[derive(Debug, Clone, Copy)]
pub struct WebInteractionDirectiveMeta<SpecParty: Copy> {
    pub requesting_party: SpecParty,
    pub responding_party: SpecParty,
}

/// An interaction directive in a web spec. It directs http interaction between two spec-parties
pub trait WebInteractionDirective: Directive {
    fn interaction_meta(&self) -> WebInteractionDirectiveMeta<Self::SpecParty>;
}

/// An interaction directive, that directs responding party of interaction.
pub trait DirectiveWithResponse: WebInteractionDirective {
    fn response_params(&self) -> HttpResponseParams;
}

/// An interaction directive that directs requesting party, and also suggests responding party with appropriate recourse in case of directive violation by requesting party.
pub trait DirectiveWithViolationRecourse: WebInteractionDirective {
    type RecourseDirective: DirectiveWithResponse<SpecParty = Self::SpecParty>;
    fn violation_recourse(&self) -> Lazy<Self::RecourseDirective>;
}

/// Ann error that records violation of interaction directive.
#[derive(Error, Debug)]
pub struct WebInteractionDirectiveViolation<T: DirectiveWithViolationRecourse> {
    pub msg: String,
    #[source]
    pub source_error: Box<dyn std::error::Error>,
    // TODO Backtrace
    pub violated: Option<Lazy<T>>,
    pub status_code_override: Option<StatusCode>,
}

impl<T: DirectiveWithViolationRecourse> WebInteractionDirectiveViolation<T> {
    pub fn recourse_response(&self) -> HttpResponseParams {
        let mut response = match &self.violated {
            Some(directive) => directive.violation_recourse().response_params(),
            None => HttpResponseParams { status_code: StatusCode::INTERNAL_SERVER_ERROR, message: None }
        };
        if let Some(status_code_override) = self.status_code_override {
            response.status_code = status_code_override;
        }
        response
    }
}

impl<T: DirectiveWithViolationRecourse> Display for WebInteractionDirectiveViolation<T> {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        let spec_statement = match &self.violated {
            Some(directive) => directive.meta().statement,
            None => "".into(),
        };
        write!(
            f,
            "error: {}\n, spec: {}\n, status_code: {}",
            self.msg,
            spec_statement,
            self.recourse_response().status_code,
        )
    }
}
