//! Simple rust models for common concepts found in web specifications

use url::Url;

pub mod web;

/// Information about a spec
#[derive(Debug)]
pub struct Spec {
    pub id: Url,
    pub label: String,
    pub description: String,
}

/// Requirement Levels as specified in [RFC2119](https://datatracker.ietf.org/doc/html/rfc2119)
#[allow(non_camel_case_types)]
#[derive(Debug, Clone, Copy)]
pub enum RequirementLevel {
    MUST,
    MUST_NOT,
    REQUIRED,
    SHALL,
    SHALL_NOT,
    SHOULD,
    SHOULD_NOT,
    RECOMMENDED,
    MAY,
    OPTIONAL,
}

/// A trait for party types that are directed in a spec
pub trait TSpecParty: Copy {

}

/// Metadata associated with a spec directive
#[derive(Debug, Clone)]
pub struct DirectiveMeta<SpecParty: TSpecParty> {
    pub id: Url,
    pub spec: Url,
    pub level: RequirementLevel,
    pub directed_to: SpecParty,
    pub statement: String,
    pub see_also: Option<Vec<Url>>,
}

/// A spec directive
pub trait Directive {
    type SpecParty: TSpecParty;

    fn meta(&self) -> DirectiveMeta<Self::SpecParty>;
}
