//! The SpamAssassin Milter application library.

macro_rules! verbose {
    ($config:ident, $($arg:tt)*) => {
        if $config.verbose() {
            ::std::eprintln!($($arg)*);
        }
    };
}

mod callbacks;
mod client;
mod collections;
mod config;
mod email;
mod error;

pub use crate::config::{Config, ConfigBuilder};
use indymilter::IntoListener;
use std::{future::Future, io};

/// The name of the SpamAssassin Milter application.
pub const MILTER_NAME: &str = "SpamAssassin Milter";

/// The current version string of SpamAssassin Milter.
pub const VERSION: &str = env!("CARGO_PKG_VERSION");

/// Starts SpamAssassin Milter listening on the given socket using the supplied
/// configuration.
///
/// # Errors
///
/// If execution of the milter fails, an error is returned.
///
/// # Examples
///
/// ```
/// # async fn f() -> std::io::Result<()> {
/// use std::process;
/// use tokio::{net::TcpListener, signal};
///
/// let listener = TcpListener::bind("127.0.0.1:3000").await?;
/// let config = Default::default();
/// let shutdown = signal::ctrl_c();
///
/// if let Err(e) = spamassassin_milter::run(listener, config, shutdown).await {
///     eprintln!("failed to run spamassassin-milter: {}", e);
///     process::exit(1);
/// }
/// # Ok(())
/// # }
/// ```
pub async fn run(
    listener: impl IntoListener,
    config: Config,
    shutdown: impl Future,
) -> io::Result<()> {
    let callbacks = callbacks::make_callbacks(config);
    let config = Default::default();

    indymilter::run(listener, callbacks, config, shutdown).await
}
