use clap::{App, Arg, ArgMatches, Error, ErrorKind, Result};
use spamassassin_milter::Config;
use std::{net::IpAddr, process};

const ARG_AUTH_UNTRUSTED: &str = "AUTH_UNTRUSTED";
const ARG_DRY_RUN: &str = "DRY_RUN";
const ARG_MAX_MESSAGE_SIZE: &str = "MAX_MESSAGE_SIZE";
const ARG_MILTER_DEBUG_LEVEL: &str = "MILTER_DEBUG_LEVEL";
const ARG_PRESERVE_BODY: &str = "PRESERVE_BODY";
const ARG_PRESERVE_HEADERS: &str = "PRESERVE_HEADERS";
const ARG_REJECT_SPAM: &str = "REJECT_SPAM";
const ARG_REPLY_CODE: &str = "REPLY_CODE";
const ARG_REPLY_STATUS_CODE: &str = "REPLY_STATUS_CODE";
const ARG_REPLY_TEXT: &str = "REPLY_TEXT";
const ARG_TRUSTED_NETWORKS: &str = "TRUSTED_NETWORKS";
const ARG_VERBOSE: &str = "VERBOSE";
const ARG_SOCKET: &str = "SOCKET";
const ARG_SPAMC_ARGS: &str = "SPAMC_ARGS";

fn main() {
    use spamassassin_milter::{MILTER_NAME, VERSION};

    let matches = App::new(MILTER_NAME)
        .version(VERSION)
        .arg(Arg::with_name(ARG_AUTH_UNTRUSTED)
            .short("a")
            .long("auth-untrusted")
            .help("Treat authenticated senders as untrusted"))
        .arg(Arg::with_name(ARG_DRY_RUN)
            .short("n")
            .long("dry-run")
            .help("Process messages without applying any changes"))
        .arg(Arg::with_name(ARG_MAX_MESSAGE_SIZE)
            .short("s")
            .long("max-message-size")
            .value_name("BYTES")
            .help("Maximum message size to process"))
        .arg(Arg::with_name(ARG_MILTER_DEBUG_LEVEL)
            .long("milter-debug-level")
            .value_name("LEVEL")
            .possible_values(&["0", "1", "2", "3", "4", "5", "6"])
            .help("Set the milter library debug level")
            .hidden(true)  // not documented for now
            .hide_possible_values(true))
        .arg(Arg::with_name(ARG_PRESERVE_BODY)
            .short("B")
            .long("preserve-body")
            .help("Suppress rewriting of message body"))
        .arg(Arg::with_name(ARG_PRESERVE_HEADERS)
            .short("H")
            .long("preserve-headers")
            .help("Suppress rewriting of Subject/From/To headers"))
        .arg(Arg::with_name(ARG_REJECT_SPAM)
            .short("r")
            .long("reject-spam")
            .help("Reject messages flagged as spam"))
        .arg(Arg::with_name(ARG_REPLY_CODE)
            .short("C")
            .long("reply-code")
            .value_name("CODE")
            .help("Reply code when rejecting messages"))
        .arg(Arg::with_name(ARG_REPLY_STATUS_CODE)
            .short("S")
            .long("reply-status-code")
            .value_name("CODE")
            .help("Status code when rejecting messages"))
        .arg(Arg::with_name(ARG_REPLY_TEXT)
            .short("R")
            .long("reply-text")
            .value_name("MSG")
            .help("Reply text when rejecting messages"))
        .arg(Arg::with_name(ARG_TRUSTED_NETWORKS)
            .short("t")
            .long("trusted-networks")
            .value_name("NETS")
            .use_delimiter(true)
            .help("Trust connections from these networks"))
        .arg(Arg::with_name(ARG_VERBOSE)
            .short("v")
            .long("verbose")
            .help("Enable verbose operation logging"))
        .arg(Arg::with_name(ARG_SOCKET)
            .required(true)
            .help("Listening socket of the milter"))
        .arg(Arg::with_name(ARG_SPAMC_ARGS)
            .last(true)
            .multiple(true)
            .help("Additional arguments to pass to spamc"))
        .get_matches();

    let socket = matches.value_of(ARG_SOCKET).unwrap();
    let config = match build_config(&matches) {
        Ok(config) => config,
        Err(e) => {
            e.exit();
        }
    };

    eprintln!("{} {} starting", MILTER_NAME, VERSION);

    match spamassassin_milter::run(socket, config) {
        Ok(_) => {
            eprintln!("{} {} shut down", MILTER_NAME, VERSION);
        }
        Err(e) => {
            eprintln!("{} {} terminated with error: {}", MILTER_NAME, VERSION, e);
            process::exit(1);
        }
    }
}

fn build_config(matches: &ArgMatches<'_>) -> Result<Config> {
    let mut config = Config::builder();

    if let Some(bytes) = matches.value_of(ARG_MAX_MESSAGE_SIZE) {
        match bytes.parse() {
            Ok(bytes) => {
                config.max_message_size(bytes);
            }
            Err(_) => {
                return Err(Error::with_description(
                    &format!("Invalid value for max message size: \"{}\"", bytes),
                    ErrorKind::InvalidValue,
                ));
            }
        }
    }

    if let Some(nets) = matches.values_of(ARG_TRUSTED_NETWORKS) {
        config.use_trusted_networks(true);

        for net in nets.filter(|n| !n.is_empty()) {
            // Both `ipnet::IpNet` and `std::net::IpAddr` inputs are supported.
            match net.parse().or_else(|_| net.parse::<IpAddr>().map(From::from)) {
                Ok(net) => {
                    config.trusted_network(net);
                }
                Err(_) => {
                    return Err(Error::with_description(
                        &format!("Invalid value for trusted network address: \"{}\"", net),
                        ErrorKind::InvalidValue,
                    ));
                }
            }
        }
    }

    let reply_code = matches.value_of(ARG_REPLY_CODE);
    let reply_status_code = matches.value_of(ARG_REPLY_STATUS_CODE);
    validate_reply_codes(reply_code, reply_status_code)?;

    if matches.is_present(ARG_AUTH_UNTRUSTED) {
        config.auth_untrusted(true);
    }
    if matches.is_present(ARG_DRY_RUN) {
        config.dry_run(true);
    }
    if matches.is_present(ARG_PRESERVE_BODY) {
        config.preserve_body(true);
    }
    if matches.is_present(ARG_PRESERVE_HEADERS) {
        config.preserve_headers(true);
    }
    if matches.is_present(ARG_REJECT_SPAM) {
        config.reject_spam(true);
    }
    if matches.is_present(ARG_VERBOSE) {
        config.verbose(true);
    }
    if let Some(code) = reply_code {
        config.reply_code(code.to_owned());
    }
    if let Some(code) = reply_status_code {
        config.reply_status_code(code.to_owned());
    }
    if let Some(msg) = matches.value_of(ARG_REPLY_TEXT) {
        config.reply_text(msg.to_owned());
    }
    if let Some(level) = matches.value_of(ARG_MILTER_DEBUG_LEVEL) {
        config.milter_debug_level(level.parse().unwrap());
    }
    if let Some(spamc_args) = matches.values_of(ARG_SPAMC_ARGS) {
        config.spamc_args(spamc_args);
    };

    Ok(config.build())
}

fn validate_reply_codes(reply_code: Option<&str>, reply_status_code: Option<&str>) -> Result<()> {
    match (reply_code, reply_status_code) {
        (Some(c1), Some(c2))
            if !((c1.starts_with('4') || c1.starts_with('5')) && c2.starts_with(&c1[..1])) =>
        {
            Err(Error::with_description(
                &format!(
                    "Invalid or incompatible values for reply code and status code: \"{}\", \"{}\"",
                    c1, c2
                ),
                ErrorKind::InvalidValue,
            ))
        }
        (Some(c), None) if !c.starts_with('5') => Err(Error::with_description(
            &format!("Invalid value for reply code (5XX): \"{}\"", c),
            ErrorKind::InvalidValue,
        )),
        (None, Some(c)) if !c.starts_with('5') => Err(Error::with_description(
            &format!("Invalid value for reply status code (5.X.X): \"{}\"", c),
            ErrorKind::InvalidValue,
        )),
        _ => Ok(()),
    }
}
