use {
  crate::{
    argmax2d::{Argmax2D, ArgmaxResult},
    geometry::{BoundingBox, PixelSpace, WorldSpace, Circle},
    sdf::SDF
  },
  euclid::{Box2D, Point2D, Size2D, Vector2D, Rect, Angle},
  image::{
    ImageBuffer, Luma, Pixel
  },
  std::{
    ops::Deref
  }
};

mod impl_draw_rgbaimage;
pub use impl_draw_rgbaimage::draw_parallel;

pub trait Draw<Backend>: Shape {
  fn draw(&self, image: &mut Backend);
}

pub trait DrawSync<Backend>: Draw<Backend> + Send + Sync {}
impl <T, Backend> DrawSync<Backend> for T where T: Draw<Backend> + Send + Sync {}

/// Something inside a rectangular area.
pub trait Shape: SDF<f32> + BoundingBox<f32, WorldSpace> {
  fn texture<T>(self, texture: T) -> Texture<Self, T> where Self: Sized {
    Texture { shape: self, texture }
  }
  fn rotate<T>(self, angle: Angle<T>) -> Rotation<Self, T> where Self: Sized {
    Rotation { shape: self, angle }
  }
}
impl <T> Shape for T where T: SDF<f32> + BoundingBox<f32, WorldSpace> {}

impl<B> SDF<f32> for Box<dyn Draw<B>> {
  fn sdf(&self, pixel: Point2D<f32, WorldSpace>) -> f32 { self.deref().sdf(pixel) } }
impl<B> BoundingBox<f32, WorldSpace> for Box<dyn Draw<B>> {
  fn bounding_box(&self) -> Box2D<f32, WorldSpace> { self.deref().bounding_box() } }

impl<B> Draw<B> for Circle<f32, WorldSpace> { fn draw(&self, _: &mut B) { unreachable!(); } }
impl<B> Draw<B> for Rect<f32, WorldSpace> { fn draw(&self, _: &mut B) { unreachable!(); } }

#[derive(Clone)]
pub struct Texture<S, T> {
  pub shape: S,
  pub texture: T
}
impl <S, T> SDF<f32> for Texture<S, T> where S: SDF<f32> {
  fn sdf(&self, pixel: Point2D<f32, WorldSpace>) -> f32 { self.shape.sdf(pixel) } }
impl <S, T> BoundingBox<f32, WorldSpace> for Texture<S, T> where S: BoundingBox<f32, WorldSpace> {
  fn bounding_box(&self) -> Box2D<f32, WorldSpace> { self.shape.bounding_box() } }

#[derive(Clone)]
pub struct Rotation<S, T> {
  pub shape: S,
  pub angle: Angle<T>
}
impl <S> SDF<f32> for Rotation<S, f32>
  where S: Shape {
  fn sdf(&self, pixel: Point2D<f32, WorldSpace>) -> f32 {
    let pivot = self.shape.bounding_box().center();
    let pixel = euclid::Rotation2D::new(self.angle)
      .transform_point( (pixel - pivot).to_point())
      + pivot.to_vector();

    self.shape.sdf(pixel)
  }
}
impl <S> BoundingBox<f32, WorldSpace> for Rotation<S, f32> where S: BoundingBox<f32, WorldSpace> {
  fn bounding_box(&self) -> Box2D<f32, WorldSpace> {
    let bounding = self.shape.bounding_box();
    let pivot = bounding.center();
    let pts = [
      [bounding.min.x, bounding.min.y],
      [bounding.max.x, bounding.min.y],
      [bounding.max.x, bounding.max.y],
      [bounding.min.x, bounding.max.y],
    ];
    let rot = |point: Point2D<_, _>| euclid::Rotation2D::new(self.angle)
      .transform_point( (point - pivot).to_point())
      + pivot.to_vector();
    let pts = pts.iter().cloned()
      .map(|p| rot(p.into()));
    Box2D::from_points(pts)
  }
}

// try to fit world in the center of image, preserving aspect ratio
fn rescale_bounding_box(
  bounding_box: Box2D<f32, WorldSpace>,
  resolution: Size2D<u32, PixelSpace>
) -> (
  Option<Box2D<u32, PixelSpace>>, // bounding_box,
  Vector2D<f32, PixelSpace>, // offset
  f32 // min_side
) {
  let min_side = resolution.width.min(resolution.height) as f32;
  let offset = (resolution.to_vector().to_f32() - Vector2D::splat(min_side)) / 2.0;
  let bounding_box = bounding_box
    .scale(min_side, min_side).cast_unit()
    .round_out()
    .translate(offset)
    .intersection(&Box2D::from_size(resolution.to_f32()))
    .map(|x| x.to_u32());
  (bounding_box, offset, min_side)
}

/// Draw shapes, parallel.
/// Faster compared to [`draw_parallel`], low memory usage.
/// Will cause undefined behaviour if two shapes intersect.
pub fn draw_parallel_unsafe<B>(
  framebuffer: &mut B,
  shapes: impl rayon::iter::ParallelIterator<Item = Box<dyn DrawSync<B>>>
) -> &mut B where B: Sync + Send {
  shapes.for_each(|shape|
    shape.draw(unsafe { &mut *(framebuffer as *const _ as *mut B) })
  );
  framebuffer
}

impl Argmax2D {
  pub fn display_debug(&self) -> image::RgbImage {
    let mut image = ImageBuffer::<image::Rgb<u8>, _>::new(
      self.resolution as u32,
      self.resolution as u32
    );
    let max_dist = self.find_max().distance;
    self.pixels().for_each(|ArgmaxResult { distance, point }| {
      let color = Luma::from([(distance / max_dist * 255.0) as u8]);
      *image.get_pixel_mut(point.x as u32, point.y as u32) = color.to_rgb();
    });
    image
  }
}

#[cfg(test)] mod test {
  use {
    super::*,
    crate::{geometry::Circle, error::Result},
    euclid::Rect, image::RgbaImage
  };

  #[test] fn texture() -> Result<()> {
    let mut image = RgbaImage::new(128, 128);
    Circle {
      xy: [0.5, 0.5].into(),
      r: 0.25
    } .texture(&image::open("doc/embedded.jpg")?)
      .draw(&mut image);
    //image.save("test_texture.png")?;
    Ok(())
  }

  #[test] fn polymorphic_a() -> Result<()> {
    let mut image = RgbaImage::new(128, 128);
    let shapes: Vec<Box<dyn Draw<RgbaImage>>> = vec![
      Box::new(Circle { xy: [0.25, 0.25].into(), r: 0.25 }),
      Box::new(Rect { origin: [0.5, 0.5].into(), size: [0.5, 0.5].into() })
    ];
    shapes.into_iter()
      .map(|shape| Box::new(shape
        .rotate(Angle::degrees(45.0))
        .texture(Luma([255u8]).to_rgba())
      ) as Box<dyn Draw<RgbaImage>>)
      .for_each(|shape| shape.draw(&mut image));
    //image.save("test_polymorphic_a.png")?;
    Ok(())
  }

  #[test] fn polymorphic_b() -> Result<()> {
    let mut image = RgbaImage::new(128, 128);
    let shapes: Vec<Box<dyn Draw<_>>> = vec![
      Box::new(Circle { xy: [0.25, 0.25].into(), r: 0.25 }
        .texture(Luma([255u8]).to_rgba())),
      Box::new(Rect { origin: [0.5, 0.5].into(), size: [0.5, 0.5].into() }
        .texture(Luma([127u8]).to_rgba()))
    ];
    shapes.into_iter()
      .for_each(|shape| shape.draw(&mut image));
    //image.save("test_polymorphic_b.png")?;
    Ok(())
  }
}