/*
 * Copyright (c) 2021, 2022  StorPool <support@storpool.com>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
//! Partially-generated data for handling build variants.
//!
//! The data is autogenerated from the supported StorPool build variants.

use std::collections::HashMap;
use std::str::FromStr;

use serde::{Deserialize, Serialize};

use crate::{
    Builder, DebRepo, Detect, Repo, Variant, VariantDefTop, VariantError, VariantFormat,
    VariantFormatVersion, YumRepo,
};

/// The supported StorPool build variants (OS distribution, version, etc).
#[derive(Debug, Clone, PartialEq, Hash, Eq, Serialize, Deserialize)]
#[non_exhaustive]
pub enum VariantKind {
    /// AlmaLinux 8.x
    ALMA8,
    /// CentOS 6.x
    CENTOS6,
    /// CentOS 7.x
    CENTOS7,
    /// CentOS 8.x
    CENTOS8,
    /// Debian 9.x (stretch)
    DEBIAN9,
    /// Debian 10.x (buster)
    DEBIAN10,
    /// Debian 11.x (bullseye)
    DEBIAN11,
    /// Debian 12.x (bookworm/unstable)
    DEBIAN12,
    /// Oracle Linux 7.x
    ORACLE7,
    /// RedHat Enterprise Linux 8.x
    RHEL8,
    /// Rocky Linux 8.x
    ROCKY8,
    /// Ubuntu 16.04 LTS (Xenial Xerus)
    UBUNTU1604,
    /// Ubuntu 18.04 LTS (Bionic Beaver)
    UBUNTU1804,
    /// Ubuntu 20.04 LTS (Focal Fossa)
    UBUNTU2004,
    /// Ubuntu 21.10 LTS (Impish Indri)
    UBUNTU2110,
    /// Ubuntu 22.04 LTS (Jammy Jellyfish)
    UBUNTU2204,
}

impl VariantKind {
    const ALMA8_NAME: &'static str = "ALMA8";
    const CENTOS6_NAME: &'static str = "CENTOS6";
    const CENTOS7_NAME: &'static str = "CENTOS7";
    const CENTOS8_NAME: &'static str = "CENTOS8";
    const DEBIAN9_NAME: &'static str = "DEBIAN9";
    const DEBIAN10_NAME: &'static str = "DEBIAN10";
    const DEBIAN11_NAME: &'static str = "DEBIAN11";
    const DEBIAN12_NAME: &'static str = "DEBIAN12";
    const ORACLE7_NAME: &'static str = "ORACLE7";
    const RHEL8_NAME: &'static str = "RHEL8";
    const ROCKY8_NAME: &'static str = "ROCKY8";
    const UBUNTU1604_NAME: &'static str = "UBUNTU1604";
    const UBUNTU1804_NAME: &'static str = "UBUNTU1804";
    const UBUNTU2004_NAME: &'static str = "UBUNTU2004";
    const UBUNTU2110_NAME: &'static str = "UBUNTU2110";
    const UBUNTU2204_NAME: &'static str = "UBUNTU2204";
}

impl AsRef<str> for VariantKind {
    #[inline]
    fn as_ref(&self) -> &str {
        match *self {
            Self::ALMA8 => Self::ALMA8_NAME,
            Self::CENTOS6 => Self::CENTOS6_NAME,
            Self::CENTOS7 => Self::CENTOS7_NAME,
            Self::CENTOS8 => Self::CENTOS8_NAME,
            Self::DEBIAN9 => Self::DEBIAN9_NAME,
            Self::DEBIAN10 => Self::DEBIAN10_NAME,
            Self::DEBIAN11 => Self::DEBIAN11_NAME,
            Self::DEBIAN12 => Self::DEBIAN12_NAME,
            Self::ORACLE7 => Self::ORACLE7_NAME,
            Self::RHEL8 => Self::RHEL8_NAME,
            Self::ROCKY8 => Self::ROCKY8_NAME,
            Self::UBUNTU1604 => Self::UBUNTU1604_NAME,
            Self::UBUNTU1804 => Self::UBUNTU1804_NAME,
            Self::UBUNTU2004 => Self::UBUNTU2004_NAME,
            Self::UBUNTU2110 => Self::UBUNTU2110_NAME,
            Self::UBUNTU2204 => Self::UBUNTU2204_NAME,
        }
    }
}

impl FromStr for VariantKind {
    type Err = VariantError;

    #[inline]
    fn from_str(s: &str) -> Result<Self, Self::Err> {
        match s {
            Self::ALMA8_NAME => Ok(Self::ALMA8),
            Self::CENTOS6_NAME => Ok(Self::CENTOS6),
            Self::CENTOS7_NAME => Ok(Self::CENTOS7),
            Self::CENTOS8_NAME => Ok(Self::CENTOS8),
            Self::DEBIAN9_NAME => Ok(Self::DEBIAN9),
            Self::DEBIAN10_NAME => Ok(Self::DEBIAN10),
            Self::DEBIAN11_NAME => Ok(Self::DEBIAN11),
            Self::DEBIAN12_NAME => Ok(Self::DEBIAN12),
            Self::ORACLE7_NAME => Ok(Self::ORACLE7),
            Self::RHEL8_NAME => Ok(Self::RHEL8),
            Self::ROCKY8_NAME => Ok(Self::ROCKY8),
            Self::UBUNTU1604_NAME => Ok(Self::UBUNTU1604),
            Self::UBUNTU1804_NAME => Ok(Self::UBUNTU1804),
            Self::UBUNTU2004_NAME => Ok(Self::UBUNTU2004),
            Self::UBUNTU2110_NAME => Ok(Self::UBUNTU2110),
            Self::UBUNTU2204_NAME => Ok(Self::UBUNTU2204),
            other => Err(VariantError::BadVariant(other.to_owned())),
        }
    }
}

/// Return the definition of the StorPool variants.
#[allow(clippy::panic)]
#[allow(clippy::too_many_lines)]
#[must_use]
pub fn get_variants() -> &'static VariantDefTop {
    lazy_static! {
        static ref DEF_TOP: VariantDefTop = VariantDefTop {
            format: VariantFormat {
                version: VariantFormatVersion {
                    major: 1,
                    minor: 3,
                },
            },
            order: vec![
                    VariantKind::ALMA8,
                    VariantKind::ROCKY8,
                    VariantKind::RHEL8,
                    VariantKind::ORACLE7,
                    VariantKind::CENTOS6,
                    VariantKind::CENTOS7,
                    VariantKind::CENTOS8,
                    VariantKind::UBUNTU1604,
                    VariantKind::UBUNTU1804,
                    VariantKind::UBUNTU2004,
                    VariantKind::UBUNTU2110,
                    VariantKind::UBUNTU2204,
                    VariantKind::DEBIAN9,
                    VariantKind::DEBIAN10,
                    VariantKind::DEBIAN11,
                    VariantKind::DEBIAN12,
            ],
            variants: HashMap::from(
                [
                    (
                            VariantKind::ALMA8,
                            Variant {
                                kind: VariantKind::ALMA8,
                                descr: "AlmaLinux 8.x".to_owned(),
                                family: "redhat".to_owned(),
                                parent: "CENTOS8".to_owned(),
                                detect: Detect {
                                    filename: "/etc/redhat-release".to_owned(),
                                    regex: r"^ AlmaLinux \s .* \s 8 \. (?: [4-9] | [1-9][0-9] )".to_owned(),
                                    os_id: "alma".to_owned(),
                                    os_version_regex: r"^8(?:$|\.[4-9]|\.[1-9][0-9])".to_owned(),
                                },
                                commands: HashMap::from(
                                    [
                                        (
                                            "package".to_owned(),
                                            HashMap::from(
                                                [
                                                    (
                                                        "install".to_owned(),
                                                        vec![
                                                            "dnf".to_owned(),
                                                            "--enablerepo=storpool-contrib".to_owned(),
                                                            "--enablerepo=powertools".to_owned(),
                                                            "install".to_owned(),
                                                            "-q".to_owned(),
                                                            "-y".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "list_all".to_owned(),
                                                        vec![
                                                            "rpm".to_owned(),
                                                            "-qa".to_owned(),
                                                            "--qf".to_owned(),
                                                            "%{Name}\\t%{EVR}\\t%{Arch}\\tii\\n".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "purge".to_owned(),
                                                        vec![
                                                            "yum".to_owned(),
                                                            "remove".to_owned(),
                                                            "-q".to_owned(),
                                                            "-y".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "remove".to_owned(),
                                                        vec![
                                                            "yum".to_owned(),
                                                            "remove".to_owned(),
                                                            "-q".to_owned(),
                                                            "-y".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "remove_impl".to_owned(),
                                                        vec![
                                                            "rpm".to_owned(),
                                                            "-e".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "update_db".to_owned(),
                                                        vec![
                                                            "true".to_owned(),
                                                        ],
                                                    ),
                                                ]
                                            ),
                                        ),
                                        (
                                            "pkgfile".to_owned(),
                                            HashMap::from(
                                                [
                                                    (
                                                        "dep_query".to_owned(),
                                                        vec![
                                                            "sh".to_owned(),
                                                            "-c".to_owned(),
                                                            "rpm -qpR -- \"$pkg\"".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "install".to_owned(),
                                                        vec![
                                                            "sh".to_owned(),
                                                            "-c".to_owned(),
                                                            "
unset to_install to_reinstall
for f in $packages; do
    package=\"$(rpm -qp \"$f\")\"
    if rpm -q -- \"$package\"; then
        to_reinstall=\"$to_reinstall ./$f\"
    else
        to_install=\"$to_install ./$f\"
    fi
done

if [ -n \"$to_install\" ]; then
    dnf install -y --enablerepo=storpool-contrib,powertools --setopt=localpkg_gpgcheck=0 -- $to_install
fi
if [ -n \"$to_reinstall\" ]; then
    dnf reinstall -y --enablerepo=storpool-contrib,powertools --setopt=localpkg_gpgcheck=0 -- $to_reinstall
fi
".to_owned(),
                                                        ],
                                                    ),
                                                ]
                                            ),
                                        ),
                                    ]
                                ),
                                min_sys_python: "2.7".to_owned(),
                                repo:
                                    Repo::Yum(YumRepo {
                                        yumdef: "redhat/repo/storpool-centos.repo".to_owned(),
                                        keyring: "redhat/repo/RPM-GPG-KEY-StorPool".to_owned(),
                                    }),

                                package: HashMap::from(
                                    [
                                        ("KMOD".to_owned(), "kmod".to_owned()),
                                        ("LIBCGROUP".to_owned(), "libcgroup-tools".to_owned()),
                                        ("LIBUDEV".to_owned(), "systemd-libs".to_owned()),
                                        ("OPENSSL".to_owned(), "openssl-libs".to_owned()),
                                        ("PERL_AUTODIE".to_owned(), "perl-autodie".to_owned()),
                                        ("PERL_FILE_PATH".to_owned(), "perl-File-Path".to_owned()),
                                        ("PERL_LWP_PROTO_HTTPS".to_owned(), "perl-LWP-Protocol-https".to_owned()),
                                        ("PERL_SYS_SYSLOG".to_owned(), "perl-Sys-Syslog".to_owned()),
                                        ("PROCPS".to_owned(), "procps-ng".to_owned()),
                                        ("PYTHON_SIMPLEJSON".to_owned(), "python2-simplejson".to_owned()),
                                        ("UDEV".to_owned(), "systemd".to_owned()),
                                    ]
                                ),
                                systemd_lib: "usr/lib/systemd/system".to_owned(),
                                file_ext: "rpm".to_owned(),
                                initramfs_flavor: "mkinitrd".to_owned(),
                                builder: Builder {
                                    alias: "alma8".to_owned(),
                                    base_image: "almalinux/almalinux:8".to_owned(),
                                    branch: "".to_owned(),
                                    kernel_package: "kernel-core".to_owned(),
                                    utf8_locale: "C.utf8".to_owned(),
                                },
                            },
                    ),
                    (
                            VariantKind::CENTOS6,
                            Variant {
                                kind: VariantKind::CENTOS6,
                                descr: "CentOS 6.x".to_owned(),
                                family: "redhat".to_owned(),
                                parent: "CENTOS7".to_owned(),
                                detect: Detect {
                                    filename: "/etc/redhat-release".to_owned(),
                                    regex: r"^ CentOS \s .* \s 6 \.".to_owned(),
                                    os_id: "centos".to_owned(),
                                    os_version_regex: r"^6(?:$|\.[0-9])".to_owned(),
                                },
                                commands: HashMap::from(
                                    [
                                        (
                                            "package".to_owned(),
                                            HashMap::from(
                                                [
                                                    (
                                                        "install".to_owned(),
                                                        vec![
                                                            "yum".to_owned(),
                                                            "--enablerepo=storpool-contrib".to_owned(),
                                                            "install".to_owned(),
                                                            "-q".to_owned(),
                                                            "-y".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "list_all".to_owned(),
                                                        vec![
                                                            "rpm".to_owned(),
                                                            "-qa".to_owned(),
                                                            "--qf".to_owned(),
                                                            "%{Name}\\t%{EVR}\\t%{Arch}\\tii\\n".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "purge".to_owned(),
                                                        vec![
                                                            "yum".to_owned(),
                                                            "remove".to_owned(),
                                                            "-q".to_owned(),
                                                            "-y".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "remove".to_owned(),
                                                        vec![
                                                            "yum".to_owned(),
                                                            "remove".to_owned(),
                                                            "-q".to_owned(),
                                                            "-y".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "remove_impl".to_owned(),
                                                        vec![
                                                            "rpm".to_owned(),
                                                            "-e".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "update_db".to_owned(),
                                                        vec![
                                                            "true".to_owned(),
                                                        ],
                                                    ),
                                                ]
                                            ),
                                        ),
                                        (
                                            "pkgfile".to_owned(),
                                            HashMap::from(
                                                [
                                                    (
                                                        "dep_query".to_owned(),
                                                        vec![
                                                            "sh".to_owned(),
                                                            "-c".to_owned(),
                                                            "rpm -qpR -- \"$pkg\"".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "install".to_owned(),
                                                        vec![
                                                            "
unset to_install to_reinstall
for f in $packages; do
    package=\"$(rpm -qp \"$f\")\"
    if rpm -q -- \"$package\"; then
        to_reinstall=\"$to_reinstall ./$f\"
    else
        to_install=\"$to_install ./$f\"
    fi
done

if [ -n \"$to_install\" ]; then
    yum install -y --enablerepo=storpool-contrib --setopt=localpkg_gpgcheck=0 -- $to_install
fi
if [ -n \"$to_reinstall\" ]; then
    yum reinstall -y --enablerepo=storpool-contrib --setopt=localpkg_gpgcheck=0 -- $to_reinstall
fi
".to_owned(),
                                                        ],
                                                    ),
                                                ]
                                            ),
                                        ),
                                    ]
                                ),
                                min_sys_python: "2.6".to_owned(),
                                repo:
                                    Repo::Yum(YumRepo {
                                        yumdef: "redhat/repo/storpool-centos.repo".to_owned(),
                                        keyring: "redhat/repo/RPM-GPG-KEY-StorPool".to_owned(),
                                    }),

                                package: HashMap::from(
                                    [
                                        ("KMOD".to_owned(), "module-init-tools".to_owned()),
                                        ("LIBCGROUP".to_owned(), "libcgroup".to_owned()),
                                        ("LIBUDEV".to_owned(), "libudev".to_owned()),
                                        ("OPENSSL".to_owned(), "openssl".to_owned()),
                                        ("PERL_AUTODIE".to_owned(), "perl".to_owned()),
                                        ("PERL_FILE_PATH".to_owned(), "perl".to_owned()),
                                        ("PERL_LWP_PROTO_HTTPS".to_owned(), "perl".to_owned()),
                                        ("PERL_SYS_SYSLOG".to_owned(), "perl".to_owned()),
                                        ("PROCPS".to_owned(), "procps".to_owned()),
                                        ("PYTHON_SIMPLEJSON".to_owned(), "python-simplejson".to_owned()),
                                        ("UDEV".to_owned(), "udev".to_owned()),
                                    ]
                                ),
                                systemd_lib: "usr/lib/systemd/system".to_owned(),
                                file_ext: "rpm".to_owned(),
                                initramfs_flavor: "mkinitrd".to_owned(),
                                builder: Builder {
                                    alias: "centos6".to_owned(),
                                    base_image: "centos:6".to_owned(),
                                    branch: "centos/6".to_owned(),
                                    kernel_package: "kernel".to_owned(),
                                    utf8_locale: "C".to_owned(),
                                },
                            },
                    ),
                    (
                            VariantKind::CENTOS7,
                            Variant {
                                kind: VariantKind::CENTOS7,
                                descr: "CentOS 7.x".to_owned(),
                                family: "redhat".to_owned(),
                                parent: "CENTOS8".to_owned(),
                                detect: Detect {
                                    filename: "/etc/redhat-release".to_owned(),
                                    regex: r"^ (?: CentOS | Virtuozzo ) \s .* \s 7 \.".to_owned(),
                                    os_id: "centos".to_owned(),
                                    os_version_regex: r"^7(?:$|\.[0-9])".to_owned(),
                                },
                                commands: HashMap::from(
                                    [
                                        (
                                            "package".to_owned(),
                                            HashMap::from(
                                                [
                                                    (
                                                        "install".to_owned(),
                                                        vec![
                                                            "yum".to_owned(),
                                                            "--enablerepo=storpool-contrib".to_owned(),
                                                            "install".to_owned(),
                                                            "-q".to_owned(),
                                                            "-y".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "list_all".to_owned(),
                                                        vec![
                                                            "rpm".to_owned(),
                                                            "-qa".to_owned(),
                                                            "--qf".to_owned(),
                                                            "%{Name}\\t%{EVR}\\t%{Arch}\\tii\\n".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "purge".to_owned(),
                                                        vec![
                                                            "yum".to_owned(),
                                                            "remove".to_owned(),
                                                            "-q".to_owned(),
                                                            "-y".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "remove".to_owned(),
                                                        vec![
                                                            "yum".to_owned(),
                                                            "remove".to_owned(),
                                                            "-q".to_owned(),
                                                            "-y".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "remove_impl".to_owned(),
                                                        vec![
                                                            "rpm".to_owned(),
                                                            "-e".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "update_db".to_owned(),
                                                        vec![
                                                            "true".to_owned(),
                                                        ],
                                                    ),
                                                ]
                                            ),
                                        ),
                                        (
                                            "pkgfile".to_owned(),
                                            HashMap::from(
                                                [
                                                    (
                                                        "dep_query".to_owned(),
                                                        vec![
                                                            "sh".to_owned(),
                                                            "-c".to_owned(),
                                                            "rpm -qpR -- \"$pkg\"".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "install".to_owned(),
                                                        vec![
                                                            "
unset to_install to_reinstall
for f in $packages; do
    package=\"$(rpm -qp \"$f\")\"
    if rpm -q -- \"$package\"; then
        to_reinstall=\"$to_reinstall ./$f\"
    else
        to_install=\"$to_install ./$f\"
    fi
done

if [ -n \"$to_install\" ]; then
    yum install -y --enablerepo=storpool-contrib --setopt=localpkg_gpgcheck=0 -- $to_install
fi
if [ -n \"$to_reinstall\" ]; then
    yum reinstall -y --enablerepo=storpool-contrib --setopt=localpkg_gpgcheck=0 -- $to_reinstall
fi
".to_owned(),
                                                        ],
                                                    ),
                                                ]
                                            ),
                                        ),
                                    ]
                                ),
                                min_sys_python: "2.7".to_owned(),
                                repo:
                                    Repo::Yum(YumRepo {
                                        yumdef: "redhat/repo/storpool-centos.repo".to_owned(),
                                        keyring: "redhat/repo/RPM-GPG-KEY-StorPool".to_owned(),
                                    }),

                                package: HashMap::from(
                                    [
                                        ("KMOD".to_owned(), "kmod".to_owned()),
                                        ("LIBCGROUP".to_owned(), "libcgroup-tools".to_owned()),
                                        ("LIBUDEV".to_owned(), "systemd-libs".to_owned()),
                                        ("OPENSSL".to_owned(), "openssl-libs".to_owned()),
                                        ("PERL_AUTODIE".to_owned(), "perl-autodie".to_owned()),
                                        ("PERL_FILE_PATH".to_owned(), "perl-File-Path".to_owned()),
                                        ("PERL_LWP_PROTO_HTTPS".to_owned(), "perl-LWP-Protocol-https".to_owned()),
                                        ("PERL_SYS_SYSLOG".to_owned(), "perl-Sys-Syslog".to_owned()),
                                        ("PROCPS".to_owned(), "procps-ng".to_owned()),
                                        ("PYTHON_SIMPLEJSON".to_owned(), "python2-simplejson".to_owned()),
                                        ("UDEV".to_owned(), "systemd".to_owned()),
                                    ]
                                ),
                                systemd_lib: "usr/lib/systemd/system".to_owned(),
                                file_ext: "rpm".to_owned(),
                                initramfs_flavor: "mkinitrd".to_owned(),
                                builder: Builder {
                                    alias: "centos7".to_owned(),
                                    base_image: "centos:7".to_owned(),
                                    branch: "centos/7".to_owned(),
                                    kernel_package: "kernel".to_owned(),
                                    utf8_locale: "C".to_owned(),
                                },
                            },
                    ),
                    (
                            VariantKind::CENTOS8,
                            Variant {
                                kind: VariantKind::CENTOS8,
                                descr: "CentOS 8.x".to_owned(),
                                family: "redhat".to_owned(),
                                parent: "".to_owned(),
                                detect: Detect {
                                    filename: "/etc/redhat-release".to_owned(),
                                    regex: r"^ CentOS \s .* \s 8 \. (?: [3-9] | (?: [12][0-9] ) )".to_owned(),
                                    os_id: "centos".to_owned(),
                                    os_version_regex: r"^8(?:$|\.[4-9]|\.[1-9][0-9])".to_owned(),
                                },
                                commands: HashMap::from(
                                    [
                                        (
                                            "package".to_owned(),
                                            HashMap::from(
                                                [
                                                    (
                                                        "install".to_owned(),
                                                        vec![
                                                            "dnf".to_owned(),
                                                            "--enablerepo=storpool-contrib".to_owned(),
                                                            "--enablerepo=powertools".to_owned(),
                                                            "install".to_owned(),
                                                            "-q".to_owned(),
                                                            "-y".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "list_all".to_owned(),
                                                        vec![
                                                            "rpm".to_owned(),
                                                            "-qa".to_owned(),
                                                            "--qf".to_owned(),
                                                            "%{Name}\\t%{EVR}\\t%{Arch}\\tii\\n".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "purge".to_owned(),
                                                        vec![
                                                            "yum".to_owned(),
                                                            "remove".to_owned(),
                                                            "-q".to_owned(),
                                                            "-y".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "remove".to_owned(),
                                                        vec![
                                                            "yum".to_owned(),
                                                            "remove".to_owned(),
                                                            "-q".to_owned(),
                                                            "-y".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "remove_impl".to_owned(),
                                                        vec![
                                                            "rpm".to_owned(),
                                                            "-e".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "update_db".to_owned(),
                                                        vec![
                                                            "true".to_owned(),
                                                        ],
                                                    ),
                                                ]
                                            ),
                                        ),
                                        (
                                            "pkgfile".to_owned(),
                                            HashMap::from(
                                                [
                                                    (
                                                        "dep_query".to_owned(),
                                                        vec![
                                                            "sh".to_owned(),
                                                            "-c".to_owned(),
                                                            "rpm -qpR -- \"$pkg\"".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "install".to_owned(),
                                                        vec![
                                                            "sh".to_owned(),
                                                            "-c".to_owned(),
                                                            "
unset to_install to_reinstall
for f in $packages; do
    package=\"$(rpm -qp \"$f\")\"
    if rpm -q -- \"$package\"; then
        to_reinstall=\"$to_reinstall ./$f\"
    else
        to_install=\"$to_install ./$f\"
    fi
done

if [ -n \"$to_install\" ]; then
    dnf install -y --enablerepo=storpool-contrib,powertools --setopt=localpkg_gpgcheck=0 -- $to_install
fi
if [ -n \"$to_reinstall\" ]; then
    dnf reinstall -y --enablerepo=storpool-contrib,powertools --setopt=localpkg_gpgcheck=0 -- $to_reinstall
fi
".to_owned(),
                                                        ],
                                                    ),
                                                ]
                                            ),
                                        ),
                                    ]
                                ),
                                min_sys_python: "2.7".to_owned(),
                                repo:
                                    Repo::Yum(YumRepo {
                                        yumdef: "redhat/repo/storpool-centos.repo".to_owned(),
                                        keyring: "redhat/repo/RPM-GPG-KEY-StorPool".to_owned(),
                                    }),

                                package: HashMap::from(
                                    [
                                        ("KMOD".to_owned(), "kmod".to_owned()),
                                        ("LIBCGROUP".to_owned(), "libcgroup-tools".to_owned()),
                                        ("LIBUDEV".to_owned(), "systemd-libs".to_owned()),
                                        ("OPENSSL".to_owned(), "openssl-libs".to_owned()),
                                        ("PERL_AUTODIE".to_owned(), "perl-autodie".to_owned()),
                                        ("PERL_FILE_PATH".to_owned(), "perl-File-Path".to_owned()),
                                        ("PERL_LWP_PROTO_HTTPS".to_owned(), "perl-LWP-Protocol-https".to_owned()),
                                        ("PERL_SYS_SYSLOG".to_owned(), "perl-Sys-Syslog".to_owned()),
                                        ("PROCPS".to_owned(), "procps-ng".to_owned()),
                                        ("PYTHON_SIMPLEJSON".to_owned(), "python2-simplejson".to_owned()),
                                        ("UDEV".to_owned(), "systemd".to_owned()),
                                    ]
                                ),
                                systemd_lib: "usr/lib/systemd/system".to_owned(),
                                file_ext: "rpm".to_owned(),
                                initramfs_flavor: "mkinitrd".to_owned(),
                                builder: Builder {
                                    alias: "centos8".to_owned(),
                                    base_image: "centos:8".to_owned(),
                                    branch: "centos/8".to_owned(),
                                    kernel_package: "kernel-core".to_owned(),
                                    utf8_locale: "C.utf8".to_owned(),
                                },
                            },
                    ),
                    (
                            VariantKind::DEBIAN9,
                            Variant {
                                kind: VariantKind::DEBIAN9,
                                descr: "Debian 9.x (stretch)".to_owned(),
                                family: "debian".to_owned(),
                                parent: "DEBIAN10".to_owned(),
                                detect: Detect {
                                    filename: "/etc/os-release".to_owned(),
                                    regex: r"^
                    PRETTY_NAME= .*
                    Debian \s+ GNU/Linux \s+
                    (?: stretch | 9 ) (?: \s | / )
                ".to_owned(),
                                    os_id: "debian".to_owned(),
                                    os_version_regex: r"^9$".to_owned(),
                                },
                                commands: HashMap::from(
                                    [
                                        (
                                            "package".to_owned(),
                                            HashMap::from(
                                                [
                                                    (
                                                        "install".to_owned(),
                                                        vec![
                                                            "env".to_owned(),
                                                            "DEBIAN_FRONTEND=noninteractive".to_owned(),
                                                            "apt-get".to_owned(),
                                                            "-q".to_owned(),
                                                            "-y".to_owned(),
                                                            "--no-install-recommends".to_owned(),
                                                            "install".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "list_all".to_owned(),
                                                        vec![
                                                            "dpkg-query".to_owned(),
                                                            "-W".to_owned(),
                                                            "-f".to_owned(),
                                                            "${Package}\\t${Version}\\t${Architecture}\\t${db:Status-Abbrev}\\n".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "purge".to_owned(),
                                                        vec![
                                                            "env".to_owned(),
                                                            "DEBIAN_FRONTEND=noninteractive".to_owned(),
                                                            "apt-get".to_owned(),
                                                            "-q".to_owned(),
                                                            "-y".to_owned(),
                                                            "purge".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "remove".to_owned(),
                                                        vec![
                                                            "env".to_owned(),
                                                            "DEBIAN_FRONTEND=noninteractive".to_owned(),
                                                            "apt-get".to_owned(),
                                                            "-q".to_owned(),
                                                            "-y".to_owned(),
                                                            "remove".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "remove_impl".to_owned(),
                                                        vec![
                                                            "env".to_owned(),
                                                            "DEBIAN_FRONTEND=noninteractive".to_owned(),
                                                            "dpkg".to_owned(),
                                                            "-r".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "update_db".to_owned(),
                                                        vec![
                                                            "apt-get".to_owned(),
                                                            "-q".to_owned(),
                                                            "-y".to_owned(),
                                                            "update".to_owned(),
                                                        ],
                                                    ),
                                                ]
                                            ),
                                        ),
                                        (
                                            "pkgfile".to_owned(),
                                            HashMap::from(
                                                [
                                                    (
                                                        "dep_query".to_owned(),
                                                        vec![
                                                            "sh".to_owned(),
                                                            "-c".to_owned(),
                                                            "dpkg-deb -f -- \"$pkg\" 'Depends' | sed -e 's/ *, */,/g' | tr ',' \"\\n\"".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "install".to_owned(),
                                                        vec![
                                                            "sh".to_owned(),
                                                            "-c".to_owned(),
                                                            "env DEBIAN_FRONTEND=noninteractive apt-get install --no-install-recommends --reinstall -y -o DPkg::Options::=--force-confnew -- $packages".to_owned(),
                                                        ],
                                                    ),
                                                ]
                                            ),
                                        ),
                                    ]
                                ),
                                min_sys_python: "2.7".to_owned(),
                                repo:
                                    Repo::Deb(DebRepo {
                                        codename: "stretch".to_owned(),
                                        vendor: "debian".to_owned(),
                                        sources: "debian/repo/storpool.sources".to_owned(),
                                        keyring: "debian/repo/storpool-keyring.gpg".to_owned(),
                                        req_packages: vec![
                                            "apt-transport-https".to_owned(),
                                            "ca-certificates".to_owned(),
                                        ],
                                    }),

                                package: HashMap::from(
                                    [
                                        ("BINDINGS_PYTHON".to_owned(), "python".to_owned()),
                                        ("BINDINGS_PYTHON_CONFGET".to_owned(), "python-confget".to_owned()),
                                        ("BINDINGS_PYTHON_SIMPLEJSON".to_owned(), "python-simplejson".to_owned()),
                                        ("CGROUP".to_owned(), "cgroup-tools".to_owned()),
                                        ("CPUPOWER".to_owned(), "linux-cpupower".to_owned()),
                                        ("LIBSSL".to_owned(), "libssl1.1".to_owned()),
                                        ("MCELOG".to_owned(), "mcelog".to_owned()),
                                    ]
                                ),
                                systemd_lib: "lib/systemd/system".to_owned(),
                                file_ext: "deb".to_owned(),
                                initramfs_flavor: "update-initramfs".to_owned(),
                                builder: Builder {
                                    alias: "debian9".to_owned(),
                                    base_image: "debian:stretch".to_owned(),
                                    branch: "debian/stretch".to_owned(),
                                    kernel_package: "linux-headers".to_owned(),
                                    utf8_locale: "C.UTF-8".to_owned(),
                                },
                            },
                    ),
                    (
                            VariantKind::DEBIAN10,
                            Variant {
                                kind: VariantKind::DEBIAN10,
                                descr: "Debian 10.x (buster)".to_owned(),
                                family: "debian".to_owned(),
                                parent: "DEBIAN11".to_owned(),
                                detect: Detect {
                                    filename: "/etc/os-release".to_owned(),
                                    regex: r"^
                    PRETTY_NAME= .*
                    Debian \s+ GNU/Linux \s+
                    (?: buster | 10 ) (?: \s | / )
                ".to_owned(),
                                    os_id: "debian".to_owned(),
                                    os_version_regex: r"^10$".to_owned(),
                                },
                                commands: HashMap::from(
                                    [
                                        (
                                            "package".to_owned(),
                                            HashMap::from(
                                                [
                                                    (
                                                        "install".to_owned(),
                                                        vec![
                                                            "env".to_owned(),
                                                            "DEBIAN_FRONTEND=noninteractive".to_owned(),
                                                            "apt-get".to_owned(),
                                                            "-q".to_owned(),
                                                            "-y".to_owned(),
                                                            "--no-install-recommends".to_owned(),
                                                            "install".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "list_all".to_owned(),
                                                        vec![
                                                            "dpkg-query".to_owned(),
                                                            "-W".to_owned(),
                                                            "-f".to_owned(),
                                                            "${Package}\\t${Version}\\t${Architecture}\\t${db:Status-Abbrev}\\n".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "purge".to_owned(),
                                                        vec![
                                                            "env".to_owned(),
                                                            "DEBIAN_FRONTEND=noninteractive".to_owned(),
                                                            "apt-get".to_owned(),
                                                            "-q".to_owned(),
                                                            "-y".to_owned(),
                                                            "purge".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "remove".to_owned(),
                                                        vec![
                                                            "env".to_owned(),
                                                            "DEBIAN_FRONTEND=noninteractive".to_owned(),
                                                            "apt-get".to_owned(),
                                                            "-q".to_owned(),
                                                            "-y".to_owned(),
                                                            "remove".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "remove_impl".to_owned(),
                                                        vec![
                                                            "env".to_owned(),
                                                            "DEBIAN_FRONTEND=noninteractive".to_owned(),
                                                            "dpkg".to_owned(),
                                                            "-r".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "update_db".to_owned(),
                                                        vec![
                                                            "apt-get".to_owned(),
                                                            "-q".to_owned(),
                                                            "-y".to_owned(),
                                                            "update".to_owned(),
                                                        ],
                                                    ),
                                                ]
                                            ),
                                        ),
                                        (
                                            "pkgfile".to_owned(),
                                            HashMap::from(
                                                [
                                                    (
                                                        "dep_query".to_owned(),
                                                        vec![
                                                            "sh".to_owned(),
                                                            "-c".to_owned(),
                                                            "dpkg-deb -f -- \"$pkg\" 'Depends' | sed -e 's/ *, */,/g' | tr ',' \"\\n\"".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "install".to_owned(),
                                                        vec![
                                                            "sh".to_owned(),
                                                            "-c".to_owned(),
                                                            "env DEBIAN_FRONTEND=noninteractive apt-get install --no-install-recommends --reinstall -y -o DPkg::Options::=--force-confnew -- $packages".to_owned(),
                                                        ],
                                                    ),
                                                ]
                                            ),
                                        ),
                                    ]
                                ),
                                min_sys_python: "2.7".to_owned(),
                                repo:
                                    Repo::Deb(DebRepo {
                                        codename: "buster".to_owned(),
                                        vendor: "debian".to_owned(),
                                        sources: "debian/repo/storpool.sources".to_owned(),
                                        keyring: "debian/repo/storpool-keyring.gpg".to_owned(),
                                        req_packages: vec![
                                            "ca-certificates".to_owned(),
                                        ],
                                    }),

                                package: HashMap::from(
                                    [
                                        ("BINDINGS_PYTHON".to_owned(), "python".to_owned()),
                                        ("BINDINGS_PYTHON_CONFGET".to_owned(), "python-confget".to_owned()),
                                        ("BINDINGS_PYTHON_SIMPLEJSON".to_owned(), "python-simplejson".to_owned()),
                                        ("CGROUP".to_owned(), "cgroup-tools".to_owned()),
                                        ("CPUPOWER".to_owned(), "linux-cpupower".to_owned()),
                                        ("LIBSSL".to_owned(), "libssl1.1".to_owned()),
                                        ("MCELOG".to_owned(), "mcelog".to_owned()),
                                    ]
                                ),
                                systemd_lib: "lib/systemd/system".to_owned(),
                                file_ext: "deb".to_owned(),
                                initramfs_flavor: "update-initramfs".to_owned(),
                                builder: Builder {
                                    alias: "debian10".to_owned(),
                                    base_image: "debian:buster".to_owned(),
                                    branch: "debian/buster".to_owned(),
                                    kernel_package: "linux-headers".to_owned(),
                                    utf8_locale: "C.UTF-8".to_owned(),
                                },
                            },
                    ),
                    (
                            VariantKind::DEBIAN11,
                            Variant {
                                kind: VariantKind::DEBIAN11,
                                descr: "Debian 11.x (bullseye)".to_owned(),
                                family: "debian".to_owned(),
                                parent: "DEBIAN12".to_owned(),
                                detect: Detect {
                                    filename: "/etc/os-release".to_owned(),
                                    regex: r"^
                    PRETTY_NAME= .*
                    Debian \s+ GNU/Linux \s+
                    (?: bullseye | 11 ) (?: \s | / )
                ".to_owned(),
                                    os_id: "debian".to_owned(),
                                    os_version_regex: r"^11$".to_owned(),
                                },
                                commands: HashMap::from(
                                    [
                                        (
                                            "package".to_owned(),
                                            HashMap::from(
                                                [
                                                    (
                                                        "install".to_owned(),
                                                        vec![
                                                            "env".to_owned(),
                                                            "DEBIAN_FRONTEND=noninteractive".to_owned(),
                                                            "apt-get".to_owned(),
                                                            "-q".to_owned(),
                                                            "-y".to_owned(),
                                                            "--no-install-recommends".to_owned(),
                                                            "install".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "list_all".to_owned(),
                                                        vec![
                                                            "dpkg-query".to_owned(),
                                                            "-W".to_owned(),
                                                            "-f".to_owned(),
                                                            "${Package}\\t${Version}\\t${Architecture}\\t${db:Status-Abbrev}\\n".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "purge".to_owned(),
                                                        vec![
                                                            "env".to_owned(),
                                                            "DEBIAN_FRONTEND=noninteractive".to_owned(),
                                                            "apt-get".to_owned(),
                                                            "-q".to_owned(),
                                                            "-y".to_owned(),
                                                            "purge".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "remove".to_owned(),
                                                        vec![
                                                            "env".to_owned(),
                                                            "DEBIAN_FRONTEND=noninteractive".to_owned(),
                                                            "apt-get".to_owned(),
                                                            "-q".to_owned(),
                                                            "-y".to_owned(),
                                                            "remove".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "remove_impl".to_owned(),
                                                        vec![
                                                            "env".to_owned(),
                                                            "DEBIAN_FRONTEND=noninteractive".to_owned(),
                                                            "dpkg".to_owned(),
                                                            "-r".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "update_db".to_owned(),
                                                        vec![
                                                            "apt-get".to_owned(),
                                                            "-q".to_owned(),
                                                            "-y".to_owned(),
                                                            "update".to_owned(),
                                                        ],
                                                    ),
                                                ]
                                            ),
                                        ),
                                        (
                                            "pkgfile".to_owned(),
                                            HashMap::from(
                                                [
                                                    (
                                                        "dep_query".to_owned(),
                                                        vec![
                                                            "sh".to_owned(),
                                                            "-c".to_owned(),
                                                            "dpkg-deb -f -- \"$pkg\" 'Depends' | sed -e 's/ *, */,/g' | tr ',' \"\\n\"".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "install".to_owned(),
                                                        vec![
                                                            "sh".to_owned(),
                                                            "-c".to_owned(),
                                                            "env DEBIAN_FRONTEND=noninteractive apt-get install --no-install-recommends --reinstall -y -o DPkg::Options::=--force-confnew -- $packages".to_owned(),
                                                        ],
                                                    ),
                                                ]
                                            ),
                                        ),
                                    ]
                                ),
                                min_sys_python: "3.9".to_owned(),
                                repo:
                                    Repo::Deb(DebRepo {
                                        codename: "bullseye".to_owned(),
                                        vendor: "debian".to_owned(),
                                        sources: "debian/repo/storpool.sources".to_owned(),
                                        keyring: "debian/repo/storpool-keyring.gpg".to_owned(),
                                        req_packages: vec![
                                            "ca-certificates".to_owned(),
                                        ],
                                    }),

                                package: HashMap::from(
                                    [
                                        ("BINDINGS_PYTHON".to_owned(), "python3".to_owned()),
                                        ("BINDINGS_PYTHON_CONFGET".to_owned(), "python3-confget".to_owned()),
                                        ("BINDINGS_PYTHON_SIMPLEJSON".to_owned(), "python3-simplejson".to_owned()),
                                        ("CGROUP".to_owned(), "cgroup-tools".to_owned()),
                                        ("CPUPOWER".to_owned(), "linux-cpupower".to_owned()),
                                        ("LIBSSL".to_owned(), "libssl1.1".to_owned()),
                                        ("MCELOG".to_owned(), "mcelog".to_owned()),
                                    ]
                                ),
                                systemd_lib: "lib/systemd/system".to_owned(),
                                file_ext: "deb".to_owned(),
                                initramfs_flavor: "update-initramfs".to_owned(),
                                builder: Builder {
                                    alias: "debian11".to_owned(),
                                    base_image: "debian:bullseye".to_owned(),
                                    branch: "debian/bullseye".to_owned(),
                                    kernel_package: "linux-headers".to_owned(),
                                    utf8_locale: "C.UTF-8".to_owned(),
                                },
                            },
                    ),
                    (
                            VariantKind::DEBIAN12,
                            Variant {
                                kind: VariantKind::DEBIAN12,
                                descr: "Debian 12.x (bookworm/unstable)".to_owned(),
                                family: "debian".to_owned(),
                                parent: "".to_owned(),
                                detect: Detect {
                                    filename: "/etc/os-release".to_owned(),
                                    regex: r"^
                    PRETTY_NAME= .*
                    Debian \s+ GNU/Linux \s+
                    (?: bookworm | 12 ) (?: \s | / )
                ".to_owned(),
                                    os_id: "debian".to_owned(),
                                    os_version_regex: r"^12$".to_owned(),
                                },
                                commands: HashMap::from(
                                    [
                                        (
                                            "package".to_owned(),
                                            HashMap::from(
                                                [
                                                    (
                                                        "install".to_owned(),
                                                        vec![
                                                            "env".to_owned(),
                                                            "DEBIAN_FRONTEND=noninteractive".to_owned(),
                                                            "apt-get".to_owned(),
                                                            "-q".to_owned(),
                                                            "-y".to_owned(),
                                                            "--no-install-recommends".to_owned(),
                                                            "install".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "list_all".to_owned(),
                                                        vec![
                                                            "dpkg-query".to_owned(),
                                                            "-W".to_owned(),
                                                            "-f".to_owned(),
                                                            "${Package}\\t${Version}\\t${Architecture}\\t${db:Status-Abbrev}\\n".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "purge".to_owned(),
                                                        vec![
                                                            "env".to_owned(),
                                                            "DEBIAN_FRONTEND=noninteractive".to_owned(),
                                                            "apt-get".to_owned(),
                                                            "-q".to_owned(),
                                                            "-y".to_owned(),
                                                            "purge".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "remove".to_owned(),
                                                        vec![
                                                            "env".to_owned(),
                                                            "DEBIAN_FRONTEND=noninteractive".to_owned(),
                                                            "apt-get".to_owned(),
                                                            "-q".to_owned(),
                                                            "-y".to_owned(),
                                                            "remove".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "remove_impl".to_owned(),
                                                        vec![
                                                            "env".to_owned(),
                                                            "DEBIAN_FRONTEND=noninteractive".to_owned(),
                                                            "dpkg".to_owned(),
                                                            "-r".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "update_db".to_owned(),
                                                        vec![
                                                            "apt-get".to_owned(),
                                                            "-q".to_owned(),
                                                            "-y".to_owned(),
                                                            "update".to_owned(),
                                                        ],
                                                    ),
                                                ]
                                            ),
                                        ),
                                        (
                                            "pkgfile".to_owned(),
                                            HashMap::from(
                                                [
                                                    (
                                                        "dep_query".to_owned(),
                                                        vec![
                                                            "sh".to_owned(),
                                                            "-c".to_owned(),
                                                            "dpkg-deb -f -- \"$pkg\" 'Depends' | sed -e 's/ *, */,/g' | tr ',' \"\\n\"".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "install".to_owned(),
                                                        vec![
                                                            "sh".to_owned(),
                                                            "-c".to_owned(),
                                                            "env DEBIAN_FRONTEND=noninteractive apt-get install --no-install-recommends --reinstall -y -o DPkg::Options::=--force-confnew -- $packages".to_owned(),
                                                        ],
                                                    ),
                                                ]
                                            ),
                                        ),
                                    ]
                                ),
                                min_sys_python: "3.9".to_owned(),
                                repo:
                                    Repo::Deb(DebRepo {
                                        codename: "unstable".to_owned(),
                                        vendor: "debian".to_owned(),
                                        sources: "debian/repo/storpool.sources".to_owned(),
                                        keyring: "debian/repo/storpool-keyring.gpg".to_owned(),
                                        req_packages: vec![
                                            "ca-certificates".to_owned(),
                                        ],
                                    }),

                                package: HashMap::from(
                                    [
                                        ("BINDINGS_PYTHON".to_owned(), "python3".to_owned()),
                                        ("BINDINGS_PYTHON_CONFGET".to_owned(), "python3-confget".to_owned()),
                                        ("BINDINGS_PYTHON_SIMPLEJSON".to_owned(), "python3-simplejson".to_owned()),
                                        ("CGROUP".to_owned(), "cgroup-tools".to_owned()),
                                        ("CPUPOWER".to_owned(), "linux-cpupower".to_owned()),
                                        ("LIBSSL".to_owned(), "libssl1.1".to_owned()),
                                        ("MCELOG".to_owned(), "mcelog".to_owned()),
                                    ]
                                ),
                                systemd_lib: "lib/systemd/system".to_owned(),
                                file_ext: "deb".to_owned(),
                                initramfs_flavor: "update-initramfs".to_owned(),
                                builder: Builder {
                                    alias: "debian12".to_owned(),
                                    base_image: "debian:unstable".to_owned(),
                                    branch: "debian/unstable".to_owned(),
                                    kernel_package: "linux-headers".to_owned(),
                                    utf8_locale: "C.UTF-8".to_owned(),
                                },
                            },
                    ),
                    (
                            VariantKind::ORACLE7,
                            Variant {
                                kind: VariantKind::ORACLE7,
                                descr: "Oracle Linux 7.x".to_owned(),
                                family: "redhat".to_owned(),
                                parent: "CENTOS7".to_owned(),
                                detect: Detect {
                                    filename: "/etc/oracle-release".to_owned(),
                                    regex: r"^ Oracle \s+ Linux \s .* \s 7 \.".to_owned(),
                                    os_id: "ol".to_owned(),
                                    os_version_regex: r"^7(?:$|\.[0-9])".to_owned(),
                                },
                                commands: HashMap::from(
                                    [
                                        (
                                            "package".to_owned(),
                                            HashMap::from(
                                                [
                                                    (
                                                        "install".to_owned(),
                                                        vec![
                                                            "yum".to_owned(),
                                                            "--enablerepo=storpool-contrib".to_owned(),
                                                            "install".to_owned(),
                                                            "-q".to_owned(),
                                                            "-y".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "list_all".to_owned(),
                                                        vec![
                                                            "rpm".to_owned(),
                                                            "-qa".to_owned(),
                                                            "--qf".to_owned(),
                                                            "%{Name}\\t%{EVR}\\t%{Arch}\\tii\\n".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "purge".to_owned(),
                                                        vec![
                                                            "yum".to_owned(),
                                                            "remove".to_owned(),
                                                            "-q".to_owned(),
                                                            "-y".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "remove".to_owned(),
                                                        vec![
                                                            "yum".to_owned(),
                                                            "remove".to_owned(),
                                                            "-q".to_owned(),
                                                            "-y".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "remove_impl".to_owned(),
                                                        vec![
                                                            "rpm".to_owned(),
                                                            "-e".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "update_db".to_owned(),
                                                        vec![
                                                            "true".to_owned(),
                                                        ],
                                                    ),
                                                ]
                                            ),
                                        ),
                                        (
                                            "pkgfile".to_owned(),
                                            HashMap::from(
                                                [
                                                    (
                                                        "dep_query".to_owned(),
                                                        vec![
                                                            "sh".to_owned(),
                                                            "-c".to_owned(),
                                                            "rpm -qpR -- \"$pkg\"".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "install".to_owned(),
                                                        vec![
                                                            "
unset to_install to_reinstall
for f in $packages; do
    package=\"$(rpm -qp \"$f\")\"
    if rpm -q -- \"$package\"; then
        to_reinstall=\"$to_reinstall ./$f\"
    else
        to_install=\"$to_install ./$f\"
    fi
done

if [ -n \"$to_install\" ]; then
    yum install -y --enablerepo=storpool-contrib --setopt=localpkg_gpgcheck=0 -- $to_install
fi
if [ -n \"$to_reinstall\" ]; then
    yum reinstall -y --enablerepo=storpool-contrib --setopt=localpkg_gpgcheck=0 -- $to_reinstall
fi
".to_owned(),
                                                        ],
                                                    ),
                                                ]
                                            ),
                                        ),
                                    ]
                                ),
                                min_sys_python: "2.7".to_owned(),
                                repo:
                                    Repo::Yum(YumRepo {
                                        yumdef: "redhat/repo/storpool-centos.repo".to_owned(),
                                        keyring: "redhat/repo/RPM-GPG-KEY-StorPool".to_owned(),
                                    }),

                                package: HashMap::from(
                                    [
                                        ("KMOD".to_owned(), "kmod".to_owned()),
                                        ("LIBCGROUP".to_owned(), "libcgroup-tools".to_owned()),
                                        ("LIBUDEV".to_owned(), "systemd-libs".to_owned()),
                                        ("OPENSSL".to_owned(), "openssl-libs".to_owned()),
                                        ("PERL_AUTODIE".to_owned(), "perl-autodie".to_owned()),
                                        ("PERL_FILE_PATH".to_owned(), "perl-File-Path".to_owned()),
                                        ("PERL_LWP_PROTO_HTTPS".to_owned(), "perl-LWP-Protocol-https".to_owned()),
                                        ("PERL_SYS_SYSLOG".to_owned(), "perl-Sys-Syslog".to_owned()),
                                        ("PROCPS".to_owned(), "procps-ng".to_owned()),
                                        ("PYTHON_SIMPLEJSON".to_owned(), "python2-simplejson".to_owned()),
                                        ("UDEV".to_owned(), "systemd".to_owned()),
                                    ]
                                ),
                                systemd_lib: "usr/lib/systemd/system".to_owned(),
                                file_ext: "rpm".to_owned(),
                                initramfs_flavor: "mkinitrd".to_owned(),
                                builder: Builder {
                                    alias: "oracle7".to_owned(),
                                    base_image: "IGNORE".to_owned(),
                                    branch: "".to_owned(),
                                    kernel_package: "kernel".to_owned(),
                                    utf8_locale: "C".to_owned(),
                                },
                            },
                    ),
                    (
                            VariantKind::RHEL8,
                            Variant {
                                kind: VariantKind::RHEL8,
                                descr: "RedHat Enterprise Linux 8.x".to_owned(),
                                family: "redhat".to_owned(),
                                parent: "CENTOS8".to_owned(),
                                detect: Detect {
                                    filename: "/etc/redhat-release".to_owned(),
                                    regex: r"^ Red \s+ Hat \s+ Enterprise \s+ Linux \s .* \s 8 \. (?: [4-9] | [1-9][0-9] )".to_owned(),
                                    os_id: "rhel".to_owned(),
                                    os_version_regex: r"^8(?:$|\.[4-9]|\.[1-9][0-9])".to_owned(),
                                },
                                commands: HashMap::from(
                                    [
                                        (
                                            "package".to_owned(),
                                            HashMap::from(
                                                [
                                                    (
                                                        "install".to_owned(),
                                                        vec![
                                                            "dnf".to_owned(),
                                                            "--enablerepo=storpool-contrib".to_owned(),
                                                            "--enablerepo=codeready-builder-for-rhel-8-x86_64-rpms".to_owned(),
                                                            "install".to_owned(),
                                                            "-q".to_owned(),
                                                            "-y".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "list_all".to_owned(),
                                                        vec![
                                                            "rpm".to_owned(),
                                                            "-qa".to_owned(),
                                                            "--qf".to_owned(),
                                                            "%{Name}\\t%{EVR}\\t%{Arch}\\tii\\n".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "purge".to_owned(),
                                                        vec![
                                                            "yum".to_owned(),
                                                            "remove".to_owned(),
                                                            "-q".to_owned(),
                                                            "-y".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "remove".to_owned(),
                                                        vec![
                                                            "yum".to_owned(),
                                                            "remove".to_owned(),
                                                            "-q".to_owned(),
                                                            "-y".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "remove_impl".to_owned(),
                                                        vec![
                                                            "rpm".to_owned(),
                                                            "-e".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "update_db".to_owned(),
                                                        vec![
                                                            "true".to_owned(),
                                                        ],
                                                    ),
                                                ]
                                            ),
                                        ),
                                        (
                                            "pkgfile".to_owned(),
                                            HashMap::from(
                                                [
                                                    (
                                                        "dep_query".to_owned(),
                                                        vec![
                                                            "sh".to_owned(),
                                                            "-c".to_owned(),
                                                            "rpm -qpR -- \"$pkg\"".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "install".to_owned(),
                                                        vec![
                                                            "sh".to_owned(),
                                                            "-c".to_owned(),
                                                            "
unset to_install to_reinstall
for f in $packages; do
    package=\"$(rpm -qp \"$f\")\"
    if rpm -q -- \"$package\"; then
        to_reinstall=\"$to_reinstall ./$f\"
    else
        to_install=\"$to_install ./$f\"
    fi
done

if [ -n \"$to_install\" ]; then
    dnf install -y --enablerepo=storpool-contrib,codeready-builder-for-rhel-8-x86_64-rpms --setopt=localpkg_gpgcheck=0 -- $to_install
fi
if [ -n \"$to_reinstall\" ]; then
    dnf reinstall -y --enablerepo=storpool-contrib,codeready-builder-for-rhel-8-x86_64-rpms --setopt=localpkg_gpgcheck=0 -- $to_reinstall
fi
".to_owned(),
                                                        ],
                                                    ),
                                                ]
                                            ),
                                        ),
                                    ]
                                ),
                                min_sys_python: "2.7".to_owned(),
                                repo:
                                    Repo::Yum(YumRepo {
                                        yumdef: "redhat/repo/storpool-centos.repo".to_owned(),
                                        keyring: "redhat/repo/RPM-GPG-KEY-StorPool".to_owned(),
                                    }),

                                package: HashMap::from(
                                    [
                                        ("KMOD".to_owned(), "kmod".to_owned()),
                                        ("LIBCGROUP".to_owned(), "libcgroup-tools".to_owned()),
                                        ("LIBUDEV".to_owned(), "systemd-libs".to_owned()),
                                        ("OPENSSL".to_owned(), "openssl-libs".to_owned()),
                                        ("PERL_AUTODIE".to_owned(), "perl-autodie".to_owned()),
                                        ("PERL_FILE_PATH".to_owned(), "perl-File-Path".to_owned()),
                                        ("PERL_LWP_PROTO_HTTPS".to_owned(), "perl-LWP-Protocol-https".to_owned()),
                                        ("PERL_SYS_SYSLOG".to_owned(), "perl-Sys-Syslog".to_owned()),
                                        ("PROCPS".to_owned(), "procps-ng".to_owned()),
                                        ("PYTHON_SIMPLEJSON".to_owned(), "python2-simplejson".to_owned()),
                                        ("UDEV".to_owned(), "systemd".to_owned()),
                                    ]
                                ),
                                systemd_lib: "usr/lib/systemd/system".to_owned(),
                                file_ext: "rpm".to_owned(),
                                initramfs_flavor: "mkinitrd".to_owned(),
                                builder: Builder {
                                    alias: "rhel8".to_owned(),
                                    base_image: "redhat/ubi8:reg".to_owned(),
                                    branch: "".to_owned(),
                                    kernel_package: "kernel-core".to_owned(),
                                    utf8_locale: "C.utf8".to_owned(),
                                },
                            },
                    ),
                    (
                            VariantKind::ROCKY8,
                            Variant {
                                kind: VariantKind::ROCKY8,
                                descr: "Rocky Linux 8.x".to_owned(),
                                family: "redhat".to_owned(),
                                parent: "CENTOS8".to_owned(),
                                detect: Detect {
                                    filename: "/etc/redhat-release".to_owned(),
                                    regex: r"^ Rocky \s+ Linux \s .* \s 8 \. (?: [4-9] | [1-9][0-9] )".to_owned(),
                                    os_id: "rocky".to_owned(),
                                    os_version_regex: r"^8(?:$|\.[4-9]|\.[1-9][0-9])".to_owned(),
                                },
                                commands: HashMap::from(
                                    [
                                        (
                                            "package".to_owned(),
                                            HashMap::from(
                                                [
                                                    (
                                                        "install".to_owned(),
                                                        vec![
                                                            "dnf".to_owned(),
                                                            "--enablerepo=storpool-contrib".to_owned(),
                                                            "--enablerepo=powertools".to_owned(),
                                                            "install".to_owned(),
                                                            "-q".to_owned(),
                                                            "-y".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "list_all".to_owned(),
                                                        vec![
                                                            "rpm".to_owned(),
                                                            "-qa".to_owned(),
                                                            "--qf".to_owned(),
                                                            "%{Name}\\t%{EVR}\\t%{Arch}\\tii\\n".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "purge".to_owned(),
                                                        vec![
                                                            "yum".to_owned(),
                                                            "remove".to_owned(),
                                                            "-q".to_owned(),
                                                            "-y".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "remove".to_owned(),
                                                        vec![
                                                            "yum".to_owned(),
                                                            "remove".to_owned(),
                                                            "-q".to_owned(),
                                                            "-y".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "remove_impl".to_owned(),
                                                        vec![
                                                            "rpm".to_owned(),
                                                            "-e".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "update_db".to_owned(),
                                                        vec![
                                                            "true".to_owned(),
                                                        ],
                                                    ),
                                                ]
                                            ),
                                        ),
                                        (
                                            "pkgfile".to_owned(),
                                            HashMap::from(
                                                [
                                                    (
                                                        "dep_query".to_owned(),
                                                        vec![
                                                            "sh".to_owned(),
                                                            "-c".to_owned(),
                                                            "rpm -qpR -- \"$pkg\"".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "install".to_owned(),
                                                        vec![
                                                            "sh".to_owned(),
                                                            "-c".to_owned(),
                                                            "
unset to_install to_reinstall
for f in $packages; do
    package=\"$(rpm -qp \"$f\")\"
    if rpm -q -- \"$package\"; then
        to_reinstall=\"$to_reinstall ./$f\"
    else
        to_install=\"$to_install ./$f\"
    fi
done

if [ -n \"$to_install\" ]; then
    dnf install -y --enablerepo=storpool-contrib,powertools --setopt=localpkg_gpgcheck=0 -- $to_install
fi
if [ -n \"$to_reinstall\" ]; then
    dnf reinstall -y --enablerepo=storpool-contrib,powertools --setopt=localpkg_gpgcheck=0 -- $to_reinstall
fi
".to_owned(),
                                                        ],
                                                    ),
                                                ]
                                            ),
                                        ),
                                    ]
                                ),
                                min_sys_python: "2.7".to_owned(),
                                repo:
                                    Repo::Yum(YumRepo {
                                        yumdef: "redhat/repo/storpool-centos.repo".to_owned(),
                                        keyring: "redhat/repo/RPM-GPG-KEY-StorPool".to_owned(),
                                    }),

                                package: HashMap::from(
                                    [
                                        ("KMOD".to_owned(), "kmod".to_owned()),
                                        ("LIBCGROUP".to_owned(), "libcgroup-tools".to_owned()),
                                        ("LIBUDEV".to_owned(), "systemd-libs".to_owned()),
                                        ("OPENSSL".to_owned(), "openssl-libs".to_owned()),
                                        ("PERL_AUTODIE".to_owned(), "perl-autodie".to_owned()),
                                        ("PERL_FILE_PATH".to_owned(), "perl-File-Path".to_owned()),
                                        ("PERL_LWP_PROTO_HTTPS".to_owned(), "perl-LWP-Protocol-https".to_owned()),
                                        ("PERL_SYS_SYSLOG".to_owned(), "perl-Sys-Syslog".to_owned()),
                                        ("PROCPS".to_owned(), "procps-ng".to_owned()),
                                        ("PYTHON_SIMPLEJSON".to_owned(), "python2-simplejson".to_owned()),
                                        ("UDEV".to_owned(), "systemd".to_owned()),
                                    ]
                                ),
                                systemd_lib: "usr/lib/systemd/system".to_owned(),
                                file_ext: "rpm".to_owned(),
                                initramfs_flavor: "mkinitrd".to_owned(),
                                builder: Builder {
                                    alias: "rocky8".to_owned(),
                                    base_image: "rockylinux/rockylinux:8".to_owned(),
                                    branch: "".to_owned(),
                                    kernel_package: "kernel-core".to_owned(),
                                    utf8_locale: "C.utf8".to_owned(),
                                },
                            },
                    ),
                    (
                            VariantKind::UBUNTU1604,
                            Variant {
                                kind: VariantKind::UBUNTU1604,
                                descr: "Ubuntu 16.04 LTS (Xenial Xerus)".to_owned(),
                                family: "debian".to_owned(),
                                parent: "UBUNTU1804".to_owned(),
                                detect: Detect {
                                    filename: "/etc/os-release".to_owned(),
                                    regex: r"^ PRETTY_NAME= .* Ubuntu \s+ 16 \. 04 ".to_owned(),
                                    os_id: "ubuntu".to_owned(),
                                    os_version_regex: r"^16\.04$".to_owned(),
                                },
                                commands: HashMap::from(
                                    [
                                        (
                                            "package".to_owned(),
                                            HashMap::from(
                                                [
                                                    (
                                                        "install".to_owned(),
                                                        vec![
                                                            "env".to_owned(),
                                                            "DEBIAN_FRONTEND=noninteractive".to_owned(),
                                                            "apt-get".to_owned(),
                                                            "-q".to_owned(),
                                                            "-y".to_owned(),
                                                            "--no-install-recommends".to_owned(),
                                                            "install".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "list_all".to_owned(),
                                                        vec![
                                                            "dpkg-query".to_owned(),
                                                            "-W".to_owned(),
                                                            "-f".to_owned(),
                                                            "${Package}\\t${Version}\\t${Architecture}\\t${db:Status-Abbrev}\\n".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "purge".to_owned(),
                                                        vec![
                                                            "env".to_owned(),
                                                            "DEBIAN_FRONTEND=noninteractive".to_owned(),
                                                            "apt-get".to_owned(),
                                                            "-q".to_owned(),
                                                            "-y".to_owned(),
                                                            "purge".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "remove".to_owned(),
                                                        vec![
                                                            "env".to_owned(),
                                                            "DEBIAN_FRONTEND=noninteractive".to_owned(),
                                                            "apt-get".to_owned(),
                                                            "-q".to_owned(),
                                                            "-y".to_owned(),
                                                            "remove".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "remove_impl".to_owned(),
                                                        vec![
                                                            "env".to_owned(),
                                                            "DEBIAN_FRONTEND=noninteractive".to_owned(),
                                                            "dpkg".to_owned(),
                                                            "-r".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "update_db".to_owned(),
                                                        vec![
                                                            "apt-get".to_owned(),
                                                            "-q".to_owned(),
                                                            "-y".to_owned(),
                                                            "update".to_owned(),
                                                        ],
                                                    ),
                                                ]
                                            ),
                                        ),
                                        (
                                            "pkgfile".to_owned(),
                                            HashMap::from(
                                                [
                                                    (
                                                        "dep_query".to_owned(),
                                                        vec![
                                                            "sh".to_owned(),
                                                            "-c".to_owned(),
                                                            "dpkg-deb -f -- \"$pkg\" 'Depends' | sed -e 's/ *, */,/g' | tr ',' \"\\n\"".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "install".to_owned(),
                                                        vec![
                                                            "sh".to_owned(),
                                                            "-c".to_owned(),
                                                            "env DEBIAN_FRONTEND=noninteractive apt-get install --no-install-recommends --reinstall -y -o DPkg::Options::=--force-confnew -- $packages".to_owned(),
                                                        ],
                                                    ),
                                                ]
                                            ),
                                        ),
                                    ]
                                ),
                                min_sys_python: "2.7".to_owned(),
                                repo:
                                    Repo::Deb(DebRepo {
                                        codename: "xenial".to_owned(),
                                        vendor: "ubuntu".to_owned(),
                                        sources: "debian/repo/storpool.sources".to_owned(),
                                        keyring: "debian/repo/storpool-keyring.gpg".to_owned(),
                                        req_packages: vec![
                                            "apt-transport-https".to_owned(),
                                            "ca-certificates".to_owned(),
                                        ],
                                    }),

                                package: HashMap::from(
                                    [
                                        ("BINDINGS_PYTHON".to_owned(), "python".to_owned()),
                                        ("BINDINGS_PYTHON_CONFGET".to_owned(), "python-confget".to_owned()),
                                        ("BINDINGS_PYTHON_SIMPLEJSON".to_owned(), "python-simplejson".to_owned()),
                                        ("CGROUP".to_owned(), "cgroup-tools".to_owned()),
                                        ("CPUPOWER".to_owned(), "linux-tools-generic".to_owned()),
                                        ("LIBSSL".to_owned(), "libssl1.0.0".to_owned()),
                                        ("MCELOG".to_owned(), "bash".to_owned()),
                                        ("mcelog".to_owned(), "mcelog".to_owned()),
                                    ]
                                ),
                                systemd_lib: "lib/systemd/system".to_owned(),
                                file_ext: "deb".to_owned(),
                                initramfs_flavor: "update-initramfs".to_owned(),
                                builder: Builder {
                                    alias: "ubuntu-16.04".to_owned(),
                                    base_image: "ubuntu:xenial".to_owned(),
                                    branch: "ubuntu/xenial".to_owned(),
                                    kernel_package: "linux-headers".to_owned(),
                                    utf8_locale: "C.UTF-8".to_owned(),
                                },
                            },
                    ),
                    (
                            VariantKind::UBUNTU1804,
                            Variant {
                                kind: VariantKind::UBUNTU1804,
                                descr: "Ubuntu 18.04 LTS (Bionic Beaver)".to_owned(),
                                family: "debian".to_owned(),
                                parent: "UBUNTU2004".to_owned(),
                                detect: Detect {
                                    filename: "/etc/os-release".to_owned(),
                                    regex: r"^ PRETTY_NAME= .* Ubuntu \s+ 18 \. 04 ".to_owned(),
                                    os_id: "ubuntu".to_owned(),
                                    os_version_regex: r"^18\.04$".to_owned(),
                                },
                                commands: HashMap::from(
                                    [
                                        (
                                            "package".to_owned(),
                                            HashMap::from(
                                                [
                                                    (
                                                        "install".to_owned(),
                                                        vec![
                                                            "env".to_owned(),
                                                            "DEBIAN_FRONTEND=noninteractive".to_owned(),
                                                            "apt-get".to_owned(),
                                                            "-q".to_owned(),
                                                            "-y".to_owned(),
                                                            "--no-install-recommends".to_owned(),
                                                            "install".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "list_all".to_owned(),
                                                        vec![
                                                            "dpkg-query".to_owned(),
                                                            "-W".to_owned(),
                                                            "-f".to_owned(),
                                                            "${Package}\\t${Version}\\t${Architecture}\\t${db:Status-Abbrev}\\n".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "purge".to_owned(),
                                                        vec![
                                                            "env".to_owned(),
                                                            "DEBIAN_FRONTEND=noninteractive".to_owned(),
                                                            "apt-get".to_owned(),
                                                            "-q".to_owned(),
                                                            "-y".to_owned(),
                                                            "purge".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "remove".to_owned(),
                                                        vec![
                                                            "env".to_owned(),
                                                            "DEBIAN_FRONTEND=noninteractive".to_owned(),
                                                            "apt-get".to_owned(),
                                                            "-q".to_owned(),
                                                            "-y".to_owned(),
                                                            "remove".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "remove_impl".to_owned(),
                                                        vec![
                                                            "env".to_owned(),
                                                            "DEBIAN_FRONTEND=noninteractive".to_owned(),
                                                            "dpkg".to_owned(),
                                                            "-r".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "update_db".to_owned(),
                                                        vec![
                                                            "apt-get".to_owned(),
                                                            "-q".to_owned(),
                                                            "-y".to_owned(),
                                                            "update".to_owned(),
                                                        ],
                                                    ),
                                                ]
                                            ),
                                        ),
                                        (
                                            "pkgfile".to_owned(),
                                            HashMap::from(
                                                [
                                                    (
                                                        "dep_query".to_owned(),
                                                        vec![
                                                            "sh".to_owned(),
                                                            "-c".to_owned(),
                                                            "dpkg-deb -f -- \"$pkg\" 'Depends' | sed -e 's/ *, */,/g' | tr ',' \"\\n\"".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "install".to_owned(),
                                                        vec![
                                                            "sh".to_owned(),
                                                            "-c".to_owned(),
                                                            "env DEBIAN_FRONTEND=noninteractive apt-get install --no-install-recommends --reinstall -y -o DPkg::Options::=--force-confnew -- $packages".to_owned(),
                                                        ],
                                                    ),
                                                ]
                                            ),
                                        ),
                                    ]
                                ),
                                min_sys_python: "2.7".to_owned(),
                                repo:
                                    Repo::Deb(DebRepo {
                                        codename: "bionic".to_owned(),
                                        vendor: "ubuntu".to_owned(),
                                        sources: "debian/repo/storpool.sources".to_owned(),
                                        keyring: "debian/repo/storpool-keyring.gpg".to_owned(),
                                        req_packages: vec![
                                            "ca-certificates".to_owned(),
                                        ],
                                    }),

                                package: HashMap::from(
                                    [
                                        ("BINDINGS_PYTHON".to_owned(), "python".to_owned()),
                                        ("BINDINGS_PYTHON_CONFGET".to_owned(), "python-confget".to_owned()),
                                        ("BINDINGS_PYTHON_SIMPLEJSON".to_owned(), "python-simplejson".to_owned()),
                                        ("CGROUP".to_owned(), "cgroup-tools".to_owned()),
                                        ("CPUPOWER".to_owned(), "linux-tools-generic".to_owned()),
                                        ("LIBSSL".to_owned(), "libssl1.1".to_owned()),
                                        ("MCELOG".to_owned(), "bash".to_owned()),
                                    ]
                                ),
                                systemd_lib: "lib/systemd/system".to_owned(),
                                file_ext: "deb".to_owned(),
                                initramfs_flavor: "update-initramfs".to_owned(),
                                builder: Builder {
                                    alias: "ubuntu-18.04".to_owned(),
                                    base_image: "ubuntu:bionic".to_owned(),
                                    branch: "ubuntu/bionic".to_owned(),
                                    kernel_package: "linux-headers".to_owned(),
                                    utf8_locale: "C.UTF-8".to_owned(),
                                },
                            },
                    ),
                    (
                            VariantKind::UBUNTU2004,
                            Variant {
                                kind: VariantKind::UBUNTU2004,
                                descr: "Ubuntu 20.04 LTS (Focal Fossa)".to_owned(),
                                family: "debian".to_owned(),
                                parent: "UBUNTU2110".to_owned(),
                                detect: Detect {
                                    filename: "/etc/os-release".to_owned(),
                                    regex: r"^ PRETTY_NAME= .* (?: Ubuntu \s+ 20 \. 04 | Mint \s+ 20 ) ".to_owned(),
                                    os_id: "ubuntu".to_owned(),
                                    os_version_regex: r"^20\.04$".to_owned(),
                                },
                                commands: HashMap::from(
                                    [
                                        (
                                            "package".to_owned(),
                                            HashMap::from(
                                                [
                                                    (
                                                        "install".to_owned(),
                                                        vec![
                                                            "env".to_owned(),
                                                            "DEBIAN_FRONTEND=noninteractive".to_owned(),
                                                            "apt-get".to_owned(),
                                                            "-q".to_owned(),
                                                            "-y".to_owned(),
                                                            "--no-install-recommends".to_owned(),
                                                            "install".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "list_all".to_owned(),
                                                        vec![
                                                            "dpkg-query".to_owned(),
                                                            "-W".to_owned(),
                                                            "-f".to_owned(),
                                                            "${Package}\\t${Version}\\t${Architecture}\\t${db:Status-Abbrev}\\n".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "purge".to_owned(),
                                                        vec![
                                                            "env".to_owned(),
                                                            "DEBIAN_FRONTEND=noninteractive".to_owned(),
                                                            "apt-get".to_owned(),
                                                            "-q".to_owned(),
                                                            "-y".to_owned(),
                                                            "purge".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "remove".to_owned(),
                                                        vec![
                                                            "env".to_owned(),
                                                            "DEBIAN_FRONTEND=noninteractive".to_owned(),
                                                            "apt-get".to_owned(),
                                                            "-q".to_owned(),
                                                            "-y".to_owned(),
                                                            "remove".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "remove_impl".to_owned(),
                                                        vec![
                                                            "env".to_owned(),
                                                            "DEBIAN_FRONTEND=noninteractive".to_owned(),
                                                            "dpkg".to_owned(),
                                                            "-r".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "update_db".to_owned(),
                                                        vec![
                                                            "apt-get".to_owned(),
                                                            "-q".to_owned(),
                                                            "-y".to_owned(),
                                                            "update".to_owned(),
                                                        ],
                                                    ),
                                                ]
                                            ),
                                        ),
                                        (
                                            "pkgfile".to_owned(),
                                            HashMap::from(
                                                [
                                                    (
                                                        "dep_query".to_owned(),
                                                        vec![
                                                            "sh".to_owned(),
                                                            "-c".to_owned(),
                                                            "dpkg-deb -f -- \"$pkg\" 'Depends' | sed -e 's/ *, */,/g' | tr ',' \"\\n\"".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "install".to_owned(),
                                                        vec![
                                                            "sh".to_owned(),
                                                            "-c".to_owned(),
                                                            "env DEBIAN_FRONTEND=noninteractive apt-get install --no-install-recommends --reinstall -y -o DPkg::Options::=--force-confnew -- $packages".to_owned(),
                                                        ],
                                                    ),
                                                ]
                                            ),
                                        ),
                                    ]
                                ),
                                min_sys_python: "3.8".to_owned(),
                                repo:
                                    Repo::Deb(DebRepo {
                                        codename: "focal".to_owned(),
                                        vendor: "ubuntu".to_owned(),
                                        sources: "debian/repo/storpool.sources".to_owned(),
                                        keyring: "debian/repo/storpool-keyring.gpg".to_owned(),
                                        req_packages: vec![
                                            "ca-certificates".to_owned(),
                                        ],
                                    }),

                                package: HashMap::from(
                                    [
                                        ("BINDINGS_PYTHON".to_owned(), "python3".to_owned()),
                                        ("BINDINGS_PYTHON_CONFGET".to_owned(), "python3-confget".to_owned()),
                                        ("BINDINGS_PYTHON_SIMPLEJSON".to_owned(), "python3-simplejson".to_owned()),
                                        ("CGROUP".to_owned(), "cgroup-tools".to_owned()),
                                        ("CPUPOWER".to_owned(), "linux-tools-generic".to_owned()),
                                        ("LIBSSL".to_owned(), "libssl1.1".to_owned()),
                                        ("MCELOG".to_owned(), "bash".to_owned()),
                                    ]
                                ),
                                systemd_lib: "lib/systemd/system".to_owned(),
                                file_ext: "deb".to_owned(),
                                initramfs_flavor: "update-initramfs".to_owned(),
                                builder: Builder {
                                    alias: "ubuntu-20.04".to_owned(),
                                    base_image: "ubuntu:focal".to_owned(),
                                    branch: "ubuntu/focal".to_owned(),
                                    kernel_package: "linux-headers".to_owned(),
                                    utf8_locale: "C.UTF-8".to_owned(),
                                },
                            },
                    ),
                    (
                            VariantKind::UBUNTU2110,
                            Variant {
                                kind: VariantKind::UBUNTU2110,
                                descr: "Ubuntu 21.10 LTS (Impish Indri)".to_owned(),
                                family: "debian".to_owned(),
                                parent: "UBUNTU2204".to_owned(),
                                detect: Detect {
                                    filename: "/etc/os-release".to_owned(),
                                    regex: r"^ PRETTY_NAME= .* (?: Ubuntu \s+ 21 \. 10 | Mint \s+ 21 ) ".to_owned(),
                                    os_id: "ubuntu".to_owned(),
                                    os_version_regex: r"^21\.10$".to_owned(),
                                },
                                commands: HashMap::from(
                                    [
                                        (
                                            "package".to_owned(),
                                            HashMap::from(
                                                [
                                                    (
                                                        "install".to_owned(),
                                                        vec![
                                                            "env".to_owned(),
                                                            "DEBIAN_FRONTEND=noninteractive".to_owned(),
                                                            "apt-get".to_owned(),
                                                            "-q".to_owned(),
                                                            "-y".to_owned(),
                                                            "--no-install-recommends".to_owned(),
                                                            "install".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "list_all".to_owned(),
                                                        vec![
                                                            "dpkg-query".to_owned(),
                                                            "-W".to_owned(),
                                                            "-f".to_owned(),
                                                            "${Package}\\t${Version}\\t${Architecture}\\t${db:Status-Abbrev}\\n".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "purge".to_owned(),
                                                        vec![
                                                            "env".to_owned(),
                                                            "DEBIAN_FRONTEND=noninteractive".to_owned(),
                                                            "apt-get".to_owned(),
                                                            "-q".to_owned(),
                                                            "-y".to_owned(),
                                                            "purge".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "remove".to_owned(),
                                                        vec![
                                                            "env".to_owned(),
                                                            "DEBIAN_FRONTEND=noninteractive".to_owned(),
                                                            "apt-get".to_owned(),
                                                            "-q".to_owned(),
                                                            "-y".to_owned(),
                                                            "remove".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "remove_impl".to_owned(),
                                                        vec![
                                                            "env".to_owned(),
                                                            "DEBIAN_FRONTEND=noninteractive".to_owned(),
                                                            "dpkg".to_owned(),
                                                            "-r".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "update_db".to_owned(),
                                                        vec![
                                                            "apt-get".to_owned(),
                                                            "-q".to_owned(),
                                                            "-y".to_owned(),
                                                            "update".to_owned(),
                                                        ],
                                                    ),
                                                ]
                                            ),
                                        ),
                                        (
                                            "pkgfile".to_owned(),
                                            HashMap::from(
                                                [
                                                    (
                                                        "dep_query".to_owned(),
                                                        vec![
                                                            "sh".to_owned(),
                                                            "-c".to_owned(),
                                                            "dpkg-deb -f -- \"$pkg\" 'Depends' | sed -e 's/ *, */,/g' | tr ',' \"\\n\"".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "install".to_owned(),
                                                        vec![
                                                            "sh".to_owned(),
                                                            "-c".to_owned(),
                                                            "env DEBIAN_FRONTEND=noninteractive apt-get install --no-install-recommends --reinstall -y -o DPkg::Options::=--force-confnew -- $packages".to_owned(),
                                                        ],
                                                    ),
                                                ]
                                            ),
                                        ),
                                    ]
                                ),
                                min_sys_python: "3.9".to_owned(),
                                repo:
                                    Repo::Deb(DebRepo {
                                        codename: "impish".to_owned(),
                                        vendor: "ubuntu".to_owned(),
                                        sources: "debian/repo/storpool.sources".to_owned(),
                                        keyring: "debian/repo/storpool-keyring.gpg".to_owned(),
                                        req_packages: vec![
                                            "ca-certificates".to_owned(),
                                        ],
                                    }),

                                package: HashMap::from(
                                    [
                                        ("BINDINGS_PYTHON".to_owned(), "python3".to_owned()),
                                        ("BINDINGS_PYTHON_CONFGET".to_owned(), "python3-confget".to_owned()),
                                        ("BINDINGS_PYTHON_SIMPLEJSON".to_owned(), "python3-simplejson".to_owned()),
                                        ("CGROUP".to_owned(), "cgroup-tools".to_owned()),
                                        ("CPUPOWER".to_owned(), "linux-tools-generic".to_owned()),
                                        ("LIBSSL".to_owned(), "libssl1.1".to_owned()),
                                        ("MCELOG".to_owned(), "bash".to_owned()),
                                    ]
                                ),
                                systemd_lib: "lib/systemd/system".to_owned(),
                                file_ext: "deb".to_owned(),
                                initramfs_flavor: "update-initramfs".to_owned(),
                                builder: Builder {
                                    alias: "ubuntu-21.10".to_owned(),
                                    base_image: "ubuntu:impish".to_owned(),
                                    branch: "ubuntu/impish".to_owned(),
                                    kernel_package: "linux-headers".to_owned(),
                                    utf8_locale: "C.UTF-8".to_owned(),
                                },
                            },
                    ),
                    (
                            VariantKind::UBUNTU2204,
                            Variant {
                                kind: VariantKind::UBUNTU2204,
                                descr: "Ubuntu 22.04 LTS (Jammy Jellyfish)".to_owned(),
                                family: "debian".to_owned(),
                                parent: "DEBIAN12".to_owned(),
                                detect: Detect {
                                    filename: "/etc/os-release".to_owned(),
                                    regex: r"^ PRETTY_NAME= .* (?: Ubuntu \s+ 22 \. 04 | Mint \s+ 22 ) ".to_owned(),
                                    os_id: "ubuntu".to_owned(),
                                    os_version_regex: r"^22\.04$".to_owned(),
                                },
                                commands: HashMap::from(
                                    [
                                        (
                                            "package".to_owned(),
                                            HashMap::from(
                                                [
                                                    (
                                                        "install".to_owned(),
                                                        vec![
                                                            "env".to_owned(),
                                                            "DEBIAN_FRONTEND=noninteractive".to_owned(),
                                                            "apt-get".to_owned(),
                                                            "-q".to_owned(),
                                                            "-y".to_owned(),
                                                            "--no-install-recommends".to_owned(),
                                                            "install".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "list_all".to_owned(),
                                                        vec![
                                                            "dpkg-query".to_owned(),
                                                            "-W".to_owned(),
                                                            "-f".to_owned(),
                                                            "${Package}\\t${Version}\\t${Architecture}\\t${db:Status-Abbrev}\\n".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "purge".to_owned(),
                                                        vec![
                                                            "env".to_owned(),
                                                            "DEBIAN_FRONTEND=noninteractive".to_owned(),
                                                            "apt-get".to_owned(),
                                                            "-q".to_owned(),
                                                            "-y".to_owned(),
                                                            "purge".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "remove".to_owned(),
                                                        vec![
                                                            "env".to_owned(),
                                                            "DEBIAN_FRONTEND=noninteractive".to_owned(),
                                                            "apt-get".to_owned(),
                                                            "-q".to_owned(),
                                                            "-y".to_owned(),
                                                            "remove".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "remove_impl".to_owned(),
                                                        vec![
                                                            "env".to_owned(),
                                                            "DEBIAN_FRONTEND=noninteractive".to_owned(),
                                                            "dpkg".to_owned(),
                                                            "-r".to_owned(),
                                                            "--".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "update_db".to_owned(),
                                                        vec![
                                                            "apt-get".to_owned(),
                                                            "-q".to_owned(),
                                                            "-y".to_owned(),
                                                            "update".to_owned(),
                                                        ],
                                                    ),
                                                ]
                                            ),
                                        ),
                                        (
                                            "pkgfile".to_owned(),
                                            HashMap::from(
                                                [
                                                    (
                                                        "dep_query".to_owned(),
                                                        vec![
                                                            "sh".to_owned(),
                                                            "-c".to_owned(),
                                                            "dpkg-deb -f -- \"$pkg\" 'Depends' | sed -e 's/ *, */,/g' | tr ',' \"\\n\"".to_owned(),
                                                        ],
                                                    ),
                                                    (
                                                        "install".to_owned(),
                                                        vec![
                                                            "sh".to_owned(),
                                                            "-c".to_owned(),
                                                            "env DEBIAN_FRONTEND=noninteractive apt-get install --no-install-recommends --reinstall -y -o DPkg::Options::=--force-confnew -- $packages".to_owned(),
                                                        ],
                                                    ),
                                                ]
                                            ),
                                        ),
                                    ]
                                ),
                                min_sys_python: "3.9".to_owned(),
                                repo:
                                    Repo::Deb(DebRepo {
                                        codename: "jammy".to_owned(),
                                        vendor: "ubuntu".to_owned(),
                                        sources: "debian/repo/storpool.sources".to_owned(),
                                        keyring: "debian/repo/storpool-keyring.gpg".to_owned(),
                                        req_packages: vec![
                                            "ca-certificates".to_owned(),
                                        ],
                                    }),

                                package: HashMap::from(
                                    [
                                        ("BINDINGS_PYTHON".to_owned(), "python3".to_owned()),
                                        ("BINDINGS_PYTHON_CONFGET".to_owned(), "python3-confget".to_owned()),
                                        ("BINDINGS_PYTHON_SIMPLEJSON".to_owned(), "python3-simplejson".to_owned()),
                                        ("CGROUP".to_owned(), "cgroup-tools".to_owned()),
                                        ("CPUPOWER".to_owned(), "linux-tools-generic".to_owned()),
                                        ("LIBSSL".to_owned(), "libssl1.1".to_owned()),
                                        ("MCELOG".to_owned(), "bash".to_owned()),
                                    ]
                                ),
                                systemd_lib: "lib/systemd/system".to_owned(),
                                file_ext: "deb".to_owned(),
                                initramfs_flavor: "update-initramfs".to_owned(),
                                builder: Builder {
                                    alias: "ubuntu-22.04".to_owned(),
                                    base_image: "ubuntu:jammy".to_owned(),
                                    branch: "ubuntu/jammy".to_owned(),
                                    kernel_package: "linux-headers".to_owned(),
                                    utf8_locale: "C.UTF-8".to_owned(),
                                },
                            },
                    ),
                ]
            ),
            version: "2.3.2".to_owned(),
        };
    }
    assert!(
        DEF_TOP.format.version.major == 1,
        "Internal error: JSON variant definition: version {:?}",
        DEF_TOP.format.version
    );
    &DEF_TOP
}
