// Take a look at the license at the top of the repository in the LICENSE file.

// rustdoc-stripper-ignore-next
//! Traits intended for subclassing [`GutterRenderer`](crate::GutterRenderer).

use crate::{Buffer, GutterLines, GutterRenderer, View};
use glib::translate::*;
use glib::Cast;
use gtk::subclass::prelude::*;

pub trait GutterRendererImpl: WidgetImpl {
    fn query_data(&self, gutter_renderer: &Self::Type, lines: &GutterLines, line: u32) {
        self.parent_query_data(gutter_renderer, lines, line)
    }

    fn begin(&self, gutter_renderer: &Self::Type, lines: &GutterLines) {
        self.parent_begin(gutter_renderer, lines)
    }

    fn end(&self, gutter_renderer: &Self::Type) {
        self.parent_end(gutter_renderer)
    }

    fn change_buffer(&self, gutter_renderer: &Self::Type, old_buffer: Option<&Buffer>) {
        self.parent_change_buffer(gutter_renderer, old_buffer)
    }

    fn change_view(&self, gutter_renderer: &Self::Type, old_view: Option<&View>) {
        self.parent_change_view(gutter_renderer, old_view)
    }

    fn query_activatable(
        &self,
        gutter_renderer: &Self::Type,
        iter: &gtk::TextIter,
        area: &gdk::Rectangle,
    ) -> bool {
        self.parent_query_activatable(gutter_renderer, iter, area)
    }

    fn activate(
        &self,
        gutter_renderer: &Self::Type,
        iter: &gtk::TextIter,
        area: &gdk::Rectangle,
        button: u32,
        state: gdk::ModifierType,
        n_presses: i32,
    ) {
        self.parent_activate(gutter_renderer, iter, area, button, state, n_presses)
    }

    fn snapshot_line(
        &self,
        gutter_renderer: &Self::Type,
        snapshot: &gtk::Snapshot,
        lines: &GutterLines,
        line: u32,
    ) {
        self.parent_snapshot_line(gutter_renderer, snapshot, lines, line)
    }
}

pub trait GutterRendererImplExt: ObjectSubclass {
    fn parent_query_data(&self, gutter_renderer: &Self::Type, lines: &GutterLines, line: u32);
    fn parent_begin(&self, gutter_renderer: &Self::Type, lines: &GutterLines);
    fn parent_end(&self, gutter_renderer: &Self::Type);
    fn parent_change_buffer(&self, gutter_renderer: &Self::Type, old_buffer: Option<&Buffer>);
    fn parent_change_view(&self, gutter_renderer: &Self::Type, old_view: Option<&View>);
    fn parent_query_activatable(
        &self,
        gutter_renderer: &Self::Type,
        iter: &gtk::TextIter,
        area: &gdk::Rectangle,
    ) -> bool;
    fn parent_activate(
        &self,
        gutter_renderer: &Self::Type,
        iter: &gtk::TextIter,
        area: &gdk::Rectangle,
        button: u32,
        state: gdk::ModifierType,
        n_presses: i32,
    );
    fn parent_snapshot_line(
        &self,
        gutter_renderer: &Self::Type,
        snapshot: &gtk::Snapshot,
        lines: &GutterLines,
        line: u32,
    );
}

impl<T: GutterRendererImpl> GutterRendererImplExt for T {
    fn parent_query_data(&self, gutter_renderer: &Self::Type, lines: &GutterLines, line: u32) {
        unsafe {
            let data = T::type_data();
            let parent_class =
                data.as_ref().parent_class() as *mut ffi::GtkSourceGutterRendererClass;
            if let Some(f) = (*parent_class).query_data {
                f(
                    gutter_renderer
                        .unsafe_cast_ref::<GutterRenderer>()
                        .to_glib_none()
                        .0,
                    lines.to_glib_none().0,
                    line,
                )
            }
        }
    }

    fn parent_begin(&self, gutter_renderer: &Self::Type, lines: &GutterLines) {
        unsafe {
            let data = T::type_data();
            let parent_class =
                data.as_ref().parent_class() as *mut ffi::GtkSourceGutterRendererClass;
            if let Some(f) = (*parent_class).begin {
                f(
                    gutter_renderer
                        .unsafe_cast_ref::<GutterRenderer>()
                        .to_glib_none()
                        .0,
                    lines.to_glib_none().0,
                )
            }
        }
    }

    fn parent_end(&self, gutter_renderer: &Self::Type) {
        unsafe {
            let data = T::type_data();
            let parent_class =
                data.as_ref().parent_class() as *mut ffi::GtkSourceGutterRendererClass;
            if let Some(f) = (*parent_class).end {
                f(gutter_renderer
                    .unsafe_cast_ref::<GutterRenderer>()
                    .to_glib_none()
                    .0)
            }
        }
    }

    fn parent_change_buffer(&self, gutter_renderer: &Self::Type, old_buffer: Option<&Buffer>) {
        unsafe {
            let data = T::type_data();
            let parent_class =
                data.as_ref().parent_class() as *mut ffi::GtkSourceGutterRendererClass;
            if let Some(f) = (*parent_class).change_buffer {
                f(
                    gutter_renderer
                        .unsafe_cast_ref::<GutterRenderer>()
                        .to_glib_none()
                        .0,
                    old_buffer.to_glib_none().0,
                )
            }
        }
    }

    fn parent_change_view(&self, gutter_renderer: &Self::Type, old_view: Option<&View>) {
        unsafe {
            let data = T::type_data();
            let parent_class =
                data.as_ref().parent_class() as *mut ffi::GtkSourceGutterRendererClass;
            if let Some(f) = (*parent_class).change_view {
                f(
                    gutter_renderer
                        .unsafe_cast_ref::<GutterRenderer>()
                        .to_glib_none()
                        .0,
                    old_view.to_glib_none().0,
                )
            }
        }
    }

    fn parent_query_activatable(
        &self,
        gutter_renderer: &Self::Type,
        iter: &gtk::TextIter,
        area: &gdk::Rectangle,
    ) -> bool {
        unsafe {
            let data = T::type_data();
            let parent_class =
                data.as_ref().parent_class() as *mut ffi::GtkSourceGutterRendererClass;
            let f = (*parent_class)
                .query_activatable
                .expect("no parent \"query_activatable\" implementation");
            from_glib(f(
                gutter_renderer
                    .unsafe_cast_ref::<GutterRenderer>()
                    .to_glib_none()
                    .0,
                mut_override(iter.to_glib_none().0),
                mut_override(area.to_glib_none().0),
            ))
        }
    }

    fn parent_activate(
        &self,
        gutter_renderer: &Self::Type,
        iter: &gtk::TextIter,
        area: &gdk::Rectangle,
        button: u32,
        state: gdk::ModifierType,
        n_presses: i32,
    ) {
        unsafe {
            let data = T::type_data();
            let parent_class =
                data.as_ref().parent_class() as *mut ffi::GtkSourceGutterRendererClass;
            if let Some(f) = (*parent_class).activate {
                f(
                    gutter_renderer
                        .unsafe_cast_ref::<GutterRenderer>()
                        .to_glib_none()
                        .0,
                    mut_override(iter.to_glib_none().0),
                    mut_override(area.to_glib_none().0),
                    button,
                    state.into_glib(),
                    n_presses,
                )
            }
        }
    }

    fn parent_snapshot_line(
        &self,
        gutter_renderer: &Self::Type,
        snapshot: &gtk::Snapshot,
        lines: &GutterLines,
        line: u32,
    ) {
        unsafe {
            let data = T::type_data();
            let parent_class =
                data.as_ref().parent_class() as *mut ffi::GtkSourceGutterRendererClass;
            if let Some(f) = (*parent_class).snapshot_line {
                f(
                    gutter_renderer
                        .unsafe_cast_ref::<GutterRenderer>()
                        .to_glib_none()
                        .0,
                    snapshot.to_glib_none().0,
                    lines.to_glib_none().0,
                    line,
                )
            }
        }
    }
}

unsafe impl<T: GutterRendererImpl> IsSubclassable<T> for GutterRenderer {
    fn class_init(class: &mut glib::Class<Self>) {
        Self::parent_class_init::<T>(class);

        let klass = class.as_mut();
        klass.query_data = Some(gutter_renderer_query_data::<T>);
        klass.begin = Some(gutter_renderer_begin::<T>);
        klass.end = Some(gutter_renderer_end::<T>);
        klass.change_buffer = Some(gutter_renderer_change_buffer::<T>);
        klass.change_view = Some(gutter_renderer_change_view::<T>);
        klass.query_activatable = Some(gutter_renderer_query_activatable::<T>);
        klass.activate = Some(gutter_renderer_activate::<T>);
        klass.snapshot_line = Some(gutter_renderer_snapshot_line::<T>);
    }
}

unsafe extern "C" fn gutter_renderer_query_data<T: GutterRendererImpl>(
    ptr: *mut ffi::GtkSourceGutterRenderer,
    lines: *mut ffi::GtkSourceGutterLines,
    line: u32,
) {
    let instance = &*(ptr as *mut T::Instance);
    let imp = instance.imp();
    let wrap: Borrowed<GutterRenderer> = from_glib_borrow(ptr);
    imp.query_data(wrap.unsafe_cast_ref(), &from_glib_borrow(lines), line);
}

unsafe extern "C" fn gutter_renderer_begin<T: GutterRendererImpl>(
    ptr: *mut ffi::GtkSourceGutterRenderer,
    lines: *mut ffi::GtkSourceGutterLines,
) {
    let instance = &*(ptr as *mut T::Instance);
    let imp = instance.imp();
    let wrap: Borrowed<GutterRenderer> = from_glib_borrow(ptr);
    imp.begin(wrap.unsafe_cast_ref(), &from_glib_borrow(lines));
}

unsafe extern "C" fn gutter_renderer_end<T: GutterRendererImpl>(
    ptr: *mut ffi::GtkSourceGutterRenderer,
) {
    let instance = &*(ptr as *mut T::Instance);
    let imp = instance.imp();
    let wrap: Borrowed<GutterRenderer> = from_glib_borrow(ptr);
    imp.end(wrap.unsafe_cast_ref());
}

unsafe extern "C" fn gutter_renderer_change_buffer<T: GutterRendererImpl>(
    ptr: *mut ffi::GtkSourceGutterRenderer,
    buffer: *mut ffi::GtkSourceBuffer,
) {
    let instance = &*(ptr as *mut T::Instance);
    let imp = instance.imp();
    let wrap: Borrowed<GutterRenderer> = from_glib_borrow(ptr);
    let buffer: Borrowed<Option<Buffer>> = from_glib_borrow(buffer);
    imp.change_buffer(wrap.unsafe_cast_ref(), buffer.as_ref().as_ref());
}

unsafe extern "C" fn gutter_renderer_change_view<T: GutterRendererImpl>(
    ptr: *mut ffi::GtkSourceGutterRenderer,
    view: *mut ffi::GtkSourceView,
) {
    let instance = &*(ptr as *mut T::Instance);
    let imp = instance.imp();
    let wrap: Borrowed<GutterRenderer> = from_glib_borrow(ptr);
    let view: Borrowed<Option<View>> = from_glib_borrow(view);
    imp.change_view(wrap.unsafe_cast_ref(), view.as_ref().as_ref());
}

unsafe extern "C" fn gutter_renderer_query_activatable<T: GutterRendererImpl>(
    ptr: *mut ffi::GtkSourceGutterRenderer,
    iter: *mut gtk::ffi::GtkTextIter,
    rect: *mut gdk::ffi::GdkRectangle,
) -> glib::ffi::gboolean {
    let instance = &*(ptr as *mut T::Instance);
    let imp = instance.imp();
    let wrap: Borrowed<GutterRenderer> = from_glib_borrow(ptr);
    imp.query_activatable(
        wrap.unsafe_cast_ref(),
        &from_glib_borrow(iter),
        &from_glib_borrow(rect),
    )
    .into_glib()
}

unsafe extern "C" fn gutter_renderer_activate<T: GutterRendererImpl>(
    ptr: *mut ffi::GtkSourceGutterRenderer,
    iter: *mut gtk::ffi::GtkTextIter,
    rect: *mut gdk::ffi::GdkRectangle,
    button: u32,
    state: gdk::ffi::GdkModifierType,
    n_presses: i32,
) {
    let instance = &*(ptr as *mut T::Instance);
    let imp = instance.imp();
    let wrap: Borrowed<GutterRenderer> = from_glib_borrow(ptr);
    imp.activate(
        wrap.unsafe_cast_ref(),
        &from_glib_borrow(iter),
        &from_glib_borrow(rect),
        button,
        from_glib(state),
        n_presses,
    )
}

unsafe extern "C" fn gutter_renderer_snapshot_line<T: GutterRendererImpl>(
    ptr: *mut ffi::GtkSourceGutterRenderer,
    snapshot: *mut gtk::ffi::GtkSnapshot,
    lines: *mut ffi::GtkSourceGutterLines,
    line: u32,
) {
    let instance = &*(ptr as *mut T::Instance);
    let imp = instance.imp();
    let wrap: Borrowed<GutterRenderer> = from_glib_borrow(ptr);
    imp.snapshot_line(
        wrap.unsafe_cast_ref(),
        &from_glib_borrow(snapshot),
        &from_glib_borrow(lines),
        line,
    )
}
