// Take a look at the license at the top of the repository in the LICENSE file.

// rustdoc-stripper-ignore-next
//! Traits intended for implementing the [`ContentProvider`](crate::ContentProvider) interface.

use crate::{CompletionCell, CompletionContext, CompletionProposal, CompletionProvider};
use glib::subclass::prelude::*;
use glib::translate::*;
use glib::{Cast, IsA};
use std::{future::Future, pin::Pin};

pub trait CompletionProviderImpl: ObjectImpl {
    fn activate(
        &self,
        provider: &Self::Type,
        context: &CompletionContext,
        proposal: &CompletionProposal,
    ) {
        self.parent_activate(provider, context, proposal)
    }

    fn display(
        &self,
        provider: &Self::Type,
        context: &CompletionContext,
        proposal: &CompletionProposal,
        cell: &CompletionCell,
    ) {
        self.parent_display(provider, context, proposal, cell)
    }

    fn title(&self, provider: &Self::Type) -> Option<glib::GString> {
        self.parent_title(provider)
    }

    fn priority(&self, provider: &Self::Type, context: &CompletionContext) -> i32 {
        self.parent_priority(provider, context)
    }

    fn is_trigger(&self, provider: &Self::Type, iter: &gtk::TextIter, c: char) -> bool {
        self.parent_is_trigger(provider, iter, c)
    }

    fn key_activates(
        &self,
        provider: &Self::Type,
        context: &CompletionContext,
        proposal: &CompletionProposal,
        keyval: gdk::Key,
        state: gdk::ModifierType,
    ) -> bool {
        self.parent_key_activates(provider, context, proposal, keyval, state)
    }

    fn refilter(&self, provider: &Self::Type, context: &CompletionContext, model: &gio::ListModel) {
        self.parent_refilter(provider, context, model)
    }

    fn list_alternates(
        &self,
        provider: &Self::Type,
        context: &CompletionContext,
        proposal: &CompletionProposal,
    ) -> Vec<CompletionProposal> {
        self.parent_list_alternates(provider, context, proposal)
    }

    fn populate(
        &self,
        provider: &Self::Type,
        context: &CompletionContext,
    ) -> Result<gio::ListModel, glib::Error> {
        self.parnet_populate(provider, context)
    }

    fn populate_future(
        &self,
        provider: &Self::Type,
        context: &CompletionContext,
    ) -> Pin<Box<dyn Future<Output = Result<gio::ListModel, glib::Error>>>> {
        self.parent_pouplate_future(provider, context)
    }
}

pub trait CompletionProviderImplExt: ObjectSubclass {
    fn parent_activate(
        &self,
        provider: &Self::Type,
        context: &CompletionContext,
        proposal: &CompletionProposal,
    );
    fn parent_display(
        &self,
        provider: &Self::Type,
        context: &CompletionContext,
        proposal: &CompletionProposal,
        cell: &CompletionCell,
    );
    fn parent_title(&self, provider: &Self::Type) -> Option<glib::GString>;
    fn parent_priority(&self, provider: &Self::Type, context: &CompletionContext) -> i32;
    fn parent_is_trigger(&self, provider: &Self::Type, iter: &gtk::TextIter, c: char) -> bool;
    fn parent_refilter(
        &self,
        provider: &Self::Type,
        context: &CompletionContext,
        model: &gio::ListModel,
    );
    fn parent_list_alternates(
        &self,
        provider: &Self::Type,
        context: &CompletionContext,
        proposal: &CompletionProposal,
    ) -> Vec<CompletionProposal>;
    fn parent_key_activates(
        &self,
        provider: &Self::Type,
        context: &CompletionContext,
        proposal: &CompletionProposal,
        keyval: gdk::Key,
        state: gdk::ModifierType,
    ) -> bool;
    fn parnet_populate(
        &self,
        provider: &Self::Type,
        context: &CompletionContext,
    ) -> Result<gio::ListModel, glib::Error>;
    fn parent_pouplate_async<
        C: IsA<gio::Cancellable>,
        Q: FnOnce(Result<gio::ListModel, glib::Error>) + Send + 'static,
    >(
        &self,
        provider: &Self::Type,
        context: &CompletionContext,
        cancellable: Option<&C>,
        callback: Q,
    );
    fn parent_pouplate_future(
        &self,
        provider: &Self::Type,
        context: &CompletionContext,
    ) -> Pin<Box<dyn Future<Output = Result<gio::ListModel, glib::Error>>>>;
}

impl<T: CompletionProviderImpl> CompletionProviderImplExt for T {
    fn parent_activate(
        &self,
        provider: &Self::Type,
        context: &CompletionContext,
        proposal: &CompletionProposal,
    ) {
        unsafe {
            let type_data = Self::type_data();
            let parent_iface = type_data.as_ref().parent_interface::<CompletionProvider>()
                as *const ffi::GtkSourceCompletionProviderInterface;

            let func = (*parent_iface)
                .activate
                .expect("no parent \"activate\" implementation");

            func(
                provider
                    .unsafe_cast_ref::<CompletionProvider>()
                    .to_glib_none()
                    .0,
                context.to_glib_none().0,
                proposal.to_glib_none().0,
            )
        }
    }

    fn parent_title(&self, provider: &Self::Type) -> Option<glib::GString> {
        unsafe {
            let type_data = Self::type_data();
            let parent_iface = type_data.as_ref().parent_interface::<CompletionProvider>()
                as *const ffi::GtkSourceCompletionProviderInterface;

            let func = (*parent_iface)
                .get_title
                .expect("no parent \"get_title\" implementation");

            from_glib_full(func(
                provider
                    .unsafe_cast_ref::<CompletionProvider>()
                    .to_glib_none()
                    .0,
            ))
        }
    }

    fn parent_display(
        &self,
        provider: &Self::Type,
        context: &CompletionContext,
        proposal: &CompletionProposal,
        cell: &CompletionCell,
    ) {
        unsafe {
            let type_data = Self::type_data();
            let parent_iface = type_data.as_ref().parent_interface::<CompletionProvider>()
                as *const ffi::GtkSourceCompletionProviderInterface;

            let func = (*parent_iface)
                .display
                .expect("no parent \"display\" implementation");

            func(
                provider
                    .unsafe_cast_ref::<CompletionProvider>()
                    .to_glib_none()
                    .0,
                context.to_glib_none().0,
                proposal.to_glib_none().0,
                cell.to_glib_none().0,
            )
        }
    }

    fn parent_priority(&self, provider: &Self::Type, context: &CompletionContext) -> i32 {
        unsafe {
            let type_data = Self::type_data();
            let parent_iface = type_data.as_ref().parent_interface::<CompletionProvider>()
                as *const ffi::GtkSourceCompletionProviderInterface;

            let func = (*parent_iface)
                .get_priority
                .expect("no parent \"get_priority\" implementation");

            func(
                provider
                    .unsafe_cast_ref::<CompletionProvider>()
                    .to_glib_none()
                    .0,
                context.to_glib_none().0,
            )
        }
    }

    fn parent_is_trigger(&self, provider: &Self::Type, iter: &gtk::TextIter, c: char) -> bool {
        unsafe {
            let type_data = Self::type_data();
            let parent_iface = type_data.as_ref().parent_interface::<CompletionProvider>()
                as *const ffi::GtkSourceCompletionProviderInterface;

            let func = (*parent_iface)
                .is_trigger
                .expect("no parent \"is_trigger\" implementation");

            from_glib(func(
                provider
                    .unsafe_cast_ref::<CompletionProvider>()
                    .to_glib_none()
                    .0,
                iter.to_glib_none().0,
                c.into_glib(),
            ))
        }
    }

    fn parent_list_alternates(
        &self,
        provider: &Self::Type,
        context: &CompletionContext,
        proposal: &CompletionProposal,
    ) -> Vec<CompletionProposal> {
        unsafe {
            let type_data = Self::type_data();
            let parent_iface = type_data.as_ref().parent_interface::<CompletionProvider>()
                as *const ffi::GtkSourceCompletionProviderInterface;

            let func = (*parent_iface)
                .list_alternates
                .expect("no parent \"list_alternates\" implementation");

            let output = func(
                provider
                    .unsafe_cast_ref::<CompletionProvider>()
                    .to_glib_none()
                    .0,
                context.to_glib_none().0,
                proposal.to_glib_none().0,
            );
            FromGlibPtrArrayContainerAsVec::from_glib_full_as_vec(output)
        }
    }

    fn parent_refilter(
        &self,
        provider: &Self::Type,
        context: &CompletionContext,
        model: &gio::ListModel,
    ) {
        unsafe {
            let type_data = Self::type_data();
            let parent_iface = type_data.as_ref().parent_interface::<CompletionProvider>()
                as *const ffi::GtkSourceCompletionProviderInterface;

            let func = (*parent_iface)
                .refilter
                .expect("no parent \"refilter\" implementation");

            func(
                provider
                    .unsafe_cast_ref::<CompletionProvider>()
                    .to_glib_none()
                    .0,
                context.to_glib_none().0,
                model.to_glib_none().0,
            )
        }
    }

    fn parent_key_activates(
        &self,
        provider: &Self::Type,
        context: &CompletionContext,
        proposal: &CompletionProposal,
        keyval: gdk::Key,
        state: gdk::ModifierType,
    ) -> bool {
        unsafe {
            let type_data = Self::type_data();
            let parent_iface = type_data.as_ref().parent_interface::<CompletionProvider>()
                as *const ffi::GtkSourceCompletionProviderInterface;

            let func = (*parent_iface)
                .key_activates
                .expect("no parent \"key_activates\" implementation");

            from_glib(func(
                provider
                    .unsafe_cast_ref::<CompletionProvider>()
                    .to_glib_none()
                    .0,
                context.to_glib_none().0,
                proposal.to_glib_none().0,
                keyval.into_glib(),
                state.into_glib(),
            ))
        }
    }

    fn parnet_populate(
        &self,
        provider: &Self::Type,
        context: &CompletionContext,
    ) -> Result<gio::ListModel, glib::Error> {
        unsafe {
            let type_data = Self::type_data();
            let parent_iface = type_data.as_ref().parent_interface::<CompletionProvider>()
                as *const ffi::GtkSourceCompletionProviderInterface;

            let func = (*parent_iface)
                .populate
                .expect("no parent \"populate\" implementation");
            let mut error = std::ptr::null_mut();
            let result = func(
                provider
                    .unsafe_cast_ref::<CompletionProvider>()
                    .to_glib_none()
                    .0,
                context.to_glib_none().0,
                &mut error,
            );
            if error.is_null() {
                Ok(from_glib_full(result))
            } else {
                Err(from_glib_full(error))
            }
        }
    }

    fn parent_pouplate_future(
        &self,
        provider: &Self::Type,
        context: &CompletionContext,
    ) -> Pin<Box<dyn Future<Output = Result<gio::ListModel, glib::Error>>>> {
        let context = context.clone();
        Box::pin(gio::GioFuture::new(
            provider,
            move |obj, cancellable, send| {
                let imp = obj.imp();
                imp.parent_pouplate_async(
                    &imp.instance(),
                    &context,
                    Some(cancellable),
                    move |res| {
                        send.resolve(res);
                    },
                );
            },
        ))
    }

    #[allow(clippy::type_complexity)]
    fn parent_pouplate_async<
        Q: IsA<gio::Cancellable>,
        C: FnOnce(Result<gio::ListModel, glib::Error>) + Send + 'static,
    >(
        &self,
        provider: &Self::Type,
        context: &CompletionContext,
        cancellable: Option<&Q>,
        callback: C,
    ) {
        unsafe {
            let type_data = Self::type_data();
            let parent_iface = type_data.as_ref().parent_interface::<CompletionProvider>()
                as *const ffi::GtkSourceCompletionProviderInterface;

            let f = (*parent_iface)
                .populate_async
                .expect("no parent \"populate_async\" implementation");
            let finish = (*parent_iface)
                .populate_finish
                .expect("no parent \"populate_finish\" implementation");

            let user_data: Box<(C, _)> = Box::new((callback, finish));

            unsafe extern "C" fn parent_populate_async_trampoline<
                C: FnOnce(Result<gio::ListModel, glib::Error>) + Send + 'static,
            >(
                source_object_ptr: *mut glib::gobject_ffi::GObject,
                res: *mut gio::ffi::GAsyncResult,
                user_data: glib::ffi::gpointer,
            ) {
                let mut error = std::ptr::null_mut();
                let cb: Box<(
                    C,
                    fn(
                        *mut ffi::GtkSourceCompletionProvider,
                        *mut gio::ffi::GAsyncResult,
                        *mut *mut glib::ffi::GError,
                    ) -> *mut gio::ffi::GListModel,
                )> = Box::from_raw(user_data as *mut _);
                let model = cb.1(source_object_ptr as _, res, &mut error);
                let result = if error.is_null() {
                    Ok(from_glib_full(model))
                } else {
                    Err(from_glib_full(error))
                };
                cb.0(result);
            }

            let cancellable = cancellable.map(|p| p.as_ref());
            let callback = parent_populate_async_trampoline::<C>;
            f(
                provider
                    .unsafe_cast_ref::<CompletionProvider>()
                    .to_glib_none()
                    .0,
                context.to_glib_none().0,
                cancellable.to_glib_none().0,
                Some(callback),
                Box::into_raw(user_data) as *mut _,
            );
        }
    }
}

unsafe impl<T: CompletionProviderImpl> IsImplementable<T> for CompletionProvider {
    fn interface_init(iface: &mut glib::Interface<Self>) {
        let iface = iface.as_mut();

        iface.activate = Some(completion_provider_activate::<T>);
        iface.display = Some(completion_provider_display::<T>);
        iface.get_title = Some(completion_provider_get_title::<T>);
        iface.get_priority = Some(completion_provider_get_priority::<T>);
        iface.is_trigger = Some(completion_provider_is_trigger::<T>);
        iface.refilter = Some(completion_provider_refilter::<T>);
        iface.key_activates = Some(completion_provider_key_activates::<T>);
        iface.list_alternates = Some(completion_provider_list_alternates::<T>);
        iface.populate = Some(completion_provider_populate::<T>);
        iface.populate_async = Some(completion_provider_populate_async::<T>);
        iface.populate_finish = Some(completion_provider_populate_finish::<T>);
    }
}

unsafe extern "C" fn completion_provider_activate<T: CompletionProviderImpl>(
    provider: *mut ffi::GtkSourceCompletionProvider,
    context: *mut ffi::GtkSourceCompletionContext,
    proposal: *mut ffi::GtkSourceCompletionProposal,
) {
    let instance = &*(provider as *mut T::Instance);
    let imp = instance.imp();

    imp.activate(
        from_glib_borrow::<_, CompletionProvider>(provider).unsafe_cast_ref(),
        &from_glib_borrow(context),
        &from_glib_borrow(proposal),
    )
}

unsafe extern "C" fn completion_provider_display<T: CompletionProviderImpl>(
    provider: *mut ffi::GtkSourceCompletionProvider,
    context: *mut ffi::GtkSourceCompletionContext,
    proposal: *mut ffi::GtkSourceCompletionProposal,
    cell: *mut ffi::GtkSourceCompletionCell,
) {
    let instance = &*(provider as *mut T::Instance);
    let imp = instance.imp();

    imp.display(
        from_glib_borrow::<_, CompletionProvider>(provider).unsafe_cast_ref(),
        &from_glib_borrow(context),
        &from_glib_borrow(proposal),
        &from_glib_borrow(cell),
    )
}

unsafe extern "C" fn completion_provider_get_title<T: CompletionProviderImpl>(
    provider: *mut ffi::GtkSourceCompletionProvider,
) -> *mut libc::c_char {
    let instance = &*(provider as *mut T::Instance);
    let imp = instance.imp();

    imp.title(from_glib_borrow::<_, CompletionProvider>(provider).unsafe_cast_ref())
        .to_glib_full()
}

unsafe extern "C" fn completion_provider_get_priority<T: CompletionProviderImpl>(
    provider: *mut ffi::GtkSourceCompletionProvider,
    context: *mut ffi::GtkSourceCompletionContext,
) -> i32 {
    let instance = &*(provider as *mut T::Instance);
    let imp = instance.imp();

    imp.priority(
        from_glib_borrow::<_, CompletionProvider>(provider).unsafe_cast_ref(),
        &from_glib_borrow(context),
    )
}

unsafe extern "C" fn completion_provider_is_trigger<T: CompletionProviderImpl>(
    provider: *mut ffi::GtkSourceCompletionProvider,
    iter: *const gtk::ffi::GtkTextIter,
    c: u32,
) -> glib::ffi::gboolean {
    let instance = &*(provider as *mut T::Instance);
    let imp = instance.imp();

    imp.is_trigger(
        from_glib_borrow::<_, CompletionProvider>(provider).unsafe_cast_ref(),
        &from_glib_borrow(iter),
        TryFrom::try_from(c).expect("Invalid character"),
    )
    .into_glib()
}

unsafe extern "C" fn completion_provider_refilter<T: CompletionProviderImpl>(
    provider: *mut ffi::GtkSourceCompletionProvider,
    context: *mut ffi::GtkSourceCompletionContext,
    model: *mut gio::ffi::GListModel,
) {
    let instance = &*(provider as *mut T::Instance);
    let imp = instance.imp();

    imp.refilter(
        from_glib_borrow::<_, CompletionProvider>(provider).unsafe_cast_ref(),
        &from_glib_borrow(context),
        &from_glib_borrow(model),
    )
}

unsafe extern "C" fn completion_provider_key_activates<T: CompletionProviderImpl>(
    provider: *mut ffi::GtkSourceCompletionProvider,
    context: *mut ffi::GtkSourceCompletionContext,
    proposal: *mut ffi::GtkSourceCompletionProposal,
    keyval: u32,
    state: gdk::ffi::GdkModifierType,
) -> glib::ffi::gboolean {
    let instance = &*(provider as *mut T::Instance);
    let imp = instance.imp();

    imp.key_activates(
        from_glib_borrow::<_, CompletionProvider>(provider).unsafe_cast_ref(),
        &from_glib_borrow(context),
        &from_glib_borrow(proposal),
        from_glib(keyval),
        from_glib(state),
    )
    .into_glib()
}

unsafe extern "C" fn completion_provider_list_alternates<T: CompletionProviderImpl>(
    provider: *mut ffi::GtkSourceCompletionProvider,
    context: *mut ffi::GtkSourceCompletionContext,
    proposal: *mut ffi::GtkSourceCompletionProposal,
) -> *mut glib::ffi::GPtrArray {
    let instance = &*(provider as *mut T::Instance);
    let imp = instance.imp();

    imp.list_alternates(
        from_glib_borrow::<_, CompletionProvider>(provider).unsafe_cast_ref(),
        &from_glib_borrow(context),
        &from_glib_borrow(proposal),
    )
    .to_glib_full()
}

unsafe extern "C" fn completion_provider_populate<T: CompletionProviderImpl>(
    provider: *mut ffi::GtkSourceCompletionProvider,
    context: *mut ffi::GtkSourceCompletionContext,
    error: *mut *mut glib::ffi::GError,
) -> *mut gio::ffi::GListModel {
    let instance = &*(provider as *mut T::Instance);
    let imp = instance.imp();

    let res = imp.populate(
        from_glib_borrow::<_, CompletionProvider>(provider).unsafe_cast_ref(),
        &from_glib_borrow(context),
    );
    match res {
        Ok(model) => {
            *error = std::ptr::null_mut();
            model.to_glib_full()
        }
        Err(err) => {
            *error = err.into_raw();
            std::ptr::null_mut()
        }
    }
}

unsafe extern "C" fn completion_provider_populate_async<T: CompletionProviderImpl>(
    ptr: *mut ffi::GtkSourceCompletionProvider,
    context: *mut ffi::GtkSourceCompletionContext,
    cancellable_ptr: *mut gio::ffi::GCancellable,
    callback: gio::ffi::GAsyncReadyCallback,
    user_data: glib::ffi::gpointer,
) {
    let instance = &*(ptr as *mut T::Instance);
    let imp = instance.imp();
    let wrap: CompletionProvider = from_glib_none(ptr);
    let context: CompletionContext = from_glib_none(context);
    let cancellable: Option<gio::Cancellable> = from_glib_none(cancellable_ptr);

    let closure = move |result: gio::LocalTask<gio::ListModel>,
                        source_object: Option<&CompletionProvider>| {
        let result: *mut gio::ffi::GAsyncResult = result
            .unsafe_cast_ref::<gio::AsyncResult>()
            .to_glib_none()
            .0;
        let source_object = source_object
            .map(|p| p.unsafe_cast_ref::<glib::Object>())
            .to_glib_none()
            .0;
        callback.unwrap()(source_object, result, user_data)
    };

    let t = gio::LocalTask::new(Some(&wrap), cancellable.as_ref(), closure);

    glib::MainContext::default().spawn_local(async move {
        let res = imp.populate_future(wrap.unsafe_cast_ref(), &context).await;
        t.return_result(res);
    });
}

unsafe extern "C" fn completion_provider_populate_finish<T: CompletionProviderImpl>(
    _ptr: *mut ffi::GtkSourceCompletionProvider,
    res_ptr: *mut gio::ffi::GAsyncResult,
    error_ptr: *mut *mut glib::ffi::GError,
) -> *mut gio::ffi::GListModel {
    let res: gio::AsyncResult = from_glib_none(res_ptr);
    let t = res.downcast::<gio::LocalTask<gio::ListModel>>().unwrap();
    let ret = t.propagate();
    match ret {
        Ok(model) => model.to_glib_full(),
        Err(e) => {
            *error_ptr = e.into_raw();
            std::ptr::null_mut()
        }
    }
}
