use anyhow::{bail, Result};
use crossbeam_channel::Receiver;
use wgpu::util::DeviceExt;
use winit::window::Window;

use std::num::NonZeroU32;

use crate::hsl;

/// The definition of a vertex for rendering the texture.
/// This is sent to the shader, see shader.wgsl.
#[repr(C)]
#[derive(Copy, Clone, Debug, bytemuck::Pod, bytemuck::Zeroable)]
struct Vertex {
    /// The drawing output coordinates.
    /// - x/y range: [-1.0, 1.0]
    /// - x increases rightward
    /// - y increases upward
    /// e.g. top left is [0.0, 1.0], bot right is [1.0, 0.0]
    position: [f32; 2],

    /// The texture input coordinates.
    /// - x/y range: [0.0, 1.0]
    /// - x increases rightward
    /// - y increases downward
    /// e.g. top left is [0.0, 0.0], bot right is [1.0, 1.0]
    tex_coords: [f32; 2],
}

impl Vertex {
    /// Describes to wgpu how Vertex is serialized
    fn desc<'a>() -> wgpu::VertexBufferLayout<'a> {
        wgpu::VertexBufferLayout {
            array_stride: std::mem::size_of::<Vertex>() as wgpu::BufferAddress,
            step_mode: wgpu::VertexStepMode::Vertex,
            attributes: &[
                // position
                wgpu::VertexAttribute {
                    offset: 0,
                    shader_location: 0,
                    format: wgpu::VertexFormat::Float32x2,
                },
                // tex_coords
                wgpu::VertexAttribute {
                    // skip past position
                    offset: std::mem::size_of::<[f32; 2]>() as wgpu::BufferAddress,
                    shader_location: 1,
                    format: wgpu::VertexFormat::Float32x2,
                },
            ],
        }
    }
}

/// Horizontal mode:
/// - texture is rotated 90 degrees counter-clockwise
/// - the analyzer is at the right and low frequencies are at the bottom
const VERTICES_HORIZ: &[Vertex] = &[
    Vertex {
        // 0
        position: [-1.0, 1.0],  // top left draw out
        tex_coords: [1.0, 0.0], // top right tex in (rotate 90 ccw)
    },
    Vertex {
        // 1
        position: [-1.0, -1.0], // bot left draw out
        tex_coords: [0.0, 0.0], // top left tex in (rotate 90 ccw)
    },
    Vertex {
        // 2
        position: [1.0, -1.0],  // bot right draw out
        tex_coords: [0.0, 1.0], // bot left tex in (rotate 90 ccw)
    },
    Vertex {
        // 3
        position: [1.0, 1.0],   // top right draw out
        tex_coords: [1.0, 1.0], // bot right tex in (rotate 90 ccw)
    },
];

/// Vertical mode:
/// - texture is mirrored vertically
/// - the analyzer is at the top and low frequencies are at the left
const VERTICES_VERT: &[Vertex] = &[
    Vertex {
        // 0
        position: [-1.0, 1.0],  // top left draw out
        tex_coords: [0.0, 1.0], // bot left tex in (mirror vertically)
    },
    Vertex {
        // 1
        position: [-1.0, -1.0], // bot left draw out
        tex_coords: [0.0, 0.0], // top left tex in (mirror vertically)
    },
    Vertex {
        // 2
        position: [1.0, -1.0],  // bot right draw out
        tex_coords: [1.0, 0.0], // top right tex in (mirror vertically)
    },
    Vertex {
        // 3
        position: [1.0, 1.0],   // top right draw out
        tex_coords: [1.0, 1.0], // bot right tex in (mirror vertically)
    },
];

/// Use two triangles to render our rectangular texture.
/// The referenced vertices are expected to be counter-clockwise from upper left,
/// but this otherwise doesn't need to change if we toggle horiz/vert mode.
const INDICES_RECT: &[u16] = &[
    0, 1, 2, // lower left triangle
    2, 3, 0, // upper right trangle
];

/// Rendering members that need to be recreated when window dimensions change.
struct SizedState {
    // The vertexes used for transforming and rendering the texture.
    // We use a different transform for vertical vs horizontal mode.
    vertex_buffer: wgpu::Buffer,
    // The GPU texture where voiceprint_buf and analyzer_buf are written to before displaying to the user.
    texture: wgpu::Texture,
    // The bind group of the GPU texture and its sampler.
    // This needs to be reinitialized whenever the texture is resized.
    bind_group: wgpu::BindGroup,

    // The number of logical rows in analyzer_buf.
    analyzer_height: usize,
    // The analyzer RGBA value data, which is only updated once per batch of inputs.
    analyzer_buf: Vec<u32>,

    // The number of logical rows in voiceprint_buf.
    voiceprint_height: usize,
    // The voiceprint RGBA value data, acts as a ring buffer where voiceprint_row is input offset.
    voiceprint_buf: Vec<u32>,
    // The "active" row to update in voiceprint_buf.
    voiceprint_row: usize,
}

/// The render configuration to be displayed
#[derive(Clone, Debug)]
pub enum Orientation {
    // - Analyzer at top
    // - Voiceprint scrolling down
    // - Low frequencies along left edge
    Vertical,

    // - Analyzer at right
    // - Voiceprint scrolling left
    // - Low frequencies along bottom edge
    Horizontal,
}

/// Required for using Orientation in commandline args.
impl std::fmt::Display for Orientation {
    fn fmt(&self, f: &mut std::fmt::Formatter) -> std::fmt::Result {
        match self {
            Orientation::Vertical => write!(f, "vertical"),
            Orientation::Horizontal => write!(f, "horizontal"),
        }
    }
}
/// Optional for better parsing of Orientation in commandline args.
impl std::str::FromStr for Orientation {
    type Err = anyhow::Error;

    fn from_str(s: &str) -> Result<Orientation> {
        match s {
            "|" | "v" | "V" | "vert" | "Vert" | "vertical" | "Vertical" => {
                Ok(Orientation::Vertical)
            }
            "-" | "h" | "H" | "horiz" | "Horiz" | "horizontal" | "Horizontal" => {
                Ok(Orientation::Horizontal)
            }
            _ => bail!("unrecognized orientation: '{}'", s),
        }
    }
}

/// The rendering state.
pub struct State {
    /// A multiplier for the speed of voiceprint scrolling should be.
    /// Higher value shrinks the texture height, which should reduce load,
    /// but it also "speeds up" voiceprint since we're doing 1 texture row per sample.
    /// This value is fixed and kept around to recalculate texture height on window resize.
    scroll_rate: f32,

    // The wgpu output components. Initialized once on startup.
    surface: wgpu::Surface,
    device: wgpu::Device,
    queue: wgpu::Queue,
    render_pipeline: wgpu::RenderPipeline,
    bind_group_layout: wgpu::BindGroupLayout,
    index_buffer: wgpu::Buffer,

    /// The wgpu window configuration, with sizes updated if the window is resized
    config: wgpu::SurfaceConfiguration,
    /// The desired render orientation, used when reinitializing sized_state
    orientation: Orientation,
    /// Various buffers/configs that need to be reinitialized if the window is resized
    sized_state: SizedState,

    /// Mapping of [0.0,1.0] audio values to RGBA colors
    hsl: hsl::HSL,
    /// Input channel for audio data, where each Vec must have length recv_processed_size
    recv_processed: Receiver<Vec<f32>>,
    /// Length of Vecs produced by recv_processed. Also used as the texture width.
    recv_processed_size: usize,

    /// Number of low frequency values that should have 4x zoom in the voiceprint/analyzer.
    /// So the resulting texture pixels used are 4x this value.
    recv_processed_4x_size: usize,

    /// Number of mid frequency values that should have 2x zoom in the voiceprint/analyzer.
    /// So the resulting texture pixels used are 2x this value.
    recv_processed_2x_size: usize,

    /// Number of high frequency values that should have 1x zoom in the voiceprint/analyzer.
    /// So the resulting texture pixels used are 1x this value.
    /// Anything past the sum of 4x_size + 2x_size + 1x_size is discarded.
    recv_processed_1x_size: usize,
}

impl State {
    /// Sets up a visualization rendering pipeline for the provided window.
    pub async fn new(
        window: &Window,
        hsl: hsl::HSL,
        recv_processed: Receiver<Vec<f32>>,
        recv_processed_size: usize,
        orientation: Orientation,
        scroll_rate: f32,
    ) -> Self {
        // The instance is a handle to our GPU
        // BackendBit::PRIMARY => Vulkan + Metal + DX12 + Browser WebGPU
        let instance = wgpu::Instance::new(wgpu::Backends::all());
        let surface = unsafe { instance.create_surface(window) };
        let adapter = instance
            .request_adapter(&wgpu::RequestAdapterOptions {
                power_preference: wgpu::PowerPreference::default(),
                compatible_surface: Some(&surface),
                force_fallback_adapter: false,
            })
            .await
            .unwrap();

        let (device, queue) = adapter
            .request_device(
                &wgpu::DeviceDescriptor {
                    label: None,
                    features: wgpu::Features::empty(),
                    limits: wgpu::Limits::default(),
                },
                None, // Trace path
            )
            .await
            .unwrap();

        let bind_group_layout = device.create_bind_group_layout(&wgpu::BindGroupLayoutDescriptor {
            entries: &[
                wgpu::BindGroupLayoutEntry {
                    binding: 0,
                    visibility: wgpu::ShaderStages::FRAGMENT,
                    ty: wgpu::BindingType::Texture {
                        multisampled: false,
                        view_dimension: wgpu::TextureViewDimension::D2,
                        sample_type: wgpu::TextureSampleType::Float { filterable: true },
                    },
                    count: None,
                },
                wgpu::BindGroupLayoutEntry {
                    binding: 1,
                    visibility: wgpu::ShaderStages::FRAGMENT,
                    ty: wgpu::BindingType::Sampler(wgpu::SamplerBindingType::Filtering),
                    count: None,
                },
            ],
            label: Some("bind_group_layout"),
        });

        let shader = device.create_shader_module(&wgpu::ShaderModuleDescriptor {
            label: Some("shader"),
            source: wgpu::ShaderSource::Wgsl(include_str!("shader.wgsl").into()),
        });

        let config_format = surface.get_preferred_format(&adapter).unwrap();

        let render_pipeline = device.create_render_pipeline(&wgpu::RenderPipelineDescriptor {
            label: Some("render_pipeline"),
            layout: Some(
                &device.create_pipeline_layout(&wgpu::PipelineLayoutDescriptor {
                    label: Some("render_pipeline_layout"),
                    bind_group_layouts: &[&bind_group_layout],
                    push_constant_ranges: &[],
                }),
            ),
            vertex: wgpu::VertexState {
                module: &shader,
                entry_point: "vs_main",
                buffers: &[Vertex::desc()],
            },
            fragment: Some(wgpu::FragmentState {
                module: &shader,
                entry_point: "fs_main",
                targets: &[wgpu::ColorTargetState {
                    format: config_format,
                    blend: Some(wgpu::BlendState {
                        color: wgpu::BlendComponent::REPLACE,
                        alpha: wgpu::BlendComponent::REPLACE,
                    }),
                    write_mask: wgpu::ColorWrites::ALL,
                }],
            }),
            primitive: wgpu::PrimitiveState {
                topology: wgpu::PrimitiveTopology::TriangleList,
                strip_index_format: None,
                front_face: wgpu::FrontFace::Ccw,
                cull_mode: Some(wgpu::Face::Back),
                // Setting this to anything other than Fill requires Features::NON_FILL_POLYGON_MODE
                polygon_mode: wgpu::PolygonMode::Fill,
                // Requires Features::DEPTH_CLIP_CONTROL
                unclipped_depth: false,
                // Requires Features::CONSERVATIVE_RASTERIZATION
                conservative: false,
            },
            depth_stencil: None,
            multisample: wgpu::MultisampleState {
                count: 1,
                mask: !0,
                alpha_to_coverage_enabled: false,
            },
            // If the pipeline will be used with a multiview render pass, this
            // indicates how many array layers the attachments will have.
            multiview: None,
        });

        let index_buffer = device.create_buffer_init(&wgpu::util::BufferInitDescriptor {
            label: Some("index_buffer"),
            contents: bytemuck::cast_slice(INDICES_RECT),
            usage: wgpu::BufferUsages::INDEX,
        });

        let window_size = window.inner_size();
        let sized_state = init_sized_state(
            &device,
            &bind_group_layout,
            &orientation,
            window_size.height,
            window_size.width,
            scroll_rate,
            recv_processed_size,
        );

        let config = wgpu::SurfaceConfiguration {
            usage: wgpu::TextureUsages::RENDER_ATTACHMENT,
            format: config_format,
            width: window_size.width,
            height: window_size.height,
            present_mode: wgpu::PresentMode::Fifo,
        };
        // Call configure() once up-front, needed for tiling WMs like sway
        surface.configure(&device, &config);

        State {
            scroll_rate,

            surface,
            device,
            queue,
            render_pipeline,
            bind_group_layout,
            index_buffer,

            config,
            orientation,
            sized_state,

            hsl,
            recv_processed,
            recv_processed_size,

            // low 1/16 x 4 => bottom 1/4 (cumulative: 25%)
            recv_processed_4x_size: recv_processed_size / 16,
            // mid 1/8 x 2 => next 1/4 (cumulative: 50%)
            recv_processed_2x_size: recv_processed_size / 8,
            // high 1/2 x 1 => remaining 1/2 (cumulative: 100%)
            recv_processed_1x_size: recv_processed_size / 2,
            // Note: remaining 5/16 at the top of the spectrum is discarded
        }
    }

    /// Switches the render orientation between horizontal and vertical
    pub fn toggle_orientation(&mut self) {
        let orientation = match self.orientation {
            Orientation::Vertical => Orientation::Horizontal,
            Orientation::Horizontal => Orientation::Vertical,
        };
        // No changes needed to wgpu config, since the surface dimensions aren't changing.
        // Reinitialize sized_state with new buffers to reflect the new orientation.
        self.sized_state = init_sized_state(
            &self.device,
            &self.bind_group_layout,
            &orientation,
            self.config.height,
            self.config.width,
            self.scroll_rate,
            self.recv_processed_size,
        );
        // Update desired orientation so that it's available in future resize() calls.
        self.orientation = orientation;
    }

    /// Updates the render size to reflect a change in window size
    pub fn resize(&mut self, new_size: Option<winit::dpi::PhysicalSize<u32>>) {
        // Update width/height in wgpu config.
        // This is separate from sized_state to avoid rebuilding the whole config.
        if let Some(new_size) = new_size {
            if new_size.width > 0 && new_size.height > 0 {
                self.config.width = new_size.width;
                self.config.height = new_size.height;
            }
        }
        // Update surface with the change to config.
        self.surface.configure(&self.device, &self.config);

        // Reinitialize sized_state with new buffers to reflect the new dimensions.
        self.sized_state = init_sized_state(
            &self.device,
            &self.bind_group_layout,
            &self.orientation,
            self.config.height,
            self.config.width,
            self.scroll_rate,
            self.recv_processed_size,
        );
    }

    /// Returns the underlying surface texture for rendering.
    pub fn surface_texture(&mut self) -> Result<wgpu::SurfaceTexture, wgpu::SurfaceError> {
        self.surface.get_current_texture()
    }

    /// Re-renders the display:
    /// - Collects a batch of audio frequency data from recv_processed
    /// - For each audio entry in the batch, adds it to the voiceprint to be rendered
    /// - For the last audio entry in the batch, updates the analyzer to be rendered
    /// - Writes the resulting buffers to a texture which is then rotated/scaled and rendered to the output
    pub fn render(&mut self, output: wgpu::SurfaceTexture) -> Result<()> {
        // Collect a batch of data. The batch is capped at the capacity of the channel.
        let audio_vec: Vec<Vec<f32>> = self.recv_processed.try_iter().collect();

        // Here's how the voiceprint/analyzer buffers are mapped to the texture.
        // - The scrolling of "voiceprint_row" allows us to only need to update one row of the vector at a time.
        // - Keeping texture in this orientation when writing to it enables passing subranges of voiceprint_buf to write_texture().
        //
        //  lowfreq   hifreq
        // |----------------|
        // | old voiceprint |
        // |   [row, end]   |
        // |----------------| <- start/end of voiceprint_buf
        // | new voiceprint |
        // |    [0, row]    |
        // |----------------| <- current voiceprint_row (increments forward through voiceprint_buf)
        // |    analyzer    |
        // |----------------|
        //
        // Horizontal mode:
        // - texture is rotated 90 degrees counter-clockwise
        // - the analyzer is at the right and low frequencies are at the bottom
        // Vertical mode:
        // - texture is mirrored vertically
        // - the analyzer is at the top and low frequencies are at the left

        // Iterate over the batch. only render analyzer on the last entry of the batch
        for (audio_idx, audio) in audio_vec.iter().enumerate() {
            // Set to buffer index for the start of the current row in voiceprint_buf
            // row index in voiceprint_buf * width of voiceprint_buf
            let mut voiceprint_idx = self.sized_state.voiceprint_row * self.recv_processed_size;

            // Iterator contains frequency data from low to high frequency
            let mut audio_iter = audio.into_iter();

            if audio_idx == audio_vec.len() - 1 {
                // We're at the last entry of the batch (common case unless we're falling behind).
                // Render both the voiceprint and the analyzer.
                let mut analyzer_idx = 0;
                // analyzer: Reset to opaque black (r=0,g=0,b=0,a=255)
                self.sized_state.analyzer_buf.clear();
                self.sized_state
                    .analyzer_buf
                    .resize(self.sized_state.analyzer_buf.capacity(), 255 * 16777216);

                for _ in 0..self.recv_processed_4x_size {
                    // Draw low frequencies as 4x zoom
                    self.draw_voiceprint_analyzer(
                        *audio_iter.next().unwrap(),
                        &mut voiceprint_idx,
                        &mut analyzer_idx,
                        4,
                    );
                }
                for _ in 0..self.recv_processed_2x_size {
                    // Draw mid frequencies as 2x zoom
                    self.draw_voiceprint_analyzer(
                        *audio_iter.next().unwrap(),
                        &mut voiceprint_idx,
                        &mut analyzer_idx,
                        2,
                    );
                }
                for _ in 0..self.recv_processed_1x_size {
                    // Draw high frequencies as 1x zoom
                    self.draw_voiceprint_analyzer(
                        *audio_iter.next().unwrap(),
                        &mut voiceprint_idx,
                        &mut analyzer_idx,
                        1,
                    );
                }
                // Highest frequencies are discarded
            } else {
                // We're not at the last entry of the batch yet. Only update the voiceprint buffer.
                // We skip updating the analyzer since only the last batch entry will be displayed there.
                for _ in 0..self.recv_processed_4x_size {
                    // Draw low frequencies as 4x zoom
                    self.draw_voiceprint(*audio_iter.next().unwrap(), &mut voiceprint_idx, 4);
                }
                for _ in 0..self.recv_processed_2x_size {
                    // Draw mid frequencies as 2x zoom
                    self.draw_voiceprint(*audio_iter.next().unwrap(), &mut voiceprint_idx, 2);
                }
                for _ in 0..self.recv_processed_1x_size {
                    // Draw high frequencies as 1x zoom
                    self.draw_voiceprint(*audio_iter.next().unwrap(), &mut voiceprint_idx, 1);
                }
                // Highest frequencies are discarded
            }

            // Increment voiceprint row, with wraparound to row 0 when we reach the height of the voiceprint.
            self.sized_state.voiceprint_row =
                (self.sized_state.voiceprint_row + 1) % self.sized_state.voiceprint_height;
        }

        // Now that we've updated the voiceprint and analyzer buffers, write them to the texture to be displayed.

        // Write the older data below voiceprint_row to the top of the voiceprint region
        self.queue.write_texture(
            wgpu::ImageCopyTexture {
                aspect: wgpu::TextureAspect::All,
                texture: &self.sized_state.texture,
                mip_level: 0,
                origin: wgpu::Origin3d::ZERO,
            },
            as_u8_slice(&self.sized_state.voiceprint_buf),
            wgpu::ImageDataLayout {
                // 4x values to get byte offsets (rather than u32 offsets)
                offset: (self.sized_state.voiceprint_row * 4 * self.recv_processed_size) as u64,
                bytes_per_row: NonZeroU32::new(4 * self.recv_processed_size as u32),
                rows_per_image: NonZeroU32::new(self.sized_state.voiceprint_height as u32),
            },
            wgpu::Extent3d {
                width: self.recv_processed_size as u32,
                height: (self.sized_state.voiceprint_height - self.sized_state.voiceprint_row)
                    as u32,
                depth_or_array_layers: 1,
            },
        );

        // Write the newer data above voiceprint_row to the bottom of the voiceprint region
        self.queue.write_texture(
            wgpu::ImageCopyTexture {
                aspect: wgpu::TextureAspect::All,
                texture: &self.sized_state.texture,
                mip_level: 0,
                origin: wgpu::Origin3d {
                    x: 0,
                    y: (self.sized_state.voiceprint_height - self.sized_state.voiceprint_row)
                        as u32,
                    z: 0,
                },
            },
            as_u8_slice(&self.sized_state.voiceprint_buf),
            wgpu::ImageDataLayout {
                // 4x values to get byte offsets (rather than u32 offsets)
                offset: 0,
                bytes_per_row: NonZeroU32::new(4 * self.recv_processed_size as u32),
                rows_per_image: NonZeroU32::new(self.sized_state.voiceprint_row as u32),
            },
            wgpu::Extent3d {
                width: self.recv_processed_size as u32,
                height: self.sized_state.voiceprint_row as u32,
                depth_or_array_layers: 1,
            },
        );

        // Write analyzer_buf to the bottom 20% of the texture
        self.queue.write_texture(
            wgpu::ImageCopyTexture {
                aspect: wgpu::TextureAspect::All,
                texture: &self.sized_state.texture,
                mip_level: 0,
                origin: wgpu::Origin3d {
                    x: 0,
                    y: self.sized_state.voiceprint_height as u32,
                    z: 0,
                },
            },
            as_u8_slice(&self.sized_state.analyzer_buf),
            wgpu::ImageDataLayout {
                // 4x values to get byte offsets (rather than u32 offsets)
                offset: 0,
                bytes_per_row: NonZeroU32::new(4 * self.recv_processed_size as u32),
                rows_per_image: NonZeroU32::new(self.sized_state.analyzer_height as u32),
            },
            wgpu::Extent3d {
                width: self.recv_processed_size as u32,
                height: self.sized_state.analyzer_height as u32,
                depth_or_array_layers: 1,
            },
        );

        let view = output
            .texture
            .create_view(&wgpu::TextureViewDescriptor::default());
        let mut command = self
            .device
            .create_command_encoder(&wgpu::CommandEncoderDescriptor {
                label: Some("Render Encoder"),
            });

        // Render the texture, with vertex_buffer controlling rotation/mirroring to match the display mode.
        {
            let mut render_pass = command.begin_render_pass(&wgpu::RenderPassDescriptor {
                label: Some("render_pass"),
                color_attachments: &[wgpu::RenderPassColorAttachment {
                    view: &view,
                    resolve_target: None,
                    ops: wgpu::Operations {
                        load: wgpu::LoadOp::Clear(wgpu::Color {
                            r: 0.1,
                            g: 0.2,
                            b: 0.3,
                            a: 1.0,
                        }),
                        store: true,
                    },
                }],
                depth_stencil_attachment: None,
            });
            render_pass.set_pipeline(&self.render_pipeline);
            render_pass.set_bind_group(0, &self.sized_state.bind_group, &[]);
            render_pass.set_vertex_buffer(0, self.sized_state.vertex_buffer.slice(..));
            render_pass.set_index_buffer(self.index_buffer.slice(..), wgpu::IndexFormat::Uint16);
            render_pass.draw_indexed(0..INDICES_RECT.len() as u32, 0, 0..1);
        }

        self.queue.submit(std::iter::once(command.finish()));
        output.present();

        Ok(())
    }

    /// Updates a value in voiceprint_buf only
    /// This is used when we know there's pending frequency data to be written,
    /// in which case writing this data to the analyzer buffer is a waste of time.
    fn draw_voiceprint(&mut self, val: f32, voiceprint_idx: &mut usize, times: usize) {
        // Convert value to rgba color. val is scaled to [0.0, 1.0] by fourier.rs.
        let pixel = self.hsl.value_to_rgba(val);

        // Repeat the value N times to zoom in on lower frequencies.
        for _ in 0..times {
            // voiceprint: draw rgba pixel on row
            self.sized_state.voiceprint_buf[*voiceprint_idx] = pixel;
            (*voiceprint_idx) += 1; // seek to next rgba pixel on row
        }
    }

    /// Updates a value in both voiceprint_buf and analyzer_buf
    /// Structured this way to share a little work between voiceprint/analyzer
    fn draw_voiceprint_analyzer(
        &mut self,
        val: f32,
        voiceprint_idx: &mut usize,
        analyzer_idx: &mut usize,
        times: usize,
    ) {
        // Convert value to rgba color. val is scaled to [0.0, 1.0] by fourier.rs.
        let pixel = self.hsl.value_to_rgba(val);

        // analyzer: line height is proportional to value
        let bar_width = (val * self.sized_state.analyzer_height as f32) as usize;

        // Repeat the value N times to zoom in on lower frequencies.
        for _ in 0..times {
            // voiceprint: draw rgba pixel on row
            self.sized_state.voiceprint_buf[*voiceprint_idx] = pixel;
            (*voiceprint_idx) += 1; // seek to next rgba pixel on row

            // analyzer: draw vertical rgba line on single column (rotated to horizontal later)
            let mut col_idx = *analyzer_idx; // top of column
            for _pixel_col in 0..bar_width {
                self.sized_state.analyzer_buf[col_idx] = pixel;
                col_idx += self.recv_processed_size; // next row in column
            }
            (*analyzer_idx) += 1; // seek to next col to draw another vertical line
        }
    }
}

/// Recasts a &[u32] as a &[u8] for passing to gfx
fn as_u8_slice(v: &[u32]) -> &[u8] {
    unsafe {
        std::slice::from_raw_parts(
            v.as_ptr() as *const u8,
            v.len() * std::mem::size_of::<u32>(),
        )
    }
}

/// (Re)initializes the render state to reflect a change in window size.
/// This avoids being a method of State to keep it clear what information is needed.
fn init_sized_state(
    // The device to create textures/buffers against.
    device: &wgpu::Device,
    // The bind group layout for recreating the bind group.
    bind_group_layout: &wgpu::BindGroupLayout,
    // The desired orientation to use in rendering. Affects texture size.
    orientation: &Orientation,
    // The window height, used for the texture height when orientation is vertical.
    window_height: u32,
    // The window width, used for the texture height when orientation is horizontal.
    window_width: u32,
    // Multiplier for the texture height, reducing render size while speeding up voiceprint scroll
    scroll_rate: f32,
    // The number of frequency values being received per sample, proportional to the texture width.
    recv_processed_size: usize,
) -> SizedState {
    // Scale texture height according to the window size in the correct direction
    // Texture width meanwhile is always locked to recv_processed_size.
    let texture_height = match orientation {
        Orientation::Vertical => (window_height as f32 / scroll_rate) as u32,
        Orientation::Horizontal => (window_width as f32 / scroll_rate) as u32,
    };

    // The transform to use when rendering the texture, depends on horiz/vert orientation
    let vertex_buffer = device.create_buffer_init(&wgpu::util::BufferInitDescriptor {
        label: Some("vertex_buffer"),
        contents: bytemuck::cast_slice(match orientation {
            Orientation::Vertical => VERTICES_VERT,
            Orientation::Horizontal => VERTICES_HORIZ,
        }),
        usage: wgpu::BufferUsages::VERTEX,
    });

    // Texture that has both voiceprint and analyzer buffers written to it
    let texture = device.create_texture(&wgpu::TextureDescriptor {
        label: Some("buf_texture"),
        size: wgpu::Extent3d {
            width: recv_processed_size as u32, // render the audio data at full resolution
            height: texture_height,
            depth_or_array_layers: 1,
        },
        mip_level_count: 1,
        sample_count: 1,
        dimension: wgpu::TextureDimension::D2,
        format: wgpu::TextureFormat::Rgba8UnormSrgb,
        usage: wgpu::TextureUsages::TEXTURE_BINDING | wgpu::TextureUsages::COPY_DST,
    });

    // Reinitialize this when the texture is recreated
    let bind_group = device.create_bind_group(&wgpu::BindGroupDescriptor {
        label: Some("bind_group"),
        layout: &bind_group_layout,
        entries: &[
            wgpu::BindGroupEntry {
                binding: 0,
                resource: wgpu::BindingResource::TextureView(
                    &texture.create_view(&wgpu::TextureViewDescriptor::default()),
                ),
            },
            wgpu::BindGroupEntry {
                binding: 1,
                resource: wgpu::BindingResource::Sampler(&device.create_sampler(
                    &wgpu::SamplerDescriptor {
                        address_mode_u: wgpu::AddressMode::ClampToEdge, // x
                        address_mode_v: wgpu::AddressMode::ClampToEdge, // y
                        address_mode_w: wgpu::AddressMode::ClampToEdge, // z
                        mag_filter: wgpu::FilterMode::Nearest,
                        min_filter: wgpu::FilterMode::Nearest,
                        mipmap_filter: wgpu::FilterMode::Nearest,
                        ..Default::default()
                    },
                )),
            },
        ],
    });

    // Circular voiceprint buffer written to top 80% of texture
    let voiceprint_height = (0.8 * (texture_height as f32)) as usize;
    let mut voiceprint_buf = Vec::with_capacity(voiceprint_height * recv_processed_size);
    // Ensure the buffer is opaque black rather than transparent (r=0,g=0,b=0,a=255)
    voiceprint_buf.resize(voiceprint_buf.capacity(), 255 * 16777216);

    // Analyzer written to bottom 20% of texture
    let analyzer_height = texture_height as usize - voiceprint_height;
    let mut analyzer_buf = Vec::with_capacity(analyzer_height * recv_processed_size);
    // Ensure the buffer is opaque black rather than transparent (r=0,g=0,b=0,a=255)
    analyzer_buf.resize(analyzer_buf.capacity(), 255 * 16777216);

    SizedState {
        vertex_buffer,
        texture,
        bind_group,
        analyzer_height,
        analyzer_buf,
        voiceprint_height,
        voiceprint_buf,
        voiceprint_row: 0,
    }
}
