use anyhow::{bail, Context, Result};
use clap::{Arg, Command};
use git_testament::{git_testament, render_testament};
use regex::Regex;
use tracing::{debug, info};

use soundview::{events, fourier, logging, recorder, renderer};

use std::str::FromStr;
use std::thread;

git_testament!(TESTAMENT);

#[derive(Debug)]
struct Args {
    buckets: usize,
    scroll: f32,

    device: Option<Regex>,
    frequency: i32,

    orientation: renderer::Orientation,
    fullscreen: bool,
}

/// Parses commandline arguments and returns the parsed result.
fn get_args(testament_version: &String) -> Result<Args> {
    let matches = Command::new("soundview")
        .author("Nick Parker, nick@nickbp.com")
        .version(testament_version.as_str())
        .about("Audio Voiceprint and Analyzer - don't miss a beat")
        .args(&[
            // Audio config
            Arg::new("buckets")
                .display_order(1)
                .long("buckets")
                .long_help("The number of frequency buckets to calculate and display. Higher values will increase granularity and graphics load. Must be a power of two, with a theoretical max of 32768. ")
                .takes_value(true)
                .default_value("2048")
                .validator(|v| {
                    let mut buckets = usize::from_str(v)
                        .with_context(|| format!("not an unsigned int: '{:?}'", v))?;
                    if buckets < 32 {
                        // Arbitrary minimum, avoid weird rendering issues
                        bail!("cannot be less than 32");
                    }
                    if buckets > 32768 {
                        bail!("cannot be greater than 32768");
                    }
                    while buckets > 1 {
                        if buckets % 2 != 0 {
                            bail!("must be a power of two (e.g. 256, 2048, 16384)");
                        }
                        buckets /= 2;
                    }
                    Ok(())
                }),
            Arg::new("scroll")
                .display_order(2)
                .long("scroll")
                .long_help("A separate adjustment of the voiceprint scroll speed, in pixels/sample. Higher values will increase voiceprint speed. ")
                .takes_value(true)
                .default_value("2")
                .validator(|v| {
                    let scroll = f32::from_str(v)
                        .with_context(|| format!("not a floating point number: '{:?}'", v))?;
                    // Values less than 1 are okay, they just mean the texture is really big and the scroll looks weird.
                    // But let's avoid having a texture that's TOO big.
                    if scroll < 0.1 {
                        // Arbitrary minimum scroll speed to avoid too many weird rendering issues
                        bail!("must be 0.1 or greater");
                    }
                    Ok(())
                }),
            // Device config
            Arg::new("device")
                .display_order(3)
                .long("device")
                .long_help("Name of the initial input device to select, as a regular expression. After startup, use leftarrow/rightarrow keys to switch between input devices, including ones that don't match this filter.")
                .takes_value(true),
            Arg::new("freq")
                .display_order(4)
                .long("freq")
                .long_help("Sample frequency in Hz for retrieving audio from all input devices. Higher values will result in faster scrolling at the same graphics load. May be <=0 to use suboptimal device-specific default sample rates. ")
                .takes_value(true)
                .default_value("96000"),
            // Display config
            Arg::new("orientation")
                .display_order(5)
                .long("orientation")
                .long_help("Initial display orientation. After startup, use the space bar or R key ('Rotate') to toggle the orientation. ")
                .takes_value(true)
                .default_value("horiz"),
            Arg::new("fullscreen")
                .display_order(6)
                .long("fullscreen")
                .long_help("Enables starting in fullscreen mode. After startup, use the F11 or F keys to toggle fullscreen mode.")
        ])
        .override_usage("[LOG_LEVEL=debug|info|warn] soundview [OPTIONS]")
        .get_matches();

    Ok(Args {
        buckets: usize::from_str(
            matches
                .value_of("buckets")
                .expect("missing default buckets"),
        )
        .expect("invalid buckets uncaught by custom validator"),
        scroll: f32::from_str(matches.value_of("scroll").expect("missing default scroll"))
            .expect("invalid scroll uncaught by custom validator"),
        device: match matches.value_of("device") {
            Some(pattern) => Some(
                Regex::new(pattern)
                    .with_context(|| format!("Invalid device regex: '{:?}'", pattern))?,
            ),
            None => None,
        },
        // we check for negative values below (interpreted as "use device default")
        frequency: i32::from_str(matches.value_of("freq").expect("missing default frequency"))
            .with_context(|| {
                format!(
                    "Invalid freq {:?}, should be an integer",
                    matches.value_of("freq").unwrap()
                )
            })?,
        orientation: renderer::Orientation::from_str(
            matches
                .value_of("orientation")
                .expect("missing default orientation"),
        )
        .with_context(|| {
            format!(
                "Invalid orientation {:?}, should be e.g. 'h','v','horiz','vert'",
                matches.value_of("orientation").unwrap()
            )
        })?,
        fullscreen: matches.is_present("fullscreen"),
    })
}

/// Main entry point. The UI and event handler runs on this thread, while separate threads
/// are launched for sampling device audio, and for running a fourier transform on that audio.
fn main() -> Result<()> {
    logging::init_logging();
    let testament_version = render_testament!(TESTAMENT);
    let args = get_args(&testament_version)?;
    debug!("{:?}", args);

    let sdl_version = sdl2::version::version();
    info!(
        "Soundview version {}, SDL {}.{}.{}",
        testament_version, sdl_version.major, sdl_version.minor, sdl_version.patch
    );

    let (send_audio, recv_audio) = crossbeam_channel::bounded::<Vec<f32>>(100);
    let (send_processed, recv_processed) = crossbeam_channel::bounded::<Vec<f32>>(100);

    let frequency = match args.frequency {
        f if f <= 0 => None,
        f => Some(f),
    };

    let mut rec = recorder::Recorder::new(
        recorder::init_audio()?,
        frequency.clone(),
        // MUST be a power of 2, and must be 2x the size of the fourier output
        // (with u16 max 65536, resulting in a fourier output max size of 32768)
        Some(2 * args.buckets as u16),
        send_audio,
    );

    let fourier_thread = thread::Builder::new()
        .name("fourier".to_string())
        .spawn(move || {
            fourier::process_audio_loop(args.buckets, frequency, recv_audio, send_processed)
        })?;

    // Recording internally runs on a separate thread
    rec.autoselect_start(args.device)?;

    events::process_event_loop(
        recv_processed,
        args.buckets,
        args.orientation,
        args.fullscreen,
        args.scroll,
        rec,
        fourier_thread,
    )
}
