use anyhow::{Context, Result};
use crossbeam_channel::Receiver;
use tracing::{debug, error, warn};
use winit::dpi::PhysicalSize;
use winit::event::*;
use winit::event_loop::{ControlFlow, EventLoop};
use winit::window::{Fullscreen, Icon, WindowBuilder};

use crate::{hsl, recorder, renderer};

/// Endless loop that takes over the main thread.
/// Drives rendering the visualization and handles any keyboard/window events.
pub fn process_event_loop(
    recv_processed: Receiver<Vec<f32>>,
    recv_processed_size: usize,
    orientation: renderer::Orientation,
    fullscreen: bool,
    scroll_rate: f32,
    mut rec: recorder::Recorder,
    fourier_thread: std::thread::JoinHandle<()>,
) -> Result<()> {
    let event_loop = EventLoop::new();

    let icon_webp = include_bytes!("soundview.webp");
    let icon_image = image::load_from_memory_with_format(icon_webp, image::ImageFormat::WebP)?;
    let icon = Icon::from_rgba(
        icon_image
            .as_rgba8()
            .context("Unable to get RGBA data for icon")?
            .to_vec(),
        icon_image.width(),
        icon_image.height(),
    )?;

    let window = WindowBuilder::new()
        .with_title("soundview")
        .with_window_icon(Some(icon))
        .with_inner_size(PhysicalSize::new(1280, 720))
        .with_min_inner_size(PhysicalSize::new(100, 100))
        .build(&event_loop)?;
    if fullscreen {
        window.set_fullscreen(Some(Fullscreen::Borderless(window.current_monitor())));
    }

    let fourier_thread_rc = std::cell::RefCell::new(Some(fourier_thread));
    let hsl = hsl::HSL::new(50, 40);

    let mut state = pollster::block_on(renderer::State::new(
        &window,
        hsl,
        recv_processed,
        recv_processed_size,
        orientation,
        scroll_rate,
    ));
    event_loop.run(move |event, _, control_flow| match event {
        Event::RedrawRequested(window_id) if window_id == window.id() => {
            match state.surface_texture() {
                Ok(output) => {
                    if let Err(e) = state.render(output) {
                        // Run any teardown operations before exiting the main thread.
                        // event_loop.run() has taken over the thread and will never return.
                        error!("shutting down: render error {}", e);
                        rec.stop();
                        match fourier_thread_rc.replace(None) {
                            Some(fourier_thread) => {
                                if let Err(e) = fourier_thread.join() {
                                    warn!("failed to wait on fourier thread to exit: {:?}", e);
                                }
                            }
                            None => {} // join already called?
                        }
                        *control_flow = ControlFlow::Exit
                    }
                }
                // Reconfigure the surface if lost
                Err(wgpu::SurfaceError::Lost) => state.resize(None),
                // The system is out of memory, we should probably quit
                Err(wgpu::SurfaceError::OutOfMemory) => {
                    // Run any teardown operations before exiting the main thread.
                    // event_loop.run() has taken over the thread and will never return.
                    error!("shutting down: wgpu is out of memory");
                    rec.stop();
                    match fourier_thread_rc.replace(None) {
                        Some(fourier_thread) => {
                            if let Err(e) = fourier_thread.join() {
                                warn!("failed to wait on fourier thread to exit: {:?}", e);
                            }
                        }
                        None => {} // join already called?
                    }
                    *control_flow = ControlFlow::Exit
                }
                // All other errors (Outdated, Timeout) should be resolved by the next frame
                Err(e) => warn!("render error: {:?}", e),
            }
        }

        Event::MainEventsCleared => {
            // RedrawRequested will only trigger once, unless we manually
            // request it.
            window.request_redraw();
        }

        Event::WindowEvent {
            ref event,
            window_id,
        } if window_id == window.id() => match event {
            WindowEvent::Resized(physical_size) => {
                state.resize(Some(*physical_size));
            }
            WindowEvent::ScaleFactorChanged { new_inner_size, .. } => {
                // new_inner_size is &mut so w have to dereference it twice
                state.resize(Some(**new_inner_size));
            }

            // <Close window>, Esc, Q: Quit
            WindowEvent::CloseRequested
            | WindowEvent::KeyboardInput {
                input:
                    KeyboardInput {
                        state: ElementState::Pressed,
                        virtual_keycode: Some(VirtualKeyCode::Escape),
                        ..
                    },
                ..
            }
            | WindowEvent::KeyboardInput {
                input:
                    KeyboardInput {
                        state: ElementState::Pressed,
                        virtual_keycode: Some(VirtualKeyCode::Q),
                        ..
                    },
                ..
            } => {
                // Run any teardown operations before exiting the main thread.
                // event_loop.run() has taken over the thread and will never return.
                debug!("shutting down");
                rec.stop();
                match fourier_thread_rc.replace(None) {
                    Some(fourier_thread) => {
                        if let Err(e) = fourier_thread.join() {
                            warn!("failed to wait on fourier thread to exit: {:?}", e);
                        }
                    }
                    None => {} // join already called?
                }
                *control_flow = ControlFlow::Exit
            }

            // F, F11: Fullscreen
            WindowEvent::KeyboardInput {
                input:
                    KeyboardInput {
                        state: ElementState::Pressed,
                        virtual_keycode: Some(VirtualKeyCode::F),
                        ..
                    },
                ..
            }
            | WindowEvent::KeyboardInput {
                input:
                    KeyboardInput {
                        state: ElementState::Pressed,
                        virtual_keycode: Some(VirtualKeyCode::F11),
                        ..
                    },
                ..
            } => {
                if let Some(_) = window.fullscreen() {
                    // toggle off
                    window.set_fullscreen(None);
                } else {
                    // toggle on
                    window.set_fullscreen(Some(Fullscreen::Borderless(window.current_monitor())));
                }
            }

            // R, Space: Rotate
            WindowEvent::KeyboardInput {
                input:
                    KeyboardInput {
                        state: ElementState::Pressed,
                        virtual_keycode: Some(VirtualKeyCode::R),
                        ..
                    },
                ..
            }
            | WindowEvent::KeyboardInput {
                input:
                    KeyboardInput {
                        state: ElementState::Pressed,
                        virtual_keycode: Some(VirtualKeyCode::Space),
                        ..
                    },
                ..
            } => {
                state.toggle_orientation();
            }

            // Left arrow: Prev device
            WindowEvent::KeyboardInput {
                input:
                    KeyboardInput {
                        state: ElementState::Pressed,
                        virtual_keycode: Some(VirtualKeyCode::Left),
                        ..
                    },
                ..
            } => {
                // TODO display device name on top corner of display, using font-rs or manual bitmap
                if let Err(e) = rec.prev_device() {
                    warn!("Failed to switch to previous device: {}", e)
                }
            }

            // Right arrow: Next device
            WindowEvent::KeyboardInput {
                input:
                    KeyboardInput {
                        state: ElementState::Pressed,
                        virtual_keycode: Some(VirtualKeyCode::Right),
                        ..
                    },
                ..
            } => {
                // TODO display device name on top corner of display, using font-rs or manual bitmap
                if let Err(e) = rec.next_device() {
                    warn!("Failed to switch to next device: {}", e)
                }
            }

            _ => {}
        },
        _ => {}
    })
}
