#![doc(test(attr(deny(warnings))))]
#![warn(missing_docs)]
//! Library to parse and iterate over weather soundings retrieved from [University of Wyoming][1].
//!
//! Retrieve a sounding from the [University of Wyoming][1] page in the "Text: List" format, and 
//! then pass the text to the `parse_text()` function and it will return an iterator. The iterator 
//! returns `(sounding_analysis::Sounding, std::collections::HashMap<&'static str, f64>)` items.
//!
//! The hash map contains indexes and values provided in the text that are not a part of the
//! `sounding_analysis::Sounding` type. The keys in the hashmap should describe the values.
//!
//! You may build a program that automatically downloads the soundings from the website or one that
//! loads the text from disk, either way just pass the text into the `parse_text()` function and 
//! it will parse it for you.
//!
//! The crate was originally built to support [sonde][2], a sounding data viewer and analysis tool.
//!
//! [1]: http://weather.uwyo.edu/upperair/sounding.html
//! [2]:https://github.com/rnleach/sonde

//
// API
//
pub use crate::sounding_iter::{parse_text, SoundingIterator};

//
// Internal use only.
//

mod parse_section;
mod sounding_iter;
mod text_iter;

#[cfg(test)]
mod test_utils {
    pub fn get_otx() -> &'static str {
        r#"
<HTML>
<TITLE>University of Wyoming - Radiosonde Data</TITLE>
<BODY BGCOLOR="white">
<H2>72786 OTX Spokane Observations at 12Z 11 Feb 2021</H2>
<PRE>
-----------------------------------------------------------------------------
   PRES   HGHT   TEMP   DWPT   RELH   MIXR   DRCT   SKNT   THTA   THTE   THTV
    hPa     m      C      C      %    g/kg    deg   knot     K      K      K 
-----------------------------------------------------------------------------
 1000.0    210                                                               
  936.0    728   -8.5  -15.5     57   1.23     20      8  269.7  273.2  269.9
  935.0    737   -8.7  -18.7     44   0.94     20      8  269.6  272.3  269.7
  925.0    824   -9.7  -19.7     44   0.87     15      9  269.4  271.9  269.5
  914.2    914  -10.4  -19.2     48   0.92     20      9  269.6  272.3  269.7
  906.0    984  -10.9  -18.9     52   0.95     18      9  269.8  272.6  269.9
  885.0   1164  -11.3  -21.3     43   0.79     12      8  271.1  273.5  271.3
  878.7   1219  -11.3  -23.7     35   0.65     10      8  271.7  273.7  271.8
  872.0   1277  -11.3  -26.3     28   0.52      2      8  272.3  273.9  272.4
  852.0   1455  -12.5  -27.5     27   0.47    337      8  272.9  274.3  272.9
  850.0   1473  -12.5  -26.5     30   0.52    335      8  273.0  274.6  273.1
  833.0   1627  -12.3  -22.3     43   0.77    326      8  274.8  277.2  275.0
  818.0   1766  -13.1  -23.1     43   0.73    319      9  275.4  277.6  275.5
  811.2   1829  -13.7  -23.3     44   0.73    315      9  275.4  277.6  275.6
  785.0   2078  -16.1  -24.1     50   0.70    319     11  275.5  277.6  275.6
  779.1   2134  -16.4  -22.5     59   0.81    320     12  275.7  278.2  275.8
  774.0   2184  -16.7  -21.1     69   0.92    312     12  275.9  278.7  276.1
  748.2   2438  -17.8  -19.7     85   1.08    270     15  277.5  280.7  277.6
  735.0   2571  -18.3  -19.0     94   1.17    277     19  278.3  281.8  278.5
  718.2   2743  -19.1  -22.0     78   0.92    285     25  279.2  282.1  279.4
  714.0   2787  -19.3  -22.8     74   0.86    287     25  279.5  282.1  279.6
  700.0   2934  -20.7  -23.5     78   0.83    295     26  279.5  282.1  279.7
  668.0   3278  -24.1  -25.0     92   0.76    299     27  279.5  281.8  279.6
  661.2   3353  -24.6  -25.5     92   0.73    300     27  279.8  282.0  279.9
  648.0   3500  -25.5  -26.6     90   0.67    302     33  280.3  282.4  280.4
  640.0   3590  -25.1  -27.3     82   0.64    304     36  281.8  283.8  281.9
  634.1   3658  -24.2  -35.1     36   0.31    305     39  283.6  284.6  283.6
  632.0   3682  -23.9  -37.9     26   0.23    305     40  284.2  284.9  284.2
  629.0   3717  -23.7  -43.7     14   0.13    305     42  284.8  285.2  284.8
  613.0   3906  -22.3  -51.3      5   0.06    305     51  288.5  288.7  288.5
  599.0   4076  -22.1  -55.1      3   0.04    305     59  290.6  290.8  290.6
  591.0   4175  -21.1  -54.1      3   0.04    305     64  292.9  293.1  292.9
  583.6   4267  -21.4  -55.4      3   0.04    305     68  293.6  293.7  293.6
  562.0   4546  -22.5  -59.5      2   0.02    303     75  295.5  295.6  295.5
  551.0   4691  -22.1  -65.1      1   0.01    302     78  297.7  297.7  297.7
  537.2   4877  -23.2  -65.3      1   0.01    300     83  298.5  298.6  298.5
  519.0   5129  -24.7  -65.7      1   0.01    302     86  299.6  299.7  299.7
  505.0   5328  -26.1  -49.1     10   0.09    304     89  300.3  300.6  300.3
  500.0   5400  -26.5  -46.5     13   0.12    305     90  300.7  301.1  300.7
  474.0   5783  -28.3  -43.3     22   0.18    300     98  303.1  303.7  303.1
  454.0   6092  -28.3  -60.3      3   0.03    295    105  306.8  306.9  306.8
  453.7   6096  -28.3  -60.3      3   0.02    295    105  306.9  307.0  306.9
  436.0   6381  -28.9  -60.9      3   0.02    295    110  309.6  309.7  309.6
  434.8   6401  -29.1  -59.9      3   0.03    295    110  309.7  309.8  309.7
  422.0   6613  -30.7  -49.7     14   0.10    299    111  310.2  310.6  310.2
  411.0   6799  -31.9  -64.9      2   0.02    302    113  311.0  311.1  311.0
  400.0   6990  -33.3  -49.3     19   0.11    305    114  311.6  312.1  311.6
  379.0   7366  -35.9  -42.9     49   0.23    302    118  313.0  313.9  313.1
  370.0   7533  -36.7  -48.7     28   0.12    301    120  314.1  314.6  314.1
  365.3   7620  -37.4  -50.1     25   0.11    300    121  314.4  314.8  314.4
  338.0   8152  -41.5  -58.5     14   0.04    300    125  315.8  316.0  315.8
  329.0   8335  -42.5  -71.5      3   0.01    300    126  316.9  316.9  316.9
  300.0   8950  -48.1  -62.1     18   0.03    300    130  317.4  317.6  317.4
  291.3   9144  -49.8  -61.5     24   0.03    300    130  317.7  317.8  317.7
  289.0   9196  -50.3  -61.3     26   0.03    300    131  317.7  317.9  317.7
  280.0   9402  -52.1  -65.1     19   0.02    301    133  318.0  318.1  318.0
  265.0   9758  -54.1  -67.1     19   0.02    303    137  320.1  320.2  320.1
  252.8  10058  -56.5  -65.8     30   0.02    305    141  320.9  321.0  320.9
  252.0  10079  -56.7  -65.7     31   0.02    305    141  320.9  321.0  320.9
  250.0  10130  -57.1  -66.1     31   0.02    305    141  321.1  321.1  321.1
  247.0  10205  -57.7  -66.7     31   0.02    305    141  321.3  321.4  321.3
  225.0  10785  -62.1  -71.1     29   0.01    300    130  323.2  323.3  323.2
  221.0  10896  -61.9  -70.9     29   0.01    301    125  325.2  325.2  325.2
  217.0  11009  -59.9  -70.9     22   0.01    302    120  330.0  330.0  330.0
  210.0  11213  -58.9  -73.9     13   0.01    303    110  334.6  334.7  334.6
  207.0  11303  -56.3  -75.3      7   0.01    304    106  340.1  340.1  340.1
  201.0  11489  -57.3  -83.3      2   0.00    305     97  341.4  341.4  341.4
  200.0  11520  -56.7  -84.7      2   0.00    305     96  342.8  342.8  342.8
  196.0  11648  -55.5  -87.5      1   0.00    303     93  346.7  346.7  346.7
  190.0  11846  -56.5  -88.5      1   0.00    301     87  348.2  348.2  348.2
  188.8  11887  -55.7  -88.1      1   0.00    300     86  350.1  350.1  350.1
  187.0  11948  -54.5  -87.5      1   0.00    300     86  353.0  353.0  353.0
  184.0  12051  -54.9  -86.9      1   0.00    299     85  354.0  354.0  354.0
  180.0  12192  -53.7  -86.7      1   0.00    298     84  358.2  358.2  358.2
  175.0  12372  -54.9  -86.9      1   0.00    296     82  359.1  359.1  359.1
  171.6  12497  -54.9  -86.9      1   0.00    295     81  361.1  361.1  361.1
  170.0  12558  -54.9  -86.9      1   0.00    296     81  362.1  362.1  362.1
  168.0  12634  -53.9  -86.9      1   0.00    297     81  365.0  365.0  365.0
  160.0  12947  -53.3  -86.3      1   0.00    300     82  371.1  371.1  371.1
  150.0  13360  -56.3  -88.3      1   0.00    305     82  372.9  372.9  372.9
  149.0  13402  -56.7  -88.7      1   0.00    305     82  372.9  372.9  372.9
  141.0  13752  -57.1  -89.1      1   0.00    308     83  378.1  378.1  378.1
  137.0  13935  -55.3  -87.3      1   0.00    309     83  384.4  384.4  384.4
  135.2  14021  -55.4  -87.4      1   0.00    310     83  385.8  385.8  385.8
  133.0  14124  -55.5  -87.5      1   0.00    310     82  387.3  387.3  387.3
  129.0  14318  -56.9  -88.9      1   0.00    309     80  388.2  388.2  388.2
  121.0  14725  -55.5  -87.5      1   0.00    307     75  397.9  397.9  397.9
  119.0  14831  -55.9  -87.9      1   0.00    307     74  399.1  399.1  399.1
  115.0  15049  -55.3  -87.3      1   0.00    306     71  404.1  404.1  404.1
  112.0  15217  -56.1  -88.1      1   0.00    305     69  405.7  405.7  405.7
  111.6  15240  -56.0  -88.0      1   0.00    305     69  406.3  406.3  406.3
  108.0  15449  -55.3  -87.3      1   0.00    305     66  411.4  411.5  411.5
  102.0  15814  -55.7  -87.7      1   0.00    305     60  417.5  417.5  417.5
  100.0  15940  -54.7  -86.7      1   0.00    305     58  421.8  421.8  421.8
</PRE><H3>Station information and sounding indices</H3><PRE>
                         Station identifier: OTX
                             Station number: 72786
                           Observation time: 210211/1200
                           Station latitude: 47.68
                          Station longitude: -117.63
                          Station elevation: 728.0
                            Showalter index: 21.63
                               Lifted index: 23.20
    LIFT computed using virtual temperature: 23.21
                                SWEAT index: 106.00
                                    K index: -15.30
                         Cross totals index: 0.00
                      Vertical totals index: 14.00
                        Totals totals index: 14.00
      Convective Available Potential Energy: 0.00
             CAPE using virtual temperature: 0.00
                      Convective Inhibition: 0.00
             CINS using virtual temperature: 0.00
                     Bulk Richardson Number: 0.00
          Bulk Richardson Number using CAPV: 0.00
  Temp [K] of the Lifted Condensation Level: 251.42
Pres [hPa] of the Lifted Condensation Level: 778.15
   Equivalent potential temp [K] of the LCL: 272.66
     Mean mixed layer potential temperature: 270.11
              Mean mixed layer mixing ratio: 0.87
              1000 hPa to 500 hPa thickness: 5190.00
Precipitable water [mm] for entire sounding: 2.71
</PRE>
<P>Description of the 
<A HREF="/upperair/columns.html">data columns</A>
or <A HREF="/upperair/indices.html">sounding indices</A>.

<P>
<FORM>
<INPUT CLASS="button" TYPE="button" VALUE=" Close this window " 
 onClick="window.close();">
<INPUT CLASS="button" TYPE="button" VALUE=" Select another map " 
 onClick="window.blur();">
</FORM>
<HR SIZE="1">
<I>Interested in graduate studies in atmospheric science?
Check out our program at the
<a href="http://www.uwyo.edu/atsc/howtoapply/"
target=_top>University of Wyoming
</a></I>
<HR SIZE="1"><FONT SIZE="-1">
Questions about the weather data provided by this site can be
addressed to <A HREF="mailto:ldoolman@uwyo.edu">
Larry Oolman (ldoolman@uwyo.edu)</A></FONT>
<HR SIZE="1">
<SCRIPT TYPE="text/javascript">
<!--
window.focus();
// -->
</SCRIPT>
</BODY>
</HTML>
        "#
    }

    pub fn get_otx2() -> &'static str {
        r#"
<html><head>
<meta http-equiv="content-type" content="text/html; charset=UTF-8"><title>University of Wyoming - Radiosonde Data</title>
</head><body bgcolor="white">
<h2>72786 OTX Spokane Observations at 12Z 13 Feb 2021</h2>
<pre>-----------------------------------------------------------------------------
   PRES   HGHT   TEMP   DWPT   RELH   MIXR   DRCT   SKNT   THTA   THTE   THTV
    hPa     m      C      C      %    g/kg    deg   knot     K      K      K 
-----------------------------------------------------------------------------
 1000.0    152                                                               
  929.0    728   -9.3  -19.3     44   0.90     35     13  269.5  272.1  269.6
  925.0    760   -9.7  -21.7     37   0.73     35     14  269.4  271.6  269.5
  906.5    914  -10.9  -22.2     39   0.71     40     20  269.7  271.8  269.8
  871.1   1219  -13.4  -23.3     43   0.68     75     29  270.2  272.2  270.3
  850.0   1406  -14.9  -23.9     46   0.66     80     25  270.5  272.5  270.6
  841.0   1486  -15.5  -24.5     46   0.63     87     23  270.7  272.6  270.8
  833.0   1559  -14.3  -24.3     42   0.65     93     21  272.7  274.7  272.8
  806.0   1809  -12.7  -21.7     47   0.84    113     14  277.0  279.6  277.1
  803.9   1829  -12.8  -22.0     46   0.82    115     13  277.1  279.6  277.2
  793.0   1933  -13.5  -23.5     43   0.73    115     12  277.4  279.7  277.6
  789.0   1972  -13.3  -23.3     43   0.75    115     12  278.1  280.4  278.2
  783.0   2030  -12.7  -18.7     61   1.12    115     11  279.3  282.7  279.5
  772.4   2134  -13.1  -18.0     67   1.21    115     10  279.9  283.6  280.1
  764.0   2217  -13.5  -17.4     72   1.28    120     11  280.4  284.3  280.6
  754.0   2318  -12.1  -14.7     81   1.63    127     12  283.0  287.9  283.3
  742.2   2438  -12.7  -15.4     80   1.56    135     14  283.6  288.3  283.9
  713.0   2743  -14.3  -17.2     79   1.40    130     18  285.1  289.4  285.3
  703.0   2851  -14.9  -17.8     79   1.35    115     13  285.6  289.8  285.8
  700.0   2883  -14.9  -17.7     79   1.37    110     12  286.0  290.2  286.2
  690.0   2992  -13.7  -15.8     84   1.63    113     11  288.5  293.5  288.8
  684.9   3048  -13.9  -16.0     84   1.62    115     11  288.9  293.8  289.1
  655.0   3387  -15.1  -17.0     85   1.55    204     14  291.2  296.1  291.5
  643.0   3527  -13.9  -15.5     88   1.79    241     15  294.1  299.7  294.4
  632.0   3658  -14.0  -15.7     87   1.79    275     16  295.4  301.1  295.7
  612.0   3902  -14.3  -16.0     87   1.80    273     18  297.8  303.6  298.2
  582.9   4267  -16.4  -18.2     86   1.57    270     22  299.5  304.6  299.8
  537.4   4877  -20.0  -21.9     84   1.23    270     24  302.3  306.4  302.6
  531.0   4966  -20.5  -22.5     84   1.19    271     25  302.7  306.7  302.9
  500.0   5410  -22.5  -24.6     83   1.05    275     28  305.6  309.1  305.7
  456.0   6083  -25.7  -28.5     77   0.80    275     31  309.7  312.5  309.8
  455.2   6096  -25.8  -28.6     77   0.80    275     31  309.7  312.5  309.9
  400.0   7020  -33.1  -36.6     71   0.42    270     35  311.9  313.4  312.0
  366.3   7620  -38.2  -41.9     68   0.26    270     41  313.1  314.1  313.1
  317.0   8607  -46.5  -50.7     62   0.12    263     53  314.7  315.2  314.7
  300.0   8970  -49.1  -53.7     58   0.09    260     58  316.0  316.4  316.1
  292.1   9144  -50.7  -54.9     61   0.08    260     56  316.2  316.5  316.2
  288.0   9236  -51.5  -55.5     62   0.07    260     56  316.3  316.6  316.3
  274.0   9557  -53.9  -58.9     54   0.05    262     58  317.4  317.6  317.4
  255.0  10015  -55.9  -60.9     53   0.04    264     60  321.0  321.2  321.0
  250.0  10140  -57.1  -62.1     53   0.04    265     61  321.1  321.2  321.1
  211.0  11192  -65.3  -69.9     53   0.01    255     77  324.2  324.3  324.2
  210.0  11222  -65.5  -70.1     53   0.01    260     77  324.3  324.4  324.3
  208.1  11278  -65.1  -70.0     51   0.01    255     77  325.8  325.8  325.8
  200.0  11520  -63.5  -69.5     44   0.02    280     65  332.1  332.1  332.1
  198.0  11582  -62.1  -69.1     39   0.02    282     62  335.2  335.3  335.2
  196.0  11645  -59.3  -68.3     30   0.02    284     60  340.7  340.8  340.7
  195.0  11678  -58.5  -67.5     30   0.02    285     58  342.4  342.5  342.4
  180.0  12179  -61.3  -74.3     16   0.01    300     37  345.8  345.8  345.8
  179.6  12192  -61.3  -74.3     16   0.01    300     36  346.1  346.1  346.1
  171.1  12497  -60.2  -74.1     15   0.01    270     47  352.6  352.7  352.6
  170.0  12536  -60.1  -74.1     14   0.01    271     48  353.5  353.5  353.5
  166.0  12686  -57.9  -73.9     11   0.01    276     53  359.6  359.6  359.6
  163.0  12802  -57.9  -73.9     11   0.01    280     56  361.4  361.5  361.5
  160.0  12918  -57.9  -73.9     11   0.01    284     55  363.4  363.4  363.4
  158.0  12998  -56.3  -73.3     10   0.01    287     55  367.4  367.4  367.4
  153.0  13203  -55.9  -73.9      9   0.01    295     53  371.4  371.5  371.5
  151.0  13287  -55.1  -73.1      9   0.01    298     52  374.2  374.3  374.2
  150.0  13330  -55.1  -73.1      9   0.01    300     52  374.9  375.0  374.9
  148.1  13411  -55.6  -74.0      8   0.01    305     50  375.4  375.5  375.4
  140.0  13767  -57.9  -77.9      6   0.01    296     45  377.5  377.5  377.5
  134.5  14021  -57.0  -76.2      7   0.01    290     41  383.5  383.6  383.5
  133.0  14091  -56.7  -75.7      7   0.01    291     42  385.2  385.3  385.2
  123.0  14584  -57.9  -75.9      8   0.01    295     47  391.7  391.8  391.7
  121.0  14687  -57.3  -76.3      7   0.01    295     48  394.6  394.7  394.6
  110.8  15240  -59.0  -77.4      8   0.01    300     54  401.5  401.5  401.5
  106.0  15517  -59.9  -77.9      8   0.01    300     54  404.9  405.0  404.9
  102.0  15757  -60.5  -78.5      8   0.01    300     55  408.2  408.3  408.2
  100.0  15880  -59.9  -78.9      7   0.01    300     55  411.7  411.8  411.7
</pre><h3>Station information and sounding indices</h3><pre>                         Station identifier: OTX
                             Station number: 72786
                           Observation time: 210213/1200
                           Station latitude: 47.68
                          Station longitude: -117.63
                          Station elevation: 728.0
                            Showalter index: 27.33
                               Lifted index: 27.83
    LIFT computed using virtual temperature: 27.98
                                SWEAT index: 77.99
                                    K index: -19.10
                         Cross totals index: -1.40
                      Vertical totals index: 7.60
                        Totals totals index: 6.20
      Convective Available Potential Energy: 0.00
             CAPE using virtual temperature: 0.00
                      Convective Inhibition: 0.00
             CINS using virtual temperature: 0.00
                     Bulk Richardson Number: 0.00
          Bulk Richardson Number using CAPV: 0.00
  Temp [K] of the Lifted Condensation Level: 248.79
Pres [hPa] of the Lifted Condensation Level: 753.37
   Equivalent potential temp [K] of the LCL: 271.88
     Mean mixed layer potential temperature: 269.78
              Mean mixed layer mixing ratio: 0.71
              1000 hPa to 500 hPa thickness: 5258.00
Precipitable water [mm] for entire sounding: 6.39
</pre>
<p>Description of the 
<a href="http://weather.uwyo.edu/upperair/columns.html">data columns</a>
or <a href="http://weather.uwyo.edu/upperair/indices.html">sounding indices</a>.

</p><p>
</p><form>
<input class="button" type="button" value=" Close this window " onclick="window.close();">
<input class="button" type="button" value=" Select another map " onclick="window.blur();">
</form>
<hr size="1">
<i>Interested in graduate studies in atmospheric science?
Check out our program at the
<a href="http://www.uwyo.edu/atsc/howtoapply/" target="_top">University of Wyoming
</a></i>
<hr size="1"><font size="-1">
Questions about the weather data provided by this site can be
addressed to <a href="mailto:ldoolman@uwyo.edu">
Larry Oolman (ldoolman@uwyo.edu)</a></font>
<hr size="1">
<script type="text/javascript">
<!--
window.focus();
// -->
</script>


</body></html>
        "#
    }

    pub fn get_tfx() -> &'static str {
        r#"
<HTML>
<TITLE>University of Wyoming - Radiosonde Data</TITLE>
<BODY BGCOLOR="white">
<H2>72776 TFX Great Falls Observations at 12Z 01 Feb 2021</H2>
<PRE>
-----------------------------------------------------------------------------
   PRES   HGHT   TEMP   DWPT   RELH   MIXR   DRCT   SKNT   THTA   THTE   THTV
    hPa     m      C      C      %    g/kg    deg   knot     K      K      K 
-----------------------------------------------------------------------------
 1000.0    154                                                               
  888.0   1134    4.0   -8.0     41   2.37    225     15  286.7  293.8  287.1
  879.0   1218    4.6   -8.4     38   2.32                288.2  295.2  288.6
  868.0   1322    4.6   -8.4     38   2.35                289.2  296.3  289.6
  859.0   1407    3.8   -9.2     38   2.23                289.2  296.0  289.6
  852.0   1475    4.8   -9.2     36   2.25                291.0  297.8  291.4
  850.0   1494    5.0   -9.0     36   2.29                291.4  298.4  291.8
  839.0   1600    5.0   -9.0     36   2.32                292.5  299.6  292.9
  811.0   1877    4.4   -5.6     48   3.12                294.7  304.2  295.2
  764.0   2359    0.6   -5.4     64   3.36                295.6  305.9  296.2
  748.0   2530    2.2   -7.8     48   2.85                299.2  308.1  299.7
  700.0   3061   -1.9  -10.9     50   2.39                300.4  307.9  300.8
  646.0   3692   -7.3  -13.3     62   2.13                301.2  308.0  301.6
  619.0   4023   -9.7  -15.7     62   1.83                302.1  308.1  302.5
  612.0   4111   -9.7  -20.7     40   1.21                303.1  307.1  303.4
  594.0   4340  -11.7  -20.7     47   1.25                303.4  307.6  303.6
  592.0   4366  -11.9  -20.9     47   1.23                303.5  307.6  303.7
  588.0   4417  -11.9  -31.9     17   0.45                304.1  305.6  304.1
  559.0   4802  -14.5  -33.5     18   0.41                305.4  306.9  305.5
  554.0   4870  -14.5  -23.5     46   1.04                306.2  309.7  306.4
  545.0   4994  -14.1  -42.1      7   0.17                308.1  308.8  308.1
  538.0   5092  -14.7  -47.7      4   0.10                308.5  308.9  308.6
  522.0   5319  -16.7  -40.7     11   0.21                308.8  309.6  308.8
  513.0   5449  -17.9  -33.9     23   0.43                308.9  310.4  309.0
  500.0   5640  -19.7  -35.7     23   0.37                309.0  310.3  309.0
  484.0   5881  -21.9  -36.9     24   0.34                309.1  310.4  309.2
  465.0   6175  -24.7  -33.7     43   0.48                309.2  310.9  309.3
  444.0   6511  -27.1  -38.1     34   0.32                310.3  311.5  310.4
  437.0   6626  -27.1  -49.1     11   0.10                311.7  312.1  311.7
  424.0   6844  -28.1  -54.1      6   0.06                313.1  313.4  313.1
  404.0   7189  -31.5  -46.5     21   0.15                313.1  313.6  313.1
  400.0   7260  -31.9  -46.9     21   0.14                313.4  314.0  313.5
  390.0   7438  -33.5  -49.5     18   0.11                313.6  314.1  313.6
  342.0   8345  -41.9  -49.9     41   0.12                314.2  314.7  314.2
  328.0   8627  -44.1  -49.1     57   0.13                315.0  315.5  315.0
  316.0   8876  -46.3  -51.3     57   0.11                315.3  315.7  315.3
  303.0   9154  -48.1  -52.1     63   0.10                316.6  317.0  316.6
  300.0   9220  -48.7  -53.1     60   0.09                316.6  317.0  316.6
  289.0   9464  -50.3  -55.0     57   0.08                317.7  318.0  317.7
  283.0   9599  -51.3  -54.9     65   0.08                318.2  318.5  318.2
  276.0   9761  -52.3  -57.2     55   0.06                319.0  319.3  319.1
  263.0  10070  -54.9  -61.9     42   0.04                319.6  319.8  319.7
  250.0  10390  -57.9  -64.9     40   0.02                319.9  320.0  319.9
  249.0  10415  -58.3  -65.3     40   0.02                319.6  319.7  319.6
  220.0  11185  -63.5  -68.4     51   0.02                323.1  323.2  323.1
  211.0  11441  -62.7  -67.7     51   0.02                328.2  328.4  328.3
  204.0  11649  -63.9  -69.9     44   0.02                329.5  329.6  329.6
  200.0  11770  -63.9  -70.9     38   0.01                331.4  331.5  331.4
  184.0  12278  -65.3  -74.3     28   0.01                337.1  337.2  337.1
  178.0  12481  -63.7  -75.7     18   0.01                343.0  343.0  343.0
  176.0  12550  -61.9  -75.9     14   0.01                347.0  347.1  347.0
  169.0  12803  -58.7  -81.7      4   0.00                356.4  356.4  356.4
  157.0  13263  -59.7  -89.7      1   0.00                362.3  362.3  362.3
  156.0  13303  -59.3  -89.3      1   0.00                363.6  363.6  363.6
  154.0  13384  -57.7  -89.7      1   0.00                367.7  367.7  367.7
  151.0  13508  -57.7  -89.7      1   0.00                369.8  369.8  369.8
  150.0  13550  -56.9  -88.9      1   0.00                371.8  371.9  371.8
  149.0  13593  -55.5  -87.5      1   0.00                375.0  375.0  375.0
  141.0  13945  -55.3  -87.3      1   0.00                381.3  381.3  381.3
  139.0  14036  -54.5  -87.5      1   0.00                384.2  384.2  384.2
  129.0  14513  -56.1  -88.1      1   0.00                389.6  389.7  389.6
  127.0  14612  -55.7  -87.7      1   0.00                392.1  392.1  392.1
  126.0  14663  -54.5  -87.5      1   0.00                395.2  395.2  395.2
  125.0  14714  -52.7  -85.7      1   0.00                399.3  399.4  399.3
  122.0  14872  -51.7  -84.7      1   0.00                403.9  403.9  403.9
  108.0  15653  -57.1  -89.1      1   0.00                408.1  408.1  408.1
  103.0  15953  -57.1  -89.1      1   0.00                413.6  413.6  413.6
  100.0  16140  -58.5  -89.5      1   0.00                414.4  414.4  414.4
   94.8  16475  -60.7  -91.7      1   0.00                416.5  416.5  416.5
   88.9  16877  -58.9  -89.9      1   0.00                427.8  427.8  427.8
   86.5  17051  -55.3  -87.3      1   0.00                438.4  438.4  438.4
   84.5  17200  -54.5  -87.5      1   0.00                443.0  443.0  443.0
   75.3  17932  -59.5  -90.5      1   0.00                447.3  447.3  447.3
   70.0  18390  -58.7  -89.7      1   0.00                458.5  458.5  458.5
   68.7  18507  -59.3  -90.3      1   0.00                459.6  459.6  459.6
   67.1  18655  -58.5  -89.5      1   0.00                464.5  464.5  464.5
   65.9  18768  -56.1  -88.1      1   0.00                472.1  472.1  472.1
   62.1  19143  -57.9  -89.9      1   0.00                476.2  476.2  476.2
   59.6  19402  -55.9  -87.9      1   0.00                486.3  486.3  486.3
   54.6  19955  -57.7  -89.7      1   0.00                494.5  494.5  494.5
   50.0  20510  -56.7  -88.7      1   0.00                509.4  509.4  509.4
   44.6  21236  -56.5  -88.5      1   0.00                526.8  526.9  526.8
   43.7  21367  -54.5  -87.5      1   0.00                534.8  534.8  534.8
   41.9  21636  -55.9  -87.9      1   0.00                537.8  537.8  537.8
   36.4  22535  -54.9  -86.9      1   0.01                562.4  562.5  562.4
   34.5  22877  -56.5  -88.5      1   0.00                566.9  567.0  566.9
   32.3  23298  -54.9  -86.9      1   0.01                582.0  582.0  582.0
   30.0  23770  -56.1  -88.1      1   0.01                591.1  591.2  591.1
   26.9  24458  -57.9  -89.9      1   0.00                604.8  604.8  604.8
   25.7  24747  -55.5  -87.5      1   0.01                619.5  619.6  619.5
   22.3  25647  -56.3  -88.3      1   0.01                642.8  642.9  642.8
   20.0  26330  -59.9  -90.9      1   0.01                652.1  652.1  652.1
   19.5  26488  -60.7  -91.7      1   0.00                654.4  654.4  654.4
   16.6  27487  -61.7  -91.7      1   0.01                681.9  682.0  682.0
</PRE><H3>Station information and sounding indices</H3><PRE>
                         Station identifier: TFX
                             Station number: 72776
                           Observation time: 210201/1200
                           Station latitude: 47.46
                          Station longitude: -111.39
                          Station elevation: 1134.0
                            Showalter index: 10.42
                               Lifted index: 11.55
    LIFT computed using virtual temperature: 11.53
                                    K index: 6.70
                         Cross totals index: 10.70
                      Vertical totals index: 24.70
                        Totals totals index: 35.40
      Convective Available Potential Energy: 0.00
             CAPE using virtual temperature: 0.00
                      Convective Inhibition: 0.00
             CINS using virtual temperature: 0.00
  Temp [K] of the Lifted Condensation Level: 261.80
Pres [hPa] of the Lifted Condensation Level: 701.03
   Equivalent potential temp [K] of the LCL: 296.75
     Mean mixed layer potential temperature: 289.79
              Mean mixed layer mixing ratio: 2.31
              1000 hPa to 500 hPa thickness: 5486.00
Precipitable water [mm] for entire sounding: 8.23
</PRE>
<H2>72776 TFX Great Falls Observations at 00Z 02 Feb 2021</H2>
<PRE>
-----------------------------------------------------------------------------
   PRES   HGHT   TEMP   DWPT   RELH   MIXR   DRCT   SKNT   THTA   THTE   THTV
    hPa     m      C      C      %    g/kg    deg   knot     K      K      K 
-----------------------------------------------------------------------------
 1000.0     98                                                               
  925.0    752                                                               
  883.0   1134   11.0   -6.0     30   2.78    210     13  294.4  303.0  294.9
  875.0   1210   10.8   -6.2     30   2.76    210     15  295.0  303.5  295.5
  850.0   1453    8.6   -7.4     31   2.59    210     22  295.1  303.1  295.6
  811.7   1829    6.0   -7.8     36   2.64    220     26  296.4  304.5  296.8
  782.0   2134    4.0   -8.1     41   2.68    230     34  297.3  305.6  297.8
  753.4   2438    1.9   -8.3     46   2.71    240     37  298.2  306.7  298.8
  749.0   2485    1.6   -8.4     47   2.72    241     37  298.4  306.9  298.9
  725.2   2743   -0.5   -9.1     52   2.66    245     38  298.8  307.1  299.3
  700.0   3026   -2.9   -9.9     59   2.59    255     41  299.2  307.4  299.7
  698.0   3048   -3.1   -9.9     59   2.60    245     43  299.3  307.4  299.7
  681.0   3243   -4.9   -9.8     69   2.68    242     39  299.4  307.8  299.9
  674.0   3325   -4.5   -8.9     71   2.91    240     38  300.7  309.8  301.2
  671.6   3353   -4.3   -9.4     67   2.80    240     37  301.2  310.1  301.8
  662.0   3466   -3.5  -11.5     54   2.41    246     37  303.4  311.1  303.8
  650.0   3611   -3.9  -12.9     50   2.19    253     36  304.5  311.6  304.9
  646.1   3658   -4.2  -13.3     49   2.13    255     36  304.7  311.6  305.1
  621.6   3962   -6.2  -16.1     45   1.76    265     30  305.8  311.6  306.1
  618.0   4008   -6.5  -16.5     45   1.71    265     30  306.0  311.6  306.3
  597.6   4267   -8.8  -17.1     51   1.68    265     29  306.2  311.8  306.5
  560.0   4768  -13.3  -18.3     66   1.62    261     27  306.7  312.1  307.0
  552.0   4877  -14.1  -19.1     66   1.54    260     27  307.0  312.1  307.2
  535.0   5114  -15.9  -20.9     65   1.36    252     30  307.6  312.2  307.8
  515.0   5400  -18.3  -21.6     75   1.33    243     33  308.1  312.5  308.3
  509.1   5486  -19.0  -22.2     76   1.28    240     34  308.2  312.5  308.5
  500.0   5620  -20.1  -23.1     77   1.20    240     35  308.5  312.5  308.7
  485.0   5845  -21.9  -24.4     80   1.10    240     37  308.9  312.7  309.2
  472.0   6044  -23.7  -24.4     94   1.13    240     39  309.1  313.0  309.3
  468.6   6096  -24.1  -24.7     94   1.11    240     40  309.3  313.1  309.5
  454.0   6327  -25.7  -26.2     96   1.00    242     41  310.1  313.5  310.3
  434.0   6651  -28.3  -31.1     77   0.66    245     43  310.8  313.1  310.9
  417.0   6936  -30.5  -32.1     86   0.63    247     45  311.5  313.8  311.7
  413.0   7004  -31.3  -34.9     70   0.48    248     46  311.4  313.1  311.5
  407.0   7108  -32.1  -36.6     64   0.41    249     46  311.6  313.1  311.7
  401.0   7212  -32.7  -42.7     36   0.22    250     47  312.2  313.0  312.2
  400.0   7230  -32.7  -45.7     26   0.16    250     47  312.4  313.0  312.4
  399.0   7248  -32.5  -48.5     19   0.12    250     47  312.9  313.4  312.9
  391.0   7390  -32.9  -44.9     29   0.18    248     48  314.2  314.9  314.2
  385.0   7499  -33.5  -46.5     26   0.15    247     49  314.8  315.4  314.8
  378.4   7620  -34.4  -49.1     21   0.12    245     50  315.2  315.7  315.2
  370.0   7777  -35.5  -52.5     16   0.08    246     51  315.7  316.1  315.7
  357.0   8026  -37.5  -54.5     15   0.07    246     53  316.3  316.6  316.3
  331.0   8543  -42.3  -53.3     29   0.08    248     57  316.6  316.9  316.6
  307.0   9047  -46.7  -57.7     27   0.05    250     60  317.3  317.5  317.3
  304.0   9112  -47.3  -56.3     35   0.06    250     61  317.4  317.6  317.4
  302.6   9144  -47.5  -55.4     40   0.07    250     61  317.5  317.8  317.5
  300.0   9200  -47.9  -53.9     50   0.08    250     62  317.7  318.1  317.8
  287.0   9491  -50.1  -53.4     68   0.09    252     65  318.6  319.0  318.6
  259.0  10155  -54.9  -59.9     54   0.05    258     71  321.1  321.2  321.1
  250.0  10380  -56.9  -61.9     53   0.04    260     73  321.4  321.5  321.4
  231.0  10875  -61.1  -66.1     51   0.02    260     77  322.3  322.4  322.3
  216.4  11278  -63.7  -68.7     50   0.02    260     80  324.4  324.4  324.4
  211.0  11433  -64.7  -69.7     50   0.02    260     80  325.1  325.2  325.1
  205.9  11582  -64.3  -69.3     50   0.02    265     95  328.0  328.1  328.0
  203.0  11669  -64.1  -69.1     50   0.02    265     95  329.7  329.8  329.7
  202.0  11699  -64.1  -69.4     48   0.02    265     95  330.2  330.2  330.2
  200.0  11760  -64.1  -70.1     44   0.02    265     93  331.1  331.2  331.1
  192.0  12009  -64.3  -72.3     33   0.01    263     83  334.7  334.7  334.7
  187.0  12171  -62.9  -73.9     21   0.01    262     77  339.4  339.5  339.5
  180.0  12405  -63.5  -77.5     13   0.01    261     68  342.2  342.2  342.2
  177.3  12497  -63.0  -78.8     10   0.00    260     64  344.5  344.6  344.6
  177.0  12508  -62.9  -78.9     10   0.00    260     64  344.8  344.9  344.8
  174.0  12614  -61.1  -79.1      8   0.00    259     67  349.5  349.5  349.5
  168.0  12831  -61.1  -83.1      4   0.00    256     71  353.0  353.0  353.0
  159.0  13173  -59.9  -85.9      2   0.00    253     79  360.6  360.6  360.6
  153.1  13411  -57.0  -85.9      1   0.00    250     84  369.6  369.6  369.6
  151.0  13498  -55.9  -85.9      1   0.00    253     81  372.9  372.9  372.9
  150.0  13540  -55.9  -85.9      1   0.00    255     80  373.6  373.6  373.6
  148.0  13625  -56.1  -87.1      1   0.00    257     79  374.6  374.7  374.6
  145.9  13716  -55.5  -87.0      1   0.00    260     77  377.3  377.3  377.3
  141.0  13934  -53.9  -86.9      1   0.00    261     71  383.7  383.7  383.7
  134.0  14261  -53.7  -86.7      1   0.00    264     62  389.7  389.7  389.7
  132.0  14358  -52.5  -85.5      1   0.00    264     60  393.5  393.5  393.5
  128.0  14557  -51.9  -84.9      1   0.00    266     54  398.1  398.1  398.1
  126.0  14659  -50.7  -84.7      1   0.00    266     52  402.0  402.1  402.0
  124.0  14763  -50.3  -84.3      1   0.00    267     49  404.6  404.6  404.6
  115.2  15240  -52.0  -85.2      1   0.00    270     36  409.9  410.0  409.9
  114.0  15309  -52.3  -85.3      1   0.00    270     33  410.7  410.7  410.7
  110.0  15540  -52.1  -85.1      1   0.00    270     23  415.3  415.3  415.3
  104.8  15850  -54.2  -86.7      1   0.00    270      9  417.0  417.1  417.0
  100.0  16150  -56.3  -88.3      1   0.00    265     12  418.7  418.7  418.7
   84.9  17182  -60.7  -91.7      1   0.00    260     37  429.8  429.8  429.8
   83.7  17271  -60.5  -91.5      1   0.00    262     38  432.0  432.0  432.0
   82.3  17374  -58.8  -90.2      1   0.00    265     39  437.4  437.4  437.4
   79.9  17564  -55.7  -87.7      1   0.00    279     32  447.6  447.6  447.6
   74.8  17983  -58.2  -89.6      1   0.00    310     16  450.9  450.9  450.9
   71.5  18267  -59.9  -90.9      1   0.00    268     17  453.1  453.1  453.1
   71.3  18288  -59.9  -90.9      1   0.00    265     17  453.6  453.6  453.6
   70.0  18400  -59.7  -90.7      1   0.00    270     20  456.3  456.3  456.3
   69.3  18463  -59.1  -90.1      1   0.00    275     21  458.9  458.9  458.9
   67.9  18593  -57.4  -88.8      1   0.00    285     23  465.4  465.4  465.4
   65.9  18780  -54.9  -86.9      1   0.00    300     11  474.7  474.7  474.7
   64.7  18898  -55.5  -87.5      1   0.00    310      4  475.9  475.9  475.9
   61.0  19269  -57.5  -89.5      1   0.00    300      3  479.5  479.5  479.5
   57.2  19675  -55.1  -87.1      1   0.00    289      1  493.8  493.9  493.8
   56.0  19812  -55.2  -87.2      1   0.00    285      1  496.7  496.8  496.7
   50.0  20530  -55.5  -87.5      1   0.00    350      5  512.2  512.3  512.2
   46.2  21031  -55.7  -87.7      1   0.00    350     10  523.4  523.4  523.4
   44.1  21336  -55.8  -87.8      1   0.00     20     14  530.3  530.3  530.3
   40.0  21946  -56.1  -88.1      1   0.00     45     20  544.4  544.4  544.4
   39.7  22001  -56.1  -88.1      1   0.00     45     21  545.6  545.7  545.6
   38.2  22247  -54.9  -86.9      1   0.01     46     23  554.7  554.8  554.7
   33.1  23165  -55.6  -87.6      1   0.01     50     31  576.1  576.1  576.1
   30.1  23774  -56.1  -88.1      1   0.01     65     24  590.7  590.8  590.7
   30.0  23790  -56.1  -88.1      1   0.01     65     24  591.1  591.2  591.1
   29.5  23896  -56.5  -88.5      1   0.01     59     27  592.9  592.9  592.9
   28.7  24079  -56.4  -88.4      1   0.01     50     31  598.0  598.0  598.0
   27.3  24384  -56.3  -88.3      1   0.01     55     36  606.7  606.7  606.7
   26.0  24689  -56.1  -88.1      1   0.01     70     33  615.5  615.5  615.5
   24.8  24994  -56.0  -88.0      1   0.01     65     27  624.4  624.5  624.4
   21.1  26020  -55.5  -87.5      1   0.01     52     35  655.4  655.5  655.5
   20.5  26213  -56.0  -88.0      1   0.01     50     36  659.8  659.9  659.8
   20.0  26360  -56.3  -88.3      1   0.01     55     37  663.1  663.2  663.1
   16.9  27432  -57.2  -89.2      1   0.01     55     52  693.0  693.1  693.0
   16.1  27737  -57.5  -89.5      1   0.01     55     57  701.8  701.8  701.8
   14.9  28224  -57.9  -89.9      1   0.01     56     57  716.0  716.1  716.0
   13.0  29079  -61.1  -92.1      1   0.01     58     56  733.4  733.4  733.4
   12.1  29527  -59.3  -90.3      1   0.01     59     55  754.9  755.0  754.9
   11.4  29870  -60.5  -91.2      1   0.01     60     55  762.5  762.6  762.5
   10.4  30480  -62.7  -92.9      1   0.01     60     74  776.3  776.4  776.3
   10.1  30648  -63.3  -93.3      1   0.01     60     74  780.1  780.2  780.1
   10.0  30710  -63.5  -93.5      1   0.01     60     73  781.5  781.6  781.5
    9.5  31023  -65.3  -94.3      1   0.01     56     64  786.2  786.3  786.2
    9.4  31090  -65.0  -94.2      1   0.01     55     62  789.7  789.8  789.7
    8.9  31422  -63.7  -93.7      1   0.01     55     71  807.2  807.3  807.2
    8.1  32004  -65.0  -95.0      1   0.01     55     87  824.5  824.6  824.5
    8.0  32073  -65.1  -95.1      1   0.01                826.6  826.7  826.6
</PRE><H3>Station information and sounding indices</H3><PRE>
                         Station identifier: TFX
                             Station number: 72776
                           Observation time: 210202/0000
                           Station latitude: 47.46
                          Station longitude: -111.39
                          Station elevation: 1134.0
                            Showalter index: 6.73
                               Lifted index: 6.59
    LIFT computed using virtual temperature: 6.65
                                SWEAT index: 166.52
                                    K index: 14.30
                         Cross totals index: 12.70
                      Vertical totals index: 28.70
                        Totals totals index: 41.40
      Convective Available Potential Energy: 0.00
             CAPE using virtual temperature: 0.00
                      Convective Inhibition: 0.00
             CINS using virtual temperature: 0.00
                     Bulk Richardson Number: 0.00
          Bulk Richardson Number using CAPV: 0.00
  Temp [K] of the Lifted Condensation Level: 263.00
Pres [hPa] of the Lifted Condensation Level: 667.99
   Equivalent potential temp [K] of the LCL: 303.31
     Mean mixed layer potential temperature: 295.14
              Mean mixed layer mixing ratio: 2.66
              1000 hPa to 500 hPa thickness: 5522.00
Precipitable water [mm] for entire sounding: 9.77
</PRE>
<H2>72776 TFX Great Falls Observations at 12Z 02 Feb 2021</H2>
<PRE>
-----------------------------------------------------------------------------
   PRES   HGHT   TEMP   DWPT   RELH   MIXR   DRCT   SKNT   THTA   THTE   THTV
    hPa     m      C      C      %    g/kg    deg   knot     K      K      K 
-----------------------------------------------------------------------------
 1000.0     67                                                               
  925.0    725                                                               
  881.0   1134    7.6   -5.4     39   2.92    215     14  291.1  299.9  291.6
  880.0   1143    8.6   -5.4     37   2.92    215     14  292.2  301.1  292.8
  878.0   1162    9.0   -6.0     34   2.79    215     15  292.8  301.4  293.3
  867.0   1265    8.4   -6.6     34   2.70    215     21  293.3  301.5  293.8
  851.0   1417    9.2   -5.8     34   2.93    215     29  295.7  304.7  296.2
  850.0   1427    9.2   -5.8     34   2.93    215     29  295.8  304.8  296.3
  808.9   1829    6.1   -6.9     39   2.83    230     40  296.6  305.4  297.2
  779.0   2134    3.7   -7.7     43   2.76    240     38  297.3  305.9  297.8
  750.3   2438    1.3   -8.6     48   2.68    250     39  297.9  306.3  298.4
  722.6   2743   -1.1   -9.4     53   2.61    255     38  298.5  306.7  299.0
  700.0   3000   -3.1  -10.1     58   2.55    255     40  299.0  307.0  299.5
  688.0   3137   -4.1  -10.1     63   2.59    256     40  299.4  307.5  299.9
  661.0   3452   -5.9  -12.9     58   2.15    258     41  300.8  307.7  301.2
  643.8   3658   -6.0  -15.9     45   1.72    260     41  302.9  308.6  303.3
  632.0   3803   -6.1  -18.1     38   1.46    260     41  304.5  309.3  304.7
  595.1   4267   -9.9  -20.9     40   1.22    260     40  305.4  309.5  305.6
  558.0   4764  -13.9  -23.9     43   1.00    248     40  306.3  309.7  306.5
  551.0   4860  -13.9  -27.9     30   0.70    245     40  307.4  309.8  307.5
  549.8   4877  -14.0  -28.1     30   0.69    245     40  307.4  309.8  307.5
  500.0   5590  -19.9  -34.9     25   0.40    250     36  308.7  310.1  308.8
  493.0   5694  -20.9  -34.9     27   0.40    252     36  308.7  310.2  308.8
  469.0   6060  -23.3  -41.3     18   0.22    259     34  310.2  311.0  310.2
  466.7   6096  -23.6  -41.3     18   0.22    260     34  310.2  311.1  310.3
  447.0   6408  -26.5  -41.5     23   0.23    259     41  310.4  311.3  310.5
  439.0   6538  -26.9  -45.9     15   0.14    258     44  311.6  312.1  311.6
  400.0   7200  -32.3  -48.3     19   0.12    255     58  312.9  313.4  312.9
  397.0   7253  -32.7  -47.7     21   0.13    253     61  313.1  313.6  313.1
  393.5   7315  -33.0  -48.8     19   0.12    250     64  313.5  314.0  313.5
  377.0   7615  -34.3  -54.3     11   0.06    250     66  315.6  315.9  315.6
  376.7   7620  -34.3  -54.3     11   0.06    250     66  315.6  315.9  315.7
  327.0   8598  -39.5  -63.5      6   0.02    250     74  321.6  321.7  321.6
  301.6   9144  -44.6  -61.1     14   0.03    250     78  321.9  322.1  321.9
  300.0   9180  -44.9  -60.9     15   0.04    250     78  322.0  322.1  322.0
  282.0   9590  -48.7  -61.7     21   0.03    247     81  322.2  322.4  322.3
  269.0   9898  -51.1  -65.1     17   0.02    245     83  323.1  323.2  323.1
  266.0   9971  -51.7  -64.7     20   0.02    245     84  323.2  323.3  323.2
  262.4  10058  -52.5  -64.2     23   0.03    245     84  323.3  323.4  323.3
  258.0  10168  -53.5  -63.5     28   0.03    247     83  323.5  323.6  323.5
  252.0  10319  -54.7  -62.7     36   0.03    249     82  323.9  324.0  323.9
  250.0  10370  -54.9  -63.9     32   0.03    250     82  324.3  324.4  324.3
  245.0  10499  -56.1  -66.1     27   0.02    250     81  324.4  324.5  324.4
  200.0  11760  -65.5  -73.5     32   0.01    245     76  328.9  328.9  328.9
  197.0  11852  -66.3  -74.3     32   0.01    245     76  329.0  329.1  329.0
  182.0  12333  -67.1  -75.1     31   0.01    248     76  335.3  335.3  335.3
  174.0  12607  -64.5  -75.5     21   0.01    250     76  343.9  343.9  343.9
  167.0  12859  -64.1  -78.1     13   0.01    252     76  348.6  348.6  348.6
  165.0  12933  -62.3  -79.3      9   0.00    252     76  352.8  352.8  352.8
  162.0  13047  -61.7  -81.7      5   0.00    253     76  355.7  355.7  355.7
  159.0  13164  -59.7  -81.7      4   0.00    253     76  361.0  361.0  361.0
  152.9  13411  -59.6  -87.0      2   0.00    255     76  365.3  365.3  365.3
  150.0  13530  -59.5  -89.5      1   0.00    255     76  367.4  367.4  367.4
  146.0  13700  -57.5  -89.5      1   0.00    255     73  373.7  373.7  373.7
  144.0  13788  -55.5  -87.5      1   0.00    255     72  378.6  378.6  378.6
  139.0  14012  -56.3  -88.3      1   0.00    255     69  381.1  381.1  381.1
  134.0  14246  -54.1  -87.1      1   0.00    255     65  389.0  389.0  389.0
  133.0  14295  -51.3  -84.3      1   0.00    255     64  394.8  394.8  394.8
  132.4  14326  -50.5  -83.9      1   0.00    255     64  396.7  396.7  396.7
  131.0  14394  -48.9  -82.9      1   0.00    259     60  400.8  400.8  400.8
  129.0  14495  -49.1  -83.1      1   0.00    264     55  402.2  402.2  402.2
  126.0  14650  -47.5  -82.5      1   0.00    273     47  407.8  407.9  407.8
  116.0  15192  -51.5  -84.5      1   0.00    302     18  410.2  410.2  410.2
  115.1  15240  -51.4  -84.6      1   0.00    305     15  411.3  411.3  411.3
  112.0  15420  -50.9  -84.9      1   0.00    296     17  415.4  415.4  415.4
  102.0  16023  -54.9  -86.9      1   0.00    266     23  419.0  419.0  419.0
  100.0  16150  -54.3  -87.3      1   0.00    260     24  422.5  422.6  422.5
   98.7  16234  -53.1  -86.1      1   0.00    263     24  426.4  426.5  426.4
   91.1  16745  -56.5  -88.5      1   0.00    279     24  429.6  429.6  429.6
   83.4  17303  -57.7  -89.7      1   0.00    298     23  438.1  438.1  438.1
   82.5  17374  -57.8  -89.8      1   0.00    300     23  439.4  439.4  439.4
   80.7  17510  -57.9  -89.9      1   0.00    288     19  441.8  441.9  441.8
   79.5  17604  -56.5  -88.5      1   0.00    279     16  446.6  446.6  446.6
   74.8  17983  -59.1  -90.6      1   0.00    245      4  449.0  449.0  449.0
   71.3  18288  -61.2  -92.2      1   0.00    240      9  450.9  450.9  450.9
   70.7  18338  -61.5  -92.5      1   0.00    250     10  451.2  451.2  451.2
   70.0  18400  -60.7  -91.7      1   0.00    265     12  454.2  454.2  454.2
   68.9  18499  -58.5  -89.5      1   0.00    267     12  461.0  461.0  461.0
   64.7  18898  -58.4  -89.4      1   0.00    275     13  469.7  469.7  469.7
   62.8  19082  -58.3  -89.3      1   0.00    281     15  473.8  473.8  473.8
   61.6  19202  -56.4  -87.9      1   0.00    285     16  480.6  480.6  480.6
   60.6  19308  -54.7  -86.7      1   0.00    280     15  486.6  486.7  486.6
   58.8  19507  -54.2  -86.8      1   0.00    270     12  492.1  492.1  492.1
   57.7  19622  -53.9  -86.9      1   0.00    283     11  495.3  495.4  495.3
   53.4  20117  -54.3  -87.3      1   0.00    340      6  505.4  505.4  505.4
   50.1  20527  -54.7  -87.7      1   0.00      4     11  513.8  513.9  513.8
   50.0  20540  -54.7  -86.7      1   0.00      5     11  514.1  514.2  514.1
   47.2  20907  -56.9  -88.9      1   0.00     12     13  517.4  517.4  517.4
   44.1  21336  -55.8  -88.1      1   0.00     20     15  530.2  530.3  530.2
   39.6  22027  -53.9  -86.9      1   0.01     27     18  551.6  551.6  551.6
   30.0  23800  -56.7  -88.7      1   0.01     45     26  589.5  589.5  589.5
   27.9  24259  -57.3  -89.3      1   0.00     53     25  600.2  600.2  600.2
   27.4  24384  -56.9  -88.9      1   0.01     55     25  604.7  604.7  604.7
   25.6  24805  -55.5  -87.5      1   0.01     52     28  620.2  620.3  620.2
   21.9  25796  -57.3  -89.3      1   0.01     44     34  643.1  643.2  643.1
   20.0  26370  -56.7  -88.7      1   0.01     40     37  661.9  662.0  661.9
   16.9  27432  -56.5  -88.5      1   0.01     45     45  695.1  695.2  695.1
   14.9  28234  -56.3  -88.3      1   0.01     48     52  721.3  721.4  721.3
   13.9  28651  -56.9  -88.8      1   0.01     50     56  732.9  733.1  732.9
   10.0  30740  -60.1  -91.1      1   0.01                794.2  794.3  794.2
    9.1  31326  -61.3  -92.3      1   0.01                811.3  811.4  811.3
</PRE><H3>Station information and sounding indices</H3><PRE>
                         Station identifier: TFX
                             Station number: 72776
                           Observation time: 210202/1200
                           Station latitude: 47.46
                          Station longitude: -111.39
                          Station elevation: 1134.0
                            Showalter index: 5.83
                               Lifted index: 6.76
    LIFT computed using virtual temperature: 6.69
                                SWEAT index: 190.69
                                    K index: 16.30
                         Cross totals index: 14.10
                      Vertical totals index: 29.10
                        Totals totals index: 43.20
      Convective Available Potential Energy: 0.00
             CAPE using virtual temperature: 0.00
                      Convective Inhibition: 0.00
             CINS using virtual temperature: 0.00
                     Bulk Richardson Number: 0.00
          Bulk Richardson Number using CAPV: 0.00
  Temp [K] of the Lifted Condensation Level: 264.07
Pres [hPa] of the Lifted Condensation Level: 681.27
   Equivalent potential temp [K] of the LCL: 303.36
     Mean mixed layer potential temperature: 294.69
              Mean mixed layer mixing ratio: 2.84
              1000 hPa to 500 hPa thickness: 5523.00
Precipitable water [mm] for entire sounding: 8.16
</PRE>
<H2>72776 TFX Great Falls Observations at 00Z 03 Feb 2021</H2>
<PRE>
-----------------------------------------------------------------------------
   PRES   HGHT   TEMP   DWPT   RELH   MIXR   DRCT   SKNT   THTA   THTE   THTV
    hPa     m      C      C      %    g/kg    deg   knot     K      K      K 
-----------------------------------------------------------------------------
 1000.0     34                                                               
  925.0    692                                                               
  877.0   1134   10.6   -0.4     46   4.26    210     22  294.6  307.4  295.4
  864.0   1259    9.2   -1.8     46   3.90    210     27  294.4  306.1  295.1
  850.0   1395    8.0   -3.0     46   3.62    210     32  294.5  305.5  295.2
  806.0   1829    4.8   -4.4     51   3.43    215     31  295.6  306.0  296.2
  778.0   2118    2.6   -5.4     56   3.30    224     34  296.2  306.4  296.9
  776.5   2134    2.5   -5.5     56   3.28    225     34  296.3  306.4  296.9
  747.6   2438    0.4   -7.6     55   2.90    235     40  297.2  306.2  297.8
  719.7   2743   -1.7   -9.7     54   2.55    245     41  298.2  306.2  298.6
  707.0   2885   -2.7  -10.7     54   2.40    245     42  298.6  306.2  299.1
  700.0   2964   -3.5  -11.5     54   2.28    245     43  298.6  305.8  299.0
  673.0   3273   -5.9  -13.9     53   1.95    245     43  299.3  305.5  299.6
  666.1   3353   -6.7  -14.1     55   1.94    245     43  299.3  305.5  299.7
  640.5   3658   -9.5  -14.8     65   1.90    245     43  299.4  305.5  299.8
  637.0   3700   -9.9  -14.9     67   1.90    245     43  299.4  305.5  299.8
  603.0   4121  -11.9  -20.9     47   1.21    249     48  301.9  305.9  302.1
  591.6   4267  -13.2  -21.7     49   1.15    250     49  302.0  305.9  302.2
  580.0   4417  -14.5  -22.5     51   1.09    249     50  302.2  305.8  302.4
  569.0   4562  -15.7  -24.7     46   0.91    248     51  302.5  305.5  302.6
  545.5   4877  -18.5  -25.7     53   0.87    245     54  302.8  305.7  302.9
  531.0   5078  -20.3  -26.3     59   0.85    240     53  303.0  305.9  303.1
  502.3   5486  -23.4  -28.4     64   0.74    230     51  304.0  306.5  304.1
  500.0   5520  -23.7  -28.6     64   0.73    230     51  304.1  306.6  304.2
  482.0   5788  -25.7  -30.2     66   0.65    232     51  304.8  307.1  304.9
  471.0   5956  -26.9  -31.9     63   0.56    234     52  305.4  307.3  305.5
  461.9   6096  -28.0  -32.8     64   0.53    235     52  305.7  307.5  305.8
  450.0   6285  -29.5  -33.9     66   0.49    235     53  306.1  307.8  306.2
  442.7   6401  -30.2  -34.9     63   0.45    235     53  306.6  308.2  306.7
  416.0   6844  -32.9  -38.9     55   0.32    241     63  308.7  309.8  308.7
  400.0   7120  -34.7  -40.7     54   0.27    245     69  309.8  310.8  309.9
  392.0   7261  -35.9  -40.8     61   0.28    245     70  310.0  311.1  310.1
  388.9   7315  -36.2  -41.1     60   0.27    245     71  310.3  311.3  310.4
  372.0   7620  -38.0  -42.9     60   0.23    245     77  311.9  312.8  312.0
  344.0   8157  -41.1  -46.1     58   0.18    243     80  314.8  315.5  314.8
  324.0   8561  -44.7  -51.7     45   0.10    242     83  315.2  315.6  315.3
  300.0   9070  -49.5  -55.5     49   0.07    240     86  315.5  315.8  315.5
  296.6   9144  -50.2  -56.2     49   0.06    245     86  315.5  315.8  315.6
  296.0   9157  -50.3  -56.3     49   0.06    245     86  315.6  315.8  315.6
  286.0   9380  -51.3  -56.0     57   0.07    244     89  317.2  317.5  317.2
  275.0   9632  -53.1  -61.1     37   0.04    243     93  318.2  318.4  318.2
  270.0   9749  -53.7  -64.7     25   0.02    242     94  319.0  319.1  319.0
  258.0  10039  -54.3  -70.3     12   0.01    241     98  322.3  322.4  322.3
  253.0  10164  -53.5  -72.5      8   0.01    240    100  325.3  325.3  325.3
  250.0  10240  -53.9  -73.9      7   0.01    240    101  325.8  325.8  325.8
  245.2  10363  -54.5  -74.5      7   0.01    240    101  326.6  326.6  326.6
  233.0  10692  -56.3  -76.3      6   0.01    240     98  328.8  328.8  328.8
  222.0  11001  -54.7  -77.7      4   0.00    242     98  335.8  335.8  335.8
  219.0  11088  -55.3  -80.3      3   0.00    242     98  336.2  336.2  336.2
  203.0  11574  -55.3  -86.3      1   0.00    245     98  343.6  343.6  343.6
  202.8  11582  -55.2  -86.4      1   0.00    245     98  343.8  343.8  343.8
  200.0  11670  -54.7  -87.7      1   0.00    245     85  346.0  346.0  346.0
  197.0  11767  -52.9  -85.9      1   0.00    245     82  350.4  350.4  350.4
  194.0  11866  -53.1  -86.1      1   0.00    245     79  351.6  351.6  351.6
  193.4  11887  -53.0  -86.0      1   0.00    245     78  352.0  352.0  352.0
  183.0  12243  -51.9  -84.9      1   0.00    248     74  359.4  359.4  359.4
  163.0  12988  -54.9  -86.9      1   0.00    254     65  366.5  366.5  366.5
  160.0  13106  -54.7  -87.0      1   0.00    255     64  368.8  368.8  368.8
  154.0  13351  -54.3  -87.3      1   0.00    243     55  373.5  373.5  373.5
  152.6  13411  -53.8  -86.8      1   0.00    240     53  375.3  375.3  375.3
  151.0  13477  -53.3  -86.3      1   0.00    240     53  377.3  377.3  377.3
  150.0  13520  -53.3  -86.3      1   0.00    240     53  378.0  378.0  378.0
  145.0  13738  -53.1  -86.1      1   0.00    233     55  382.1  382.1  382.1
  140.0  13963  -54.9  -86.9      1   0.00    227     57  382.8  382.8  382.8
  138.7  14021  -54.9  -86.9      1   0.00    225     58  383.8  383.8  383.8
  135.0  14195  -54.9  -86.9      1   0.00    235     60  386.8  386.8  386.8
  133.0  14291  -52.1  -85.1      1   0.00    241     61  393.4  393.4  393.4
  128.0  14539  -51.1  -85.1      1   0.00    255     63  399.5  399.5  399.5
  126.2  14630  -51.3  -85.0      1   0.00    260     64  400.7  400.8  400.7
  123.0  14797  -51.7  -84.7      1   0.00    266     57  403.0  403.0  403.0
  120.0  14958  -49.7  -83.7      1   0.00    271     51  409.5  409.5  409.5
  116.0  15179  -49.1  -83.1      1   0.00    278     41  414.6  414.6  414.6
  114.9  15240  -49.4  -83.3      1   0.00    280     39  415.2  415.3  415.2
  111.0  15467  -50.3  -84.3      1   0.00    258     26  417.6  417.6  417.6
  109.7  15545  -50.9  -84.7      1   0.00    250     21  417.9  417.9  417.9
  106.0  15766  -52.7  -85.7      1   0.00    244     27  418.6  418.6  418.6
  100.0  16140  -54.1  -87.1      1   0.00    235     36  422.9  422.9  422.9
   99.8  16154  -54.2  -87.2      1   0.00    235     37  423.0  423.0  423.0
   92.1  16663  -57.5  -89.5      1   0.00    245     33  426.3  426.3  426.3
   82.5  17360  -56.1  -88.1      1   0.00    258     27  442.7  442.8  442.7
   77.7  17740  -57.7  -89.7      1   0.00    265     24  447.1  447.1  447.1
   72.1  18212  -56.9  -88.9      1   0.00    274     21  458.4  458.4  458.4
   71.2  18288  -56.4  -88.4      1   0.00    275     20  461.0  461.0  461.0
   70.0  18400  -55.7  -87.7      1   0.00    275     18  464.9  464.9  464.9
   67.7  18614  -53.9  -86.9      1   0.00    271     13  473.2  473.2  473.2
   64.8  18898  -55.5  -87.9      1   0.00    265      6  475.7  475.7  475.7
   63.1  19064  -56.5  -88.5      1   0.00    275      6  477.1  477.1  477.1
   61.0  19280  -55.1  -87.1      1   0.00    289      7  484.9  484.9  484.9
   55.1  19929  -56.5  -88.5      1   0.00    329      7  495.9  496.0  495.9
   53.6  20105  -55.3  -87.3      1   0.00    340      8  502.6  502.7  502.6
   51.0  20422  -55.3  -87.3      1   0.00      0      8  509.8  509.8  509.8
   50.0  20550  -55.3  -87.3      1   0.00    345      8  512.7  512.8  512.7
   48.6  20726  -55.0  -87.3      1   0.00    325     11  517.5  517.5  517.5
   46.4  21031  -54.5  -87.3      1   0.00    355     12  525.9  525.9  525.9
   45.7  21124  -54.3  -87.3      1   0.00      3     13  528.5  528.5  528.5
   44.2  21336  -55.1  -87.6      1   0.00     20     16  531.5  531.5  531.5
   42.9  21527  -55.9  -87.9      1   0.00     11     14  534.2  534.2  534.2
   42.1  21641  -55.7  -87.8      1   0.00      5     12  537.3  537.3  537.3
   35.2  22789  -54.1  -87.1      1   0.01     24     24  569.9  570.0  569.9
   31.1  23580  -55.3  -87.3      1   0.01     37     33  587.2  587.3  587.2
   30.2  23774  -54.8  -87.6      1   0.01     40     35  593.7  593.8  593.7
   30.0  23810  -54.7  -87.7      1   0.01     35     38  594.9  595.0  594.9
   28.8  24079  -54.4  -87.4      1   0.01     25     34  603.0  603.0  603.0
   26.2  24689  -53.6  -86.7      1   0.01     45     35  621.6  621.7  621.6
   25.0  24982  -53.3  -86.3      1   0.01     42     37  630.8  630.8  630.8
   24.2  25191  -54.5  -87.5      1   0.01     40     39  633.2  633.2  633.2
   22.7  25603  -53.6  -86.6      1   0.01     35     42  647.4  647.5  647.4
   21.6  25908  -53.0  -86.0      1   0.01     50     44  658.2  658.3  658.2
   20.3  26324  -52.1  -85.1      1   0.01     50     51  673.1  673.2  673.1
   20.0  26420  -52.3  -85.3      1   0.01     50     53  675.3  675.5  675.3
   17.1  27432  -54.6  -86.9      1   0.01     50     59  699.3  699.4  699.3
   16.3  27737  -55.2  -87.3      1   0.01     55     63  706.6  706.8  706.6
   16.0  27850  -55.5  -87.5      1   0.01     55     63  709.4  709.5  709.4
   12.2  29574  -55.7  -87.7      1   0.01     55     60  765.8  766.0  765.8
   10.6  30480  -57.7  -89.0      1   0.01     55     59  790.4  790.6  790.4
   10.0  30830  -58.5  -89.5      1   0.01     50     59  800.1  800.3  800.1
    9.8  30957  -58.7  -89.7      1   0.01     50     58  804.0  804.2  804.0
    9.6  31090  -58.5  -89.7      1   0.01     50     57  809.7  809.9  809.7
    8.9  31563  -57.7  -89.7      1   0.01     53     64  830.3  830.5  830.3
    8.0  32232  -59.7  -90.7      1   0.01     57     73  848.0  848.2  848.0
    7.5  32637  -58.5  -89.5      1   0.02                868.7  868.9  868.7
    7.5  32614  -58.6  -89.6      1   0.02     60     79  867.5  867.7  867.5
    7.2  32892  -60.1  -91.1      1   0.01                872.3  872.5  872.3
</PRE><H3>Station information and sounding indices</H3><PRE>
                         Station identifier: TFX
                             Station number: 72776
                           Observation time: 210203/0000
                           Station latitude: 47.46
                          Station longitude: -111.39
                          Station elevation: 1134.0
                            Showalter index: 1.58
                               Lifted index: 1.28
    LIFT computed using virtual temperature: 1.24
                                SWEAT index: 250.73
                                    K index: 20.70
                         Cross totals index: 20.70
                      Vertical totals index: 31.70
                        Totals totals index: 52.40
      Convective Available Potential Energy: 0.00
             CAPE using virtual temperature: 0.00
                      Convective Inhibition: 0.00
             CINS using virtual temperature: 0.00
                     Bulk Richardson Number: 0.00
          Bulk Richardson Number using CAPV: 0.00
  Temp [K] of the Lifted Condensation Level: 268.38
Pres [hPa] of the Lifted Condensation Level: 721.49
   Equivalent potential temp [K] of the LCL: 305.88
     Mean mixed layer potential temperature: 294.63
              Mean mixed layer mixing ratio: 3.75
              1000 hPa to 500 hPa thickness: 5486.00
Precipitable water [mm] for entire sounding: 9.35
</PRE>
<H2>72776 TFX Great Falls Observations at 12Z 03 Feb 2021</H2>
<PRE>
-----------------------------------------------------------------------------
   PRES   HGHT   TEMP   DWPT   RELH   MIXR   DRCT   SKNT   THTA   THTE   THTV
    hPa     m      C      C      %    g/kg    deg   knot     K      K      K 
-----------------------------------------------------------------------------
 1000.0     77                                                               
  925.0    723                                                               
  880.0   1134   -1.9   -7.9     64   2.41    250      9  281.3  288.4  281.8
  874.0   1188    0.2   -7.8     55   2.44    252     12  284.1  291.3  284.5
  852.0   1391   -0.1  -11.1     43   1.93    259     21  285.8  291.7  286.2
  850.0   1410   -0.3  -11.3     43   1.91    260     22  285.8  291.6  286.1
  805.7   1829   -4.0  -14.0     46   1.61    270     28  286.3  291.2  286.6
  774.9   2134   -6.7  -16.0     47   1.42    270     27  286.6  291.0  286.9
  745.3   2438   -9.3  -18.0     49   1.25    275     21  286.9  290.8  287.1
  718.0   2730  -11.9  -19.9     51   1.10    275     23  287.2  290.6  287.4
  716.8   2743  -11.9  -20.3     50   1.07    275     23  287.3  290.6  287.4
  700.0   2924  -12.7  -25.7     33   0.68    295     26  288.4  290.6  288.5
  690.0   3034  -13.5  -29.5     25   0.48    299     29  288.7  290.3  288.8
  688.7   3048  -13.6  -29.6     25   0.48    300     29  288.7  290.3  288.8
  634.9   3658  -17.7  -33.7     23   0.35    280     29  290.9  292.1  290.9
  596.0   4132  -20.9  -36.9     22   0.27    264     44  292.4  293.4  292.5
  585.2   4267  -21.5  -35.4     27   0.32    260     48  293.3  294.4  293.4
  570.0   4461  -22.3  -33.3     36   0.41    254     52  294.6  295.9  294.6
  561.4   4572  -23.1  -31.3     47   0.50    250     55  294.9  296.6  295.0
  555.0   4656  -23.7  -29.7     58   0.59    249     55  295.1  297.1  295.3
  538.4   4877  -25.8  -29.5     71   0.62    245     55  295.2  297.3  295.4
  525.0   5060  -27.5  -29.3     85   0.65    240     58  295.3  297.5  295.4
  518.0   5156  -27.9  -29.2     89   0.66    237     59  296.0  298.2  296.1
  504.0   5353  -28.5  -31.0     79   0.57    232     62  297.6  299.5  297.7
  500.0   5410  -28.9  -33.9     62   0.44    230     63  297.7  299.3  297.8
  499.0   5424  -28.9  -34.9     56   0.40    230     63  297.9  299.3  298.0
  492.0   5525  -29.9  -35.9     56   0.36    230     63  297.9  299.2  298.0
  475.0   5775  -31.9  -35.6     70   0.39    230     64  298.4  299.8  298.5
  455.0   6078  -34.7  -38.3     70   0.31    230     65  298.6  299.7  298.7
  453.9   6096  -34.8  -38.8     67   0.30    230     65  298.7  299.7  298.7
  450.0   6156  -35.3  -40.3     60   0.25    230     66  298.8  299.7  298.9
  444.0   6249  -35.9  -47.9     28   0.11    229     68  299.2  299.6  299.2
  439.0   6328  -36.5  -50.5     22   0.09    229     69  299.4  299.7  299.4
  430.0   6472  -37.1  -51.1     22   0.08    228     72  300.4  300.7  300.4
  421.0   6618  -37.5  -58.5      9   0.03    227     75  301.7  301.9  301.7
  400.0   6970  -39.9  -63.9      6   0.02    225     82  303.1  303.1  303.1
  397.6   7010  -40.1  -64.8      5   0.02    225     82  303.3  303.4  303.3
  379.0   7336  -41.5  -72.5      2   0.01    228     79  305.6  305.7  305.6
  363.2   7620  -43.0  -72.7      2   0.01    230     76  307.4  307.5  307.4
  346.9   7925  -44.5  -72.9      3   0.01    230     66  309.4  309.4  309.4
  316.6   8534  -47.7  -73.2      4   0.01    240     87  313.2  313.2  313.2
  311.0   8654  -48.3  -73.3      4   0.01    240     87  313.9  313.9  313.9
  300.0   8890  -48.5  -75.5      3   0.00    240     84  316.9  316.9  316.9
  291.0   9089  -49.1  -78.1      2   0.00    245     83  318.8  318.8  318.8
  288.6   9144  -48.7  -77.7      2   0.00    245     83  320.1  320.1  320.1
  287.0   9180  -48.5  -77.5      2   0.00    245     83  320.9  320.9  320.9
  250.0  10080  -50.7  -77.7      3   0.00    235     74  330.6  330.6  330.6
  238.0  10400  -51.3  -80.3      2   0.00    240     74  334.3  334.4  334.3
  236.0  10455  -50.9  -79.9      2   0.00    241     74  335.8  335.8  335.8
  234.0  10510  -49.3  -78.3      2   0.00    242     74  339.0  339.0  339.0
  230.0  10623  -48.5  -80.5      1   0.00    244     74  341.9  341.9  341.9
  228.4  10668  -48.6  -80.9      1   0.00    245     74  342.3  342.3  342.3
  218.0  10975  -49.7  -83.7      1   0.00    245     63  345.3  345.3  345.3
  210.0  11220  -48.7  -82.7      1   0.00    245     54  350.6  350.6  350.6
  208.2  11278  -48.8  -82.8      1   0.00    245     52  351.3  351.4  351.3
  200.0  11540  -49.1  -83.1      1   0.00    240     54  354.9  354.9  354.9
  195.0  11706  -49.7  -83.7      1   0.00    244     54  356.5  356.5  356.5
  182.0  12160  -48.9  -82.9      1   0.00    254     53  364.9  364.9  364.9
  181.1  12192  -49.0  -83.0      1   0.00    255     53  365.3  365.3  365.3
  175.0  12418  -49.3  -83.3      1   0.00    249     55  368.3  368.3  368.3
  173.0  12494  -48.7  -82.7      1   0.00    248     56  370.5  370.5  370.5
  168.0  12688  -48.5  -82.5      1   0.00    243     58  374.0  374.0  374.0
  166.0  12767  -47.7  -82.7      1   0.00    241     59  376.6  376.6  376.6
  165.1  12802  -47.9  -82.6      1   0.00    240     59  376.8  376.8  376.8
  163.0  12887  -48.5  -82.5      1   0.00    244     60  377.2  377.2  377.2
  157.7  13106  -48.0  -82.8      1   0.00    255     61  381.6  381.6  381.6
  156.0  13178  -47.9  -82.9      1   0.00    260     57  383.0  383.0  383.0
  154.0  13263  -45.5  -80.5      1   0.00    265     52  388.5  388.5  388.5
  150.7  13411  -44.3  -80.2      1   0.00    275     43  393.0  393.0  393.0
  150.0  13440  -44.1  -80.1      1   0.00    275     47  393.9  393.9  393.9
  149.0  13485  -43.5  -79.5      1   0.00    274     46  395.6  395.7  395.6
  142.0  13808  -44.9  -79.9      1   0.00    265     37  398.7  398.7  398.7
  141.0  13855  -44.1  -80.1      1   0.00    263     35  400.9  400.9  400.9
  137.0  14048  -44.9  -79.9      1   0.00    258     30  402.8  402.8  402.8
  131.3  14326  -47.2  -81.9      1   0.00    250     22  403.5  403.5  403.5
  123.0  14759  -50.9  -84.9      1   0.00    248     31  404.4  404.5  404.4
  118.0  15028  -52.1  -85.1      1   0.00    246     37  407.1  407.1  407.1
  114.2  15240  -51.7  -84.7      1   0.00    245     41  411.6  411.6  411.6
  114.0  15252  -51.7  -84.7      1   0.00    245     41  411.8  411.9  411.8
  107.0  15662  -52.9  -85.9      1   0.00    260     41  417.1  417.1  417.1
  103.0  15908  -51.5  -84.5      1   0.00    268     41  424.3  424.4  424.3
  100.0  16100  -51.9  -84.9      1   0.00    275     41  427.2  427.2  427.2
   95.4  16403  -53.1  -86.1      1   0.00    285     38  430.6  430.6  430.6
   92.4  16609  -51.1  -85.1      1   0.00    291     36  438.5  438.5  438.5
   84.3  17203  -51.9  -84.9      1   0.00    310     31  448.5  448.6  448.5
   82.1  17374  -52.6  -85.4      1   0.00    315     29  450.6  450.6  450.6
   78.3  17678  -53.8  -86.3      1   0.00    335     24  454.3  454.3  454.3
   74.6  17983  -55.0  -87.3      1   0.00    335     13  458.0  458.0  458.0
   71.1  18288  -56.2  -88.2      1   0.00    275     11  461.7  461.7  461.7
   70.8  18318  -56.3  -88.3      1   0.00    275     12  462.1  462.1  462.1
   70.0  18390  -55.7  -87.7      1   0.00    275     16  464.9  464.9  464.9
   67.8  18593  -54.5  -87.4      1   0.00    300     21  471.6  471.7  471.6
   67.4  18632  -54.3  -87.3      1   0.00    300     21  472.9  473.0  472.9
   61.6  19202  -54.8  -87.0      1   0.00    300     25  484.0  484.0  484.0
   60.9  19281  -54.9  -86.9      1   0.00    308     22  485.5  485.6  485.5
   58.8  19507  -53.9  -86.6      1   0.00    330     14  492.8  492.8  492.8
   58.1  19582  -53.5  -86.5      1   0.00    329     14  495.2  495.3  495.2
   56.0  19812  -54.6  -87.2      1   0.00    325     12  497.8  497.8  497.8
   53.8  20074  -55.9  -87.9      1   0.00    312     15  500.7  500.7  500.7
   53.4  20117  -55.9  -87.9      1   0.00    310     16  501.7  501.7  501.7
   50.0  20540  -55.9  -87.9      1   0.00    315     24  511.3  511.3  511.3
   48.6  20726  -56.1  -88.1      1   0.00    320     25  515.0  515.1  515.0
   47.6  20853  -56.3  -88.3      1   0.00    326     24  517.6  517.6  517.6
   45.4  21155  -54.5  -87.5      1   0.00    341     21  529.0  529.0  529.0
   44.1  21336  -54.3  -87.3      1   0.00    350     19  533.7  533.7  533.7
   39.6  22032  -53.7  -86.7      1   0.01     13     19  552.1  552.1  552.1
   38.3  22250  -53.7  -86.7      1   0.01     20     19  557.5  557.5  557.5
   36.5  22555  -53.7  -86.7      1   0.01     10     28  565.1  565.1  565.1
   36.0  22645  -53.7  -86.7      1   0.01     12     30  567.3  567.4  567.3
   33.7  23067  -56.1  -88.1      1   0.01     23     37  571.8  571.8  571.8
   33.2  23165  -55.8  -87.9      1   0.01     25     39  575.1  575.1  575.1
   30.2  23774  -54.0  -87.0      1   0.01     25     27  595.9  595.9  595.9
   30.0  23810  -53.9  -86.9      1   0.01     20     29  597.1  597.2  597.1
   27.4  24384  -53.1  -86.2      1   0.01     20     48  614.6  614.7  614.6
   24.8  25034  -52.3  -85.3      1   0.01     26     49  635.1  635.2  635.1
   21.5  25955  -52.3  -85.3      1   0.01     35     50  661.5  661.6  661.5
   20.0  26420  -53.7  -86.7      1   0.01     40     51  671.0  671.2  671.1
   18.4  26955  -54.5  -87.5      1   0.01     43     53  684.7  684.8  684.7
   17.1  27432  -54.5  -87.5      1   0.01     45     55  699.5  699.6  699.5
   16.3  27730  -54.5  -87.5      1   0.01     44     54  708.9  709.0  708.9
   15.1  28222  -52.7  -85.7      1   0.02     43     53  730.5  730.7  730.5
   12.8  29261  -53.4  -86.4      1   0.02     40     51  762.5  762.7  762.5
   12.6  29388  -53.5  -86.5      1   0.02                766.5  766.7  766.5
</PRE><H3>Station information and sounding indices</H3><PRE>
                         Station identifier: TFX
                             Station number: 72776
                           Observation time: 210203/1200
                           Station latitude: 47.46
                          Station longitude: -111.39
                          Station elevation: 1134.0
                            Showalter index: 6.14
                               Lifted index: 6.39
    LIFT computed using virtual temperature: 6.40
                                SWEAT index: 107.01
                                    K index: 4.30
                         Cross totals index: 17.60
                      Vertical totals index: 28.60
                        Totals totals index: 46.20
      Convective Available Potential Energy: 0.00
             CAPE using virtual temperature: 0.00
                      Convective Inhibition: 0.00
             CINS using virtual temperature: 0.00
                     Bulk Richardson Number: 0.00
          Bulk Richardson Number using CAPV: 0.00
  Temp [K] of the Lifted Condensation Level: 260.72
Pres [hPa] of the Lifted Condensation Level: 730.82
   Equivalent potential temp [K] of the LCL: 291.23
     Mean mixed layer potential temperature: 285.18
              Mean mixed layer mixing ratio: 2.04
              1000 hPa to 500 hPa thickness: 5333.00
Precipitable water [mm] for entire sounding: 4.01
</PRE>
<H2>72776 TFX Great Falls Observations at 00Z 04 Feb 2021</H2>
<PRE>
-----------------------------------------------------------------------------
   PRES   HGHT   TEMP   DWPT   RELH   MIXR   DRCT   SKNT   THTA   THTE   THTV
    hPa     m      C      C      %    g/kg    deg   knot     K      K      K 
-----------------------------------------------------------------------------
 1000.0    159                                                               
  925.0    781                                                               
  885.0   1134   -7.7  -10.9     78   1.89    250      4  274.9  280.3  275.2
  884.0   1143   -7.7  -12.2     70   1.70    253      4  275.0  279.9  275.2
  870.0   1265   -9.3  -12.1     80   1.75    299      5  274.6  279.6  274.9
  850.0   1443  -10.1  -11.7     88   1.85      5      6  275.6  280.9  275.9
  845.0   1488  -10.3  -11.7     89   1.86    356      7  275.8  281.2  276.1
  838.0   1553   -9.7  -12.1     83   1.81    342      8  277.1  282.4  277.4
  819.0   1730  -10.3  -13.1     80   1.71    306     10  278.3  283.3  278.6
  813.0   1786   -8.9  -11.9     79   1.90    294     11  280.4  285.9  280.7
  808.6   1829   -7.7  -12.2     70   1.86    285     12  282.1  287.6  282.4
  804.0   1873   -6.5  -12.5     62   1.83    286     13  283.8  289.3  284.1
  789.0   2020   -6.5  -12.5     62   1.86    291     17  285.3  290.9  285.6
  777.4   2134   -7.5  -13.0     65   1.82    295     20  285.5  291.0  285.8
  747.3   2438  -10.0  -14.2     71   1.71    300     21  285.9  291.1  286.2
  718.3   2743  -12.6  -15.5     79   1.60    295     22  286.4  291.3  286.6
  700.0   2942  -14.3  -16.3     85   1.54    290     23  286.6  291.3  286.9
  654.0   3452  -18.7  -19.4     94   1.27    297     29  287.3  291.2  287.5
  636.0   3658  -20.2  -23.2     77   0.93    300     31  287.9  290.8  288.0
  600.0   4088  -23.3  -31.3     48   0.47    304     28  289.1  290.7  289.2
  585.4   4267  -24.7  -34.2     41   0.36    305     27  289.6  290.8  289.6
  581.0   4322  -25.1  -35.1     39   0.33    304     27  289.7  290.8  289.7
  564.0   4537  -27.1  -35.1     47   0.34    301     29  289.8  291.0  289.9
  551.0   4704  -28.3  -39.3     34   0.23    298     30  290.3  291.1  290.4
  537.7   4877  -29.9  -39.8     38   0.23    295     31  290.4  291.2  290.5
  506.0   5307  -33.9  -40.9     49   0.21    299     31  290.7  291.4  290.7
  500.0   5390  -34.7  -42.7     44   0.18    300     31  290.7  291.3  290.7
  487.0   5574  -36.3  -43.3     48   0.17    302     30  290.9  291.5  290.9
  473.0   5777  -38.1  -42.8     61   0.19    305     29  291.1  291.8  291.1
  472.0   5791  -38.2  -42.9     62   0.18    305     29  291.1  291.8  291.1
  458.0   5999  -40.3  -44.2     66   0.17    302     30  291.1  291.6  291.1
  451.5   6096  -41.2  -44.8     67   0.16    300     31  291.2  291.7  291.2
  443.0   6226  -42.3  -45.7     69   0.15    298     32  291.3  291.8  291.3
  436.0   6334  -43.3  -48.2     58   0.11    296     32  291.4  291.8  291.4
  423.0   6538  -45.5  -50.2     59   0.09    292     34  291.1  291.4  291.1
  411.0   6730  -46.9  -51.6     59   0.08    288     35  291.7  292.0  291.7
  404.0   6844  -47.9  -51.5     66   0.08    286     36  291.8  292.1  291.8
  400.0   6910  -48.3  -51.9     66   0.08    285     36  292.1  292.4  292.1
  393.9   7010  -48.8  -52.9     62   0.07    285     38  292.7  293.0  292.7
  391.0   7059  -49.1  -53.4     61   0.07    285     40  293.0  293.2  293.0
  382.0   7212  -47.9  -58.9     27   0.04    291     41  296.5  296.7  296.5
  376.0   7315  -48.2  -62.2     18   0.02    295     41  297.5  297.6  297.5
  370.0   7421  -48.5  -65.5     12   0.02    300     41  298.4  298.5  298.5
  364.0   7528  -47.3  -67.3      8   0.01    306     40  301.4  301.5  301.5
  360.0   7601  -47.9  -69.9      6   0.01    309     40  301.6  301.6  301.6
  359.0   7620  -47.6  -69.9      6   0.01    310     40  302.2  302.2  302.2
  355.0   7693  -46.7  -69.7      6   0.01    311     40  304.4  304.5  304.4
  340.0   7977  -48.9  -72.9      4   0.01    313     39  305.2  305.2  305.2
  315.0   8479  -46.7  -76.7      2   0.00    317     38  315.0  315.0  315.0
  300.0   8800  -47.9  -77.9      2   0.00    320     37  317.7  317.8  317.7
  298.2   8839  -48.1  -78.1      2   0.00    325     37  318.0  318.0  318.0
  298.0   8844  -48.1  -78.1      2   0.00    325     37  318.1  318.1  318.1
  291.0   9002  -47.3  -79.3      1   0.00    322     39  321.4  321.4  321.4
  284.9   9144  -47.5  -79.8      1   0.00    320     40  323.0  323.0  323.0
  268.0   9549  -48.1  -81.1      1   0.00    315     40  327.9  327.9  327.9
  254.0   9904  -47.7  -82.7      1   0.00    311     39  333.5  333.5  333.5
  250.0  10010  -46.9  -81.9      1   0.00    310     39  336.2  336.2  336.2
  248.0  10063  -46.5  -81.5      1   0.00    310     39  337.6  337.6  337.6
  239.0  10308  -47.1  -82.1      1   0.00    308     41  340.3  340.3  340.3
  236.0  10392  -46.1  -81.1      1   0.00    308     42  343.0  343.0  343.0
  230.0  10563  -45.5  -80.5      1   0.00    307     44  346.4  346.5  346.4
  216.2  10973  -46.7  -81.7      1   0.00    305     47  350.8  350.8  350.8
  214.0  11041  -46.9  -81.9      1   0.00    306     45  351.5  351.5  351.5
  210.0  11166  -46.1  -81.1      1   0.00    307     43  354.6  354.6  354.6
  204.0  11359  -46.7  -81.7      1   0.00    309     38  356.6  356.6  356.6
  200.0  11490  -45.9  -80.9      1   0.00    310     35  359.9  359.9  359.9
  198.0  11557  -45.3  -80.3      1   0.00    310     34  361.9  361.9  361.9
  193.0  11728  -45.3  -80.3      1   0.00    310     32  364.6  364.6  364.6
  188.4  11887  -46.3  -81.3      1   0.00    310     30  365.4  365.4  365.4
  186.0  11974  -46.9  -81.9      1   0.00    309     32  365.9  365.9  365.9
  171.0  12531  -48.1  -82.1      1   0.00    303     43  372.8  372.8  372.8
  164.2  12802  -47.4  -81.8      1   0.00    300     49  378.3  378.3  378.3
  156.8  13106  -46.6  -81.4      1   0.00    305     30  384.6  384.7  384.6
  154.0  13225  -46.3  -81.3      1   0.00    305     31  387.1  387.2  387.1
  150.0  13400  -47.1  -82.1      1   0.00    305     33  388.7  388.7  388.7
  146.0  13579  -47.9  -82.9      1   0.00    305     38  390.3  390.3  390.3
  143.0  13716  -47.8  -82.8      1   0.00    305     41  392.9  392.9  392.9
  131.0  14295  -47.3  -82.3      1   0.00    300     29  403.7  403.7  403.7
  130.4  14326  -47.5  -82.4      1   0.00    300     28  403.9  403.9  403.9
  125.0  14604  -49.3  -83.3      1   0.00    298     33  405.5  405.5  405.5
  122.0  14764  -48.3  -82.3      1   0.00    298     35  410.1  410.2  410.1
  117.0  15038  -49.5  -83.5      1   0.00    296     40  412.9  412.9  412.9
  114.0  15209  -47.9  -81.9      1   0.00    295     43  418.9  418.9  418.9
  113.5  15240  -48.1  -82.1      1   0.00    295     43  419.1  419.2  419.1
  110.0  15444  -49.3  -83.3      1   0.00    295     46  420.6  420.6  420.6
  108.3  15545  -49.3  -83.3      1   0.00    295     47  422.4  422.4  422.4
  104.0  15812  -49.3  -83.3      1   0.00    295     38  427.4  427.4  427.4
  103.0  15876  -48.3  -82.3      1   0.00    295     36  430.5  430.5  430.5
  100.0  16070  -48.9  -82.9      1   0.00    295     30  433.0  433.0  433.0
   80.0  17519  -53.7  -86.7      1   0.00    295     40  451.6  451.6  451.6
   78.0  17678  -53.2  -86.2      1   0.00    300     41  455.9  455.9  455.9
   74.9  17944  -52.3  -85.3      1   0.00    304     37  463.1  463.1  463.1
   71.0  18288  -53.2  -86.2      1   0.00    310     31  468.2  468.2  468.2
   70.0  18380  -53.5  -86.5      1   0.00    315     29  469.6  469.6  469.6
   67.7  18593  -54.0  -86.8      1   0.00    315     24  472.9  472.9  472.9
   61.2  19238  -55.7  -87.7      1   0.00    315     28  483.1  483.1  483.1
   58.8  19493  -53.7  -86.7      1   0.00    315     29  493.1  493.1  493.1
   58.7  19507  -53.8  -86.7      1   0.00    315     29  493.3  493.3  493.3
   55.9  19812  -55.3  -87.6      1   0.00    335     17  496.6  496.6  496.6
   53.3  20117  -56.9  -88.5      1   0.00    300     11  499.9  499.9  499.9
   50.8  20422  -58.4  -89.4      1   0.00    295     20  503.2  503.2  503.2
   50.6  20445  -58.5  -89.5      1   0.00    295     20  503.5  503.5  503.5
   50.0  20520  -58.3  -89.3      1   0.00    295     22  505.7  505.7  505.7
   47.3  20872  -55.3  -87.3      1   0.00    333     33  520.9  521.0  520.9
   46.6  20967  -55.9  -87.9      1   0.00    343     36  521.7  521.7  521.7
   46.1  21031  -54.5  -87.1      1   0.00    350     38  526.5  526.5  526.5
   45.8  21078  -53.5  -86.5      1   0.00    350     37  530.1  530.1  530.1
   44.0  21336  -52.9  -85.9      1   0.01    350     34  537.5  537.6  537.6
   43.9  21351  -52.9  -85.9      1   0.01    351     34  538.0  538.0  538.0
   42.0  21641  -54.3  -87.3      1   0.00      0     31  541.4  541.4  541.4
   41.5  21713  -54.7  -87.7      1   0.00    358     30  542.2  542.3  542.2
   40.0  21946  -54.2  -87.2      1   0.00    350     26  549.1  549.1  549.1
   36.4  22555  -53.0  -86.0      1   0.01    350     31  567.3  567.4  567.3
   35.6  22699  -52.7  -85.7      1   0.01    358     31  571.7  571.8  571.7
   33.1  23165  -53.8  -86.2      1   0.01     25     31  580.8  580.9  580.8
   31.6  23470  -54.5  -86.6      1   0.01     15     37  586.9  586.9  586.9
   31.1  23569  -54.7  -86.7      1   0.01     15     38  588.8  588.9  588.8
   30.0  23800  -53.7  -86.7      1   0.01     15     39  597.6  597.7  597.6
   27.4  24384  -52.7  -85.7      1   0.01     20     44  616.0  616.1  616.0
   26.3  24647  -52.3  -85.3      1   0.01     20     44  624.5  624.6  624.5
   22.0  25795  -53.7  -86.7      1   0.01     20     46  653.0  653.1  653.0
   20.6  26213  -51.8  -85.5      1   0.01     20     46  670.9  671.0  670.9
   20.1  26378  -51.1  -85.1      1   0.01     24     49  678.0  678.2  678.0
   20.0  26410  -51.3  -85.3      1   0.01     25     49  678.4  678.5  678.4
   17.9  27127  -52.5  -86.0      1   0.01     35     53  696.6  696.7  696.6
   17.1  27432  -53.0  -86.4      1   0.01     45     52  704.5  704.6  704.5
   16.3  27737  -53.5  -86.7      1   0.01     50     45  712.4  712.6  712.4
   15.3  28141  -54.1  -87.1      1   0.01     43     42  723.1  723.3  723.1
   14.8  28346  -53.1  -86.2      1   0.02     40     40  732.9  733.0  732.9
   14.6  28443  -52.7  -85.7      1   0.02     40     41  737.5  737.7  737.5
   13.1  29142  -53.9  -86.9      1   0.02     37     50  756.6  756.8  756.6
   12.3  29566  -52.8  -86.2      1   0.02     35     56  774.8  775.0  774.8
   11.1  30213  -51.1  -85.1      1   0.03     42     55  803.4  803.7  803.4
   10.7  30480  -51.8  -85.4      1   0.03     45     54  810.3  810.6  810.3
   10.0  30890  -52.9  -85.9      1   0.02     40     54  821.0  821.3  821.0
    9.7  31086  -53.1  -86.1      1   0.02     42     51  827.4  827.7  827.4
    8.8  31717  -50.9  -84.9      1   0.03                859.3  859.7  859.3
    8.8  31699  -51.0  -84.9      1   0.03     50     41  858.4  858.8  858.4
    8.5  31942  -51.7  -84.7      1   0.04                864.7  865.2  864.7
</PRE><H3>Station information and sounding indices</H3><PRE>
                         Station identifier: TFX
                             Station number: 72776
                           Observation time: 210204/0000
                           Station latitude: 47.46
                          Station longitude: -111.39
                          Station elevation: 1134.0
                            Showalter index: 8.50
                               Lifted index: 8.62
    LIFT computed using virtual temperature: 8.62
                                SWEAT index: 43.01
                                    K index: 10.90
                         Cross totals index: 23.00
                      Vertical totals index: 24.60
                        Totals totals index: 47.60
      Convective Available Potential Energy: 0.00
             CAPE using virtual temperature: 0.00
                      Convective Inhibition: 0.00
             CINS using virtual temperature: 0.00
                     Bulk Richardson Number: 0.00
          Bulk Richardson Number using CAPV: 0.00
  Temp [K] of the Lifted Condensation Level: 260.64
Pres [hPa] of the Lifted Condensation Level: 822.94
   Equivalent potential temp [K] of the LCL: 280.73
     Mean mixed layer potential temperature: 275.58
              Mean mixed layer mixing ratio: 1.79
              1000 hPa to 500 hPa thickness: 5231.00
Precipitable water [mm] for entire sounding: 4.88
</PRE>
<H2>72776 TFX Great Falls Observations at 12Z 04 Feb 2021</H2>
<PRE>
-----------------------------------------------------------------------------
   PRES   HGHT   TEMP   DWPT   RELH   MIXR   DRCT   SKNT   THTA   THTE   THTV
    hPa     m      C      C      %    g/kg    deg   knot     K      K      K 
-----------------------------------------------------------------------------
 1000.0    176                                                               
  925.0    791                                                               
  885.0   1134   -7.7  -10.4     81   1.97    230     10  274.9  280.5  275.2
  879.0   1188   -3.9   -9.9     63   2.06    238     13  279.4  285.4  279.7
  871.0   1261   -2.3  -10.3     54   2.01    250     17  281.8  287.7  282.1
  850.0   1456   -2.7  -11.7     50   1.85    280     28  283.3  288.8  283.6
  847.0   1484   -2.7  -11.7     50   1.85    281     28  283.6  289.1  283.9
  810.7   1829   -5.5  -15.3     46   1.44    300     31  284.2  288.6  284.5
  805.0   1885   -5.9  -15.9     45   1.38    302     30  284.3  288.6  284.6
  779.6   2134   -8.1  -16.6     50   1.35    310     25  284.6  288.7  284.8
  755.0   2383  -10.3  -17.3     57   1.31    302     24  284.8  288.9  285.1
  749.5   2438  -10.8  -17.4     58   1.31    300     24  284.9  288.9  285.1
  720.1   2743  -13.7  -18.1     69   1.28    295     21  285.0  288.9  285.2
  700.0   2959  -15.7  -18.6     78   1.27    295     20  285.1  289.0  285.3
  662.0   3377  -19.7  -20.1     97   1.18    295     20  285.1  288.8  285.4
  654.0   3468  -20.5  -20.9     97   1.11    295     20  285.2  288.7  285.4
  644.0   3582  -21.3  -23.5     82   0.90    295     20  285.6  288.4  285.8
  637.4   3658  -22.0  -23.9     84   0.87    295     20  285.7  288.4  285.8
  636.0   3674  -22.1  -24.0     85   0.87    296     20  285.7  288.4  285.9
  617.0   3897  -22.7  -29.7     53   0.53    309     26  287.5  289.2  287.6
  604.0   4053  -23.3  -42.3     16   0.15    318     30  288.6  289.1  288.6
  602.0   4078  -23.3  -39.3     22   0.21    319     31  288.8  289.6  288.9
  598.0   4126  -23.3  -31.3     48   0.47    322     32  289.4  290.9  289.5
  592.0   4200  -23.3  -29.3     58   0.57    326     34  290.2  292.1  290.3
  586.6   4267  -23.7  -29.3     60   0.58    330     36  290.5  292.5  290.6
  578.0   4375  -24.3  -29.2     64   0.59    330     39  291.0  293.0  291.1
  559.0   4620  -23.7  -26.4     78   0.80    330     45  294.5  297.2  294.7
  547.0   4779  -23.3  -27.2     70   0.76    330     49  296.9  299.4  297.0
  539.7   4877  -23.9  -29.6     60   0.62    330     52  297.2  299.3  297.3
  538.0   4900  -24.1  -30.1     58   0.59    330     52  297.3  299.3  297.4
  529.0   5023  -25.3  -31.3     57   0.53    331     53  297.3  299.1  297.4
  500.0   5430  -28.9  -32.9     68   0.48    335     55  297.7  299.4  297.8
  486.0   5632  -30.9  -34.1     73   0.44    335     59  297.7  299.2  297.8
  472.0   5839  -32.3  -36.2     68   0.37    335     62  298.5  299.8  298.5
  455.0   6096  -34.5  -36.7     80   0.36    335     67  298.9  300.1  298.9
  452.0   6142  -34.9  -36.8     83   0.36    335     67  298.9  300.2  299.0
  451.0   6158  -35.0  -37.2     80   0.35    335     67  299.0  300.2  299.0
  441.0   6314  -36.3  -41.3     60   0.23    334     65  299.3  300.1  299.3
  439.0   6345  -36.3  -43.3     48   0.19    334     64  299.7  300.4  299.7
  436.0   6393  -34.5  -51.5     16   0.08    334     63  302.5  302.8  302.5
  435.0   6409  -34.3  -59.3      6   0.03    333     63  303.0  303.1  303.0
  400.0   6990  -38.7  -65.7      4   0.01    330     54  304.6  304.7  304.6
  376.0   7412  -41.7  -67.7      4   0.01    330     53  306.1  306.1  306.1
  364.4   7620  -43.0  -68.0      5   0.01    330     52  307.1  307.2  307.1
  308.0   8738  -49.9  -69.9      8   0.01    344     59  312.6  312.6  312.6
  303.3   8839  -50.1  -69.5      8   0.01    345     60  313.6  313.6  313.6
  300.0   8910  -50.3  -69.3      9   0.01    340     61  314.3  314.4  314.4
  295.0   9019  -49.9  -67.9     10   0.01    338     59  316.4  316.5  316.4
  289.4   9144  -50.6  -69.4      9   0.01    335     56  317.2  317.2  317.2
  282.0   9312  -51.5  -71.5      7   0.01    332     58  318.2  318.3  318.2
  276.1   9449  -51.3  -73.3      6   0.01    330     60  320.4  320.5  320.4
  276.0   9451  -51.3  -73.3      6   0.01    330     60  320.5  320.5  320.5
  268.0   9640  -52.1  -75.1      5   0.01    330     57  322.0  322.0  322.0
  251.0  10064  -50.7  -79.7      2   0.00    339     50  330.2  330.2  330.2
  250.0  10090  -50.7  -79.7      2   0.00    340     50  330.6  330.6  330.6
  239.8  10363  -50.8  -79.8      2   0.00    340     48  334.4  334.4  334.4
  224.0  10807  -50.9  -79.9      2   0.00    335     48  340.8  340.8  340.8
  208.5  11278  -49.7  -82.5      1   0.00    330     47  349.8  349.8  349.8
  204.0  11420  -49.3  -83.3      1   0.00    325     50  352.5  352.6  352.5
  200.0  11550  -49.5  -83.5      1   0.00    320     53  354.2  354.2  354.2
  197.0  11649  -49.7  -83.7      1   0.00    320     56  355.4  355.4  355.4
  190.0  11887  -48.6  -82.7      1   0.00    320     62  360.8  360.8  360.8
  189.0  11921  -48.5  -82.5      1   0.00    321     61  361.6  361.6  361.6
  186.0  12026  -49.1  -83.1      1   0.00    323     59  362.3  362.3  362.3
  180.0  12243  -46.7  -81.7      1   0.00    329     54  369.6  369.6  369.6
  173.2  12497  -46.1  -81.1      1   0.00    335     48  374.6  374.6  374.6
  173.0  12506  -46.1  -81.1      1   0.00    335     48  374.8  374.8  374.8
  165.4  12802  -46.8  -81.8      1   0.00    320     39  378.4  378.4  378.4
  157.0  13150  -47.7  -82.7      1   0.00    317     44  382.6  382.7  382.6
  150.0  13450  -49.3  -83.3      1   0.00    315     49  384.9  384.9  384.9
  148.0  13538  -50.1  -84.1      1   0.00    315     48  385.0  385.0  385.0
  146.0  13627  -49.7  -83.7      1   0.00    315     47  387.2  387.2  387.2
  144.0  13716  -50.2  -83.9      1   0.00    315     46  387.8  387.9  387.9
  139.0  13947  -51.5  -84.5      1   0.00    309     47  389.5  389.5  389.5
  131.1  14326  -51.2  -84.9      1   0.00    300     48  396.6  396.6  396.6
  128.0  14483  -51.1  -85.1      1   0.00    305     51  399.5  399.5  399.5
  127.0  14535  -50.1  -84.1      1   0.00    307     51  402.2  402.2  402.2
  125.2  14630  -50.1  -84.1      1   0.00    310     53  403.9  403.9  403.9
  122.0  14797  -50.1  -84.1      1   0.00    314     51  406.9  406.9  406.9
  113.9  15240  -53.3  -86.6      1   0.00    325     45  409.0  409.0  409.0
  111.0  15409  -54.5  -87.5      1   0.00    325     44  409.8  409.8  409.8
  106.0  15705  -53.9  -86.9      1   0.00    325     43  416.3  416.3  416.3
  104.0  15827  -52.9  -85.9      1   0.00    325     42  420.5  420.5  420.5
  100.0  16080  -53.5  -86.5      1   0.00    325     41  424.1  424.1  424.1
   86.7  16997  -53.1  -86.1      1   0.00    323     39  442.5  442.6  442.5
   77.2  17736  -56.9  -88.9      1   0.00    321     37  449.6  449.6  449.6
   71.1  18260  -53.9  -86.9      1   0.00    320     36  466.6  466.7  466.6
   70.8  18288  -54.0  -87.0      1   0.00    320     36  467.1  467.1  467.1
   70.0  18360  -54.1  -87.1      1   0.00    325     37  468.3  468.3  468.3
   58.4  19507  -56.8  -88.8      1   0.00    320     29  487.0  487.0  487.0
   58.1  19544  -56.9  -88.9      1   0.00    321     29  487.6  487.6  487.6
   54.8  19916  -54.3  -87.3      1   0.00    331     33  501.8  501.8  501.8
   50.6  20422  -55.3  -87.5      1   0.00    345     39  510.8  510.9  510.8
   50.0  20500  -55.5  -87.5      1   0.00    340     40  512.2  512.3  512.2
   49.2  20603  -55.9  -87.9      1   0.00    338     42  513.7  513.7  513.7
   48.3  20726  -55.7  -87.8      1   0.00    335     45  516.9  516.9  516.9
   43.9  21336  -54.9  -87.5      1   0.00    345     42  533.1  533.2  533.1
   41.8  21641  -54.5  -87.3      1   0.00    355     40  541.5  541.5  541.5
   39.9  21946  -54.1  -87.1      1   0.01    345     25  549.9  550.0  549.9
   39.7  21973  -54.1  -87.1      1   0.01    345     25  550.7  550.7  550.7
   34.2  22925  -56.3  -88.3      1   0.00    353     38  568.9  568.9  568.9
   30.0  23760  -54.9  -86.9      1   0.01      0     49  594.4  594.4  594.4
   27.2  24384  -53.5  -86.0      1   0.01      5     52  614.9  615.0  614.9
   26.0  24689  -52.8  -85.6      1   0.01      5     56  625.2  625.3  625.2
   24.7  25015  -52.1  -85.1      1   0.01     10     49  636.4  636.5  636.4
   22.6  25603  -53.7  -86.7      1   0.01     20     37  648.4  648.5  648.4
   21.8  25823  -54.3  -87.3      1   0.01     20     43  652.9  653.0  652.9
   20.5  26220  -52.5  -85.5      1   0.01     20     55  670.0  670.1  670.0
   20.5  26213  -52.5  -85.5      1   0.01     20     55  669.7  669.8  669.7
   20.0  26380  -52.5  -85.5      1   0.01     20     52  674.7  674.9  674.7
   17.2  27354  -52.7  -85.7      1   0.02     34     44  703.8  704.0  703.8
   17.0  27432  -52.3  -85.3      1   0.02     35     43  707.4  707.5  707.4
   16.4  27662  -51.3  -84.3      1   0.02     36     42  718.0  718.2  718.0
   13.6  28878  -51.1  -85.1      1   0.02     38     39  758.1  758.3  758.1
   12.3  29528  -53.1  -86.1      1   0.02     40     37  773.1  773.4  773.2
   12.2  29566  -53.0  -86.0      1   0.02     40     37  774.9  775.1  774.9
   10.9  30310  -50.7  -84.7      1   0.03     40     45  809.0  809.4  809.0
   10.6  30480  -51.0  -84.7      1   0.03     40     47  814.0  814.4  814.0
   10.0  30870  -51.7  -84.7      1   0.03     40     47  825.5  825.9  825.5
    9.8  31001  -52.1  -85.1      1   0.03     40     49  828.8  829.1  828.8
    9.2  31412  -49.9  -83.9      1   0.04     42     53  852.2  852.8  852.3
    9.0  31555  -51.1  -85.1      1   0.03     42     55  853.0  853.4  853.0
    8.3  32082  -50.1  -84.1      1   0.04     43     61  876.9  877.5  876.9
    7.6  32614  -52.6  -86.1      1   0.03     45     67  887.5  887.9  887.5
    7.2  33002  -54.5  -87.5      1   0.03                895.2  895.6  895.3
</PRE><H3>Station information and sounding indices</H3><PRE>
                         Station identifier: TFX
                             Station number: 72776
                           Observation time: 210204/1200
                           Station latitude: 47.46
                          Station longitude: -111.39
                          Station elevation: 1134.0
                            Showalter index: 8.20
                               Lifted index: 9.09
    LIFT computed using virtual temperature: 9.12
                                SWEAT index: 110.98
                                    K index: 11.60
                         Cross totals index: 17.20
                      Vertical totals index: 26.20
                        Totals totals index: 43.40
      Convective Available Potential Energy: 0.00
             CAPE using virtual temperature: 0.00
                      Convective Inhibition: 0.00
             CINS using virtual temperature: 0.00
                     Bulk Richardson Number: 0.00
          Bulk Richardson Number using CAPV: 0.00
  Temp [K] of the Lifted Condensation Level: 260.31
Pres [hPa] of the Lifted Condensation Level: 755.48
   Equivalent potential temp [K] of the LCL: 287.64
     Mean mixed layer potential temperature: 282.04
              Mean mixed layer mixing ratio: 1.90
              1000 hPa to 500 hPa thickness: 5254.00
Precipitable water [mm] for entire sounding: 4.68
</PRE>
<H2>72776 TFX Great Falls Observations at 00Z 05 Feb 2021</H2>
<PRE>
-----------------------------------------------------------------------------
   PRES   HGHT   TEMP   DWPT   RELH   MIXR   DRCT   SKNT   THTA   THTE   THTV
    hPa     m      C      C      %    g/kg    deg   knot     K      K      K 
-----------------------------------------------------------------------------
 1000.0    141                                                               
  925.0    767                                                               
  883.0   1134    2.2   -9.8     41   2.07    270     17  285.3  291.5  285.7
  850.0   1444   -0.5  -11.5     43   1.88    275     30  285.6  291.3  285.9
  841.5   1524   -1.2  -11.9     44   1.84    275     30  285.6  291.2  286.0
  805.0   1875   -4.5  -13.5     49   1.68    284     29  285.8  290.9  286.1
  778.6   2134   -6.9  -14.3     56   1.63    290     29  286.0  291.0  286.3
  748.6   2438   -9.7  -15.2     64   1.57    295     33  286.2  291.0  286.4
  719.7   2743  -12.5  -16.2     74   1.51    300     34  286.3  290.9  286.6
  716.0   2783  -12.9  -16.3     76   1.50    301     35  286.3  290.9  286.6
  700.0   2955  -14.3  -16.9     81   1.46    305     37  286.6  291.1  286.9
  663.9   3353  -17.5  -18.4     92   1.36    315     37  287.4  291.6  287.6
  653.0   3478  -18.5  -18.9     97   1.32    309     35  287.6  291.7  287.9
  637.4   3658  -19.1  -19.6     96   1.28    300     32  288.9  292.9  289.1
  615.0   3924  -20.1  -20.6     96   1.21    309     42  290.8  294.6  291.0
  587.1   4267  -21.5  -23.3     85   1.00    320     55  293.0  296.2  293.2
  587.0   4268  -21.5  -23.3     85   1.00    320     55  293.0  296.3  293.2
  549.0   4758  -25.3  -26.3     91   0.82    320     56  294.2  296.9  294.3
  540.1   4877  -26.1  -27.6     87   0.74    320     56  294.6  297.0  294.7
  536.0   4932  -26.5  -28.2     86   0.70    319     57  294.8  297.1  294.9
  511.0   5275  -29.3  -29.6     97   0.65    313     60  295.4  297.6  295.5
  502.0   5402  -30.1  -31.4     88   0.56    311     62  295.9  297.8  296.0
  500.0   5430  -30.3  -32.4     82   0.51    310     62  296.0  297.8  296.1
  498.0   5459  -30.5  -33.5     75   0.46    310     62  296.1  297.7  296.2
  489.0   5588  -31.3  -34.0     77   0.44    312     62  296.7  298.2  296.8
  475.2   5791  -32.3  -35.5     73   0.39    315     63  297.9  299.3  298.0
  464.0   5959  -33.1  -36.7     70   0.36    318     77  298.9  300.2  299.0
  455.1   6096  -33.4  -37.0     70   0.35    320     88  300.3  301.5  300.3
  413.0   6777  -34.7  -38.3     70   0.34    324     96  307.0  308.2  307.1
  400.0   7000  -35.9  -39.8     67   0.30    325     99  308.2  309.4  308.3
  380.0   7354  -38.5  -43.2     61   0.22    328    100  309.4  310.2  309.4
  365.4   7620  -40.5  -44.9     62   0.19    330    100  310.2  311.0  310.3
  349.4   7925  -42.7  -46.9     63   0.16    335    103  311.2  311.8  311.2
  348.0   7952  -42.9  -47.1     63   0.16    335    105  311.3  311.9  311.3
  342.0   8069  -42.7  -46.9     63   0.16    335    114  313.1  313.8  313.1
  334.0   8228  -44.1  -47.8     67   0.15    335    126  313.3  313.9  313.4
  333.9   8230  -44.1  -47.8     66   0.15    335    126  313.4  313.9  313.4
  325.0   8411  -44.9  -49.6     59   0.13    335    127  314.7  315.2  314.7
  307.0   8788  -47.9  -51.5     66   0.11    335    131  315.6  316.1  315.7
  304.6   8839  -48.0  -51.7     66   0.11    335    131  316.1  316.6  316.2
  304.0   8853  -48.1  -51.7     66   0.11    335    131  316.3  316.7  316.3
  300.0   8940  -48.3  -52.0     65   0.10    335    131  317.2  317.6  317.2
  290.8   9144  -49.6  -53.7     62   0.09    335    126  318.2  318.6  318.2
  280.0   9391  -51.1  -55.7     58   0.07    335    124  319.4  319.8  319.5
  273.0   9555  -52.1  -55.7     65   0.07    335    122  320.3  320.6  320.3
  261.0   9844  -53.5  -60.5     42   0.04    335    120  322.4  322.6  322.4
  253.0  10044  -54.1  -67.1     19   0.02    335    118  324.4  324.5  324.4
  250.0  10120  -54.3  -67.3     19   0.02    335    117  325.2  325.3  325.2
  227.0  10735  -55.5  -72.5     10   0.01    330    103  332.5  332.5  332.5
  218.7  10973  -54.4  -72.8      9   0.01    330    105  337.7  337.8  337.7
  215.0  11082  -53.9  -72.9      8   0.01    329    100  340.1  340.2  340.1
  209.0  11265  -50.7  -70.7      8   0.01    327     91  347.9  348.0  347.9
  202.0  11486  -51.7  -71.7      7   0.01    326     80  349.7  349.8  349.7
  200.0  11550  -51.1  -71.1      7   0.01    325     77  351.7  351.8  351.7
  199.0  11583  -50.7  -70.7      8   0.01    325     76  352.8  352.9  352.8
  195.0  11715  -51.3  -70.3      9   0.01    325     73  353.9  354.0  353.9
  189.9  11887  -50.4  -70.0      8   0.02    325     69  358.0  358.1  358.0
  187.0  11989  -49.9  -69.9      8   0.02    323     71  360.4  360.5  360.4
  181.0  12202  -51.1  -71.1      7   0.01    320     74  361.9  361.9  361.9
  177.0  12348  -49.7  -70.7      7   0.02    317     77  366.5  366.6  366.5
  175.0  12422  -49.7  -70.7      7   0.02    316     78  367.7  367.8  367.7
  173.0  12498  -48.1  -69.1      7   0.02    315     79  371.5  371.6  371.5
  173.0  12497  -48.1  -69.1      7   0.02    315     79  371.5  371.6  371.5
  166.0  12771  -48.1  -70.1      6   0.02    320     66  375.9  376.0  375.9
  162.0  12932  -47.3  -69.3      6   0.02    322     58  379.9  380.0  379.9
  157.8  13106  -47.9  -69.9      6   0.02    325     49  381.7  381.8  381.7
  153.0  13310  -48.7  -70.7      6   0.02    322     44  383.8  383.9  383.8
  150.0  13440  -49.7  -72.7      5   0.01    320     40  384.2  384.3  384.2
  143.8  13716  -52.0  -75.0      5   0.01    315     37  384.8  384.9  384.8
  139.0  13934  -53.9  -76.9      4   0.01    311     43  385.3  385.3  385.3
  136.0  14074  -53.5  -77.5      4   0.01    309     47  388.4  388.4  388.4
  132.0  14266  -54.7  -77.7      4   0.01    305     53  389.6  389.6  389.6
  130.0  14364  -53.3  -77.3      4   0.01    306     55  393.8  393.9  393.8
  123.0  14721  -53.1  -78.1      3   0.01    308     60  400.4  400.5  400.4
  120.0  14880  -51.3  -78.3      3   0.01    310     63  406.6  406.6  406.6
  119.0  14935  -51.6  -78.2      3   0.01    310     64  407.1  407.1  407.1
  117.0  15045  -52.1  -78.1      3   0.01    308     61  408.1  408.1  408.1
  115.0  15157  -50.7  -76.7      3   0.01    306     57  412.7  412.7  412.7
  108.3  15545  -52.8  -78.8      3   0.01    300     45  415.8  415.9  415.8
  108.0  15564  -52.9  -78.9      3   0.01    300     45  416.0  416.0  416.0
  100.0  16060  -53.5  -79.5      3   0.01    305     50  424.1  424.1  424.1
   94.2  16444  -53.9  -79.9      3   0.01    319     53  430.6  430.6  430.6
   94.0  16459  -53.7  -79.8      3   0.01    320     53  431.3  431.3  431.3
   91.7  16617  -51.7  -78.7      3   0.01    328     40  438.3  438.3  438.3
   89.6  16764  -52.5  -79.4      3   0.01    335     28  439.5  439.5  439.5
   85.5  17069  -54.3  -80.8      2   0.01    325     23  441.9  442.0  441.9
   81.5  17374  -56.0  -82.1      2   0.01    305     30  444.4  444.4  444.4
   80.5  17452  -56.5  -82.5      2   0.01    310     31  445.0  445.1  445.0
   74.0  17983  -56.0  -83.2      2   0.01    325     32  456.8  456.9  456.8
   70.3  18313  -55.7  -83.7      2   0.01    325     35  464.3  464.3  464.3
   70.0  18340  -55.7  -83.7      2   0.01    325     35  464.9  464.9  464.9
   63.3  18979  -58.1  -86.1      2   0.00    325     41  473.1  473.2  473.1
   55.5  19812  -56.5  -86.1      1   0.00    325     49  494.9  494.9  494.9
   53.7  20022  -56.1  -86.1      1   0.00    331     44  500.5  500.6  500.5
   50.0  20480  -53.3  -84.3      1   0.01    345     33  517.4  517.5  517.4
   49.1  20597  -52.5  -82.5      2   0.01    348     30  522.0  522.1  522.0
   43.7  21336  -58.4  -87.4      1   0.00     10      9  525.4  525.4  525.4
   43.4  21381  -58.7  -87.7      1   0.00      9     10  525.5  525.6  525.5
   40.1  21876  -58.9  -87.9      1   0.00    355     25  537.1  537.1  537.1
   37.8  22250  -56.9  -87.2      1   0.01    345     36  551.2  551.3  551.2
   31.9  23333  -51.1  -85.1      1   0.01      7     29  594.2  594.3  594.2
   30.0  23730  -52.7  -85.7      1   0.01     15     27  600.4  600.5  600.4
   29.8  23774  -52.8  -85.8      1   0.01     15     27  601.3  601.4  601.3
   25.8  24699  -54.5  -87.5      1   0.01     17     38  621.7  621.8  621.7
   20.7  26125  -49.5  -83.5      1   0.02     20     55  677.2  677.4  677.2
   20.4  26213  -49.8  -83.8      1   0.02     20     56  678.9  679.0  678.9
   20.0  26350  -50.3  -84.3      1   0.02     25     55  681.5  681.6  681.5
   16.9  27432  -50.6  -84.7      1   0.02     30     56  713.4  713.6  713.4
   15.4  28042  -50.9  -84.8      1   0.02     25     60  732.0  732.2  732.0
   15.1  28185  -50.9  -84.9      1   0.02     27     58  736.4  736.7  736.4
   13.3  29020  -46.7  -81.7      1   0.04     41     46  778.1  778.5  778.1
   12.6  29376  -50.1  -84.1      1   0.03     47     41  778.3  778.6  778.3
   12.2  29588  -49.3  -83.3      1   0.03     50     38  788.3  788.7  788.4
   12.2  29566  -49.4  -83.4      1   0.03     50     38  787.3  787.7  787.3
   11.5  29975  -50.5  -84.5      1   0.03     41     41  797.5  797.8  797.5
   10.6  30480  -53.6  -85.9      1   0.02     30     44  804.1  804.4  804.1
   10.5  30564  -54.1  -86.1      1   0.02     29     46  805.2  805.5  805.2
   10.4  30626  -52.5  -85.5      1   0.03     28     47  813.3  813.6  813.3
   10.2  30785  -51.9  -84.9      1   0.03     25     51  821.4  821.8  821.4
   10.0  30880  -51.5  -84.5      1   0.03     25     52  826.2  826.6  826.2
    9.7  31078  -49.9  -83.9      1   0.04     23     51  839.5  839.9  839.5
    8.8  31699  -50.1  -84.1      1   0.04     15     47  862.0  862.5  862.0
    8.6  31865  -50.1  -84.1      1   0.04                868.0  868.6  868.1
</PRE><H3>Station information and sounding indices</H3><PRE>
                         Station identifier: TFX
                             Station number: 72776
                           Observation time: 210205/0000
                           Station latitude: 47.46
                          Station longitude: -111.39
                          Station elevation: 1134.0
                            Showalter index: 4.97
                               Lifted index: 4.95
    LIFT computed using virtual temperature: 4.96
                                SWEAT index: 121.99
                                    K index: 15.70
                         Cross totals index: 18.80
                      Vertical totals index: 29.80
                        Totals totals index: 48.60
      Convective Available Potential Energy: 0.00
             CAPE using virtual temperature: 0.00
                      Convective Inhibition: 0.00
             CINS using virtual temperature: 0.00
                     Bulk Richardson Number: 0.00
          Bulk Richardson Number using CAPV: 0.00
  Temp [K] of the Lifted Condensation Level: 259.83
Pres [hPa] of the Lifted Condensation Level: 718.86
   Equivalent potential temp [K] of the LCL: 291.29
     Mean mixed layer potential temperature: 285.54
              Mean mixed layer mixing ratio: 1.92
              1000 hPa to 500 hPa thickness: 5289.00
Precipitable water [mm] for entire sounding: 5.95
</PRE>
<H2>72776 TFX Great Falls Observations at 12Z 05 Feb 2021</H2>
<PRE>
-----------------------------------------------------------------------------
   PRES   HGHT   TEMP   DWPT   RELH   MIXR   DRCT   SKNT   THTA   THTE   THTV
    hPa     m      C      C      %    g/kg    deg   knot     K      K      K 
-----------------------------------------------------------------------------
 1000.0     98                                                               
  925.0    729                                                               
  879.0   1134   -4.5   -6.8     84   2.62     70      8  278.7  286.3  279.2
  875.0   1170   -4.7   -7.0     84   2.60     86      8  278.9  286.4  279.3
  869.0   1225   -2.9   -5.3     84   2.98    111      8  281.3  289.9  281.8
  861.0   1300   -2.5   -5.3     81   3.01    144      8  282.5  291.2  283.0
  850.0   1403   -3.3   -6.0     82   2.89    190      8  282.7  291.1  283.2
  819.0   1696   -5.5   -7.5     86   2.67    224     11  283.4  291.2  283.8
  805.1   1829   -6.2   -8.2     86   2.56    240     13  283.9  291.5  284.4
  774.0   2134   -8.0   -9.9     86   2.33    270     14  285.3  292.2  285.7
  744.3   2438   -9.7  -11.7     86   2.12    285     16  286.6  293.0  287.0
  705.0   2859  -12.1  -14.0     86   1.85    287     20  288.5  294.1  288.8
  700.0   2913  -12.5  -14.4     86   1.80    287     20  288.6  294.1  288.9
  633.8   3658  -18.0  -20.3     82   1.21    290     27  290.6  294.4  290.8
  600.0   4070  -21.1  -23.6     80   0.96    287     33  291.7  294.7  291.8
  583.9   4267  -22.4  -25.0     80   0.87    285     36  292.4  295.2  292.5
  559.8   4572  -24.5  -27.1     79   0.75    280     40  293.4  295.9  293.6
  536.7   4877  -26.6  -29.2     78   0.64    285     44  294.5  296.6  294.6
  500.0   5390  -30.1  -32.8     77   0.49    295     54  296.3  297.9  296.4
  487.0   5577  -31.5  -34.5     75   0.42    298     56  296.8  298.3  296.9
  469.0   5843  -31.5  -34.4     75   0.44    301     58  300.0  301.6  300.1
  458.0   6011  -32.5  -35.7     73   0.40    304     60  300.8  302.2  300.9
  452.4   6096  -33.2  -36.4     73   0.38    305     61  301.0  302.4  301.1
  414.3   6706  -38.0  -41.5     69   0.24    310     67  302.5  303.4  302.5
  400.0   6950  -39.9  -43.6     68   0.20    310     72  303.1  303.8  303.1
  371.0   7460  -44.3  -47.7     69   0.14    302     77  303.8  304.3  303.8
  362.2   7620  -45.5  -48.8     69   0.13    300     79  304.4  304.8  304.4
  345.9   7925  -47.6  -50.9     69   0.10    305     86  305.4  305.8  305.4
  330.0   8237  -49.9  -53.1     69   0.08    307     85  306.4  306.8  306.5
  317.0   8500  -50.3  -52.7     76   0.09    308     85  309.4  309.8  309.4
  309.0   8667  -50.7  -61.7     26   0.03    309     84  311.1  311.3  311.1
  300.0   8860  -51.3  -71.3      7   0.01    310     84  312.9  313.0  312.9
  287.1   9144  -52.8  -72.8      7   0.01    310     86  314.7  314.8  314.7
  272.0   9491  -54.7  -74.7      7   0.01    315     89  316.9  316.9  316.9
  261.0   9755  -55.3  -73.3      9   0.01    315     95  319.8  319.8  319.8
  261.0   9754  -55.3  -73.3      9   0.01    315     95  319.8  319.8  319.8
  258.0   9829  -55.1  -73.4      9   0.01    315     95  321.1  321.1  321.1
  250.0  10030  -54.7  -73.7      8   0.01    315     91  324.6  324.6  324.6
  242.0  10238  -53.9  -75.9      5   0.01    315     92  328.9  328.9  328.9
  226.5  10668  -50.7  -80.8      2   0.00    315     93  340.1  340.1  340.1
  223.0  10768  -49.9  -81.9      1   0.00    315     91  342.8  342.8  342.8
  211.0  11129  -50.3  -82.3      1   0.00    315     86  347.6  347.6  347.6
  204.0  11350  -49.1  -83.1      1   0.00    315     82  352.9  352.9  352.9
  200.0  11480  -49.3  -83.3      1   0.00    315     80  354.5  354.6  354.5
  196.0  11612  -48.9  -82.9      1   0.00    315     76  357.2  357.2  357.2
  192.0  11747  -49.5  -83.5      1   0.00    315     72  358.4  358.4  358.4
  188.0  11885  -48.5  -82.5      1   0.00    315     68  362.1  362.2  362.1
  187.9  11887  -48.5  -82.5      1   0.00    315     68  362.2  362.2  362.2
  181.0  12134  -49.5  -83.5      1   0.00    316     69  364.5  364.5  364.5
  178.0  12243  -48.3  -82.3      1   0.00    316     70  368.2  368.2  368.2
  159.0  12986  -47.7  -82.7      1   0.00    320     73  381.3  381.3  381.3
  156.1  13106  -47.9  -82.6      1   0.00    320     74  382.9  382.9  382.9
  150.0  13370  -48.3  -82.3      1   0.00    315     68  386.6  386.6  386.6
  148.0  13459  -48.3  -82.3      1   0.00    315     66  388.1  388.1  388.1
  141.0  13780  -46.5  -81.5      1   0.00    315     57  396.7  396.7  396.7
  137.0  13971  -46.9  -81.9      1   0.00    315     52  399.2  399.3  399.2
  129.8  14326  -49.1  -83.2      1   0.00    315     43  401.6  401.6  401.6
  121.0  14787  -51.9  -84.9      1   0.00    310     43  404.5  404.5  404.5
  112.8  15240  -53.5  -86.5      1   0.00    305     43  409.7  409.7  409.7
  112.0  15287  -53.7  -86.7      1   0.00    305     45  410.2  410.2  410.2
  105.0  15704  -51.9  -84.9      1   0.00    316     49  421.2  421.3  421.2
  100.0  16020  -53.3  -86.3      1   0.00    325     52  424.5  424.5  424.5
   93.4  16459  -55.5  -87.5      1   0.00    295     48  428.7  428.7  428.7
   93.2  16470  -55.5  -87.5      1   0.00    295     48  428.8  428.8  428.8
   84.9  17069  -52.5  -85.8      1   0.00    305     69  446.5  446.5  446.5
   84.1  17129  -52.2  -85.6      1   0.00    305     69  448.3  448.3  448.3
   80.9  17378  -50.9  -84.9      1   0.00    308     64  455.9  455.9  455.9
   72.6  18076  -53.9  -86.9      1   0.00    317     52  463.9  463.9  463.9
   70.2  18288  -52.8  -85.8      1   0.00    320     48  470.6  470.6  470.6
   70.0  18310  -52.7  -85.7      1   0.00    320     48  471.3  471.3  471.3
   69.2  18384  -51.9  -84.9      1   0.00    319     47  474.6  474.6  474.6
   65.4  18750  -52.3  -85.3      1   0.00    316     39  481.4  481.4  481.4
   58.1  19507  -55.9  -88.3      1   0.00    310     24  489.9  489.9  489.9
   55.4  19812  -57.4  -89.5      1   0.00    315     36  493.3  493.3  493.3
   54.4  19926  -57.9  -89.9      1   0.00    315     32  494.5  494.6  494.5
   52.8  20117  -57.3  -89.3      1   0.00    315     26  500.3  500.3  500.3
   50.0  20460  -56.1  -88.1      1   0.00    330     31  510.8  510.9  510.8
   46.1  20979  -54.5  -87.5      1   0.00    348     30  526.7  526.7  526.7
   45.7  21031  -54.7  -87.6      1   0.00    350     30  527.5  527.6  527.5
   43.6  21336  -55.6  -88.1      1   0.00    340     26  532.6  532.6  532.6
   40.6  21789  -56.9  -88.9      1   0.00    338     31  540.2  540.2  540.2
   39.2  22012  -55.7  -87.7      1   0.00    336     34  548.6  548.7  548.6
   37.8  22250  -55.7  -87.7      1   0.00    335     37  554.5  554.6  554.5
   35.8  22590  -55.7  -87.7      1   0.01    350     35  563.0  563.1  563.0
   32.7  23165  -57.3  -88.7      1   0.00     15     31  573.7  573.7  573.7
   31.2  23470  -58.1  -89.2      1   0.00    355     39  579.4  579.4  579.4
   30.8  23543  -58.3  -89.3      1   0.00    357     40  580.7  580.8  580.7
   30.0  23710  -55.7  -87.7      1   0.01      0     42  592.2  592.3  592.2
   29.9  23731  -55.5  -87.5      1   0.01      3     41  593.3  593.4  593.3
   29.7  23774  -55.5  -87.5      1   0.01     10     39  594.5  594.5  594.5
   27.5  24263  -55.5  -87.5      1   0.01      2     44  607.7  607.7  607.7
   27.0  24384  -54.6  -87.0      1   0.01      0     45  613.6  613.6  613.6
   26.4  24524  -53.5  -86.5      1   0.01      5     47  620.4  620.5  620.5
   25.7  24689  -54.0  -87.0      1   0.01     10     50  623.5  623.5  623.5
   25.2  24822  -54.5  -87.5      1   0.01     13     48  625.9  626.0  625.9
   23.4  25298  -53.0  -86.0      1   0.01     25     42  643.7  643.8  643.7
   22.4  25578  -52.1  -85.1      1   0.01     20     40  654.4  654.5  654.4
   21.3  25908  -52.1  -85.1      1   0.01     15     37  664.0  664.2  664.0
   20.0  26310  -52.1  -85.1      1   0.01     15     48  675.9  676.1  676.0
   19.4  26507  -52.7  -85.7      1   0.01     16     49  680.0  680.1  680.0
   18.0  26994  -50.3  -84.3      1   0.02     19     52  702.3  702.5  702.3
   17.6  27127  -50.5  -84.5      1   0.02     20     53  705.8  706.0  705.8
   16.8  27432  -50.9  -84.9      1   0.02     25     53  714.0  714.2  714.0
   16.4  27602  -51.1  -85.1      1   0.02     25     53  718.6  718.8  718.6
   15.7  27887  -49.9  -83.9      1   0.02     26     53  731.6  731.8  731.6
   14.4  28450  -52.1  -85.1      1   0.02     27     53  742.5  742.7  742.5
   11.5  29917  -49.7  -83.7      1   0.03     29     52  800.3  800.7  800.3
   10.6  30480  -50.6  -84.6      1   0.03     30     52  817.1  817.5  817.1
   10.0  30830  -51.1  -85.1      1   0.03     30     55  827.7  828.1  827.7
    9.3  31301  -51.9  -84.9      1   0.03     32     53  842.0  842.4  842.0
    8.6  31811  -49.1  -83.1      1   0.05     33     50  871.9  872.6  872.0
    8.0  32309  -49.1  -83.1      1   0.05     35     48  891.1  891.8  891.1
    7.5  32708  -49.1  -83.1      1   0.05                906.7  907.5  906.7
</PRE><H3>Station information and sounding indices</H3><PRE>
                         Station identifier: TFX
                             Station number: 72776
                           Observation time: 210205/1200
                           Station latitude: 47.46
                          Station longitude: -111.39
                          Station elevation: 1134.0
                            Showalter index: 5.32
                               Lifted index: 5.89
    LIFT computed using virtual temperature: 5.91
                                SWEAT index: 108.01
                                    K index: 18.90
                         Cross totals index: 24.10
                      Vertical totals index: 26.80
                        Totals totals index: 50.90
      Convective Available Potential Energy: 0.00
             CAPE using virtual temperature: 0.00
                      Convective Inhibition: 0.00
             CINS using virtual temperature: 0.00
                     Bulk Richardson Number: 0.00
          Bulk Richardson Number using CAPV: 0.00
  Temp [K] of the Lifted Condensation Level: 266.45
Pres [hPa] of the Lifted Condensation Level: 819.30
   Equivalent potential temp [K] of the LCL: 290.30
     Mean mixed layer potential temperature: 282.09
              Mean mixed layer mixing ratio: 2.84
              1000 hPa to 500 hPa thickness: 5292.00
Precipitable water [mm] for entire sounding: 7.04
</PRE>
<H2>72776 TFX Great Falls Observations at 00Z 06 Feb 2021</H2>
<PRE>
-----------------------------------------------------------------------------
   PRES   HGHT   TEMP   DWPT   RELH   MIXR   DRCT   SKNT   THTA   THTE   THTV
    hPa     m      C      C      %    g/kg    deg   knot     K      K      K 
-----------------------------------------------------------------------------
 1000.0    110                                                               
  925.0    732                                                               
  879.0   1134   -6.1   -7.3     91   2.52      0      0  277.1  284.3  277.5
  869.0   1224   -7.1   -8.2     92   2.38     58      0  276.9  283.8  277.3
  850.0   1396   -8.5   -9.0     96   2.29    170      1  277.2  283.8  277.6
  848.0   1414   -8.7   -9.1     97   2.27    175      2  277.2  283.8  277.6
  827.0   1608   -9.5   -9.7     98   2.22    232     11  278.4  284.8  278.7
  817.0   1703   -6.9   -9.5     82   2.29    259     15  282.1  288.8  282.5
  803.9   1829   -6.0   -9.1     79   2.40    295     21  284.3  291.4  284.7
  793.0   1936   -5.3   -8.8     76   2.49    297     22  286.2  293.6  286.6
  773.1   2134   -7.0  -10.1     78   2.30    300     25  286.4  293.3  286.8
  752.0   2350   -8.9  -11.6     81   2.10    311     28  286.7  293.0  287.0
  743.5   2438   -9.8  -11.9     85   2.08    315     29  286.6  292.9  286.9
  732.0   2558  -11.1  -12.3     91   2.04    321     33  286.5  292.6  286.8
  723.0   2653  -11.5  -13.9     82   1.82    326     35  287.1  292.6  287.4
  714.5   2743  -12.2  -14.2     85   1.79    330     38  287.2  292.7  287.5
  700.0   2900  -13.5  -14.8     90   1.74    325     40  287.5  292.8  287.8
  698.0   2922  -13.5  -14.8     90   1.75    325     40  287.7  293.1  288.1
  632.7   3658  -18.8  -20.2     89   1.22    310     42  289.9  293.7  290.1
  607.6   3962  -21.0  -22.4     89   1.05    305     54  290.7  294.1  290.9
  600.0   4056  -21.7  -23.1     88   1.00    305     52  291.0  294.2  291.1
  582.9   4267  -23.5  -25.3     85   0.84    305     48  291.2  294.0  291.4
  544.0   4770  -27.9  -30.6     78   0.55    305     54  291.8  293.7  291.9
  540.0   4823  -28.3  -29.9     86   0.60    305     54  292.0  294.0  292.1
  536.0   4877  -28.9  -31.2     80   0.53    305     55  291.9  293.6  292.0
  531.0   4944  -29.7  -32.9     74   0.45    305     54  291.7  293.2  291.8
  500.0   5370  -33.3  -36.1     76   0.35    305     51  292.4  293.6  292.4
  478.0   5683  -36.1  -38.4     79   0.29    305     50  292.7  293.7  292.8
  470.5   5791  -36.9  -39.8     74   0.26    305     49  293.0  293.9  293.1
  467.0   5843  -37.3  -40.5     72   0.24    306     51  293.2  294.0  293.2
  450.0   6097  -38.9  -45.9     47   0.14    310     60  294.3  294.8  294.3
  437.0   6297  -38.7  -56.7     13   0.04    315     66  297.0  297.2  297.0
  408.0   6765  -39.3  -65.3      5   0.01    327     82  302.1  302.2  302.1
  400.0   6900  -40.3  -67.3      4   0.01    330     86  302.5  302.6  302.5
  372.0   7394  -42.7  -70.7      3   0.01    333    102  305.7  305.7  305.7
  359.6   7620  -44.4  -71.4      3   0.01    335    109  306.4  306.5  306.4
  336.0   8074  -47.7  -72.8      4   0.01    335    113  307.9  307.9  307.9
  328.2   8230  -48.9  -73.2      4   0.01    335    113  308.4  308.4  308.4
  324.0   8317  -49.5  -73.5      4   0.01    335    112  308.6  308.6  308.6
  300.0   8820  -52.3  -75.3      5   0.00    335    109  311.5  311.6  311.5
  285.2   9144  -54.3  -75.3      6   0.00    330    103  313.2  313.2  313.2
  271.0   9470  -56.3  -75.3      7   0.01    325     98  314.9  314.9  314.9
  259.1   9754  -57.1  -75.3      8   0.01    320     93  317.7  317.7  317.7
  257.0   9806  -57.3  -75.3      8   0.01    320     93  318.2  318.3  318.2
  250.0   9980  -56.9  -74.9      9   0.01    320     97  321.4  321.4  321.4
  249.0  10005  -56.9  -75.9      7   0.01    320     97  321.7  321.7  321.7
  241.0  10212  -55.3  -77.3      5   0.00    321     98  327.1  327.1  327.1
  238.0  10292  -53.3  -77.3      4   0.00    322     99  331.3  331.3  331.3
  226.0  10623  -53.3  -80.3      2   0.00    323    101  336.2  336.3  336.2
  222.0  10738  -51.5  -80.5      2   0.00    324    102  340.7  340.8  340.7
  218.0  10855  -51.3  -82.3      1   0.00    324    102  342.8  342.8  342.8
  216.0  10915  -49.1  -81.1      1   0.00    325    103  347.1  347.1  347.1
  214.1  10973  -48.7  -81.4      1   0.00    325    103  348.6  348.6  348.6
  208.0  11163  -47.5  -82.5      1   0.00    323     89  353.4  353.4  353.4
  200.0  11420  -48.3  -82.3      1   0.00    320     70  356.1  356.1  356.1
  195.1  11582  -49.0  -83.0      1   0.00    320     66  357.5  357.5  357.5
  193.0  11655  -49.3  -83.3      1   0.00    320     68  358.2  358.2  358.2
  191.0  11723  -48.7  -82.7      1   0.00    320     69  360.2  360.2  360.2
  179.0  12149  -49.9  -83.9      1   0.00    320     78  365.0  365.0  365.0
  177.8  12192  -49.8  -83.8      1   0.00    320     79  365.8  365.8  365.8
  171.0  12449  -49.5  -83.5      1   0.00    323     74  370.4  370.4  370.4
  170.0  12488  -48.5  -82.5      1   0.00    324     73  372.7  372.7  372.7
  166.0  12645  -48.3  -82.3      1   0.00    326     71  375.6  375.6  375.6
  158.0  12970  -50.1  -84.1      1   0.00    330     64  377.9  377.9  377.9
  155.0  13095  -49.3  -83.3      1   0.00    332     62  381.3  381.3  381.3
  152.0  13223  -50.3  -84.3      1   0.00    334     60  381.7  381.8  381.7
  150.0  13310  -50.3  -84.3      1   0.00    335     58  383.2  383.2  383.2
  147.7  13411  -50.8  -84.8      1   0.00    335     57  384.1  384.1  384.1
  146.0  13486  -51.1  -85.1      1   0.00    334     57  384.8  384.8  384.8
  144.0  13576  -50.5  -84.5      1   0.00    332     58  387.3  387.4  387.3
  139.0  13805  -51.7  -84.7      1   0.00    327     59  389.2  389.2  389.2
  136.0  13947  -49.9  -83.9      1   0.00    324     59  394.8  394.8  394.8
  127.0  14394  -50.9  -84.9      1   0.00    316     61  400.8  400.8  400.8
  124.0  14549  -49.9  -83.9      1   0.00    313     61  405.3  405.4  405.3
  116.8  14935  -52.9  -86.1      1   0.00    305     63  406.7  406.8  406.7
  115.0  15038  -53.7  -86.7      1   0.00    307     61  407.1  407.1  407.1
  114.0  15094  -53.3  -86.3      1   0.00    308     60  408.9  408.9  408.9
  111.4  15240  -53.3  -86.3      1   0.00    310     57  411.5  411.6  411.5
  110.0  15324  -53.3  -86.3      1   0.00    310     54  413.1  413.1  413.1
  108.0  15442  -52.1  -85.1      1   0.00    310     49  417.5  417.5  417.5
  106.3  15545  -52.1  -85.1      1   0.00    310     45  419.4  419.4  419.4
  102.0  15812  -52.1  -85.1      1   0.00    307     54  424.4  424.4  424.4
  100.0  15940  -51.3  -84.3      1   0.00    305     58  428.3  428.4  428.3
   96.8  16154  -51.5  -84.5      1   0.00    300     62  432.1  432.1  432.1
   96.7  16158  -51.5  -84.5      1   0.00    300     62  432.1  432.2  432.1
   73.6  17927  -52.7  -85.7      1   0.00    325     33  464.6  464.6  464.6
   70.0  18250  -53.7  -86.7      1   0.00    330     28  469.1  469.2  469.1
   69.6  18288  -53.8  -86.7      1   0.00    320     25  469.7  469.8  469.7
   66.4  18593  -54.5  -86.8      1   0.00    305     31  474.6  474.6  474.6
   64.8  18745  -54.9  -86.9      1   0.00    314     31  477.0  477.0  477.0
   60.3  19202  -54.4  -87.2      1   0.00    340     31  487.9  488.0  487.9
   59.6  19281  -54.3  -87.3      1   0.00    336     32  489.9  489.9  489.9
   57.5  19507  -55.3  -87.9      1   0.00    325     33  492.5  492.5  492.5
   54.9  19806  -56.7  -88.7      1   0.00    327     33  496.0  496.0  496.0
   50.0  20400  -56.1  -88.1      1   0.00    330     33  510.8  510.9  510.8
   45.3  21031  -54.4  -86.9      1   0.00    335     41  529.5  529.6  529.5
   43.2  21336  -53.6  -86.4      1   0.01    340     39  538.8  538.9  538.8
   41.5  21591  -52.9  -85.9      1   0.01    336     36  546.7  546.8  546.7
   41.2  21641  -53.0  -86.0      1   0.01    335     36  547.7  547.7  547.7
   39.3  21946  -53.7  -86.5      1   0.01      0     34  553.5  553.5  553.5
   35.7  22555  -55.0  -87.5      1   0.01    355     29  565.3  565.3  565.3
   31.7  23311  -56.7  -88.7      1   0.00      5     34  580.3  580.3  580.3
   30.0  23660  -56.5  -88.5      1   0.01     10     37  590.0  590.1  590.0
   28.9  23897  -55.9  -87.9      1   0.01      8     40  598.0  598.1  598.0
   28.0  24100  -50.5  -84.5      1   0.01      7     43  618.4  618.5  618.4
   26.8  24384  -49.8  -83.8      1   0.01      5     46  628.1  628.2  628.1
   26.6  24434  -49.7  -83.7      1   0.01      8     46  629.8  630.0  629.8
   25.6  24689  -49.6  -83.6      1   0.01     25     43  637.2  637.4  637.3
   21.2  25908  -48.9  -82.9      1   0.02     25     35  674.1  674.3  674.1
   20.0  26300  -48.7  -82.7      1   0.02     25     40  686.3  686.6  686.4
   19.4  26500  -48.1  -82.1      1   0.02     25     43  694.2  694.5  694.2
   17.6  27139  -50.3  -84.3      1   0.02     25     54  706.8  707.0  706.8
   17.6  27127  -50.3  -84.3      1   0.02     25     54  706.6  706.8  706.6
   16.8  27432  -49.8  -83.8      1   0.02     35     52  717.5  717.7  717.5
   14.9  28229  -48.5  -82.5      1   0.03     35     56  747.2  747.6  747.2
   12.8  29224  -50.3  -84.3      1   0.03     35     60  774.1  774.4  774.1
   12.2  29566  -49.8  -83.8      1   0.03     35     62  787.4  787.8  787.4
   11.7  29812  -49.5  -83.5      1   0.03                797.1  797.5  797.1
</PRE><H3>Station information and sounding indices</H3><PRE>
                         Station identifier: TFX
                             Station number: 72776
                           Observation time: 210206/0000
                           Station latitude: 47.46
                          Station longitude: -111.39
                          Station elevation: 1134.0
                            Showalter index: 7.63
                               Lifted index: 7.42
    LIFT computed using virtual temperature: 7.44
                                SWEAT index: 54.97
                                    K index: 14.50
                         Cross totals index: 24.30
                      Vertical totals index: 24.80
                        Totals totals index: 49.10
      Convective Available Potential Energy: 0.00
             CAPE using virtual temperature: 0.00
                      Convective Inhibition: 0.00
             CINS using virtual temperature: 0.00
                     Bulk Richardson Number: 0.00
          Bulk Richardson Number using CAPV: 0.00
  Temp [K] of the Lifted Condensation Level: 264.20
Pres [hPa] of the Lifted Condensation Level: 842.92
   Equivalent potential temp [K] of the LCL: 284.07
     Mean mixed layer potential temperature: 277.43
              Mean mixed layer mixing ratio: 2.32
              1000 hPa to 500 hPa thickness: 5260.00
Precipitable water [mm] for entire sounding: 6.23
</PRE>
<H2>72776 TFX Great Falls Observations at 12Z 06 Feb 2021</H2>
<PRE>
-----------------------------------------------------------------------------
   PRES   HGHT   TEMP   DWPT   RELH   MIXR   DRCT   SKNT   THTA   THTE   THTV
    hPa     m      C      C      %    g/kg    deg   knot     K      K      K 
-----------------------------------------------------------------------------
 1000.0    210                                                               
  925.0    813                                                               
  887.0   1134  -19.1  -21.7     80   0.76      0      0  262.9  265.1  263.0
  879.0   1201  -19.7  -22.9     76   0.69    326      2  263.0  265.0  263.1
  861.0   1355  -19.9  -23.7     72   0.66    248      7  264.3  266.2  264.4
  850.0   1451  -19.7  -23.5     72   0.68    200     10  265.5  267.5  265.6
  840.0   1539  -19.5  -22.9     74   0.73    196      9  266.6  268.7  266.7
  830.0   1628  -18.3  -21.9     73   0.80    192      8  268.8  271.1  268.9
  815.0   1764  -18.9  -22.6     73   0.77    186      7  269.6  271.8  269.7
  802.0   1885  -16.1  -19.8     73   1.00    181      6  273.8  276.8  273.9
  778.0   2114  -14.9  -18.3     75   1.17    171      4  277.4  280.9  277.6
  775.9   2134  -15.0  -18.4     75   1.16    170      4  277.6  281.1  277.8
  756.0   2331  -15.7  -19.1     75   1.12    238     10  278.9  282.3  279.1
  745.4   2438  -15.1  -17.4     82   1.32    275     13  280.7  284.7  280.9
  730.0   2596  -14.1  -14.9     94   1.66    278     22  283.4  288.4  283.7
  716.0   2743  -14.2  -14.8     95   1.70    280     30  284.9  290.0  285.1
  710.0   2806  -14.3  -14.8     96   1.72    282     33  285.5  290.7  285.8
  700.0   2914  -14.9  -15.3     97   1.67    285     38  286.0  291.0  286.2
  660.1   3353  -18.2  -18.6     97   1.34    295     51  287.0  291.1  287.2
  633.8   3658  -20.6  -21.0     97   1.14    300     49  287.7  291.3  287.9
  628.0   3726  -21.1  -21.5     97   1.10    303     50  287.9  291.3  288.1
  607.0   3977  -22.7  -24.9     82   0.84    313     54  288.9  291.5  289.0
  601.0   4050  -22.7  -28.7     58   0.60    316     55  289.7  291.6  289.8
  587.0   4223  -22.5  -36.5     27   0.29    323     57  291.9  292.9  291.9
  583.5   4267  -22.3  -36.0     28   0.31    325     58  292.5  293.6  292.6
  578.0   4337  -22.1  -35.1     30   0.34    324     59  293.6  294.8  293.7
  564.0   4517  -22.7  -34.7     33   0.36    323     61  295.0  296.2  295.0
  540.0   4835  -25.1  -30.1     63   0.58    320     65  295.8  297.8  295.9
  536.8   4877  -25.4  -30.1     65   0.59    320     66  295.9  297.9  296.1
  521.0   5094  -26.9  -29.9     76   0.62    316     68  296.7  298.8  296.8
  500.0   5390  -28.7  -31.6     76   0.55    310     70  298.0  299.9  298.1
  490.0   5535  -29.7  -32.2     79   0.53    310     71  298.5  300.3  298.6
  455.0   6059  -34.1  -36.1     82   0.39    310     74  299.4  300.7  299.4
  452.6   6096  -34.4  -36.5     81   0.37    310     74  299.4  300.7  299.5
  433.1   6401  -37.1  -40.1     73   0.27    315     74  299.8  300.8  299.9
  423.0   6565  -38.5  -42.0     69   0.23    316     77  300.0  300.9  300.1
  413.0   6730  -38.9  -41.3     78   0.25    318     79  301.6  302.5  301.6
  407.0   6831  -38.1  -40.6     77   0.27    319     81  303.9  304.9  303.9
  400.0   6950  -38.7  -41.6     74   0.25    320     83  304.6  305.5  304.7
  387.0   7176  -39.9  -42.8     74   0.23    322     86  305.9  306.8  306.0
  362.0   7620  -43.5  -47.0     69   0.15    325     93  307.0  307.6  307.0
  316.0   8523  -50.9  -55.4     59   0.07    325    110  308.9  309.1  308.9
  305.0   8753  -50.7  -54.8     62   0.07    325    115  312.3  312.6  312.3
  300.0   8860  -51.7  -55.8     61   0.07    325    117  312.4  312.6  312.4
  293.0   9013  -52.7  -56.6     62   0.06    325    115  313.1  313.3  313.1
  287.1   9144  -53.5  -58.8     52   0.05    325    113  313.7  313.9  313.8
  280.0   9304  -54.5  -61.5     42   0.03    326    117  314.6  314.7  314.6
  269.0   9561  -54.3  -69.3     14   0.01    327    123  318.5  318.5  318.5
  250.0  10030  -54.5  -73.5      8   0.01    330    134  324.9  324.9  324.9
  248.9  10058  -54.5  -73.5      8   0.01    330    135  325.3  325.4  325.3
  247.0  10107  -54.5  -73.5      8   0.01    330    135  326.0  326.1  326.0
  246.0  10133  -54.6  -73.6      8   0.01    330    135  326.2  326.2  326.2
  226.0  10673  -57.5  -75.5      8   0.01    325    110  329.8  329.9  329.8
  220.0  10843  -57.9  -76.9      7   0.00    319    101  331.8  331.8  331.8
  216.0  10959  -57.3  -77.3      6   0.00    315     96  334.4  334.5  334.4
  215.5  10973  -56.9  -77.4      6   0.00    315     95  335.2  335.2  335.2
  214.0  11019  -55.7  -77.7      5   0.00    315     96  337.8  337.8  337.8
  209.0  11169  -56.3  -81.3      3   0.00    315     98  339.2  339.2  339.2
  201.0  11418  -55.1  -81.1      3   0.00    315    102  344.9  344.9  344.9
  200.0  11450  -54.7  -80.7      3   0.00    315    102  346.0  346.0  346.0
  197.0  11547  -52.7  -80.7      2   0.00    315    106  350.7  350.7  350.7
  195.9  11582  -52.5  -81.0      2   0.00    315    108  351.5  351.5  351.5
  187.0  11886  -50.9  -83.9      1   0.00    317    101  358.8  358.8  358.8
  182.0  12062  -52.1  -85.1      1   0.00    319     97  359.7  359.7  359.7
  162.4  12802  -50.7  -84.7      1   0.00    325     80  373.9  373.9  373.9
  162.0  12819  -50.7  -84.7      1   0.00    325     79  374.2  374.2  374.2
  150.0  13320  -52.1  -85.1      1   0.00    320     65  380.1  380.1  380.1
  147.9  13411  -52.7  -85.7      1   0.00    315     60  380.7  380.7  380.7
  147.0  13450  -52.9  -85.9      1   0.00    315     60  380.9  380.9  380.9
  141.0  13719  -52.9  -85.9      1   0.00    316     58  385.5  385.5  385.5
  137.0  13904  -51.9  -84.9      1   0.00    316     57  390.4  390.4  390.4
  129.0  14293  -52.9  -85.9      1   0.00    317     55  395.4  395.4  395.4
  119.0  14813  -52.5  -85.5      1   0.00    319     52  405.4  405.4  405.4
  112.0  15203  -53.5  -86.5      1   0.00    320     49  410.6  410.6  410.6
  111.3  15240  -53.6  -86.6      1   0.00    320     49  411.1  411.1  411.1
  110.0  15318  -53.7  -86.7      1   0.00    319     50  412.3  412.3  412.3
  100.0  15930  -53.5  -86.5      1   0.00    310     57  424.1  424.1  424.1
   92.1  16459  -53.5  -86.5      1   0.00    320     46  434.2  434.2  434.2
   87.8  16764  -53.5  -86.5      1   0.00    315     46  440.1  440.1  440.1
   83.8  17069  -53.5  -86.5      1   0.00    305     56  446.1  446.1  446.1
   82.0  17206  -53.5  -86.5      1   0.00    307     58  448.8  448.8  448.8
   77.5  17567  -55.3  -87.3      1   0.00    313     64  452.4  452.4  452.4
   76.2  17678  -54.8  -87.0      1   0.00    315     66  455.6  455.6  455.6
   75.3  17752  -54.5  -86.8      1   0.00    315     66  457.8  457.8  457.8
   70.0  18220  -52.5  -85.5      1   0.00    330     35  471.7  471.7  471.7
   69.9  18229  -52.5  -85.5      1   0.00    329     35  471.9  471.9  471.9
   69.3  18288  -52.9  -85.8      1   0.00    325     33  472.3  472.3  472.3
   66.0  18593  -55.0  -87.3      1   0.00    305     33  474.1  474.1  474.1
   64.8  18715  -55.9  -87.9      1   0.00    308     35  474.8  474.8  474.8
   63.7  18824  -54.9  -86.9      1   0.00    311     37  479.3  479.4  479.3
   60.0  19202  -56.5  -88.1      1   0.00    320     43  483.9  483.9  483.9
   55.0  19755  -58.9  -89.9      1   0.00    306     37  490.7  490.7  490.7
   54.5  19812  -58.4  -89.6      1   0.00    305     36  493.1  493.1  493.1
   50.0  20360  -53.7  -86.7      1   0.00    340     40  516.5  516.5  516.5
   48.7  20530  -52.5  -85.5      1   0.01    345     42  523.2  523.3  523.2
   47.2  20726  -52.4  -85.4      1   0.01    350     44  528.0  528.1  528.0
   45.6  20954  -52.3  -85.3      1   0.01    348     40  533.6  533.7  533.6
   43.0  21336  -54.2  -87.2      1   0.00    345     34  538.2  538.2  538.2
   42.5  21407  -54.5  -87.5      1   0.00    345     34  539.1  539.1  539.1
   37.3  22250  -53.6  -86.6      1   0.01    345     37  562.0  562.1  562.0
   35.7  22525  -53.3  -86.3      1   0.01    348     35  569.7  569.8  569.7
   32.3  23165  -54.1  -86.5      1   0.01    355     31  584.0  584.1  584.0
   30.0  23640  -54.7  -86.7      1   0.01     15     29  594.9  595.0  594.9
   29.4  23774  -54.9  -86.8      1   0.01     20     31  598.1  598.2  598.1
   28.0  24079  -55.2  -87.2      1   0.01     15     26  605.4  605.4  605.4
   26.7  24384  -55.5  -87.5      1   0.01     20     33  612.7  612.8  612.8
   26.1  24527  -55.7  -87.7      1   0.01     20     35  616.2  616.3  616.2
   24.0  25063  -53.7  -86.7      1   0.01     22     44  637.0  637.1  637.0
   23.7  25144  -50.9  -84.9      1   0.01     23     46  647.4  647.6  647.4
   21.7  25717  -51.3  -84.3      1   0.02     24     55  662.8  662.9  662.8
   21.1  25908  -49.8  -83.4      1   0.02     25     58  672.8  673.0  672.9
   20.7  26024  -48.9  -82.9      1   0.02     25     56  679.0  679.2  679.0
   20.0  26250  -48.3  -82.3      1   0.02     25     52  687.6  687.8  687.6
   19.4  26450  -49.1  -83.1      1   0.02     27     51  691.1  691.3  691.1
   18.0  26944  -47.1  -82.1      1   0.03     31     50  712.4  712.6  712.4
   16.7  27432  -48.2  -82.6      1   0.03     35     48  724.1  724.4  724.1
   15.7  27845  -49.1  -83.1      1   0.03     33     43  734.2  734.5  734.2
   13.9  28651  -49.0  -83.0      1   0.03     30     34  760.7  761.1  760.7
   12.4  29395  -48.9  -82.9      1   0.03     30     50  786.1  786.5  786.1
   12.1  29566  -48.5  -82.6      1   0.04     30     54  793.5  794.0  793.5
   10.5  30480  -46.0  -80.8      1   0.06     35     45  834.3  835.0  834.4
   10.1  30754  -45.3  -80.3      1   0.06     39     46  846.9  847.7  847.0
   10.0  30820  -45.3  -80.3      1   0.06     40     46  849.3  850.1  849.4
    9.6  31090  -46.3  -81.0      1   0.06     45     39  855.6  856.3  855.6
    8.4  32004  -49.4  -83.4      1   0.05     40     52  877.9  878.5  877.9
    8.4  31977  -49.5  -83.5      1   0.04     40     52  876.3  876.9  876.3
    8.0  32299  -47.7  -82.7      1   0.05     39     51  895.7  896.4  895.7
    7.4  32814  -48.9  -82.9      1   0.06     37     49  911.0  911.8  911.0
    7.0  33223  -48.3  -82.3      1   0.07     35     46  929.6  930.6  929.7
    7.0  33180  -48.7  -82.7      1   0.06     35     47  926.4  927.3  926.5
    6.9  33275  -47.9  -81.9      1   0.07                933.5  934.5  933.6
</PRE><H3>Station information and sounding indices</H3><PRE>
                         Station identifier: TFX
                             Station number: 72776
                           Observation time: 210206/1200
                           Station latitude: 47.46
                          Station longitude: -111.39
                          Station elevation: 1134.0
                            Showalter index: 25.16
                               Lifted index: 25.56
    LIFT computed using virtual temperature: 25.63
                                SWEAT index: 90.02
                                    K index: -14.90
                         Cross totals index: 5.20
                      Vertical totals index: 9.00
                        Totals totals index: 14.20
      Convective Available Potential Energy: 0.00
             CAPE using virtual temperature: 0.00
                      Convective Inhibition: 0.00
             CINS using virtual temperature: 0.00
                     Bulk Richardson Number: 0.00
          Bulk Richardson Number using CAPV: 0.00
  Temp [K] of the Lifted Condensation Level: 249.47
Pres [hPa] of the Lifted Condensation Level: 810.09
   Equivalent potential temp [K] of the LCL: 266.99
     Mean mixed layer potential temperature: 264.96
              Mean mixed layer mixing ratio: 0.70
              1000 hPa to 500 hPa thickness: 5180.00
Precipitable water [mm] for entire sounding: 4.36
</PRE>
<H2>72776 TFX Great Falls Observations at 00Z 07 Feb 2021</H2>
<PRE>
-----------------------------------------------------------------------------
   PRES   HGHT   TEMP   DWPT   RELH   MIXR   DRCT   SKNT   THTA   THTE   THTV
    hPa     m      C      C      %    g/kg    deg   knot     K      K      K 
-----------------------------------------------------------------------------
 1000.0    188                                                               
  925.0    777                                                               
  882.0   1134  -19.1  -21.5     81   0.78      0     12  263.3  265.6  263.4
  874.0   1201  -20.1  -22.7     80   0.71    359     11  263.0  265.0  263.1
  862.0   1303  -20.9  -23.6     79   0.67    357     10  263.2  265.1  263.3
  850.0   1406  -21.5  -23.8     82   0.66    355      8  263.6  265.5  263.7
  824.0   1634  -22.9  -24.6     86   0.64    342      5  264.5  266.4  264.6
  805.0   1806  -21.9  -24.4     80   0.66    331      2  267.3  269.3  267.4
  802.5   1829  -21.9  -24.4     80   0.66    330      2  267.5  269.5  267.6
  794.0   1907  -22.1  -24.5     81   0.67    325      5  268.1  270.1  268.3
  789.0   1954  -21.5  -23.6     83   0.73    322      7  269.3  271.4  269.4
  781.0   2029  -18.9  -20.4     88   0.97    317     10  272.9  275.7  273.0
  770.1   2134  -19.1  -20.6     87   0.97    310     14  273.8  276.6  273.9
  755.0   2281  -19.3  -21.0     86   0.95    286     19  275.1  277.9  275.2
  746.0   2371  -18.1  -19.6     88   1.09    271     22  277.3  280.6  277.5
  743.0   2401  -16.7  -17.8     91   1.28    266     23  279.2  283.0  279.4
  740.0   2431  -12.3  -12.6     98   1.97    261     24  284.3  290.2  284.6
  739.4   2438  -11.9  -12.1     98   2.05    260     24  284.8  291.0  285.2
  738.0   2452  -10.9  -11.1     98   2.23    260     24  286.0  292.7  286.4
  723.0   2610  -10.9  -12.0     92   2.12    266     26  287.7  294.1  288.1
  710.6   2743  -11.8  -13.1     90   1.98    270     28  288.2  294.2  288.5
  700.0   2858  -12.5  -14.0     89   1.86    275     30  288.6  294.3  288.9
  654.9   3353  -16.3  -18.2     86   1.40    290     38  289.8  294.2  290.1
  628.5   3658  -18.7  -20.8     84   1.17    300     39  290.5  294.2  290.8
  603.3   3962  -21.1  -23.4     82   0.97    305     38  291.2  294.3  291.4
  579.0   4267  -23.4  -26.0     80   0.80    305     33  291.9  294.5  292.0
  551.0   4636  -26.3  -29.1     77   0.63    299     36  292.7  294.8  292.8
  532.7   4877  -28.0  -30.8     77   0.55    295     38  293.4  295.3  293.5
  500.0   5330  -31.3  -34.0     77   0.43    305     35  294.8  296.3  294.9
  468.3   5791  -34.9  -38.0     74   0.31    300     38  295.9  296.9  295.9
  457.0   5964  -36.3  -39.5     72   0.27    300     43  296.2  297.2  296.3
  448.4   6096  -36.6  -40.0     71   0.26    300     46  297.4  298.4  297.5
  447.0   6118  -36.7  -40.1     71   0.26    300     46  297.6  298.5  297.7
  400.0   6880  -43.3  -46.8     68   0.14    295     54  298.6  299.2  298.7
  387.0   7101  -45.3  -49.3     64   0.11    293     57  298.8  299.3  298.9
  359.0   7596  -49.1  -51.5     76   0.09    290     63  300.2  300.6  300.2
  357.7   7620  -49.1  -51.9     72   0.09    290     65  300.6  300.9  300.6
  353.0   7706  -49.1  -53.3     61   0.08    291     67  301.7  302.0  301.7
  345.0   7856  -48.3  -61.3     21   0.03    292     71  304.8  304.9  304.8
  336.0   8029  -48.7  -69.7      7   0.01    294     75  306.5  306.6  306.5
  300.0   8770  -49.3  -75.3      3   0.00    300     92  315.8  315.8  315.8
  289.0   9014  -49.3  -74.3      4   0.01    300     97  319.1  319.2  319.1
  285.0   9105  -48.7  -74.7      3   0.01    300     99  321.3  321.3  321.3
  283.3   9144  -48.7  -74.8      3   0.01    300    100  321.9  321.9  321.9
  276.0   9316  -48.5  -75.5      3   0.00    306    103  324.5  324.6  324.5
  270.4   9449  -49.1  -76.3      3   0.00    310    106  325.6  325.6  325.6
  270.0   9459  -49.1  -76.4      3   0.00    310    106  325.6  325.6  325.6
  251.0   9934  -51.3  -79.3      2   0.00    301     88  329.3  329.3  329.3
  250.0   9960  -51.1  -79.1      2   0.00    300     87  330.0  330.0  330.0
  246.3  10058  -50.5  -80.4      2   0.00    300     83  332.2  332.2  332.2
  246.0  10065  -50.5  -80.5      2   0.00    300     83  332.4  332.4  332.4
  237.0  10308  -51.3  -82.3      1   0.00    301     87  334.7  334.8  334.7
  234.0  10391  -50.7  -81.7      1   0.00    302     88  336.9  336.9  336.9
  221.0  10762  -52.3  -83.3      1   0.00    304     94  339.9  340.0  339.9
  214.0  10971  -51.3  -83.3      1   0.00    305     97  344.6  344.6  344.6
  213.9  10973  -51.3  -83.3      1   0.00    305     97  344.7  344.7  344.7
  209.0  11125  -52.1  -85.1      1   0.00    307     93  345.7  345.7  345.7
  205.0  11250  -51.5  -84.5      1   0.00    308     89  348.6  348.6  348.6
  200.0  11410  -52.5  -84.5      1   0.00    310     84  349.5  349.5  349.5
  197.0  11508  -53.1  -85.1      1   0.00    310     83  350.0  350.0  350.0
  194.0  11607  -52.3  -85.3      1   0.00    310     81  352.8  352.9  352.8
  187.0  11845  -52.7  -85.7      1   0.00    310     78  355.9  355.9  355.9
  185.8  11887  -52.3  -85.5      1   0.00    310     77  357.2  357.2  357.2
  181.0  12057  -50.9  -84.9      1   0.00    309     78  362.2  362.2  362.2
  179.0  12129  -51.1  -85.1      1   0.00    309     79  363.0  363.0  363.0
  173.0  12351  -51.9                         308     80  365.2         365.2
  167.0  12581  -50.7                         308     81  370.9         370.9
  162.0  12779  -51.5                         307     83  372.8         372.8
  155.0  13067  -50.9                         306     85  378.6         378.6
  153.0  13151  -51.7                         305     85  378.6         378.6
  150.0  13280  -51.9                         305     86  380.4         380.4
  141.0  13680  -52.7                         310     86  385.8         385.8
  140.2  13716  -52.5                         310     86  386.8         386.8
  130.0  14207  -49.9                         310     62  399.9         399.9
  127.7  14326  -50.1                         310     56  401.6         401.6
  114.0  15062  -51.5                         306     50  412.2         412.2
  110.9  15240  -51.6                         305     49  415.2         415.2
  110.0  15294  -51.7                         305     50  416.1         416.1
  108.0  15413  -51.1                         306     52  419.4         419.4
  100.0  15910  -53.5                         310     59  424.1         424.1
   91.8  16459  -54.5                         300     47  432.6         432.6
   87.5  16764  -55.1                         300     63  437.4         437.4
   85.8  16889  -55.3                         300     63  439.4         439.4
   83.1  17094  -55.7                         305     61  442.6         442.6
   70.0  18190  -53.9                         330     52  468.7         468.7
   69.5  18236  -53.5                         330     50  470.5         470.5
   68.9  18288  -53.6                         330     48  471.5         471.5
   60.3  19148  -54.7                         325     38  487.3         487.3
   51.9  20117  -54.2                         320     27  510.1         510.1
   50.8  20248  -54.1                         317     29  513.2         513.2
   50.0  20350  -54.7                         315     30  514.1         514.1
   48.6  20531  -56.1                         314     31  515.0         515.0
   46.3  20840  -56.5                         311     34  521.2         521.2
   44.9  21031  -56.4                         310     35  526.0         526.0
   42.8  21336  -56.1                         325     36  533.8         533.8
   40.8  21641  -55.9                         330     49  541.7         541.7
   40.5  21691  -55.9                         333     48  543.0         543.0
   38.3  22049  -52.5                         354     40  560.4         560.4
   37.1  22250  -53.1                           5     36  563.9         563.9
   33.5  22911  -55.1                           1     35  575.4         575.4
   32.2  23165  -54.2                           0     34  584.2         584.2
   30.0  23620  -52.7                          20     34  600.4         600.4
   29.7  23685  -52.5                          22     34  602.6         602.6
   29.3  23774  -52.7                          25     34  604.6         604.6
   26.6  24384  -53.8                          15     29  618.2         618.2
   24.1  25026  -54.9                          29     27  632.8         632.8
   23.6  25161  -53.3                          32     26  641.2         641.2
   23.1  25298  -53.2                          35     26  645.3         645.3
   22.0  25603  -53.1                           5     26  654.4         654.4
   21.6  25730  -53.1                           8     30  658.2         658.2
   20.0  26230  -49.5                          20     46  683.9         683.9
   19.4  26430  -48.7                          20     50  692.3         692.3
   19.1  26518  -48.7                          20     52  695.0         695.0
   16.7  27432  -48.6                          30     47  723.4         723.4
   14.5  28346  -48.5                          40     35  753.0         753.0
   14.1  28530  -48.5                          37     35  759.1         759.1
   12.2  29487  -46.5                          24     33  798.2         798.2
   11.4  29936  -48.1                          18     32  808.1         808.1
   11.0  30175  -48.2                          15     32  816.3         816.3
   10.5  30480  -48.2                          15     35  826.9         826.9
   10.0  30800  -48.3                          15     34  838.1         838.1
    9.8  30933  -48.3                          15     37  843.0         843.0
    9.1  31394  -46.6                          15     49  866.4         866.4
    8.6  31799  -45.1                          28     40  887.5         887.5
    8.0  32309  -45.8                          45     28  904.5         904.5
    8.0  32281  -45.9                          44     29  902.9         902.9
    7.9  32364  -45.5                                     907.7         907.7
</PRE><H3>Station information and sounding indices</H3><PRE>
                         Station identifier: TFX
                             Station number: 72776
                           Observation time: 210207/0000
                           Station latitude: 47.46
                          Station longitude: -111.39
                          Station elevation: 1134.0
                            Showalter index: 24.13
                               Lifted index: 24.14
    LIFT computed using virtual temperature: 24.20
                                SWEAT index: 51.01
                                    K index: -15.50
                         Cross totals index: 7.50
                      Vertical totals index: 9.80
                        Totals totals index: 17.30
      Convective Available Potential Energy: 0.00
             CAPE using virtual temperature: 0.00
                      Convective Inhibition: 0.00
             CINS using virtual temperature: 0.00
                     Bulk Richardson Number: 0.00
          Bulk Richardson Number using CAPV: 0.00
  Temp [K] of the Lifted Condensation Level: 249.13
Pres [hPa] of the Lifted Condensation Level: 821.01
   Equivalent potential temp [K] of the LCL: 265.53
     Mean mixed layer potential temperature: 263.59
              Mean mixed layer mixing ratio: 0.67
              1000 hPa to 500 hPa thickness: 5142.00
Precipitable water [mm] for entire sounding: 4.39
</PRE>
<H2>72776 TFX Great Falls Observations at 12Z 07 Feb 2021</H2>
<PRE>
-----------------------------------------------------------------------------
   PRES   HGHT   TEMP   DWPT   RELH   MIXR   DRCT   SKNT   THTA   THTE   THTV
    hPa     m      C      C      %    g/kg    deg   knot     K      K      K 
-----------------------------------------------------------------------------
 1000.0    239                                                               
  925.0    822                                                               
  887.0   1134  -23.3  -26.5     75   0.50    335      8  258.6  260.0  258.6
  884.0   1159  -23.5  -27.2     72   0.47    334      8  258.6  260.0  258.7
  858.0   1375  -25.5  -28.9     73   0.41    327      9  258.7  259.9  258.8
  850.0   1443  -25.5  -29.0     72   0.41    325      9  259.4  260.6  259.5
  840.0   1529  -24.5  -28.0     73   0.46    303      8  261.4  262.7  261.4
  805.9   1829  -27.5  -29.5     83   0.41    225      4  261.3  262.5  261.4
  799.0   1891  -28.1  -29.8     85   0.41    237      4  261.3  262.5  261.3
  778.0   2082  -28.9  -30.8     84   0.38    275      6  262.4  263.6  262.5
  772.3   2134  -28.8  -30.4     86   0.40    285      6  263.1  264.2  263.1
  764.0   2212  -28.7  -29.8     90   0.43    293      7  264.0  265.3  264.1
  755.0   2297  -27.3  -28.8     87   0.47    301      7  266.4  267.8  266.5
  740.4   2438  -27.1  -29.2     82   0.46    315      8  268.1  269.5  268.2
  740.0   2441  -27.1  -29.2     82   0.46    315      8  268.1  269.6  268.2
  725.0   2589  -27.3  -28.6     89   0.50    310     15  269.5  271.0  269.6
  721.0   2629  -26.7  -28.1     88   0.53    309     17  270.6  272.2  270.7
  716.0   2679  -23.3  -23.7     97   0.79    307     20  274.9  277.3  275.0
  709.9   2743  -20.8  -21.0     98   1.01    305     23  278.4  281.4  278.5
  705.0   2794  -18.7  -18.9     98   1.23    300     29  281.2  284.9  281.4
  700.0   2847  -18.1  -18.3     98   1.30    295     35  282.4  286.4  282.6
  683.0   3031  -16.5  -17.6     91   1.41    295     50  286.2  290.5  286.4
  681.5   3048  -16.6  -17.8     90   1.39    295     51  286.2  290.5  286.5
  627.6   3658  -20.7  -25.2     67   0.79    300     42  288.4  291.0  288.6
  590.0   4116  -23.7  -30.7     53   0.51    304     52  290.0  291.7  290.1
  577.8   4267  -25.0  -31.1     57   0.50    305     56  290.3  291.9  290.3
  564.0   4443  -26.5  -31.5     63   0.49    302     57  290.5  292.1  290.6
  554.0   4572  -27.3  -37.3     38   0.28    300     58  291.0  292.0  291.1
  530.8   4877  -30.0  -41.0     33   0.20    295     60  291.4  292.1  291.4
  510.0   5161  -32.5  -44.5     29   0.14    295     63  291.7  292.2  291.7
  508.5   5182  -32.7  -44.2     31   0.15    295     63  291.7  292.3  291.8
  500.0   5300  -33.7  -42.7     40   0.18    290     62  291.9  292.5  291.9
  469.0   5745  -37.9  -42.7     61   0.19    284     62  292.1  292.7  292.1
  445.5   6096  -41.1  -45.0     66   0.16    280     62  292.4  292.9  292.4
  430.0   6337  -43.3  -46.5     71   0.14    280     60  292.5  293.0  292.6
  421.0   6479  -44.5  -48.3     66   0.11    280     59  292.8  293.2  292.8
  410.0   6656  -46.1  -48.3     78   0.12    280     58  292.9  293.4  292.9
  400.0   6820  -47.5  -49.5     80   0.10    280     57  293.2  293.6  293.2
  394.0   6919  -48.5  -50.0     84   0.10    279     57  293.1  293.5  293.2
  373.0   7277  -50.9  -53.7     72   0.07    275     56  294.6  294.8  294.6
  370.9   7315  -50.1  -54.1     63   0.07    275     53  296.1  296.3  296.1
  369.0   7348  -49.5  -54.5     56   0.06    276     54  297.4  297.6  297.4
  359.0   7527  -49.3  -63.3     18   0.02    282     60  300.0  300.1  300.0
  354.0   7619  -49.5  -68.5      9   0.01    285     63  300.9  300.9  300.9
  353.9   7620  -49.5  -68.5      9   0.01    285     63  300.9  301.0  300.9
  322.4   8230  -49.5  -75.5      3   0.00    290     78  309.1  309.1  309.1
  318.0   8320  -49.5  -76.5      3   0.00    291     78  310.3  310.3  310.3
  304.0   8614  -50.5  -77.5      3   0.00    294     78  312.9  312.9  312.9
  300.0   8700  -50.5  -77.5      3   0.00    295     78  314.1  314.1  314.1
  292.0   8877  -50.5  -78.5      2   0.00    297     80  316.5  316.5  316.5
  286.0   9012  -51.5  -79.5      2   0.00    299     81  316.9  317.0  316.9
  282.0   9104  -50.9  -78.9      2   0.00    300     82  319.1  319.1  319.1
  280.3   9144  -51.0  -79.2      2   0.00    300     82  319.5  319.5  319.5
  272.0   9340  -51.5  -80.5      2   0.00    300     81  321.5  321.5  321.5
  265.0   9510  -50.5  -80.5      2   0.00    300     81  325.4  325.4  325.4
  250.0   9890  -51.7  -81.7      2   0.00    300     79  329.1  329.1  329.1
  242.0  10100  -52.9  -81.9      2   0.00    302     82  330.4  330.4  330.4
  225.0  10572  -50.9  -82.4      1   0.00    305     90  340.3  340.3  340.3
  221.7  10668  -50.5  -82.5      1   0.00    305     90  342.4  342.4  342.4
  215.0  10867  -49.7  -82.7      1   0.00    305     86  346.7  346.7  346.7
  203.0  11242  -49.1  -83.1      1   0.00    305     80  353.4  353.4  353.4
  200.0  11340  -47.9  -82.9      1   0.00    305     78  356.8  356.8  356.8
  192.8  11582  -48.8  -83.2      1   0.00    305     70  359.1  359.1  359.1
  189.0  11712  -49.3  -83.3      1   0.00    305     72  360.3  360.3  360.3
  182.0  11959  -49.3  -83.3      1   0.00    305     75  364.2  364.2  364.2
  177.0  12142  -48.3  -82.3      1   0.00    305     77  368.8  368.8  368.8
  167.7  12497  -49.0  -83.0      1   0.00    305     82  373.3  373.3  373.3
  162.0  12724  -49.5  -83.5      1   0.00    305     77  376.2  376.2  376.2
  159.0  12846  -48.5  -82.5      1   0.00    305     74  379.9  379.9  379.9
  153.0  13100  -48.1  -82.1      1   0.00    305     68  384.8  384.8  384.8
  150.0  13230  -48.7  -82.7      1   0.00    305     65  385.9  386.0  385.9
  144.0  13499  -48.3  -82.3      1   0.00    302     61  391.2  391.2  391.2
  139.3  13716  -49.3  -83.3      1   0.00    300     58  393.1  393.1  393.1
  136.0  13874  -50.1  -84.1      1   0.00    301     58  394.4  394.4  394.4
  131.0  14120  -49.7  -83.7      1   0.00    301     59  399.4  399.4  399.4
  119.0  14746  -52.1  -85.1      1   0.00    303     59  406.1  406.1  406.1
  117.0  14856  -51.5  -84.5      1   0.00    304     60  409.2  409.2  409.2
  111.0  15196  -53.3  -86.3      1   0.00    305     60  412.0  412.0  412.0
  110.2  15240  -53.1  -86.2      1   0.00    305     60  413.1  413.1  413.1
  108.0  15373  -52.7  -85.7      1   0.00    306     62  416.4  416.4  416.4
  105.0  15555  -53.5  -86.5      1   0.00    308     65  418.2  418.2  418.2
  100.3  15850  -52.8  -85.8      1   0.00    310     70  425.1  425.2  425.1
  100.0  15870  -52.7  -85.7      1   0.00    310     70  425.6  425.6  425.6
   99.7  15889  -52.6  -85.6      1   0.00    310     70  426.1  426.1  426.1
   93.7  16290  -51.5  -84.5      1   0.00    306     59  436.0  436.0  436.0
   91.3  16459  -52.3  -85.1      1   0.00    305     54  437.6  437.6  437.6
   83.0  17069  -55.4  -87.4      1   0.00    305     67  443.5  443.5  443.5
   82.6  17099  -55.5  -87.5      1   0.00    306     66  443.8  443.8  443.8
   79.1  17374  -54.0  -86.7      1   0.00    320     60  452.5  452.5  452.5
   78.1  17457  -53.5  -86.5      1   0.00    321     59  455.1  455.1  455.1
   70.0  18160  -53.1  -86.1      1   0.00    325     55  470.4  470.4  470.4
   68.6  18288  -53.3  -86.3      1   0.00    330     51  472.7  472.8  472.7
   65.4  18593  -53.7  -86.7      1   0.00    330     48  478.2  478.3  478.2
   64.1  18726  -53.9  -86.9      1   0.00    331     45  480.7  480.7  480.7
   60.6  19089  -51.5  -84.5      1   0.01    333     37  493.8  493.8  493.8
   56.8  19507  -54.6  -86.8      1   0.00    335     28  496.1  496.1  496.1
   55.9  19608  -55.3  -87.3      1   0.00    327     25  496.6  496.7  496.6
   54.1  19812  -55.3  -87.3      1   0.00    310     20  501.2  501.2  501.2
   51.6  20117  -55.3  -87.3      1   0.00    300     27  508.1  508.1  508.1
   50.0  20320  -55.3  -87.3      1   0.00    305     28  512.7  512.8  512.7
   48.3  20541  -55.1  -87.1      1   0.00    319     31  518.3  518.3  518.3
   46.9  20726  -54.3  -86.8      1   0.00    330     34  524.5  524.5  524.5
   45.6  20910  -53.5  -86.5      1   0.00    322     35  530.7  530.8  530.7
   42.8  21316  -56.3  -88.3      1   0.00    306     36  533.5  533.6  533.5
   42.7  21336  -56.3  -88.3      1   0.00    305     36  534.1  534.1  534.1
   40.7  21641  -56.1  -88.1      1   0.00    345     38  541.8  541.9  541.9
   40.4  21683  -56.1  -88.1      1   0.00    345     38  542.9  543.0  542.9
   37.0  22250  -54.0  -86.7      1   0.01    340     40  562.2  562.3  562.2
   35.6  22493  -53.1  -86.1      1   0.01    352     36  570.7  570.7  570.7
   33.6  22860  -54.7  -86.8      1   0.01     10     30  575.8  575.9  575.8
   33.4  22903  -54.9  -86.9      1   0.01      8     29  576.4  576.5  576.4
   30.6  23470  -55.1  -87.1      1   0.01    345     17  590.8  590.8  590.8
   30.0  23590  -55.1  -87.1      1   0.01    335     23  593.8  593.9  593.8
   29.1  23774  -54.6  -87.2      1   0.01    335     27  600.1  600.1  600.1
   28.6  23896  -54.3  -87.3      1   0.01    340     28  604.2  604.3  604.2
   26.5  24384  -56.4  -88.5      1   0.01      0     33  611.7  611.7  611.7
   26.2  24454  -56.7  -88.7      1   0.01      3     33  612.7  612.8  612.7
   24.4  24908  -54.1  -87.1      1   0.01     21     36  632.8  632.9  632.8
   24.1  24994  -53.9  -86.9      1   0.01     25     36  635.9  636.0  635.9
   23.0  25298  -53.1  -86.1      1   0.01     25     44  646.8  646.9  646.8
   21.1  25844  -51.7  -84.7      1   0.01     25     42  666.9  667.0  666.9
   20.0  26190  -53.3  -86.3      1   0.01     25     40  672.3  672.4  672.3
   19.1  26486  -53.5  -86.5      1   0.01     24     40  680.6  680.7  680.6
   17.9  26904  -52.7  -85.7      1   0.01     22     39  695.8  696.0  695.8
   17.0  27239  -50.1  -84.1      1   0.02     20     39  714.5  714.7  714.5
   14.3  28346  -49.6  -83.6      1   0.03     15     37  751.4  751.7  751.4
   13.6  28698  -49.5  -83.5      1   0.03                763.6  763.9  763.6
</PRE><H3>Station information and sounding indices</H3><PRE>
                         Station identifier: TFX
                             Station number: 72776
                           Observation time: 210207/1200
                           Station latitude: 47.46
                          Station longitude: -111.39
                          Station elevation: 1134.0
                            Showalter index: 25.72
                               Lifted index: 25.56
    LIFT computed using virtual temperature: 25.58
                                SWEAT index: 79.99
                                    K index: -21.00
                         Cross totals index: 4.70
                      Vertical totals index: 8.20
                        Totals totals index: 12.90
      Convective Available Potential Energy: 0.00
             CAPE using virtual temperature: 0.00
                      Convective Inhibition: 0.00
             CINS using virtual temperature: 0.00
                     Bulk Richardson Number: 0.00
          Bulk Richardson Number using CAPV: 0.00
  Temp [K] of the Lifted Condensation Level: 244.30
Pres [hPa] of the Lifted Condensation Level: 809.12
   Equivalent potential temp [K] of the LCL: 260.82
     Mean mixed layer potential temperature: 259.56
              Mean mixed layer mixing ratio: 0.44
              1000 hPa to 500 hPa thickness: 5061.00
Precipitable water [mm] for entire sounding: 2.54
</PRE>
<H2>72776 TFX Great Falls Observations at 00Z 08 Feb 2021</H2>
<PRE>
-----------------------------------------------------------------------------
   PRES   HGHT   TEMP   DWPT   RELH   MIXR   DRCT   SKNT   THTA   THTE   THTV
    hPa     m      C      C      %    g/kg    deg   knot     K      K      K 
-----------------------------------------------------------------------------
 1000.0    220                                                               
  925.0    801                                                               
  884.0   1134  -21.7  -24.7     77   0.59     70      6  260.5  262.2  260.6
  879.0   1176  -22.7  -25.9     75   0.53     78      6  259.9  261.4  259.9
  851.0   1413  -24.9  -26.3     88   0.53    123      4  260.0  261.5  260.1
  850.0   1422  -24.9  -26.2     89   0.53    125      4  260.1  261.6  260.1
  839.0   1517  -23.5  -26.6     76   0.52    172      3  262.5  264.0  262.6
  838.2   1524  -23.5  -26.6     76   0.52    175      3  262.5  264.1  262.6
  823.0   1658  -23.9  -27.1     75   0.51    193      6  263.5  265.0  263.6
  807.0   1802  -21.7  -24.5     78   0.66    211      9  267.3  269.3  267.4
  804.0   1829  -21.8  -24.6     78   0.65    215     10  267.5  269.5  267.6
  792.0   1940  -22.1  -25.1     77   0.63    222     10  268.4  270.2  268.4
  787.0   1987  -20.7  -22.9     82   0.78    225     11  270.3  272.6  270.5
  777.0   2082  -16.5  -18.6     84   1.14    232     11  275.8  279.2  276.0
  771.7   2134  -16.4  -18.7     83   1.14    235     11  276.5  279.9  276.6
  749.0   2358  -16.1  -19.0     78   1.14    279     24  279.2  282.6  279.4
  741.1   2438  -16.6  -19.7     77   1.09    295     28  279.5  282.8  279.7
  711.5   2743  -18.5  -22.4     71   0.90    290     35  280.7  283.4  280.8
  705.0   2812  -18.9  -23.0     70   0.86    290     37  280.9  283.6  281.1
  700.0   2865  -19.5  -23.7     69   0.81    290     38  280.9  283.4  281.0
  680.0   3079  -21.3  -25.4     69   0.72    290     40  281.2  283.4  281.3
  654.9   3353  -23.6  -26.3     78   0.68    290     43  281.6  283.8  281.7
  628.1   3658  -26.2  -27.4     90   0.65    285     42  282.1  284.1  282.2
  625.0   3694  -26.5  -27.5     91   0.64    285     42  282.1  284.1  282.2
  618.0   3775  -25.7  -28.5     77   0.59    286     43  283.9  285.8  284.0
  586.0   4158  -28.5  -29.9     88   0.55    289     47  285.0  286.8  285.1
  577.1   4267  -28.6  -30.8     82   0.51    290     48  286.1  287.8  286.2
  570.0   4356  -28.7  -31.5     77   0.48    290     47  287.0  288.6  287.1
  550.0   4610  -30.7  -32.6     83   0.45    290     45  287.6  289.1  287.7
  544.0   4687  -31.1  -33.2     82   0.43    290     45  288.0  289.5  288.1
  540.0   4740  -30.9  -34.8     69   0.37    290     44  288.9  290.1  288.9
  531.0   4858  -31.9  -35.8     68   0.34    290     43  289.1  290.2  289.1
  529.6   4877  -32.0  -36.1     67   0.33    290     43  289.1  290.3  289.2
  511.0   5128  -33.7  -40.7     49   0.22    290     45  290.1  290.8  290.1
  500.0   5280  -35.1  -42.1     49   0.19    290     47  290.2  290.9  290.2
  484.0   5507  -37.1  -43.1     54   0.18    289     45  290.4  291.1  290.5
  469.0   5724  -38.7  -45.7     48   0.14    287     44  291.1  291.6  291.1
  460.0   5858  -39.7  -42.8     72   0.19    286     43  291.4  292.1  291.5
  448.0   6039  -40.9  -46.9     52   0.13    285     41  292.1  292.6  292.2
  444.2   6096  -40.9  -48.5     44   0.11    285     41  292.8  293.2  292.9
  434.0   6255  -40.9  -52.9     26   0.07    285     42  294.8  295.1  294.8
  424.8   6401  -41.8  -53.0     28   0.07    285     43  295.4  295.7  295.4
  422.0   6446  -42.1  -53.1     29   0.07    286     44  295.6  295.9  295.6
  418.0   6511  -41.5  -52.5     29   0.07    288     45  297.2  297.5  297.2
  400.0   6810  -42.5  -56.5     20   0.05    295     52  299.7  299.9  299.7
  396.0   6878  -42.5  -56.5     20   0.05    295     54  300.5  300.7  300.6
  380.0   7155  -44.3  -59.3     17   0.03    297     60  301.7  301.9  301.7
  357.0   7572  -45.1  -73.1      3   0.01    300     70  306.1  306.1  306.1
  354.4   7620  -45.4  -72.6      3   0.01    300     71  306.4  306.4  306.4
  335.0   7994  -47.5  -68.5      7   0.01    302     78  308.4  308.5  308.4
  323.0   8235  -47.7  -72.7      4   0.01    303     83  311.4  311.4  311.4
  314.0   8421  -48.7  -73.7      4   0.01    304     87  312.5  312.5  312.5
  300.0   8720  -48.9  -75.9      3   0.00    305     93  316.3  316.3  316.3
  295.0   8830  -48.9  -77.9      2   0.00    305     95  317.8  317.9  317.8
  294.6   8839  -49.0  -77.9      2   0.00    305     95  317.9  317.9  317.9
  281.1   9144  -51.0  -79.2      2   0.00    305     91  319.2  319.2  319.2
  278.0   9216  -51.5  -79.5      2   0.00    305     90  319.5  319.5  319.5
  250.0   9900  -53.7  -78.7      3   0.00    300     82  326.1  326.1  326.1
  244.0  10056  -54.3  -79.3      3   0.00    300     83  327.5  327.5  327.5
  224.0  10605  -53.1  -83.1      1   0.00    300     76  337.4  337.4  337.4
  217.0  10810  -51.7  -84.7      1   0.00    300     74  342.7  342.7  342.7
  212.0  10961  -51.9  -84.9      1   0.00    300     72  344.6  344.6  344.6
  211.6  10973  -51.9                         300     72  344.8         344.8
  211.0  10992  -51.9                         300     72  345.1         345.1
  203.0  11243  -50.7                         296     74  350.8         350.8
  200.0  11340  -50.9                         295     75  352.0         352.0
  199.0  11373  -50.9                         295     75  352.5         352.5
  196.0  11472  -50.1                         296     74  355.3         355.3
  182.0  11956  -51.9                         298     72  360.0         360.0
  179.0  12065  -50.9                         299     71  363.3         363.3
  163.0  12676  -51.7                         302     68  371.9         371.9
  160.0  12797  -50.3                         303     67  376.2         376.2
  150.0  13220  -50.5                         305     65  382.9         382.9
  144.0  13486  -50.9                         300     63  386.6         386.6
  139.0  13716  -49.7                         295     62  392.6         392.6
  138.0  13763  -49.5                         297     64  393.8         393.8
  132.6  14021  -50.1                         305     72  397.3         397.3
  110.0  15235  -52.7                         300     53  414.2         414.2
  109.9  15240  -52.7                         300     53  414.3         414.3
  104.0  15597  -52.7                         294     53  420.9         420.9
  103.0  15659  -51.9                         293     53  423.6         423.6
  101.0  15786  -52.7                         291     53  424.4         424.4
  100.0  15850  -52.3                         290     53  426.4         426.4
   91.0  16459  -52.1                         305     45  438.4         438.4
   79.0  17374  -51.8                         310     58  457.2         457.2
   70.9  18077  -51.5                         306     37  472.1         472.1
   70.0  18160  -51.9                         305     34  473.0         473.0
   68.6  18288  -52.3                         295     31  474.8         474.8
   62.4  18898  -54.3                         310     38  483.5         483.5
   59.5  19202  -55.2                         290     31  487.9         487.9
   59.4  19216  -55.3                         290     31  488.1         488.1
   54.1  19812  -54.4                         310     41  503.2         503.2
   50.0  20320  -53.7                         330     34  516.5         516.5
   47.4  20664  -53.1                         338     22  525.9         525.9
   46.9  20726  -53.4                         340     20  526.6         526.6
   44.8  21031  -54.8                         320     18  530.5         530.5
   42.7  21336  -56.2                         340     21  534.3         534.3
   41.9  21453  -56.7                         341     22  535.8         535.8
   37.0  22250  -53.7                         345     30  562.9         562.9
   36.1  22407  -53.1                         342     23  568.4         568.4
   33.9  22809  -56.3                         336      3  570.3         570.3
   33.6  22860  -55.8                         335      1  572.9         572.9
   32.9  23001  -54.5                         341      8  580.0         580.0
   31.4  23299  -55.9                         353     22  584.0         584.0
   30.6  23470  -55.3                           0     30  590.1         590.1
   30.0  23590  -54.9                           0     14  594.4         594.4
   29.8  23633  -54.5                         357     14  596.6         596.6
   29.1  23774  -54.9                         345     12  599.3         599.3
   26.5  24384  -56.7                         355     29  610.8         610.8
   24.9  24772  -57.9                           1     39  618.3         618.3
   24.0  24994  -57.4                           5     44  625.9         625.9
   23.2  25218  -56.9                         359     47  633.8         633.8
   22.5  25413  -54.3                         355     49  647.1         647.1
   22.2  25500  -51.1                         352     50  659.0         659.0
   21.9  25603  -50.6                         350     51  663.6         663.6
   21.3  25769  -49.7                         356     50  671.1         671.1
   20.0  26180  -50.7                          10     48  680.2         680.2
   19.9  26213  -50.9                          15     46  680.8         680.8
   19.6  26311  -51.3                          16     44  682.3         682.3
   17.3  27126  -49.3                          22     29  713.5         713.5
   16.5  27432  -50.5                          25     23  719.1         719.1
   15.8  27737  -51.6                          15     33  725.2         725.2
   15.8  27717  -51.7                          16     32  724.3         724.3
   13.1  28956  -48.5                         335     22  775.6         775.6
   12.6  29196  -47.9                                     786.0         786.0
</PRE><H3>Station information and sounding indices</H3><PRE>
                         Station identifier: TFX
                             Station number: 72776
                           Observation time: 210208/0000
                           Station latitude: 47.46
                          Station longitude: -111.39
                          Station elevation: 1134.0
                            Showalter index: 23.53
                               Lifted index: 22.88
    LIFT computed using virtual temperature: 22.91
                                SWEAT index: 55.01
                                    K index: -20.20
                         Cross totals index: 8.90
                      Vertical totals index: 10.20
                        Totals totals index: 19.10
      Convective Available Potential Energy: 0.00
             CAPE using virtual temperature: 0.00
                      Convective Inhibition: 0.00
             CINS using virtual temperature: 0.00
                     Bulk Richardson Number: 0.00
          Bulk Richardson Number using CAPV: 0.00
  Temp [K] of the Lifted Condensation Level: 246.44
Pres [hPa] of the Lifted Condensation Level: 819.53
   Equivalent potential temp [K] of the LCL: 262.39
     Mean mixed layer potential temperature: 260.87
              Mean mixed layer mixing ratio: 0.53
              1000 hPa to 500 hPa thickness: 5060.00
Precipitable water [mm] for entire sounding: 2.72
</PRE>
<H2>72776 TFX Great Falls Observations at 12Z 08 Feb 2021</H2>
<PRE>
-----------------------------------------------------------------------------
   PRES   HGHT   TEMP   DWPT   RELH   MIXR   DRCT   SKNT   THTA   THTE   THTV
    hPa     m      C      C      %    g/kg    deg   knot     K      K      K 
-----------------------------------------------------------------------------
 1000.0    222                                                               
  925.0    801                                                               
  884.0   1134  -24.7  -27.4     78   0.46     10      3  257.4  258.7  257.4
  880.0   1167  -24.7  -28.0     74   0.44      8      3  257.7  259.0  257.8
  850.0   1419  -26.7  -28.3     86   0.44    351      6  258.2  259.4  258.2
  833.0   1566  -24.3  -26.0     86   0.55    342      8  262.2  263.8  262.3
  821.0   1671  -24.7  -26.6     84   0.53    335      9  262.9  264.4  262.9
  806.0   1806  -22.3  -23.9     87   0.69    326     11  266.8  268.8  266.9
  778.0   2066  -22.3  -22.7     97   0.80    309     14  269.5  271.9  269.6
  774.0   2104  -21.1  -21.4     97   0.90    306     14  271.2  273.9  271.3
  765.0   2191  -15.9  -16.2     98   1.42    301     15  277.7  281.9  277.9
  758.0   2261  -15.1  -16.8     87   1.36    296     16  279.3  283.4  279.5
  740.3   2438  -16.3  -18.5     83   1.21    285     18  279.9  283.6  280.1
  710.8   2743  -18.3  -21.4     77   0.98    295     23  280.9  283.9  281.1
  700.0   2858  -19.1  -22.5     74   0.90    305     29  281.3  284.1  281.5
  639.0   3531  -23.5  -28.5     63   0.57    297     36  283.7  285.6  283.8
  628.0   3658  -24.5  -30.4     58   0.49    295     37  284.0  285.6  284.1
  613.0   3833  -25.9  -32.9     52   0.39    292     38  284.4  285.6  284.4
  593.0   4073  -28.1  -33.0     63   0.40    288     40  284.5  285.8  284.6
  577.0   4267  -29.9  -34.5     64   0.36    285     42  284.6  285.8  284.7
  545.0   4672  -33.7  -37.7     67   0.27    285     44  284.8  285.7  284.8
  529.2   4877  -35.4  -40.6     59   0.21    285     45  285.2  285.9  285.2
  519.0   5013  -36.5  -42.5     54   0.18    283     45  285.4  286.0  285.4
  500.0   5270  -38.9  -43.8     60   0.16    280     44  285.6  286.1  285.6
  492.0   5380  -39.9  -44.3     63   0.15    281     46  285.6  286.2  285.7
  465.0   5763  -42.3  -45.3     72   0.14    283     53  287.3  287.8  287.3
  451.0   5968  -43.3  -46.2     73   0.13    284     57  288.6  289.1  288.6
  442.5   6096  -43.6  -48.5     58   0.11    285     59  289.8  290.2  289.8
  438.0   6165  -43.7  -49.7     51   0.09    287     59  290.5  290.8  290.5
  432.0   6257  -44.3  -51.3     46   0.08    290     59  290.9  291.2  290.9
  425.0   6366  -44.5  -56.5     25   0.04    294     59  292.0  292.1  292.0
  422.8   6401  -44.6  -58.0     21   0.04    295     59  292.3  292.5  292.3
  419.0   6461  -44.7  -60.7     15   0.03    294     61  292.9  293.0  292.9
  406.0   6671  -45.5  -66.5      8   0.01    290     67  294.5  294.6  294.5
  400.0   6770  -45.3  -68.3      6   0.01    295     69  296.0  296.1  296.0
  388.0   6974  -44.5  -69.5      4   0.01    297     78  299.7  299.7  299.7
  352.4   7620  -43.9  -71.8      3   0.01    305    106  308.8  308.9  308.8
  351.0   7646  -43.9  -71.9      3   0.01    305    106  309.2  309.2  309.2
  301.0   8668  -48.3  -74.3      3   0.01    305    108  316.9  316.9  316.9
  300.0   8690  -48.3  -74.3      3   0.01    305    108  317.2  317.2  317.2
  293.3   8839  -49.3  -74.3      4   0.01    305    109  317.8  317.8  317.8
  289.0   8935  -50.0  -74.3      4   0.01    305    109  318.2  318.2  318.2
  287.0   8981  -50.3  -74.3      4   0.01    306    108  318.4  318.4  318.4
  279.9   9144  -50.7  -75.3      4   0.00    310    104  320.1  320.1  320.1
  257.0   9701  -51.9  -78.9      3   0.00    310     92  326.2  326.2  326.2
  250.0   9880  -51.7  -78.7      3   0.00    305     93  329.1  329.1  329.1
  244.0  10038  -50.9  -79.9      2   0.00    305     92  332.6  332.6  332.6
  235.0  10283  -51.3  -79.3      2   0.00    305     91  335.6  335.6  335.6
  225.0  10567  -49.7  -79.7      2   0.00    305     89  342.2  342.2  342.2
  211.5  10973  -49.9  -81.7      1   0.00    305     87  348.0  348.0  348.0
  210.0  11019  -49.9  -81.9      1   0.00    305     85  348.7  348.7  348.7
  205.0  11178  -48.7  -81.7      1   0.00    305     77  353.0  353.0  353.0
  200.0  11340  -49.1  -83.1      1   0.00    305     69  354.9  354.9  354.9
  192.7  11582  -50.2  -83.3      1   0.00    295     66  356.9  356.9  356.9
  192.0  11606  -50.3  -83.3      1   0.00    295     66  357.1  357.1  357.1
  190.0  11674  -49.5  -82.5      1   0.00    296     66  359.4  359.5  359.4
  182.0  11954  -50.9  -83.9      1   0.00    297     67  361.6  361.6  361.6
  181.0  11990  -49.9  -83.9      1   0.00    298     68  363.8  363.8  363.8
  174.0  12247  -49.5  -83.5      1   0.00    299     69  368.6  368.6  368.6
  169.0  12437  -50.7  -84.7      1   0.00    301     69  369.7  369.7  369.7
  161.0  12752  -50.5  -83.5      1   0.00    303     71  375.2  375.2  375.2
  155.0  12998  -51.5  -84.5      1   0.00    304     72  377.6  377.6  377.6
  152.4  13106  -51.3  -84.3      1   0.00    305     72  379.7  379.7  379.7
  150.0  13210  -51.1  -84.1      1   0.00    300     67  381.8  381.8  381.8
  148.0  13297  -50.1  -84.1      1   0.00    298     65  385.0  385.0  385.0
  138.8  13716  -51.4  -84.7      1   0.00    290     53  389.9  390.0  389.9
  135.0  13895  -51.9  -84.9      1   0.00    290     56  392.1  392.1  392.1
  127.0  14292  -50.7  -84.7      1   0.00    290     63  401.1  401.1  401.1
  118.0  14769  -51.5  -84.5      1   0.00    290     72  408.2  408.2  408.2
  117.0  14824  -50.5  -84.5      1   0.00    290     73  411.0  411.0  411.0
  115.0  14935  -50.4  -84.4      1   0.00    290     75  413.2  413.2  413.2
  112.0  15109  -50.3  -84.3      1   0.00    299     70  416.6  416.6  416.6
  109.8  15240  -49.4  -83.4      1   0.00    305     66  420.6  420.6  420.6
  108.0  15347  -48.7  -82.7      1   0.00    306     66  423.9  423.9  423.9
  103.0  15657  -50.3  -84.3      1   0.00    308     65  426.6  426.7  426.6
  100.0  15850  -50.1  -84.1      1   0.00    310     64  430.6  430.7  430.6
   95.7  16137  -51.3  -84.7      1   0.00    310     67  433.8  433.8  433.8
   95.5  16154  -51.4  -84.7      1   0.00    310     67  434.0  434.0  434.0
   92.8  16338  -52.1  -85.1      1   0.00    309     64  436.0  436.0  436.0
   70.8  18096  -51.9  -84.9      1   0.00    300     37  471.5  471.5  471.5
   70.0  18170  -52.5  -85.5      1   0.00    300     36  471.7  471.7  471.7
   68.7  18288  -52.9  -85.8      1   0.00    300     36  473.4  473.4  473.4
   65.5  18593  -53.8  -86.4      1   0.00    285     40  477.9  477.9  477.9
   59.5  19210  -55.7  -87.7      1   0.00    302     43  487.0  487.0  487.0
   56.8  19507  -54.6  -87.2      1   0.00    310     45  495.9  495.9  495.9
   54.5  19770  -53.7  -86.7      1   0.00    315     39  503.9  504.0  503.9
   50.0  20320  -56.1  -88.1      1   0.00    325     25  510.8  510.9  510.8
   49.2  20422  -56.5  -88.1      1   0.00    325     26  512.3  512.3  512.3
   47.9  20592  -57.1  -88.1      1   0.00    319     25  514.8  514.8  514.8
   46.9  20726  -56.6  -87.9      1   0.00    315     24  519.0  519.0  519.0
   44.7  21031  -55.6  -87.5      1   0.00    340     33  528.7  528.7  528.7
   44.5  21060  -55.5  -87.5      1   0.00    340     33  529.6  529.6  529.6
   42.6  21336  -56.0  -88.0      1   0.00    335     33  534.9  535.0  534.9
   42.3  21383  -56.1  -88.1      1   0.00    335     33  535.8  535.9  535.8
   40.6  21641  -54.4  -86.9      1   0.01    335     34  546.3  546.3  546.3
   39.0  21903  -52.7  -85.7      1   0.01    339     28  557.0  557.1  557.0
   38.7  21946  -53.0  -85.9      1   0.01    340     27  557.4  557.5  557.4
   36.9  22250  -54.8  -87.0      1   0.01    315     17  560.3  560.4  560.3
   34.1  22760  -57.9  -88.9      1   0.00    340     23  565.1  565.2  565.1
   33.6  22860  -57.3  -88.9      1   0.00    345     24  569.3  569.3  569.3
   33.2  22929  -56.9  -88.9      1   0.00    339     24  572.1  572.2  572.1
   32.0  23165  -58.1  -89.3      1   0.00    320     25  575.0  575.1  575.0
   31.6  23241  -58.5  -89.5      1   0.00    325     27  576.0  576.0  576.0
   30.5  23470  -56.3  -88.0      1   0.01    340     34  588.0  588.0  588.0
   30.0  23570  -55.3  -87.3      1   0.01    340     33  593.3  593.4  593.3
   29.4  23699  -53.9  -86.9      1   0.01    350     30  600.6  600.6  600.6
   27.7  24079  -54.4  -87.2      1   0.01     20     21  609.6  609.6  609.6
   26.4  24384  -54.7  -87.4      1   0.01     15     16  616.9  617.0  616.9
   25.2  24689  -55.1  -87.6      1   0.01    310     17  624.3  624.4  624.3
   22.9  25298  -55.8  -88.0      1   0.01    335     35  639.4  639.5  639.4
   21.5  25700  -56.3  -88.3      1   0.01    354     28  649.5  649.6  649.5
   20.0  26160  -55.9  -87.9      1   0.01     15     19  664.3  664.4  664.3
   19.8  26213  -55.9  -87.9      1   0.01     10     18  665.9  666.0  665.9
   18.0  26829  -55.9  -87.9      1   0.01    356     29  684.6  684.7  684.6
   17.2  27127  -53.3  -86.3      1   0.01    350     34  702.1  702.2  702.1
   17.2  27120  -53.3  -86.3      1   0.01    350     34  701.9  702.0  701.9
   16.4  27432  -53.8  -86.5      1   0.01    350     42  710.1  710.3  710.1
   14.9  28042  -54.8  -86.8      1   0.01     10     43  726.5  726.6  726.5
   14.7  28126  -54.9  -86.9      1   0.01      9     44  728.8  728.9  728.8
   13.6  28624  -53.5  -86.5      1   0.02      3     51  749.9  750.1  749.9
   12.3  29261  -49.7  -83.4      1   0.03    355     59  784.6  785.0  784.6
   11.9  29492  -48.3  -82.3      1   0.04    355     52  797.5  798.0  797.5
   11.5  29717  -49.1  -83.1      1   0.03    355     46  802.5  802.9  802.5
   11.2  29870  -48.3  -82.7      1   0.04    355     41  810.8  811.2  810.8
   10.8  30130  -46.9  -81.9      1   0.05    346     38  825.0  825.6  825.0
   10.2  30480  -46.6  -81.6      1   0.05    335     34  838.6  839.2  838.6
   10.0  30640  -46.5  -81.5      1   0.05    325     28  844.9  845.5  844.9
    9.8  30785  -46.3  -81.3      1   0.06    320     25  850.9  851.6  851.0
    9.6  30911  -46.1  -81.1      1   0.06    321     26  856.3  857.0  856.3
    8.7  31563  -48.9  -82.9      1   0.05    324     34  869.8  870.5  869.9
    8.2  31953  -47.9  -81.9      1   0.06    325     38  888.6  889.4  888.7
    7.4  32622  -53.7  -86.7      1   0.03    329     46  891.5  891.9  891.5
    7.1  32918  -52.7  -85.7      1   0.04    330     49  907.4  907.9  907.4
    7.0  32980  -52.5  -85.5      1   0.04    340     40  910.7  911.3  910.8
    6.8  33167  -52.5  -85.5      1   0.04                918.3  918.9  918.3
</PRE><H3>Station information and sounding indices</H3><PRE>
                         Station identifier: TFX
                             Station number: 72776
                           Observation time: 210208/1200
                           Station latitude: 47.46
                          Station longitude: -111.39
                          Station elevation: 1134.0
                            Showalter index: 21.48
                               Lifted index: 20.64
    LIFT computed using virtual temperature: 20.66
                                SWEAT index: 56.62
                                    K index: -19.50
                         Cross totals index: 10.60
                      Vertical totals index: 12.20
                        Totals totals index: 22.80
      Convective Available Potential Energy: 0.00
             CAPE using virtual temperature: 0.00
                      Convective Inhibition: 0.00
             CINS using virtual temperature: 0.00
                     Bulk Richardson Number: 0.00
          Bulk Richardson Number using CAPV: 0.00
  Temp [K] of the Lifted Condensation Level: 245.19
Pres [hPa] of the Lifted Condensation Level: 824.11
   Equivalent potential temp [K] of the LCL: 260.48
     Mean mixed layer potential temperature: 259.14
              Mean mixed layer mixing ratio: 0.47
              1000 hPa to 500 hPa thickness: 5048.00
Precipitable water [mm] for entire sounding: 2.56
</PRE>
<H2>72776 TFX Great Falls Observations at 00Z 09 Feb 2021</H2>
<PRE>
-----------------------------------------------------------------------------
   PRES   HGHT   TEMP   DWPT   RELH   MIXR   DRCT   SKNT   THTA   THTE   THTV
    hPa     m      C      C      %    g/kg    deg   knot     K      K      K 
-----------------------------------------------------------------------------
 1000.0    283                                                               
  925.0    859                                                               
  891.0   1134  -23.7  -27.2     73   0.46    345      7  257.8  259.2  257.9
  887.0   1167  -24.5  -28.0     73   0.43    345      7  257.3  258.6  257.4
  853.0   1450  -26.7  -29.1     80   0.41    350      7  257.9  259.1  258.0
  850.0   1475  -26.7  -29.5     77   0.39    350      7  258.2  259.3  258.2
  844.0   1526  -26.3  -30.2     70   0.37    340      6  259.1  260.2  259.2
  809.2   1829  -27.5  -30.8     73   0.36    280      3  261.0  262.1  261.1
  775.6   2134  -28.6  -31.5     77   0.36    275      5  262.9  264.0  263.0
  774.0   2149  -28.7  -31.5     77   0.36    275      6  263.0  264.1  263.1
  762.0   2261  -26.3  -30.3     69   0.41    279     15  266.8  268.0  266.9
  752.0   2357  -25.7  -29.4     71   0.45    282     23  268.4  269.8  268.5
  743.7   2438  -22.0  -24.9     77   0.69    285     29  273.3  275.4  273.4
  739.0   2484  -19.9  -22.3     81   0.87    285     29  276.1  278.8  276.2
  731.0   2565  -18.9  -21.2     82   0.97    285     30  278.1  281.0  278.2
  724.0   2637  -18.9  -21.7     79   0.94    285     30  278.8  281.7  279.0
  713.7   2743  -19.7  -22.4     79   0.89    285     31  279.1  281.9  279.3
  700.0   2887  -20.7  -23.4     79   0.83    285     33  279.5  282.1  279.7
  629.9   3658  -26.2  -27.9     85   0.61    280     54  281.9  283.8  282.0
  609.0   3904  -27.9  -29.4     87   0.55    276     61  282.6  284.4  282.7
  604.1   3962  -28.4  -29.8     88   0.54    275     63  282.6  284.4  282.7
  580.0   4253  -31.1  -31.7     94   0.47    280     63  282.8  284.3  282.9
  578.8   4267  -31.0  -32.0     91   0.46    280     63  283.1  284.6  283.1
  577.0   4289  -30.9  -32.4     87   0.44    281     63  283.5  284.9  283.5
  570.0   4376  -29.9  -40.9     34   0.19    283     64  285.6  286.3  285.7
  567.0   4414  -28.9  -47.9     14   0.09    284     65  287.2  287.6  287.2
  562.0   4478  -27.9  -53.9      6   0.04    285     65  289.1  289.3  289.1
  552.0   4607  -28.7  -54.7      6   0.04    288     67  289.7  289.8  289.7
  543.0   4724  -29.3  -43.3     25   0.15    291     68  290.3  290.9  290.4
  531.5   4877  -30.5  -43.6     27   0.15    295     70  290.6  291.2  290.7
  530.0   4897  -30.7  -43.7     27   0.15    295     70  290.7  291.2  290.7
  524.0   4978  -31.1  -44.1     27   0.15    297     69  291.1  291.7  291.2
  519.0   5046  -31.1  -58.1      5   0.03    298     68  291.9  292.1  291.9
  509.1   5182  -31.8  -54.7      9   0.04    300     66  292.7  292.8  292.7
  500.0   5310  -32.5  -51.5     13   0.07    300     70  293.4  293.6  293.4
  487.0   5495  -33.3  -47.3     23   0.11    300     70  294.6  295.0  294.6
  477.0   5640  -34.1  -56.1      9   0.04    300     70  295.4  295.5  295.4
  460.0   5892  -34.9  -57.9      8   0.03    300     70  297.4  297.6  297.4
  446.5   6096  -36.3  -57.4     10   0.04    300     70  298.2  298.4  298.2
  400.0   6850  -41.5  -55.5     20   0.05    300     67  301.0  301.2  301.0
  399.0   6867  -41.7  -55.7     20   0.05    300     67  300.9  301.1  300.9
  356.6   7620  -46.5  -62.8     14   0.02    300     69  304.3  304.4  304.3
  345.0   7843  -47.9  -64.9     12   0.02    300     69  305.3  305.4  305.3
  340.7   7925  -48.3  -66.5     10   0.01    300     69  305.8  305.9  305.8
  323.0   8277  -50.1  -73.1      5   0.01    300     78  308.1  308.1  308.1
  300.0   8760  -50.9  -73.9      5   0.01    300     91  313.5  313.5  313.5
  284.0   9113  -52.3  -74.3      5   0.01    300     93  316.4  316.5  316.4
  282.6   9144  -52.2  -74.2      5   0.01    300     94  317.0  317.0  317.0
  276.0   9297  -52.0  -74.0      5   0.01    295     98  319.6  319.6  319.6
  275.0   9320  -51.9  -73.9      5   0.01    295     98  319.9  320.0  319.9
  269.5   9449  -52.5  -74.5      5   0.01    295     98  320.8  320.9  320.8
  265.0   9558  -53.1  -75.1      5   0.01    296     97  321.6  321.6  321.6
  254.0   9829  -53.5  -75.5      5   0.01    299     93  324.9  324.9  324.9
  250.0   9930  -52.9  -75.9      5   0.01    300     92  327.3  327.3  327.3
  236.0  10304  -51.9  -79.9      2   0.00    299     92  334.2  334.2  334.2
  229.0  10499  -52.5  -81.5      2   0.00    298     92  336.2  336.2  336.2
  211.0  11032  -51.3  -82.3      1   0.00    296     91  346.0  346.1  346.0
  207.0  11156  -51.7  -82.7      1   0.00    296     91  347.3  347.3  347.3
  200.0  11380  -52.1                         295     91  350.1         350.1
  196.0  11511  -52.7                         296     91  351.2         351.2
  191.0  11678  -52.1                         298     91  354.7         354.7
  187.0  11814  -53.3                         299     91  354.9         354.9
  184.9  11887  -52.8                         300     91  356.9         356.9
  182.0  11989  -52.1                         300     89  359.7         359.7
  175.0  12243  -52.5                         300     84  363.1         363.1
  173.0  12317  -51.9                         300     83  365.2         365.2
  166.0  12584  -52.7                         300     77  368.2         368.2
  159.0  12863  -51.7                         300     72  374.5         374.5
  155.0  13028  -52.5                         300     69  375.9         375.9
  153.1  13106  -52.4                         300     67  377.3         377.3
  150.0  13240  -52.3                         295     71  379.8         379.8
  139.3  13716  -53.7                         295     80  385.5         385.5
  139.0  13731  -53.7                         295     80  385.6         385.6
  130.0  14162  -52.7                         297     72  394.9         394.9
  122.0  14571  -53.5                         300     65  400.7         400.7
  120.9  14630  -53.1                         300     64  402.4         402.4
  118.0  14786  -52.1                         300     65  407.1         407.1
  110.0  15240  -53.7                         300     67  412.4         412.4
  108.0  15356  -54.1                         300     67  413.7         413.7
  100.0  15850  -53.9                         300     66  423.3         423.3
   86.7  16770  -52.1                         294     50  444.5         444.5
   79.9  17295  -55.1                         291     41  448.9         448.9
   78.9  17374  -55.1                         290     40  450.5         450.5
   75.2  17678  -55.0                         285     46  456.8         456.8
   71.7  17983  -54.9                         305     54  463.3         463.3
   70.7  18076  -54.9                         308     54  465.3         465.3
   70.0  18140  -54.7                         310     54  467.0         467.0
   68.4  18288  -52.9                         315     49  474.1         474.1
   67.1  18412  -51.3                         314     48  480.0         480.0
   60.5  19076  -56.3                         312     40  483.3         483.3
   56.5  19507  -56.1                         310     35  493.2         493.2
   56.0  19566  -56.1                         308     36  494.6         494.6
   51.7  20070  -58.9                         292     43  499.4         499.4
   51.3  20117  -58.7                         290     44  501.0         501.0
   50.0  20280  -57.9                         305     49  506.6         506.6
   45.9  20821  -56.3                         319     43  523.0         523.0
   44.6  21003  -57.3                         324     40  524.9         524.9
   44.4  21031  -57.0                         325     40  526.2         526.2
   42.3  21336  -54.1                         320     49  540.5         540.5
   41.6  21447  -53.1                         323     45  545.8         545.8
   40.4  21636  -53.1                         327     37  550.4         550.4
   38.8  21895  -54.7                         334     27  552.8         552.8
   38.5  21946  -54.7                         335     25  553.9         553.9
   35.0  22555  -55.2                         300     21  568.0         568.0
   34.3  22682  -55.3                         306     21  571.0         571.0
   31.8  23162  -57.9                         326     23  576.5         576.5
   30.3  23470  -57.0                         340     24  587.2         587.2
   30.2  23488  -56.9                         339     23  587.8         587.8
   30.0  23530  -57.1                         335     21  588.4         588.4
   28.1  23942  -58.7                         316     22  595.0         595.0
   27.5  24079  -58.3                         310     22  599.8         599.8
   26.2  24384  -57.5                         320     23  610.5         610.5
   25.0  24689  -56.6                         345     27  621.4         621.4
   24.5  24808  -56.3                         349     26  625.8         625.8
   22.7  25298  -57.1                           5     23  637.3         637.3
   21.6  25603  -57.7                         345     32  644.6         644.6
   20.0  26090  -58.5                         345     31  656.4         656.4
   19.9  26122  -58.5                         345     32  657.3         657.3
   18.7  26518  -57.2                         350     39  673.2         673.2
   16.4  27349  -54.5                         359     23  707.6         707.6
   16.2  27432  -54.7                           0     21  709.6         709.6
   15.4  27737  -55.5                          15     21  716.7         716.7
   14.0  28346  -57.1                         350     31  731.2         731.2
   13.4  28651  -57.9                          10     18  738.6         738.6
   12.9  28872  -58.5                         348     23  744.0         744.0
   12.7  28956  -58.2                         340     25  747.7         747.7
   10.1  30426  -53.9                         330     53  815.0         815.0
   10.0  30490  -53.5                         330     54  818.8         818.8
   10.0  30480  -53.6                         330     54  818.2         818.2
    9.6  30785  -51.4                         325     57  837.4         837.4
    9.6  30753  -52.1                         326     57  833.6         833.6
    9.4  30890  -49.1                         324     57  850.1         850.1
    8.6  31474  -49.3                         316     54  871.2         871.2
    8.0  31949  -47.5                         310     52  896.5         896.5
    7.6  32309  -48.7                         305     50  905.8         905.8
    7.5  32374  -48.9                                     907.5         907.5
    7.3  32552  -47.1                                     921.9         921.9
</PRE><H3>Station information and sounding indices</H3><PRE>
                         Station identifier: TFX
                             Station number: 72776
                           Observation time: 210209/0000
                           Station latitude: 47.46
                          Station longitude: -111.39
                          Station elevation: 1134.0
                            Showalter index: 27.99
                               Lifted index: 28.01
    LIFT computed using virtual temperature: 28.02
                                SWEAT index: 83.99
                                    K index: -26.40
                         Cross totals index: 3.00
                      Vertical totals index: 5.80
                        Totals totals index: 8.80
      Convective Available Potential Energy: 0.00
             CAPE using virtual temperature: 0.00
                      Convective Inhibition: 0.00
             CINS using virtual temperature: 0.00
                     Bulk Richardson Number: 0.00
          Bulk Richardson Number using CAPV: 0.00
  Temp [K] of the Lifted Condensation Level: 243.55
Pres [hPa] of the Lifted Condensation Level: 816.23
   Equivalent potential temp [K] of the LCL: 259.29
     Mean mixed layer potential temperature: 258.12
              Mean mixed layer mixing ratio: 0.41
              1000 hPa to 500 hPa thickness: 5027.00
Precipitable water [mm] for entire sounding: 1.97
</PRE>
<H2>72776 TFX Great Falls Observations at 12Z 09 Feb 2021</H2>
<PRE>
-----------------------------------------------------------------------------
   PRES   HGHT   TEMP   DWPT   RELH   MIXR   DRCT   SKNT   THTA   THTE   THTV
    hPa     m      C      C      %    g/kg    deg   knot     K      K      K 
-----------------------------------------------------------------------------
 1000.0    281                                                               
  925.0    856                                                               
  890.0   1134  -25.5  -28.2     78   0.42    325      3  256.0  257.3  256.1
  887.0   1159  -25.7  -28.9     74   0.40    328      3  256.1  257.2  256.1
  850.0   1471  -27.5  -30.4     76   0.36      5      3  257.3  258.4  257.4
  844.0   1522  -27.7  -30.7     76   0.35    355      3  257.6  258.7  257.7
  808.7   1829  -27.4  -30.7     73   0.37    295      1  261.1  262.2  261.2
  802.0   1889  -27.3  -30.7     73   0.37    284      3  261.9  263.0  261.9
  791.0   1989  -26.3  -28.8     79   0.45    266      7  263.9  265.3  264.0
  785.0   2044  -22.9  -24.5     87   0.67    256      9  268.2  270.2  268.3
  775.5   2134  -20.6  -21.8     90   0.87    240     12  271.6  274.2  271.8
  772.0   2167  -19.7  -20.8     91   0.95    246     12  272.9  275.7  273.1
  761.0   2274  -18.3  -20.4     84   1.00    263     14  275.5  278.5  275.7
  749.0   2393  -18.1  -20.4     82   1.01    283     15  277.0  280.1  277.2
  744.5   2438  -18.4  -20.7     82   0.99    290     16  277.1  280.1  277.3
  714.5   2743  -20.6  -22.6     84   0.87    275     29  278.0  280.7  278.1
  700.0   2895  -21.7  -23.6     85   0.82    275     32  278.4  280.9  278.6
  685.5   3048  -23.0  -24.7     86   0.76    270     34  278.6  281.0  278.8
  652.0   3414  -26.1  -27.2     90   0.63    273     39  279.2  281.1  279.3
  632.0   3639  -28.1  -28.2     99   0.60    275     42  279.4  281.2  279.5
  630.4   3658  -27.4  -27.8     97   0.62    275     42  280.3  282.3  280.4
  627.0   3697  -26.1  -27.0     92   0.67    276     43  282.3  284.4  282.4
  623.0   3743  -25.7  -28.6     77   0.58    278     44  283.3  285.1  283.4
  621.0   3766  -25.5  -33.5     47   0.37    279     45  283.8  285.0  283.8
  619.0   3790  -25.5  -39.5     26   0.20    279     45  284.0  284.7  284.1
  611.0   3884  -25.7  -39.7     26   0.20    282     47  284.9  285.5  284.9
  604.5   3962  -25.5  -42.8     18   0.15    285     49  285.9  286.4  285.9
  603.0   3980  -25.5  -43.5     17   0.14    285     49  286.2  286.6  286.2
  579.6   4267  -25.8  -50.0      8   0.07    285     49  289.1  289.3  289.1
  573.0   4350  -25.9  -51.9      7   0.06    285     50  289.9  290.1  289.9
  545.0   4713  -27.1  -57.1      4   0.03    285     53  292.6  292.8  292.6
  540.0   4779  -27.7  -55.7      5   0.04    285     54  292.7  292.8  292.7
  534.0   4859  -28.3  -45.3     18   0.13    285     55  292.9  293.4  292.9
  532.7   4877  -28.3  -47.9     13   0.09    285     55  293.1  293.5  293.1
  530.0   4913  -28.3  -53.3      7   0.05    285     55  293.6  293.7  293.6
  518.0   5078  -28.9  -45.9     18   0.12    287     58  294.8  295.2  294.8
  500.0   5330  -30.7  -47.7     17   0.10    290     61  295.6  295.9  295.6
  453.0   6021  -36.1  -52.1     18   0.07    286     66  297.2  297.5  297.2
  448.1   6096  -36.3  -53.6     15   0.06    285     66  297.9  298.1  297.9
  434.0   6317  -36.9  -57.9      9   0.04    285     80  299.9  300.0  299.9
  425.0   6461  -35.9  -65.9      3   0.01    285     89  303.0  303.0  303.0
  410.2   6706  -36.5  -62.8      5   0.02    285    105  305.3  305.4  305.3
  409.0   6727  -36.5  -62.5      5   0.02    285    105  305.5  305.6  305.5
  400.0   6880  -37.7  -60.7      7   0.03    285    106  305.9  306.0  305.9
  358.9   7620  -43.2  -64.4      8   0.02    290    111  308.2  308.2  308.2
  354.0   7713  -43.9  -64.9      8   0.02    290    112  308.4  308.5  308.4
  341.0   7964  -44.9  -65.9      8   0.02    289    113  310.4  310.5  310.4
  300.0   8810  -51.7  -69.7     10   0.01    285    119  312.4  312.4  312.4
  298.6   8839  -51.8  -69.9     10   0.01    285    122  312.6  312.6  312.6
  293.0   8960  -52.4  -70.5      9   0.01    285    122  313.5  313.6  313.5
  284.7   9144  -53.2  -71.5      9   0.01    285    122  314.9  315.0  314.9
  252.0   9920  -56.7  -75.7      7   0.01    290    104  320.9  320.9  320.9
  250.0   9970  -56.5  -75.5      7   0.01    290    101  321.9  322.0  321.9
  246.6  10058  -56.4  -75.4      7   0.01    290     97  323.4  323.4  323.4
  234.0  10391  -55.9  -74.9      8   0.01    290     98  329.0  329.0  329.0
  231.0  10473  -55.1  -75.1      7   0.01    290     98  331.4  331.4  331.4
  226.0  10613  -55.7  -75.7      7   0.01    290     99  332.6  332.6  332.6
  215.0  10933  -52.9  -77.9      3   0.00    290    100  341.7  341.7  341.7
  203.8  11278  -53.3  -81.1      2   0.00    290    101  346.4  346.4  346.4
  203.0  11304  -53.3  -81.3      2   0.00    290    100  346.7  346.7  346.7
  200.0  11400  -52.3  -81.3      2   0.00    290     96  349.8  349.8  349.8
  197.0  11498  -51.7  -81.7      2   0.00    290     95  352.2  352.3  352.3
  191.0  11699  -53.1  -84.1      1   0.00    291     93  353.1  353.1  353.1
  183.0  11975  -53.3  -84.3      1   0.00    292     90  357.1  357.2  357.1
  168.0  12528  -52.7  -83.7      1   0.00    293     83  367.0  367.0  367.0
  155.0  13049  -53.9  -84.9      1   0.00    295     78  373.5  373.5  373.5
  153.6  13106  -53.9  -84.8      1   0.00    295     77  374.5  374.5  374.5
  150.0  13260  -53.7  -84.7      1   0.00    290     82  377.3  377.4  377.4
  145.0  13478  -53.5  -84.5      1   0.00    291     83  381.4  381.4  381.4
  141.0  13657  -54.9  -85.9      1   0.00    291     84  382.0  382.0  382.0
  136.0  13888  -54.1  -85.1      1   0.00    292     85  387.4  387.4  387.4
  130.0  14177  -55.5  -86.5      1   0.00    293     87  389.9  389.9  389.9
  121.1  14630  -55.1  -86.1      1   0.00    295     89  398.6  398.6  398.6
  121.0  14634  -55.1  -86.1      1   0.00    295     89  398.7  398.7  398.7
  117.0  14848  -56.5  -86.5      1   0.00    297     83  399.9  399.9  399.9
  110.0  15240  -55.1  -86.4      1   0.00    300     72  409.6  409.6  409.6
  106.0  15477  -54.3  -86.3      1   0.00    300     75  415.6  415.6  415.6
  100.0  15850  -54.9  -86.9      1   0.00    300     79  421.4  421.4  421.4
   79.8  17298  -52.7  -85.7      1   0.00    292     55  454.0  454.0  454.0
   75.2  17678  -53.5  -86.5      1   0.00    290     49  460.0  460.0  460.0
   71.1  18040  -54.3  -87.3      1   0.00    294     47  465.8  465.8  465.8
   70.0  18140  -54.9  -86.9      1   0.00    295     46  466.6  466.6  466.6
   68.4  18288  -55.9  -87.4      1   0.00    295     48  467.6  467.6  467.6
   63.6  18747  -58.9  -88.9      1   0.00    302     47  470.8  470.8  470.8
   56.4  19507  -58.6  -88.6      1   0.00    315     44  487.9  488.0  487.9
   53.5  19833  -58.5  -88.5      1   0.00    304     44  495.5  495.5  495.5
   51.9  20024  -57.1  -88.1      1   0.00    298     44  503.1  503.1  503.1
   51.1  20117  -57.3  -88.3      1   0.00    295     44  504.6  504.7  504.7
   50.0  20260  -57.7  -88.7      1   0.00    300     40  507.1  507.1  507.1
   45.3  20881  -58.3  -89.3      1   0.00    317     43  520.1  520.1  520.1
   44.4  21008  -55.7  -87.7      1   0.00    321     43  529.5  529.5  529.5
   43.0  21212  -56.3  -88.3      1   0.00    327     44  532.8  532.9  532.8
   42.2  21336  -55.1  -87.1      1   0.00    330     45  538.7  538.7  538.7
   42.0  21362  -54.9  -86.9      1   0.00    330     43  539.9  539.9  539.9
   40.2  21641  -55.2  -87.2      1   0.00    335     27  545.8  545.9  545.8
   38.3  21946  -55.6  -87.6      1   0.00    320     18  552.4  552.4  552.4
   37.0  22169  -55.9  -87.9      1   0.00    309     18  557.2  557.3  557.2
   36.5  22250  -56.3  -88.2      1   0.00    305     18  558.3  558.4  558.3
   33.2  22860  -59.0  -90.1      1   0.00    350     20  566.6  566.7  566.6
   33.1  22872  -59.1  -90.1      1   0.00    349     20  566.8  566.8  566.8
   31.2  23244  -57.7  -88.7      1   0.00    328     28  580.2  580.3  580.2
   30.2  23448  -59.1  -89.1      1   0.00    316     32  581.8  581.9  581.8
   30.1  23470  -59.0  -89.5      1   0.00    315     32  582.7  582.7  582.7
   30.0  23490  -58.9  -89.9      1   0.00    320     36  583.5  583.5  583.5
   26.0  24384  -57.4  -89.3      1   0.01    330     38  611.9  611.9  611.9
   25.8  24441  -57.3  -89.3      1   0.01    330     35  613.7  613.8  613.7
   25.3  24565  -55.1  -87.1      1   0.01    330     29  623.5  623.5  623.5
   24.8  24689  -55.8  -87.5      1   0.01    330     22  624.8  624.9  624.8
   22.5  25298  -59.5  -89.7      1   0.01      0     14  631.6  631.6  631.6
   21.4  25603  -61.3  -90.7      1   0.00    340     18  634.9  635.0  634.9
   20.9  25767  -62.3  -91.3      1   0.00    340     22  636.7  636.7  636.7
   20.0  26040  -60.3  -90.3      1   0.01    340     29  650.9  650.9  650.9
   19.5  26213  -59.1  -90.2      1   0.01    345     31  659.7  659.7  659.7
   19.0  26361  -58.1  -90.1      1   0.01    340     30  667.3  667.4  667.3
   18.5  26518  -57.9  -89.8      1   0.01    335     28  672.8  672.9  672.8
   17.7  26822  -57.4  -89.4      1   0.01    355     22  683.7  683.8  683.7
   17.2  26990  -57.1  -89.1      1   0.01    347     25  689.8  689.8  689.8
   16.8  27127  -57.5  -89.4      1   0.01    340     27  692.6  692.7  692.6
   16.0  27432  -58.5  -89.9      1   0.01    355     24  699.0  699.1  699.0
   15.3  27737  -59.5  -90.5      1   0.01      5     21  705.8  705.8  705.8
   15.3  27727  -59.5  -90.5      1   0.01      5     21  705.3  705.4  705.3
   14.6  28042  -58.3  -89.8      1   0.01    350     34  719.6  719.7  719.6
   14.0  28285  -57.3  -89.3      1   0.01      4     30  730.9  731.0  730.9
   13.2  28651  -57.5  -89.5      1   0.01     25     24  742.4  742.6  742.4
   13.1  28705  -57.5  -89.5      1   0.01     26     22  744.2  744.3  744.2
   12.6  28956  -58.8  -89.9      1   0.01     30     15  748.4  748.5  748.4
   11.9  29308  -60.5  -90.5      1   0.01      1     27  754.2  754.4  754.2
   11.6  29468  -57.7  -89.7      1   0.01    348     33  769.8  769.9  769.8
   11.4  29566  -57.7  -89.7      1   0.01    340     36  773.2  773.3  773.2
   11.0  29803  -57.7  -89.7      1   0.01    336     36  781.5  781.7  781.5
   10.7  29979  -54.7  -86.7      1   0.02    333     36  798.7  799.0  798.7
   10.2  30284  -56.3  -88.3      1   0.02    327     36  803.8  804.0  803.8
   10.0  30410  -55.5  -87.5      1   0.02    325     36  811.3  811.5  811.3
    9.9  30480  -55.6  -87.6      1   0.02    330     34  813.5  813.7  813.5
    9.8  30539  -55.7  -87.7      1   0.02                815.3  815.5  815.3
    9.6  30671  -53.7  -86.7      1   0.02                827.6  827.9  827.6
</PRE><H3>Station information and sounding indices</H3><PRE>
                         Station identifier: TFX
                             Station number: 72776
                           Observation time: 210209/1200
                           Station latitude: 47.46
                          Station longitude: -111.39
                          Station elevation: 1134.0
                            Showalter index: 30.54
                               Lifted index: 30.72
    LIFT computed using virtual temperature: 30.74
                                SWEAT index: 66.98
                                    K index: -29.10
                         Cross totals index: 0.30
                      Vertical totals index: 3.20
                        Totals totals index: 3.50
      Convective Available Potential Energy: 0.00
             CAPE using virtual temperature: 0.00
                      Convective Inhibition: 0.00
             CINS using virtual temperature: 0.00
                     Bulk Richardson Number: 0.00
          Bulk Richardson Number using CAPV: 0.00
  Temp [K] of the Lifted Condensation Level: 242.69
Pres [hPa] of the Lifted Condensation Level: 817.49
   Equivalent potential temp [K] of the LCL: 258.17
     Mean mixed layer potential temperature: 257.09
              Mean mixed layer mixing ratio: 0.37
              1000 hPa to 500 hPa thickness: 5049.00
Precipitable water [mm] for entire sounding: 1.97
</PRE>
<H2>72776 TFX Great Falls Observations at 00Z 10 Feb 2021</H2>
<PRE>
-----------------------------------------------------------------------------
   PRES   HGHT   TEMP   DWPT   RELH   MIXR   DRCT   SKNT   THTA   THTE   THTV
    hPa     m      C      C      %    g/kg    deg   knot     K      K      K 
-----------------------------------------------------------------------------
 1000.0    291                                                               
  925.0    864                                                               
  891.0   1134  -25.1  -29.6     66   0.37    355      3  256.4  257.4  256.4
  887.0   1167  -25.7  -30.2     66   0.35    354      3  256.1  257.1  256.1
  868.0   1325  -26.9  -30.3     73   0.36    349      2  256.4  257.5  256.5
  859.0   1401  -26.7  -31.7     63   0.32    347      1  257.4  258.3  257.4
  850.0   1478  -27.3  -32.3     62   0.30    345      1  257.5  258.4  257.6
  843.0   1537  -27.5  -32.4     63   0.30    335      1  257.9  258.8  258.0
  809.6   1829  -26.7  -31.7     63   0.33    285      2  261.8  262.8  261.8
  809.0   1834  -26.7  -31.7     63   0.34    286      2  261.8  262.9  261.9
  800.0   1915  -23.9  -28.9     63   0.44    295      9  265.7  267.0  265.7
  793.0   1980  -22.7  -27.7     64   0.50    302     14  267.6  269.1  267.7
  788.0   2026  -21.1  -27.1     58   0.53    308     18  269.8  271.4  269.9
  785.0   2054  -19.5  -27.5     49   0.51    311     20  271.8  273.4  271.9
  777.0   2131  -17.3  -35.3     19   0.25    320     26  275.0  275.8  275.0
  776.6   2134  -17.3  -35.3     19   0.24    320     26  275.0  275.8  275.0
  745.6   2438  -19.2  -39.1     15   0.17    310     27  276.2  276.8  276.2
  728.0   2616  -20.3  -41.3     14   0.14    304     30  276.9  277.3  276.9
  722.0   2677  -20.7  -36.7     22   0.23    302     31  277.1  277.8  277.1
  715.6   2743  -21.1  -39.4     18   0.18    300     32  277.4  278.0  277.4
  708.0   2821  -21.5  -42.5     13   0.13    298     34  277.7  278.2  277.8
  700.0   2905  -21.9  -42.9     13   0.12    295     36  278.2  278.6  278.2
  687.0   3043  -22.7  -43.7     13   0.12    292     37  278.8  279.2  278.8
  631.0   3658  -28.1  -44.0     20   0.12    280     40  279.5  279.9  279.5
  621.0   3773  -29.1  -44.1     22   0.12    280     40  279.6  280.1  279.7
  612.0   3877  -30.1  -40.1     37   0.19    280     40  279.7  280.3  279.7
  604.8   3962  -30.6  -41.1     35   0.17    280     40  280.0  280.6  280.0
  596.0   4065  -31.3  -42.3     33   0.16    280     42  280.4  280.9  280.4
  582.0   4233  -31.9  -41.9     36   0.17    280     46  281.6  282.2  281.6
  579.2   4267  -31.9  -42.7     33   0.15    280     47  282.0  282.5  282.0
  572.0   4356  -31.9  -44.9     26   0.12    280     49  283.0  283.4  283.0
  553.0   4594  -32.3  -45.3     26   0.12    280     56  285.3  285.7  285.3
  531.2   4877  -33.4  -46.4     26   0.11    280     63  287.3  287.7  287.3
  524.0   4973  -33.7  -46.7     26   0.11    280     65  288.0  288.4  288.0
  511.0   5148  -34.7  -53.7     13   0.05    280     69  288.9  289.1  288.9
  500.0   5300  -35.3  -54.3     13   0.05    280     72  289.9  290.1  289.9
  445.5   6096  -39.7  -64.4      5   0.01    280     86  294.1  294.2  294.1
  443.0   6134  -39.9  -64.9      5   0.01    280     87  294.3  294.4  294.3
  412.0   6629  -40.3  -71.3      2   0.01    284    102  300.0  300.0  300.0
  400.0   6830  -40.9  -71.9      2   0.01    285    108  301.8  301.8  301.8
  389.5   7010  -41.4  -72.0      2   0.01    280    114  303.4  303.4  303.4
  371.0   7340  -42.3  -72.3      2   0.01    280    117  306.4  306.5  306.5
  355.8   7620  -44.3  -72.8      3   0.01    280    119  307.5  307.5  307.5
  340.0   7924  -46.4  -73.2      3   0.01    280    121  308.6  308.7  308.6
  339.9   7925  -46.4  -73.2      3   0.01    280    121  308.6  308.7  308.6
  332.0   8083  -47.5  -73.5      3   0.01    281    117  309.2  309.2  309.2
  300.0   8750  -49.1  -76.1      3   0.00    285    101  316.0  316.1  316.0
  295.9   8839  -49.4  -76.4      3   0.00    285     94  316.8  316.8  316.8
  286.0   9063  -50.3  -77.3      3   0.00    281     94  318.7  318.7  318.7
  283.0   9132  -49.9  -76.9      3   0.00    280     94  320.2  320.2  320.2
  282.5   9144  -50.0  -76.9      3   0.00    280     94  320.3  320.3  320.3
  271.0   9414  -51.1  -77.1      3   0.00    280     95  322.4  322.5  322.4
  264.0   9584  -51.1  -78.1      3   0.00    280     93  324.9  324.9  324.9
  259.0   9709  -50.1  -78.1      2   0.00    280     91  328.1  328.1  328.1
  250.0   9940  -50.1  -78.1      2   0.00    280     88  331.4  331.5  331.4
  245.0  10072  -49.5  -77.5      2   0.00    280     86  334.3  334.3  334.3
  234.3  10363  -50.4  -78.4      2   0.00    280     83  337.2  337.2  337.2
  233.0  10400  -50.5  -78.5      2   0.00    280     83  337.6  337.6  337.6
  225.0  10628  -49.9  -80.9      1   0.00    279     85  341.9  341.9  341.9
  221.0  10746  -48.7  -81.7      1   0.00    278     85  345.5  345.5  345.5
  216.0  10896  -49.5  -82.5      1   0.00    277     86  346.5  346.5  346.5
  213.0  10988  -48.5  -82.5      1   0.00    277     87  349.5  349.5  349.5
  203.0  11303  -50.1  -84.1      1   0.00    275     88  351.8  351.8  351.8
  200.0  11400  -49.9  -82.9      1   0.00    275     89  353.6  353.6  353.6
  194.5  11582  -49.3  -83.1      1   0.00    275     93  357.4  357.4  357.4
  193.0  11634  -49.1  -83.1      1   0.00    275     93  358.5  358.5  358.5
  182.0  12019  -49.5  -83.5      1   0.00    279     89  363.9  363.9  363.9
  160.0  12861  -51.7  -84.7      1   0.00    286     82  373.8  373.8  373.8
  150.0  13280  -52.3  -85.3      1   0.00    290     78  379.8  379.8  379.8
  147.0  13411  -52.2  -85.2      1   0.00    295     75  382.2  382.2  382.2
  145.0  13499  -52.1  -85.1      1   0.00    294     76  383.8  383.8  383.8
  137.0  13865  -54.1  -86.1      1   0.00    291     79  386.5  386.6  386.5
  135.0  13959  -53.5  -85.5      1   0.00    291     80  389.2  389.2  389.2
  131.0  14152  -54.3  -86.3      1   0.00    289     82  391.2  391.2  391.2
  123.0  14556  -53.9  -85.9      1   0.00    286     86  399.0  399.0  399.0
  122.0  14609  -52.7  -85.7      1   0.00    285     87  402.1  402.1  402.1
  121.6  14630  -52.6  -85.6      1   0.00    285     87  402.6  402.6  402.6
  120.0  14715  -52.3  -85.3      1   0.00    286     85  404.8  404.8  404.8
  116.0  14934  -53.3  -86.3      1   0.00    287     81  406.8  406.9  406.8
  110.6  15240  -52.8  -85.8      1   0.00    290     74  413.4  413.4  413.4
  110.0  15276  -52.7  -85.7      1   0.00    290     74  414.2  414.2  414.2
  106.0  15514  -54.1  -87.1      1   0.00    292     72  415.9  415.9  415.9
  103.0  15699  -52.3  -85.3      1   0.00    294     71  422.8  422.8  422.8
  100.0  15890  -53.1  -86.1      1   0.00    295     70  424.9  424.9  424.9
   92.6  16386  -52.5  -85.5      1   0.00    296     66  435.5  435.5  435.5
   85.8  16875  -55.1  -87.1      1   0.00    297     62  439.8  439.8  439.8
   70.0  18170  -56.3  -87.3      1   0.00    300     51  463.6  463.6  463.6
   68.7  18288  -56.2  -87.5      1   0.00    300     48  466.3  466.3  466.3
   65.5  18593  -55.9  -87.9      1   0.00    300     47  473.4  473.4  473.4
   65.4  18601  -55.9  -87.9      1   0.00    300     47  473.6  473.6  473.6
   59.9  19156  -58.3  -88.3      1   0.00    300     53  480.2  480.2  480.2
   56.8  19491  -56.1  -87.1      1   0.00    300     57  492.6  492.6  492.6
   56.7  19507  -56.1  -87.1      1   0.00    300     57  492.8  492.9  492.8
   53.2  19906  -56.9  -87.9      1   0.00    305     51  500.0  500.0  500.0
   50.0  20300  -54.7  -86.7      1   0.00    310     46  514.1  514.2  514.1
   48.6  20482  -54.1  -87.1      1   0.00    311     44  519.7  519.8  519.8
   42.5  21336  -56.6  -88.3      1   0.00    315     37  533.9  534.0  533.9
   39.2  21849  -58.1  -89.1      1   0.00    319     29  542.6  542.6  542.6
   38.6  21946  -57.8  -88.9      1   0.00    320     28  545.8  545.8  545.8
   35.5  22476  -56.1  -88.1      1   0.00    355     20  563.4  563.4  563.4
   35.1  22555  -56.4  -88.2      1   0.00      0     19  564.6  564.7  564.6
   33.4  22860  -57.5  -88.7      1   0.00    330     15  569.5  569.6  569.6
   32.8  22977  -57.9  -88.9      1   0.00    324     16  571.5  571.5  571.5
   31.8  23165  -57.7  -88.7      1   0.00    315     17  576.8  576.8  576.8
   30.6  23415  -57.5  -88.5      1   0.01    327     19  584.0  584.0  584.0
   30.3  23470  -57.8  -88.8      1   0.00    330     20  584.7  584.8  584.7
   30.0  23540  -58.1  -89.1      1   0.00    325     17  585.7  585.7  585.7
   28.2  23928  -60.1  -90.1      1   0.00    325     19  590.6  590.6  590.6
   27.5  24079  -59.8  -90.0      1   0.00    325     20  595.5  595.6  595.5
   26.2  24384  -59.1  -89.7      1   0.00    340     25  605.7  605.7  605.7
   25.0  24684  -58.5  -89.5      1   0.01    345     29  615.8  615.9  615.8
   22.7  25298  -59.1  -90.1      1   0.01    355     38  631.6  631.7  631.6
   22.5  25346  -59.1  -90.1      1   0.01    356     36  632.9  632.9  632.9
   21.6  25603  -58.4  -89.4      1   0.01      0     24  642.4  642.4  642.4
   21.2  25721  -58.1  -89.1      1   0.01    350     19  646.7  646.8  646.7
   20.7  25872  -56.3  -88.3      1   0.01    338     12  656.6  656.7  656.6
   20.6  25908  -56.5  -88.3      1   0.01    335     10  657.1  657.2  657.1
   20.0  26090  -57.5  -88.5      1   0.01    340      6  659.4  659.5  659.4
   19.6  26213  -58.5  -89.1      1   0.01    345      4  660.0  660.1  660.0
   18.7  26518  -61.0  -90.7      1   0.01    315     10  661.5  661.6  661.5
   18.0  26748  -62.9  -91.9      1   0.00    330     18  662.6  662.6  662.6
   17.8  26822  -62.7  -91.7      1   0.00    335     20  665.5  665.6  665.5
   16.8  27174  -61.7  -90.7      1   0.01    335     26  679.6  679.7  679.6
   16.1  27432  -59.3  -90.1      1   0.01    335     31  695.6  695.7  695.6
   15.7  27598  -57.7  -89.7      1   0.01    340     34  706.0  706.1  706.0
   15.4  27737  -57.4  -89.4      1   0.01    345     37  711.4  711.5  711.4
   14.1  28278  -56.3  -88.3      1   0.01     21     29  732.8  732.9  732.8
   13.9  28346  -56.2  -88.2      1   0.01     25     28  735.2  735.3  735.2
   12.7  28956  -55.8  -87.7      1   0.01     15      9  757.3  757.5  757.3
   12.7  28944  -55.7  -87.7      1   0.01     15      9  757.1  757.2  757.1
   10.9  29909  -59.9  -89.9      1   0.01    350     19  775.6  775.7  775.6
   10.0  30450  -57.7  -89.7      1   0.01    335     25  803.1  803.3  803.1
    9.9  30480  -57.8  -89.7      1   0.01    335     25  804.0  804.1  804.0
    8.9  31183  -59.1  -90.1      1   0.01    335     33  824.9  825.1  824.9
    8.6  31394  -58.1  -89.8      1   0.01    335     35  836.9  837.0  836.9
    8.5  31472  -57.7  -89.7      1   0.01                841.3  841.5  841.3
</PRE><H3>Station information and sounding indices</H3><PRE>
                         Station identifier: TFX
                             Station number: 72776
                           Observation time: 210210/0000
                           Station latitude: 47.46
                          Station longitude: -111.39
                          Station elevation: 1134.0
                            Showalter index: 25.91
                               Lifted index: 26.18
    LIFT computed using virtual temperature: 26.18
                                SWEAT index: 73.99
                                    K index: -45.30
                         Cross totals index: 3.00
                      Vertical totals index: 8.00
                        Totals totals index: 11.00
      Convective Available Potential Energy: 0.00
             CAPE using virtual temperature: 0.00
                      Convective Inhibition: 0.00
             CINS using virtual temperature: 0.00
                     Bulk Richardson Number: 0.00
          Bulk Richardson Number using CAPV: 0.00
  Temp [K] of the Lifted Condensation Level: 241.12
Pres [hPa] of the Lifted Condensation Level: 798.54
   Equivalent potential temp [K] of the LCL: 258.10
     Mean mixed layer potential temperature: 257.14
              Mean mixed layer mixing ratio: 0.33
              1000 hPa to 500 hPa thickness: 5009.00
Precipitable water [mm] for entire sounding: 0.85
</PRE>
<H2>72776 TFX Great Falls Observations at 00Z 11 Feb 2021</H2>
<PRE>
-----------------------------------------------------------------------------
   PRES   HGHT   TEMP   DWPT   RELH   MIXR   DRCT   SKNT   THTA   THTE   THTV
    hPa     m      C      C      %    g/kg    deg   knot     K      K      K 
-----------------------------------------------------------------------------
 1000.0    313                                                               
  925.0    886                                                               
  894.0   1134  -25.1  -29.1     69   0.39    345      5  256.1  257.2  256.2
  887.0   1191  -25.9  -30.2     67   0.35    341      5  255.9  256.9  255.9
  864.0   1381  -28.1  -31.6     72   0.32    328      5  255.5  256.4  255.6
  850.0   1499  -28.7  -32.5     70   0.30    320      5  256.1  256.9  256.1
  825.0   1712  -30.3  -33.4     74   0.28    314      5  256.6  257.4  256.6
  811.5   1829  -31.0  -33.6     77   0.28    310      5  257.1  257.9  257.1
  809.0   1851  -31.1  -33.7     78   0.28    306      5  257.2  258.0  257.2
  789.0   2029  -30.9  -33.9     75   0.28    274      4  259.2  260.1  259.3
  780.0   2110  -28.9  -31.9     75   0.34    259      4  262.2  263.2  262.3
  777.4   2134  -28.8  -31.8     75   0.34    255      4  262.6  263.6  262.6
  769.0   2212  -28.5  -31.6     75   0.36    255      7  263.7  264.8  263.8
  767.0   2231  -27.1  -29.7     79   0.43    255      7  265.4  266.7  265.5
  761.0   2288  -20.1  -20.9     93   0.96    255      9  273.6  276.4  273.8
  758.0   2317  -19.1  -21.1     84   0.94    255     10  275.0  277.8  275.1
  751.0   2386  -19.7  -22.5     78   0.84    255     12  275.1  277.6  275.2
  745.8   2438  -19.9  -22.7     78   0.83    255     14  275.4  277.9  275.5
  716.0   2740  -21.1  -23.9     78   0.78    275     16  277.3  279.7  277.4
  715.7   2743  -21.1  -23.9     78   0.78    275     16  277.3  279.7  277.4
  700.0   2906  -22.5  -25.2     79   0.71    275     15  277.5  279.7  277.7
  686.5   3048  -23.6  -25.8     82   0.69    265     21  277.9  280.1  278.0
  662.0   3313  -25.5  -26.8     89   0.65    269     25  278.6  280.6  278.7
  643.0   3524  -27.1  -30.0     76   0.50    273     29  279.1  280.7  279.2
  631.0   3658  -28.3  -30.9     78   0.46    275     31  279.2  280.7  279.3
  599.0   4030  -31.7  -33.4     85   0.38    275     34  279.5  280.8  279.6
  579.2   4267  -33.6  -36.6     74   0.29    275     36  280.0  281.0  280.1
  578.0   4281  -33.7  -36.8     74   0.28    275     36  280.1  281.0  280.1
  564.0   4453  -35.1  -37.3     80   0.28    278     41  280.4  281.3  280.4
  561.0   4490  -34.9  -36.4     86   0.30    279     41  281.0  282.0  281.1
  559.0   4515  -34.9  -37.0     81   0.29    279     42  281.3  282.3  281.4
  555.0   4565  -34.1  -49.1     20   0.08    280     43  282.8  283.1  282.9
  552.0   4603  -33.9  -58.9      6   0.02    281     44  283.5  283.6  283.5
  530.9   4877  -32.2  -62.1      3   0.02    285     51  288.7  288.8  288.7
  522.0   4996  -31.5  -63.5      3   0.01    287     55  291.0  291.0  291.0
  508.4   5182  -32.6  -59.7      5   0.02    290     60  291.8  291.9  291.8
  500.0   5300  -33.3  -57.3      7   0.03    290     60  292.4  292.5  292.4
  483.0   5543  -35.5  -55.5     11   0.04    290     62  292.6  292.7  292.6
  458.0   5912  -37.7  -58.7      9   0.03    290     66  294.3  294.4  294.3
  445.9   6096  -38.4  -59.4      9   0.03    290     68  295.6  295.7  295.6
  414.0   6606  -40.5  -61.5      9   0.02    290     76  299.3  299.4  299.3
  400.0   6840  -42.1  -63.1      8   0.02    290     80  300.2  300.3  300.2
  361.0   7526  -44.9  -73.9      2   0.00    294     93  305.4  305.4  305.4
  355.9   7620  -45.4  -73.2      3   0.01    295     95  305.9  305.9  305.9
  333.0   8059  -47.9  -69.9      6   0.01    295     98  308.4  308.4  308.4
  309.6   8534  -49.7  -72.4      5   0.01    295    101  312.3  312.4  312.3
  307.0   8590  -49.9  -72.7      5   0.01    295    101  312.8  312.8  312.8
  300.0   8740  -50.5  -73.5      5   0.01    295     98  314.1  314.1  314.1
  298.0   8784  -50.5  -72.5      6   0.01    295     98  314.7  314.7  314.7
  282.1   9144  -50.8  -73.4      5   0.01    295     95  319.3  319.3  319.3
  274.0   9333  -50.9  -73.9      5   0.01    295     97  321.7  321.8  321.7
  266.0   9527  -52.3  -74.3      5   0.01    295    100  322.4  322.4  322.4
  257.0   9751  -51.9  -74.9      5   0.01    298     95  326.2  326.2  326.2
  250.0   9930  -52.7  -76.7      4   0.00    300     91  327.6  327.6  327.6
  243.0  10114  -51.3  -79.3      2   0.00    300     89  332.4  332.4  332.4
  234.0  10359  -51.9  -82.9      1   0.00    300     85  335.1  335.1  335.1
  232.0  10415  -51.3  -83.3      1   0.00    300     85  336.8  336.8  336.8
  220.0  10760  -52.3  -84.3      1   0.00    300     80  340.4  340.4  340.4
  208.0  11124  -51.5  -84.5      1   0.00    300     75  347.1  347.1  347.1
  204.0  11251  -50.3  -84.3      1   0.00    300     74  351.0  351.0  351.0
  200.0  11380  -50.9  -84.9      1   0.00    300     72  352.0  352.0  352.0
  195.0  11544  -51.5  -84.5      1   0.00    298     71  353.6  353.6  353.6
  190.0  11712  -50.1  -84.1      1   0.00    297     71  358.5  358.5  358.5
  185.0  11887  -50.9  -84.9      1   0.00    295     70  360.0  360.0  360.0
  185.0  11886  -50.9  -84.9      1   0.00    295     70  359.9  359.9  359.9
  180.0  12064  -50.1  -84.1      1   0.00    294     71  364.1  364.1  364.1
  168.3  12497  -51.8  -84.9      1   0.00    290     74  368.4  368.4  368.4
  166.0  12588  -52.1  -85.1      1   0.00    291     73  369.2  369.3  369.2
  155.0  13029  -52.9  -85.9      1   0.00    294     68  375.2  375.2  375.2
  150.0  13240  -52.7  -85.7      1   0.00    295     66  379.1  379.1  379.1
  147.0  13371  -52.5  -85.5      1   0.00    297     66  381.6  381.6  381.6
  142.0  13594  -53.5  -86.5      1   0.00    300     66  383.6  383.7  383.6
  132.9  14021  -52.2  -85.2      1   0.00    305     67  393.2  393.2  393.2
  132.0  14065  -52.1  -85.1      1   0.00    304     67  394.2  394.2  394.2
  124.0  14468  -53.9  -86.9      1   0.00    294     68  398.1  398.1  398.1
  120.9  14630  -53.6  -86.6      1   0.00    290     68  401.5  401.5  401.5
  118.0  14787  -53.3  -86.3      1   0.00    291     69  404.9  404.9  404.9
  114.0  15009  -54.1  -87.1      1   0.00    293     71  407.4  407.4  407.4
  110.0  15240  -53.7  -86.7      1   0.00    295     73  412.3  412.4  412.3
  110.0  15238  -53.7  -86.7      1   0.00    295     73  412.3  412.3  412.3
  100.0  15850  -54.1  -87.1      1   0.00    305     78  422.9  422.9  422.9
   93.1  16307  -55.7  -87.7      1   0.00    305     74  428.5  428.5  428.5
   86.7  16764  -53.7  -86.5      1   0.00    305     71  441.3  441.3  441.3
   78.0  17448  -50.7  -84.7      1   0.00    298     50  461.1  461.1  461.1
   75.3  17678  -51.4  -85.1      1   0.00    295     43  464.3  464.3  464.3
   70.0  18150  -52.9  -85.9      1   0.00    300     49  470.9  470.9  470.9
   68.5  18288  -53.4  -86.3      1   0.00    295     46  472.7  472.8  472.7
   56.9  19472  -57.5  -89.5      1   0.00    290     42  489.1  489.2  489.1
   56.6  19507  -57.5  -89.5      1   0.00    290     42  490.0  490.0  490.0
   50.0  20290  -56.5  -88.5      1   0.00    315     50  509.9  509.9  509.9
   48.6  20469  -57.1  -89.1      1   0.00    317     50  512.6  512.6  512.6
   46.7  20726  -55.3  -88.0      1   0.00    320     51  522.9  522.9  522.9
   45.6  20874  -54.3  -87.3      1   0.00    321     50  528.8  528.9  528.8
   42.4  21336  -54.3  -87.3      1   0.00    325     47  539.9  539.9  539.9
   41.8  21430  -54.3  -87.3      1   0.00    319     46  542.1  542.2  542.1
   40.4  21641  -55.6  -88.2      1   0.00    305     45  544.1  544.1  544.1
   38.1  22017  -57.9  -89.9      1   0.00    315     41  547.5  547.5  547.5
   36.6  22270  -55.5  -87.5      1   0.01    322     38  560.0  560.0  560.0
   35.0  22555  -55.3  -87.3      1   0.01    330     35  567.7  567.8  567.7
   35.0  22555  -55.3  -87.3      1   0.01    330     35  567.7  567.8  567.7
   31.8  23165  -56.5  -88.5      1   0.00    320     26  580.3  580.4  580.3
   30.8  23364  -56.9  -88.9      1   0.00    307     22  584.5  584.6  584.5
   30.3  23470  -57.8  -89.2      1   0.00    300     20  584.9  585.0  584.9
   30.0  23530  -58.3  -89.3      1   0.00    300     20  585.1  585.2  585.1
   28.9  23774  -59.9  -90.4      1   0.00    305     22  587.4  587.5  587.4
   27.4  24097  -61.9  -91.9      1   0.00    329     24  590.4  590.4  590.4
   26.2  24384  -61.6  -91.6      1   0.00    350     25  599.1  599.1  599.1
   24.9  24691  -61.3  -91.3      1   0.00    340     30  608.5  608.5  608.5
   23.7  24994  -59.8  -90.6      1   0.00    330     34  621.4  621.5  621.4
   22.6  25298  -58.2  -89.8      1   0.01    335     32  634.6  634.7  634.6
   22.1  25440  -57.5  -89.5      1   0.01    349     30  640.9  641.0  640.9
   21.5  25603  -57.8  -89.5      1   0.01      5     28  644.7  644.8  644.7
   20.0  26070  -58.7  -89.7      1   0.01      0     29  655.8  655.8  655.8
   19.6  26213  -57.6  -89.1      1   0.01      0     30  663.5  663.5  663.5
   19.2  26327  -56.7  -88.7      1   0.01    356     24  669.6  669.7  669.6
   18.6  26518  -58.0  -89.5      1   0.01    350     15  671.3  671.4  671.3
   17.3  26980  -61.3  -91.3      1   0.01    346     26  675.2  675.3  675.2
   16.9  27127  -60.3  -90.7      1   0.01    345     29  683.1  683.1  683.1
   16.3  27351  -58.7  -89.7      1   0.01    360     39  695.2  695.3  695.2
   16.1  27432  -58.7  -89.7      1   0.01      5     43  697.8  697.9  697.8
   15.3  27737  -58.7  -89.7      1   0.01     20     35  707.5  707.6  707.5
   14.6  28043  -58.7  -89.7      1   0.01     15     27  717.5  717.6  717.5
   14.0  28307  -56.1  -88.1      1   0.01     28     22  734.9  735.1  734.9
   13.9  28346  -56.2  -88.2      1   0.01     30     21  735.7  735.8  735.7
   13.3  28651  -57.5  -89.0      1   0.01     10     14  741.8  741.9  741.8
   12.5  29022  -58.9  -89.9      1   0.01    340     17  749.3  749.4  749.3
   12.0  29261  -57.6  -89.2      1   0.01    320     19  762.1  762.2  762.1
   11.8  29385  -56.9  -88.9      1   0.01    302     18  768.9  769.0  768.9
   11.5  29566  -57.9  -89.5      1   0.01    275     16  771.8  772.0  771.8
   10.9  29870  -59.5  -90.4      1   0.01    285     14  776.8  776.9  776.8
   10.4  30175  -61.1  -91.3      1   0.01    325     15  781.8  781.9  781.8
   10.2  30297  -61.7  -91.7      1   0.01    315     14  783.8  783.9  783.8
   10.0  30420  -59.7  -90.7      1   0.01    305     12  795.6  795.8  795.7
    9.9  30480  -59.3  -90.5      1   0.01    275     14  799.4  799.5  799.4
    9.6  30676  -57.9  -89.9      1   0.01    298     11  811.8  811.9  811.8
    9.4  30785  -57.9  -89.9      1   0.01    310     10  815.7  815.8  815.7
    9.0  31090  -58.0  -90.0      1   0.01    305     25  826.6  826.8  826.7
    8.7  31296  -58.1  -90.1      1   0.01                834.2  834.3  834.2
</PRE><H3>Station information and sounding indices</H3><PRE>
                         Station identifier: TFX
                             Station number: 72776
                           Observation time: 210211/0000
                           Station latitude: 47.46
                          Station longitude: -111.39
                          Station elevation: 1134.0
                            Showalter index: 29.12
                               Lifted index: 29.22
    LIFT computed using virtual temperature: 29.23
                                SWEAT index: 69.99
                                    K index: -30.60
                         Cross totals index: 0.80
                      Vertical totals index: 4.60
                        Totals totals index: 5.40
      Convective Available Potential Energy: 0.00
             CAPE using virtual temperature: 0.00
                      Convective Inhibition: 0.00
             CINS using virtual temperature: 0.00
                     Bulk Richardson Number: 0.00
          Bulk Richardson Number using CAPV: 0.00
  Temp [K] of the Lifted Condensation Level: 240.96
Pres [hPa] of the Lifted Condensation Level: 810.40
   Equivalent potential temp [K] of the LCL: 256.81
     Mean mixed layer potential temperature: 255.89
              Mean mixed layer mixing ratio: 0.32
              1000 hPa to 500 hPa thickness: 4987.00
Precipitable water [mm] for entire sounding: 1.71
</PRE>
<H2>72776 TFX Great Falls Observations at 12Z 11 Feb 2021</H2>
<PRE>
-----------------------------------------------------------------------------
   PRES   HGHT   TEMP   DWPT   RELH   MIXR   DRCT   SKNT   THTA   THTE   THTV
    hPa     m      C      C      %    g/kg    deg   knot     K      K      K 
-----------------------------------------------------------------------------
 1000.0    367                                                               
  925.0    932                                                               
  899.0   1134  -31.9  -35.1     73   0.22    330      5  248.7  249.3  248.7
  895.0   1166  -30.7  -34.1     72   0.24    327      5  250.3  251.0  250.3
  850.0   1534  -31.3  -34.7     72   0.24    295      5  253.3  254.1  253.4
  845.0   1576  -31.3  -34.8     71   0.24    303      5  253.8  254.5  253.8
  828.0   1719  -32.1  -35.0     75   0.24    330      7  254.4  255.1  254.4
  816.0   1823  -31.7  -34.3     78   0.26    349      8  255.9  256.7  255.9
  815.3   1829  -31.8  -34.3     78   0.26    350      8  255.9  256.7  255.9
  780.7   2134  -33.9  -34.9     91   0.25      5     10  256.8  257.5  256.8
  775.0   2185  -34.3  -35.0     93   0.25      4     12  256.9  257.6  256.9
  761.0   2313  -33.5  -34.9     87   0.26      2     16  259.1  259.9  259.1
  747.5   2438  -34.1  -35.7     86   0.25      0     20  259.8  260.5  259.8
  739.0   2518  -34.5  -36.2     84   0.24    343     20  260.2  260.9  260.2
  732.0   2585  -32.9  -34.3     87   0.29    328     20  262.6  263.5  262.7
  725.0   2653  -32.7  -34.7     82   0.28    314     20  263.6  264.4  263.6
  719.0   2712  -27.9  -29.2     89   0.48    302     20  269.5  270.9  269.6
  716.0   2742  -26.7  -28.3     86   0.52    295     20  271.1  272.7  271.2
  715.9   2743  -26.7  -28.3     86   0.52    295     20  271.2  272.8  271.2
  705.0   2854  -24.5  -26.4     84   0.63    288     25  274.8  276.7  274.9
  700.0   2906  -24.7  -26.5     85   0.63    285     27  275.1  277.0  275.2
  667.0   3256  -26.5  -28.0     87   0.58    283     32  276.9  278.7  277.0
  648.0   3465  -28.3  -29.1     93   0.54    281     34  277.2  278.8  277.2
  636.0   3598  -29.3  -31.1     84   0.45    280     36  277.5  278.9  277.6
  630.7   3658  -29.8  -31.5     85   0.44    280     37  277.6  279.0  277.7
  610.0   3895  -31.7  -33.3     86   0.38    284     39  278.1  279.3  278.1
  605.0   3953  -31.7  -34.3     78   0.35    285     40  278.7  279.9  278.8
  598.0   4036  -31.1  -43.1     30   0.14    286     41  280.4  280.8  280.4
  591.0   4119  -30.9  -42.9     30   0.15    288     42  281.5  282.0  281.6
  578.7   4267  -31.9  -43.9     29   0.13    290     43  282.0  282.5  282.0
  541.0   4741  -35.3  -47.3     28   0.10    290     48  283.5  283.8  283.5
  532.0   4858  -35.5  -48.5     25   0.09    290     49  284.6  284.9  284.6
  530.6   4877  -35.4  -50.6     20   0.07    290     49  284.9  285.2  285.0
  526.0   4937  -35.1  -57.1      9   0.03    291     50  286.0  286.1  286.0
  514.0   5098  -35.3  -62.3      5   0.02    294     53  287.7  287.7  287.7
  507.8   5182  -35.8  -59.8      7   0.02    295     54  288.0  288.1  288.0
  500.0   5290  -36.5  -56.5     11   0.04    295     57  288.5  288.6  288.5
  446.0   6072  -40.3  -57.3     14   0.04    295     78  293.3  293.4  293.3
  444.4   6096  -40.3  -57.5     14   0.04    295     79  293.5  293.6  293.5
  411.0   6626  -41.3  -62.3      8   0.02    299    101  298.9  299.0  298.9
  400.0   6810  -40.3  -66.3      4   0.01    300    109  302.5  302.6  302.5
  386.0   7053  -41.3  -67.3      4   0.01    302    118  304.3  304.4  304.3
  376.0   7232  -40.9  -69.9      3   0.01    304    124  307.1  307.2  307.1
  371.4   7315  -41.5  -70.2      3   0.01    305    127  307.4  307.4  307.4
  354.9   7620  -43.8  -71.1      3   0.01    300    121  308.4  308.4  308.4
  339.0   7930  -46.1  -72.1      4   0.01    300    123  309.3  309.3  309.3
  300.0   8740  -48.3  -72.3      5   0.01    300    129  317.2  317.2  317.2
  295.5   8839  -48.4  -72.4      5   0.01    300    130  318.4  318.4  318.4
  295.0   8850  -48.4  -72.4      5   0.01    300    130  318.5  318.6  318.5
  292.0   8917  -48.5  -72.5      5   0.01    300    128  319.3  319.4  319.3
  282.1   9144  -49.4  -75.0      3   0.01    300    123  321.2  321.2  321.2
  273.0   9357  -50.3  -77.3      3   0.00    300    120  322.9  322.9  322.9
  252.0   9878  -50.3  -80.3      2   0.00    300    107  330.4  330.4  330.4
  250.0   9930  -50.7  -80.7      2   0.00    300    106  330.6  330.6  330.6
  239.0  10223  -51.3  -82.3      1   0.00    300    102  333.9  333.9  333.9
  230.0  10472  -50.9  -80.9      2   0.00    300     99  338.2  338.2  338.2
  218.0  10820  -52.3  -82.3      2   0.00    300     95  341.3  341.3  341.3
  210.0  11062  -50.9  -81.9      1   0.00    300     92  347.1  347.1  347.1
  200.0  11380  -50.9  -83.9      1   0.00    300     88  352.0  352.0  352.0
  195.0  11545  -50.3  -84.3      1   0.00    300     81  355.5  355.5  355.5
  193.9  11582  -50.5  -84.5      1   0.00    300     80  355.9  355.9  355.9
  188.0  11783  -51.3  -85.3      1   0.00    301     79  357.6  357.6  357.6
  175.0  12249  -50.7  -84.7      1   0.00    302     76  366.0  366.0  366.0
  162.0  12751  -51.7  -84.7      1   0.00    304     72  372.5  372.5  372.5
  156.0  12996  -51.5  -84.5      1   0.00    304     71  376.9  376.9  376.9
  151.0  13207  -52.3  -85.3      1   0.00    305     69  379.0  379.0  379.0
  150.0  13250  -52.1  -85.1      1   0.00    305     69  380.1  380.1  380.1
  139.0  13744  -51.9  -84.9      1   0.00    304     69  388.8  388.8  388.8
  129.0  14226  -54.3  -87.3      1   0.00    303     68  392.9  392.9  392.9
  125.0  14429  -53.3  -86.3      1   0.00    302     68  398.2  398.3  398.2
  110.2  15240  -55.0  -87.0      1   0.00    300     67  409.8  409.8  409.8
  109.0  15309  -55.1  -87.1      1   0.00    300     67  410.8  410.8  410.8
  100.0  15860  -55.1  -87.1      1   0.00    300     69  421.0  421.0  421.0
   96.8  16067  -55.2  -87.2      1   0.00    300     73  424.6  424.7  424.6
   95.5  16154  -55.3  -87.3      1   0.00    300     73  426.2  426.2  426.2
   82.9  17054  -55.9  -87.9      1   0.00    300     61  442.5  442.6  442.5
   76.2  17593  -52.7  -85.7      1   0.00    300     53  460.0  460.0  460.0
   71.7  17983  -53.0  -86.0      1   0.00    300     48  467.4  467.4  467.4
   70.1  18131  -53.1  -86.1      1   0.00    300     49  470.2  470.3  470.2
   70.0  18140  -53.1  -86.1      1   0.00    300     49  470.4  470.4  470.4
   68.4  18288  -53.7  -86.5      1   0.00    300     51  472.3  472.3  472.3
   65.2  18593  -54.9  -87.5      1   0.00    290     49  476.2  476.2  476.2
   58.7  19263  -57.5  -89.5      1   0.00    299     47  484.8  484.8  484.8
   52.0  20031  -55.7  -87.7      1   0.00    309     44  506.1  506.1  506.1
   51.3  20117  -56.1  -88.1      1   0.00    310     44  507.1  507.1  507.1
   50.0  20280  -56.9  -88.9      1   0.00    310     43  508.9  509.0  509.0
   46.6  20726  -57.8  -89.3      1   0.00    295     49  517.2  517.2  517.2
   43.3  21187  -58.7  -89.7      1   0.00    299     51  525.9  525.9  525.9
   42.3  21336  -58.1  -89.4      1   0.00    300     52  530.9  531.0  530.9
   40.3  21639  -56.9  -88.9      1   0.00    305     52  541.3  541.3  541.3
   38.4  21946  -58.0  -89.5      1   0.00    310     53  546.2  546.2  546.2
   36.1  22332  -59.3  -90.3      1   0.00    314     45  552.4  552.4  552.4
   34.4  22635  -57.7  -89.7      1   0.00    318     39  564.2  564.3  564.2
   33.2  22860  -58.7  -90.2      1   0.00    320     34  567.4  567.5  567.5
   30.0  23490  -61.5  -91.5      1   0.00    295     34  576.4  576.4  576.4
   28.6  23774  -62.8  -92.8      1   0.00    290     32  580.6  580.6  580.6
   28.5  23807  -62.9  -92.9      1   0.00    291     33  581.0  581.1  581.0
   26.0  24384  -61.6  -92.0      1   0.00    305     46  600.4  600.4  600.4
   24.7  24689  -60.9  -91.6      1   0.00    325     52  610.9  610.9  610.9
   23.0  25136  -59.9  -90.9      1   0.00    323     49  626.6  626.6  626.6
   22.4  25302  -56.9  -88.9      1   0.01    322     47  640.2  640.3  640.2
   21.4  25590  -58.1  -89.1      1   0.01    321     45  645.0  645.1  645.0
   20.4  25908  -55.1  -87.0      1   0.01    320     43  663.5  663.6  663.5
   20.3  25925  -54.9  -86.9      1   0.01    316     41  664.5  664.7  664.5
   20.0  26020  -55.1  -87.1      1   0.01    295     32  666.8  666.9  666.8
   19.8  26084  -54.9  -86.9      1   0.01    295     32  669.3  669.4  669.3
   17.0  27043  -61.9  -91.9      1   0.00    299     36  676.7  676.7  676.7
   16.5  27228  -60.1  -91.1      1   0.01    299     37  688.3  688.4  688.3
   16.0  27432  -61.0  -91.7      1   0.01    300     38  691.9  691.9  691.9
   15.0  27819  -62.7  -92.7      1   0.00    315     34  698.7  698.7  698.7
   14.3  28115  -60.7  -91.7      1   0.01    326     31  715.0  715.1  715.0
   13.8  28346  -60.9  -91.9      1   0.01    335     28  721.9  721.9  721.9
   13.4  28518  -61.1  -92.1      1   0.01                727.0  727.1  727.0
</PRE><H3>Station information and sounding indices</H3><PRE>
                         Station identifier: TFX
                             Station number: 72776
                           Observation time: 210211/1200
                           Station latitude: 47.46
                          Station longitude: -111.39
                          Station elevation: 1134.0
                            Showalter index: 28.28
                               Lifted index: 29.38
    LIFT computed using virtual temperature: 29.39
                                SWEAT index: 66.98
                                    K index: -31.30
                         Cross totals index: 1.80
                      Vertical totals index: 5.20
                        Totals totals index: 7.00
      Convective Available Potential Energy: 0.00
             CAPE using virtual temperature: 0.00
                      Convective Inhibition: 0.00
             CINS using virtual temperature: 0.00
                     Bulk Richardson Number: 0.00
          Bulk Richardson Number using CAPV: 0.00
  Temp [K] of the Lifted Condensation Level: 238.07
Pres [hPa] of the Lifted Condensation Level: 819.54
   Equivalent potential temp [K] of the LCL: 252.70
     Mean mixed layer potential temperature: 252.03
              Mean mixed layer mixing ratio: 0.24
              1000 hPa to 500 hPa thickness: 4923.00
Precipitable water [mm] for entire sounding: 1.23
</PRE>
<P>Description of the 
<A HREF="/upperair/columns.html">data columns</A>
or <A HREF="/upperair/indices.html">sounding indices</A>.

<P>
<FORM>
<INPUT CLASS="button" TYPE="button" VALUE=" Close this window " 
 onClick="window.close();">
<INPUT CLASS="button" TYPE="button" VALUE=" Select another map " 
 onClick="window.blur();">
</FORM>
<HR SIZE="1">
<I>Interested in graduate studies in atmospheric science?
Check out our program at the
<a href="http://www.uwyo.edu/atsc/howtoapply/"
target=_top>University of Wyoming
</a></I>
<HR SIZE="1"><FONT SIZE="-1">
Questions about the weather data provided by this site can be
addressed to <A HREF="mailto:ldoolman@uwyo.edu">
Larry Oolman (ldoolman@uwyo.edu)</A></FONT>
<HR SIZE="1">
<SCRIPT TYPE="text/javascript">
<!--
window.focus();
// -->
</SCRIPT>
</BODY>
</HTML>
        "#
    }
}
