//! Deals with the text and parsing of the upper section in a bufkit file.

use crate::bufkit_data::upper_air::UpperAir;
use std::error::Error;

/// Represents the section of a string that represents sounding data in a bufkit file.
pub struct UpperAirSection<'a> {
    raw_text: &'a str,
}

impl<'a> UpperAirSection<'a> {
    /// Create a new UpperAirSection.
    pub fn new(text: &'a str) -> UpperAirSection<'a> {
        UpperAirSection { raw_text: text }
    }

    /// Validate the upper air section
    pub fn validate_section(&self) -> Result<(), Box<dyn Error>> {
        let mut iter = self.into_iter();

        while let Some(chunk) = iter.get_next_chunk() {
            let ua = UpperAir::parse(chunk)?;
            ua.validate()?;
        }
        Ok(())
    }
}

impl<'a> IntoIterator for &'a UpperAirSection<'a> {
    type Item = UpperAir;
    type IntoIter = UpperAirIterator<'a>;

    fn into_iter(self) -> Self::IntoIter {
        UpperAirIterator {
            remaining: self.raw_text,
        }
    }
}

/// Iterator struct that parses one entry at a time.
///
/// If there is a parsing error, it skips the entry that caused it and moves on.
pub struct UpperAirIterator<'a> {
    remaining: &'a str,
}

impl<'a> UpperAirIterator<'a> {
    fn get_next_chunk(&mut self) -> Option<&'a str> {
        // Cordon off the text I want
        let next_chunk: &str;
        if let Some(start) = self.remaining.find("STID =") {
            let end = self.remaining[(start + 2)..]
                .find("STID =")
                .unwrap_or_else(|| self.remaining[(start + 2)..].len())
                + (start + 2);
            next_chunk = &self.remaining[start..end];
            self.remaining = &self.remaining[end..];
        } else {
            return None;
        }
        Some(next_chunk)
    }
}

impl<'a> Iterator for UpperAirIterator<'a> {
    type Item = UpperAir;

    fn next(&mut self) -> Option<UpperAir> {
        // Keep trying different chunks of text while you can
        while let Some(text) = self.get_next_chunk() {
            if let Ok(snd) = UpperAir::parse(text) {
                return Some(snd);
            }
        }
        // Ran out of text to try
        None
    }
}

#[cfg(test)]
mod test {
    use super::*;

    fn get_valid_test_data() -> &'static str {
        "
        SNPARM = PRES;TMPC;TMWC;DWPC;THTE;DRCT;SKNT;OMEG;HGHT
        STNPRM = SHOW;LIFT;SWET;KINX;LCLP;PWAT;TOTL;CAPE;LCLT;CINS;EQLV;LFCT;BRCH

        STID = STNM = 727730 TIME = 170401/0000
        SLAT = 46.92 SLON = -114.08 SELV = 972.0
        STIM = 0

        SHOW = 8.73 LIFT = 8.75 SWET = 35.35 KINX = 10.91
        LCLP = 790.16 PWAT = 9.96 TOTL = 38.90 CAPE = 6.66
        LCLT = 272.76 CINS = 0.00 EQLV = 768.94 LFCT = 789.03
        BRCH = 0.34

        PRES TMPC TMWC DWPC THTE DRCT SKNT OMEG
        HGHT
        906.70 10.54 6.12 1.52 305.69 270.00 2.14 -2.00
        994.01
        901.50 10.04 5.79 1.32 305.54 274.76 2.33 -2.00
        1041.87
        895.70 9.54 5.49 1.20 305.53 274.76 2.33 -2.00
        1095.48
        889.10 8.94 5.14 1.07 305.50 279.46 2.37 -1.90
        1156.78
        881.60 8.24 4.73 0.95 305.48 285.26 2.21 -1.90
        1226.84
        873.30 7.44 4.27 0.82 305.43 291.80 2.10 -1.90
        1304.87
        863.90 6.54 3.75 0.67 305.39 303.69 2.10 -1.80
        1393.86
        853.50 5.54 3.16 0.50 305.35 315.00 2.20 -1.80
        1493.12
        841.80 4.44 2.51 0.31 305.35 329.04 2.27 -1.70
        1605.81
        828.90 3.14 1.75 0.13 305.29 333.43 2.60 -1.60
        1731.35
        814.60 1.64 0.85 -0.11 305.14 338.96 2.70 -1.60
        1872.11
        798.90 0.14 -0.09 -0.41 305.13 340.35 2.89 -1.50
        2028.66
        781.60 -1.56 -1.56 -0.82 305.03 344.05 2.84 -1.50
        2203.74
        762.80 -1.96 -3.42 -5.62 302.99 357.40 4.27 -1.40
        2397.62
        742.40 -1.76 -5.28 -11.99 301.96 351.87 4.12 -1.50
        2613.23
        720.40 -3.06 -6.67 -14.23 302.25 339.44 3.32 -1.60
        2851.90
        696.80 -4.76 -8.23 -16.16 302.59 340.35 2.89 -1.70
        3114.67
        671.70 -6.26 -10.00 -19.99 302.87 10.01 3.36 -1.50
        3402.32
        645.10 -7.16 -11.58 -26.55 303.81 19.86 7.44 -1.20
        3717.63
        617.20 -7.56 -12.67 -35.59 305.94 16.93 14.01 -0.70
        4061.71
        588.20 -8.46 -13.67 -40.54 308.74 12.83 20.12 -0.40
        4435.27
        558.20 -10.56 -15.29 -40.08 310.98 7.72 23.14 -0.30
        4839.31
        527.50 -13.46 -17.38 -37.16 312.88 3.69 24.15 -0.40
        5271.78
        496.30 -17.06 -20.05 -35.14 314.28 0.43 25.84 -0.40
        5732.11
        465.00 -20.86 -22.95 -33.69 315.75 357.06 30.34 -0.30
        6216.94
        433.70 -24.66 -25.99 -33.20 317.45 352.62 37.80 -0.20
        6727.83
        402.80 -28.46 -29.37 -35.15 319.06 347.31 46.00 -0.20
        7261.50
        372.50 -32.66 -33.44 -40.22 320.07 344.46 52.21 -0.10
        7816.95
        343.00 -37.26 -37.90 -46.06 320.96 346.25 57.19 0.00
        8392.29
        314.60 -41.96 -42.43 -51.17 322.16 351.06 66.26 0.20
        8983.17
        287.40 -46.96 -47.22 -53.64 323.36 352.98 79.47 0.40
        9588.53
        261.40 -52.16 -52.28 -56.49 324.53 349.42 89.92 0.40
        10209.16
        237.00 -57.56 -57.63 -61.69 325.46 348.22 92.27 0.30
        10835.31
        213.90 -63.06 -63.10 -67.60 326.51 352.66 85.20 0.30
        11474.23
        192.40 -66.06 -66.11 -73.17 331.69 356.81 59.34 0.10
        12121.03
        172.50 -65.16 -65.22 -73.90 343.69 336.93 45.61 0.10
        12784.30
        154.10 -60.76 -9999.00 -9999.00 -9999.00 337.13 40.48 0.00
        13478.28
        137.10 -58.86 -9999.00 -9999.00 -9999.00 330.91 31.57 0.00
        14208.24
        121.60 -58.36 -9999.00 -9999.00 -9999.00 333.24 25.45 0.00
        14961.66
        107.50 -57.96 -9999.00 -9999.00 -9999.00 328.12 18.76 0.00
        15737.25
        94.70 -57.46 -9999.00 -9999.00 -9999.00 319.01 15.70 0.00
        16536.74
        83.20 -57.76 -9999.00 -9999.00 -9999.00 314.41 13.33 0.00
        17353.56
        72.90 -58.66 -9999.00 -9999.00 -9999.00 315.00 11.81 0.00
        18185.05
        63.70 -58.56 -59.08 -80.30 471.36 322.65 9.29 0.00
        19032.24
        55.50 -58.16 -58.78 -81.15 491.20 330.02 5.83 0.00
        19898.63
        48.20 -58.56 -9999.00 -9999.00 -9999.00 342.35 4.49 0.00
        20785.28
        41.80 -58.76 -59.52 -82.87 531.15 360.00 5.24 0.00
        21679.72
        36.10 -58.56 -59.45 -83.75 554.39 26.57 6.51 0.00
        22600.16
        31.20 -58.06 -59.14 -84.61 579.33 50.39 7.30 0.00
        23517.53
        26.80 -57.86 -59.12 -85.49 605.61 66.04 7.65 0.00
        24475.07
        22.90 -57.56 -59.05 -86.40 634.33 76.29 8.20 0.00
        25466.82
        19.50 -56.96 -58.78 -87.32 665.99 87.56 9.13 0.00
        26482.50
        16.50 -56.06 -58.34 -88.26 701.45 96.58 10.16 0.00
        27541.86
        13.90 -54.66 -57.63 -89.21 741.43 100.20 9.87 0.00
        28635.00
        11.60 -53.16 -57.00 -90.21 786.11 96.34 7.03 0.00
        29795.82
        9.60 -51.46 -56.44 -91.23 836.20 71.57 2.45 0.00
        31019.14
        7.80 -49.26 -55.85 -92.34 896.11 321.34 3.73 0.00
        32373.24
        6.30 -46.36 -55.20 -93.47 964.84 311.88 7.56 0.00
        33781.99
        4.90 -42.66 -54.71 -94.77 1053.58 300.78 10.63 0.00
        35463.95
        3.70 -38.06 -54.47 -96.20 1164.41 282.01 18.67 0.00
        37378.06
        2.70 -30.66 -54.11 -97.77 1314.21 274.52 34.50 0.00
        39580.40
        1.80 -20.26 -54.35 -99.75 1538.91 269.37 53.22 0.00
        42520.13
        1.00 -13.26 -13.26 -102.52 1870.72 263.80 71.91 0.00
        46931.43
        0.30 -23.16 -23.16 -107.87 2538.27 259.15 80.50 0.00
        55916.08

        STID = STNM = 727730 TIME = 170401/0300
        SLAT = 46.92 SLON = -114.08 SELV = 972.0
        STIM = 3

        SHOW = 9.67 LIFT = 9.84 SWET = 33.41 KINX = 3.88
        LCLP = 822.95 PWAT = 9.52 TOTL = 37.25 CAPE = 0.00
        LCLT = 273.49 CINS = 0.00 EQLV = -9999.00 LFCT = -9999.00
        BRCH = 0.00

        PRES TMPC TMWC DWPC THTE DRCT SKNT OMEG
        HGHT
        906.90 8.04 4.99 1.70 303.11 250.71 4.12 -2.00
        994.72
        901.80 8.24 5.10 1.76 303.96 249.15 4.37 -2.00
        1041.29
        895.90 8.24 5.06 1.67 304.53 250.71 4.12 -2.00
        1095.52
        889.30 8.04 4.87 1.48 304.88 249.78 3.92 -2.00
        1156.58
        881.90 7.54 4.50 1.22 304.93 250.56 3.50 -2.00
        1225.49
        873.50 6.84 4.04 0.97 304.89 249.44 3.32 -1.90
        1304.25
        864.20 6.04 3.49 0.64 304.78 254.05 2.84 -1.90
        1392.10
        853.70 5.14 2.90 0.36 304.75 251.57 2.45 -1.80
        1492.12
        842.00 4.14 2.24 0.05 304.74 251.57 2.45 -1.80
        1604.64
        829.10 2.94 1.39 -0.47 304.48 259.70 2.18 -1.70
        1730.03
        814.80 1.74 0.37 -1.33 304.10 275.19 2.14 -1.60
        1870.70
        799.10 0.54 -0.80 -2.58 303.62 298.30 2.87 -1.60
        2027.30
        781.80 -0.56 -2.25 -4.68 302.87 305.22 4.04 -1.50
        2202.69
        763.00 -0.86 -3.81 -8.76 302.15 306.03 5.28 -1.60
        2397.16
        742.60 -1.36 -5.31 -13.12 301.85 300.96 5.67 -1.70
        2613.25
        720.60 -2.16 -6.67 -16.97 302.14 300.96 5.67 -1.60
        2852.39
        697.00 -3.16 -8.02 -20.81 302.80 316.33 5.91 -1.50
        3116.24
        671.80 -3.96 -9.29 -26.11 303.92 338.46 7.94 -1.20
        3406.98
        645.30 -4.86 -10.45 -31.21 305.62 349.26 11.46 -0.90
        3723.68
        617.40 -6.06 -11.53 -33.19 307.92 350.17 14.78 -0.80
        4070.09
        588.30 -7.86 -12.78 -31.48 310.40 346.91 17.15 -0.70
        4446.37
        558.30 -10.26 -14.37 -29.11 312.74 345.26 19.08 -0.70
        4851.11
        527.60 -13.26 -16.43 -27.67 314.64 346.46 21.58 -0.60
        5284.03
        496.40 -16.46 -18.86 -27.77 316.36 349.62 25.87 -0.40
        5745.11
        465.10 -19.96 -21.83 -29.76 317.61 351.28 32.03 -0.20
        6231.36
        433.80 -23.66 -25.19 -33.20 318.71 350.26 39.02 0.00
        6744.11
        402.90 -27.76 -29.00 -37.48 319.60 349.17 45.49 0.10
        7279.47
        372.60 -32.26 -33.18 -41.54 320.43 349.58 51.55 0.30
        7836.01
        343.10 -37.06 -37.69 -45.57 321.24 350.32 58.92 0.40
        8411.91
        314.60 -41.96 -42.35 -49.07 322.28 349.96 66.88 0.50
        9005.04
        287.40 -47.16 -47.41 -53.64 323.08 348.69 72.30 0.40
        9610.15
        261.50 -52.46 -52.60 -57.74 324.01 347.72 73.95 0.40
        10227.58
        237.00 -57.66 -57.73 -61.69 325.31 347.50 71.83 0.30
        10855.60
        214.00 -62.46 -62.51 -67.60 327.40 346.92 65.21 0.20
        11492.35
        192.50 -65.66 -65.68 -68.36 332.33 343.88 53.19 0.10
        12140.38
        172.50 -65.46 -65.52 -73.90 343.19 331.20 46.78 0.00
        12806.97
        154.10 -62.36 -9999.00 -9999.00 -9999.00 333.80 41.36 0.00
        13497.82
        137.10 -59.76 -9999.00 -9999.00 -9999.00 333.43 30.40 0.00
        14223.50
        121.60 -59.06 -9999.00 -9999.00 -9999.00 333.86 23.37 0.00
        14974.11
        107.50 -58.66 -9999.00 -9999.00 -9999.00 326.82 18.10 0.00
        15747.18
        94.70 -58.06 -9999.00 -9999.00 -9999.00 315.95 16.49 0.00
        16544.25
        83.20 -58.06 -9999.00 -9999.00 -9999.00 312.37 15.00 0.00
        17359.37
        72.90 -58.56 -9999.00 -9999.00 -9999.00 315.62 12.78 0.00
        18190.47
        63.70 -58.56 -59.08 -80.30 471.36 322.03 10.10 0.00
        19037.85
        55.50 -58.36 -58.97 -81.15 490.74 329.30 7.23 0.00
        19903.84
        48.20 -58.66 -9999.00 -9999.00 -9999.00 351.38 6.49 0.00
        20789.87
        41.80 -58.76 -59.52 -82.87 531.15 22.62 7.58 0.00
        21684.10
        36.10 -58.56 -59.45 -83.75 554.39 50.71 8.28 0.00
        22604.55
        31.20 -58.16 -59.23 -84.61 579.06 72.80 8.55 0.00
        23521.70
        26.80 -58.06 -59.29 -85.49 605.05 83.66 8.80 0.00
        24478.57
        22.90 -57.66 -59.14 -86.40 634.03 87.56 9.13 0.00
        25469.64
        19.50 -57.06 -58.86 -87.32 665.68 91.22 9.13 0.00
        26484.84
        16.50 -56.06 -58.34 -88.26 701.45 92.60 8.55 0.00
        27543.95
        13.90 -54.76 -57.70 -89.21 741.09 87.21 7.96 0.00
        28636.84
        11.60 -53.36 -57.14 -90.21 785.40 68.46 7.94 0.00
        29796.87
        9.60 -51.66 -56.57 -91.23 835.44 45.00 8.80 0.00
        31019.09
        7.80 -49.66 -56.08 -92.34 894.51 28.14 9.48 0.00
        32371.37
        6.30 -46.86 -55.44 -93.47 962.71 11.31 7.93 0.00
        33777.30
        4.90 -43.06 -54.87 -94.77 1051.75 313.96 7.56 0.00
        35455.95
        3.70 -38.06 -54.47 -96.20 1164.41 283.45 18.38 0.00
        37368.41
        2.70 -30.26 -54.00 -97.77 1316.37 278.99 34.81 0.00
        39572.59
        1.80 -20.06 -54.31 -99.75 1540.13 275.12 54.41 0.00
        42515.89
        1.00 -13.66 -13.66 -102.52 1867.84 272.13 73.10 0.00
        46925.47
        0.30 -23.66 -23.66 -107.87 2533.19 271.24 80.63 0.00
        55894.26

        STID = STNM = 727730 TIME = 170401/0600
        SLAT = 46.92 SLON = -114.08 SELV = 972.0
        STIM = 6

        SHOW = 11.19 LIFT = 11.70 SWET = 31.69 KINX = -4.95
        LCLP = 840.66 PWAT = 9.51 TOTL = 34.49 CAPE = 0.00
        LCLT = 272.71 CINS = 0.00 EQLV = -9999.00 LFCT = -9999.00
        BRCH = 0.00

        PRES TMPC TMWC DWPC THTE DRCT SKNT OMEG
        HGHT
        907.00 5.54 3.28 0.61 299.37 242.10 3.73 -1.50
        993.61
        901.80 5.94 3.57 0.84 300.58 240.95 4.00 -1.50
        1040.68
        895.90 6.34 3.82 0.96 301.78 243.43 3.90 -1.50
        1094.49
        889.30 6.54 3.92 0.98 302.76 242.10 3.73 -1.50
        1155.17
        881.90 6.34 3.75 0.84 303.25 240.64 3.57 -1.50
        1223.75
        873.50 6.04 3.46 0.55 303.61 236.31 3.50 -1.50
        1302.22
        864.20 5.54 3.05 0.19 303.81 236.31 3.50 -1.50
        1389.86
        853.70 4.84 2.49 -0.25 303.86 236.31 3.50 -1.40
        1489.73
        842.00 3.94 1.78 -0.82 303.73 239.53 3.83 -1.40
        1602.13
        829.10 2.94 0.95 -1.52 303.55 247.17 4.00 -1.40
        1727.45
        814.80 1.84 -0.01 -2.42 303.28 257.20 4.39 -1.40
        1868.12
        799.10 0.84 -1.08 -3.71 303.04 267.80 5.05 -1.40
        2024.80
        781.90 0.04 -2.30 -5.79 302.73 273.58 6.24 -1.50
        2199.42
        763.00 -0.46 -3.74 -9.31 302.27 277.70 7.25 -1.50
        2395.25
        742.60 -0.56 -5.23 -15.01 301.93 283.09 8.57 -1.50
        2611.79
        720.60 -0.76 -6.49 -21.46 302.32 291.80 9.42 -1.40
        2851.85
        697.00 -1.26 -7.43 -26.04 303.69 303.69 10.51 -1.20
        3117.24
        671.80 -2.06 -8.19 -27.30 305.84 316.30 12.08 -1.00
        3409.99
        645.30 -3.26 -8.94 -25.62 308.44 324.09 13.91 -0.80
        3728.77
        617.40 -4.96 -9.92 -23.57 311.02 328.30 15.52 -0.70
        4077.02
        588.30 -7.06 -11.33 -22.96 313.20 333.14 16.76 -0.60
        4454.77
        558.30 -9.56 -13.20 -23.55 314.93 340.02 18.18 -0.40
        4860.78
        527.60 -12.36 -15.51 -25.44 316.30 346.48 20.78 -0.20
        5295.10
        496.40 -15.56 -18.25 -28.05 317.41 350.61 25.00 0.00
        5757.81
        465.10 -19.26 -21.34 -30.07 318.42 351.03 29.90 0.20
        6245.57
        433.80 -23.46 -24.91 -32.11 319.18 347.93 34.36 0.30
        6759.25
        402.90 -27.76 -28.79 -35.15 319.95 344.13 38.36 0.30
        7294.86
        372.60 -32.26 -33.05 -39.62 320.65 340.82 42.58 0.30
        7851.43
        343.10 -36.96 -37.53 -44.22 321.49 338.28 48.29 0.30
        8427.46
        314.60 -42.06 -42.45 -49.07 322.14 337.71 54.80 0.30
        9020.60
        287.40 -47.46 -47.72 -54.62 322.61 337.58 60.10 0.40
        9625.17
        261.50 -52.86 -53.03 -60.93 323.34 336.52 61.42 0.30
        10241.63
        237.00 -57.66 -57.75 -63.87 325.27 334.44 59.42 0.20
        10869.07
        214.00 -61.86 -61.92 -67.60 328.33 331.36 55.11 0.10
        11506.72
        192.50 -65.26 -65.29 -68.36 332.97 327.65 50.82 0.00
        12156.30
        172.50 -66.26 -66.31 -73.90 341.87 325.39 50.27 0.00
        12822.24
        154.10 -63.06 -9999.00 -9999.00 -9999.00 327.76 40.42 0.00
        13510.61
        137.10 -60.36 -9999.00 -9999.00 -9999.00 327.33 30.23 0.00
        14234.07
        121.60 -59.56 -9999.00 -9999.00 -9999.00 324.30 24.63 0.00
        14982.75
        107.50 -59.06 -9999.00 -9999.00 -9999.00 320.75 20.57 0.00
        15754.19
        94.70 -58.46 -9999.00 -9999.00 -9999.00 313.65 17.44 0.00
        16549.78
        83.20 -57.96 -9999.00 -9999.00 -9999.00 310.52 14.04 0.00
        17364.33
        72.90 -58.36 -9999.00 -9999.00 -9999.00 312.77 10.59 0.00
        18196.01
        63.70 -58.66 -59.17 -80.30 471.14 318.95 7.98 0.00
        19043.59
        55.50 -58.56 -59.15 -81.15 490.28 334.98 6.43 0.00
        19908.98
        48.20 -58.66 -9999.00 -9999.00 -9999.00 10.95 6.14 0.00
        20794.59
        41.80 -58.86 -59.61 -82.87 530.91 47.20 7.15 0.00
        21688.62
        36.10 -58.86 -59.72 -83.75 553.62 71.08 7.19 0.00
        22608.21
        31.20 -58.36 -59.40 -84.61 578.53 83.66 7.03 0.00
        23524.29
        26.80 -58.06 -59.29 -85.49 605.05 88.49 7.38 0.00
        24480.72
        22.90 -57.76 -59.22 -86.40 633.74 88.64 8.16 0.00
        25471.55
        19.50 -57.16 -58.94 -87.32 665.37 88.75 8.94 0.00
        26486.29
        16.50 -56.36 -58.57 -88.26 700.48 85.03 8.97 0.00
        27544.42
        13.90 -55.26 -58.07 -89.21 739.39 80.54 9.46 0.00
        28635.30
        11.60 -53.76 -57.41 -90.21 783.97 75.00 11.27 0.00
        29792.95
        9.60 -51.96 -56.75 -91.23 834.31 70.51 13.38 0.00
        31013.22
        7.80 -49.66 -56.08 -92.34 894.51 66.47 13.13 0.00
        32364.59
        6.30 -46.66 -55.34 -93.47 963.56 55.12 7.81 0.00
        33771.15
        4.90 -42.66 -54.71 -94.77 1053.58 319.76 6.62 0.00
        35452.00
        3.70 -37.76 -54.37 -96.20 1165.90 295.20 18.24 0.00
        37367.34
        2.70 -30.06 -53.95 -97.77 1317.46 288.32 31.51 0.00
        39573.84
        1.80 -20.06 -54.31 -99.75 1540.13 283.53 48.15 0.00
        42518.32
        1.00 -14.46 -14.46 -102.52 1862.08 281.80 62.70 0.00
        46921.02
        0.30 -24.86 -24.86 -107.87 2521.01 284.33 65.15 0.00
        55854.56

        STID = STNM = 727730 TIME = 170401/0900
        SLAT = 46.92 SLON = -114.08 SELV = 972.0
        STIM = 9

        SHOW = 11.44 LIFT = 12.26 SWET = 26.49 KINX = -1.93
        LCLP = 850.57 PWAT = 9.92 TOTL = 34.11 CAPE = 0.00
        LCLT = 272.07 CINS = 0.00 EQLV = -9999.00 LFCT = -9999.00
        BRCH = 0.00

        PRES TMPC TMWC DWPC THTE DRCT SKNT OMEG
        HGHT
        906.80 3.94 2.11 -0.20 296.97 237.99 3.67 -0.90
        994.39
        901.70 4.34 2.37 -0.08 298.05 240.95 4.00 -0.90
        1040.28
        895.80 4.84 2.68 0.06 299.36 244.65 4.08 -1.00
        1093.79
        889.20 5.24 2.89 0.08 300.54 247.17 4.00 -1.00
        1154.17
        881.80 5.34 2.89 0.00 301.40 248.75 3.75 -1.00
        1222.46
        873.40 5.74 2.97 -0.29 302.54 251.57 3.69 -1.00
        1300.74
        864.10 5.34 2.63 -0.60 302.89 249.78 3.92 -1.00
        1388.30
        853.60 4.84 2.20 -0.99 303.22 254.05 4.23 -1.00
        1488.13
        841.90 4.04 1.56 -1.51 303.25 258.23 4.76 -1.00
        1600.55
        829.00 3.14 0.77 -2.26 303.16 262.15 5.69 -1.00
        1725.93
        814.70 2.24 -0.11 -3.24 303.08 263.29 6.64 -1.10
        1866.75
        799.00 1.34 -1.12 -4.58 302.95 265.91 8.18 -1.20
        2023.69
        781.80 0.64 -2.32 -6.86 302.69 268.92 10.30 -1.30
        2198.66
        762.90 0.14 -3.72 -10.55 302.28 274.54 12.28 -1.30
        2394.91
        742.50 -0.36 -5.24 -15.71 301.89 282.88 13.95 -1.20
        2611.78
        720.50 -0.76 -6.42 -20.88 302.49 290.98 15.19 -1.00
        2851.96
        696.90 -1.16 -6.92 -21.96 304.75 297.86 15.38 -0.80
        3117.46
        671.80 -1.86 -7.16 -19.69 307.96 304.70 15.37 -0.60
        3409.31
        645.20 -2.96 -7.79 -18.70 310.79 312.37 15.00 -0.40
        3729.73
        617.30 -4.46 -8.94 -19.22 313.00 317.73 14.43 -0.30
        4078.66
        588.20 -6.46 -10.64 -21.04 314.54 319.99 14.20 -0.30
        4457.33
        558.30 -8.96 -12.72 -23.02 315.81 321.71 14.10 -0.30
        4862.96
        527.60 -12.16 -15.18 -24.18 316.91 324.89 15.19 -0.20
        5297.97
        496.40 -15.76 -17.94 -24.97 318.00 325.97 18.05 -0.10
        5760.72
        465.10 -19.56 -21.09 -26.70 318.89 324.12 22.53 0.00
        6248.09
        433.80 -23.46 -24.69 -30.35 319.55 319.35 27.14 -0.10
        6761.52
        402.90 -27.56 -28.62 -35.15 320.21 316.25 31.47 -0.20
        7297.36
        372.50 -31.96 -32.85 -40.54 320.96 318.60 34.97 -0.10
        7856.41
        343.10 -36.76 -37.45 -46.06 321.61 324.80 38.75 0.00
        8431.15
        314.60 -41.86 -42.33 -51.17 322.30 329.55 44.83 0.20
        9024.79
        287.40 -47.26 -47.53 -54.62 322.90 329.79 53.26 0.20
        9629.88
        261.50 -52.56 -52.69 -57.74 323.86 327.53 60.78 0.20
        10247.04
        237.00 -57.66 -57.73 -61.69 325.31 324.29 67.23 0.20
        10874.91
        213.90 -62.46 -62.51 -67.60 327.44 321.85 71.39 0.10
        11514.58
        192.50 -65.36 -65.39 -68.36 332.81 323.03 67.83 0.10
        12160.22
        172.50 -65.76 -65.82 -73.90 342.70 324.17 49.11 0.00
        12826.80
        154.10 -63.86 -63.95 -74.65 357.16 317.65 38.64 0.00
        13514.68
        137.10 -60.46 -9999.00 -9999.00 -9999.00 315.79 29.82 0.00
        14236.60
        121.60 -59.76 -9999.00 -9999.00 -9999.00 314.12 26.79 0.00
        14984.75
        107.50 -59.06 -9999.00 -9999.00 -9999.00 313.59 22.26 0.00
        15755.84
        94.70 -58.66 -9999.00 -9999.00 -9999.00 307.00 17.75 0.00
        16551.05
        83.20 -58.26 -9999.00 -9999.00 -9999.00 298.67 14.16 0.00
        17364.65
        72.90 -58.66 -9999.00 -9999.00 -9999.00 297.00 11.56 0.00
        18195.18
        63.70 -59.06 -59.55 -80.30 470.26 303.69 9.81 0.00
        19041.37
        55.50 -58.66 -59.25 -81.15 490.06 320.60 7.03 0.00
        19905.75
        48.20 -58.56 -9999.00 -9999.00 -9999.00 6.01 3.71 0.00
        20791.37
        41.80 -58.96 -59.70 -82.87 530.66 69.44 3.32 0.00
        21685.39
        36.10 -58.96 -59.81 -83.75 553.36 86.82 3.50 0.00
        22604.55
        31.20 -58.36 -59.40 -84.61 578.53 85.03 4.49 0.00
        23520.42
        26.80 -58.16 -59.38 -85.49 604.77 86.19 5.85 0.00
        24476.63
        22.90 -57.96 -59.39 -86.40 633.15 88.45 7.19 0.00
        25466.77
        19.50 -57.46 -59.19 -87.32 664.45 87.40 8.55 0.00
        26480.33
        16.50 -56.56 -58.73 -88.26 699.84 84.40 9.95 0.00
        27537.24
        13.90 -55.46 -58.22 -89.21 738.71 82.01 11.19 0.00
        28627.12
        11.60 -53.86 -57.48 -90.21 783.61 84.56 12.30 0.00
        29783.97
        9.60 -52.16 -56.88 -91.23 833.56 88.29 13.01 0.00
        31003.41
        7.80 -49.96 -56.25 -92.34 893.31 91.04 10.68 0.00
        32353.26
        6.30 -47.16 -55.59 -93.47 961.43 71.57 4.29 0.00
        33757.32
        4.90 -43.36 -55.00 -94.77 1050.38 329.04 6.80 0.00
        35433.76
        3.70 -38.46 -54.61 -96.20 1162.43 307.18 14.14 0.00
        37343.34
        2.70 -30.96 -54.19 -97.77 1312.58 288.73 24.20 0.00
        39542.45
        1.80 -21.46 -54.59 -99.75 1531.61 279.83 39.82 0.00
        42473.29
        1.00 -16.36 -16.36 -102.52 1848.40 277.59 52.91 0.00
        46847.60
        0.30 -26.46 -26.46 -107.87 2504.76 277.71 52.14 0.00
        55719.47

        STID = STNM = 727730 TIME = 170401/1200
        SLAT = 46.92 SLON = -114.08 SELV = 972.0
        STIM = 12

        SHOW = 11.24 LIFT = 12.58 SWET = 27.14 KINX = 3.00
        LCLP = 861.46 PWAT = 10.48 TOTL = 34.65 CAPE = 0.00
        LCLT = 271.91 CINS = 0.00 EQLV = -9999.00 LFCT = -9999.00
        BRCH = 0.00

        PRES TMPC TMWC DWPC THTE DRCT SKNT OMEG
        HGHT
        906.40 2.74 1.32 -0.54 295.43 214.70 3.07 -0.20
        993.41
        901.20 2.94 1.48 -0.41 296.31 220.24 3.30 -0.30
        1040.00
        895.40 3.04 1.56 -0.34 297.10 225.00 3.85 -0.40
        1092.32
        888.80 3.14 1.63 -0.27 297.99 230.53 4.27 -0.50
        1152.30
        881.30 3.24 1.68 -0.26 298.94 234.16 4.31 -0.50
        1221.02
        873.00 3.94 2.02 -0.33 300.57 240.95 4.00 -0.50
        1297.87
        863.60 4.64 2.27 -0.57 302.20 246.04 3.83 -0.50
        1386.02
        853.20 4.44 2.02 -0.90 302.90 249.78 3.92 -0.50
        1484.75
        841.50 3.74 1.44 -1.41 303.06 250.02 4.55 -0.50
        1597.09
        828.60 2.94 0.67 -2.23 303.01 249.68 5.59 -0.60
        1722.42
        814.30 2.14 -0.27 -3.53 302.79 250.02 6.82 -0.60
        1863.23
        798.60 1.54 -1.45 -5.88 302.31 254.41 8.66 -0.70
        2020.26
        781.40 1.24 -2.82 -9.75 301.68 263.88 10.94 -0.70
        2195.52
        762.60 0.94 -4.17 -14.62 301.32 277.54 13.33 -0.60
        2391.25
        742.20 0.64 -5.11 -18.43 302.09 289.56 15.68 -0.40
        2608.86
        720.20 0.24 -5.43 -18.31 304.42 296.28 17.54 -0.20
        2850.01
        696.60 -0.46 -5.54 -15.94 307.57 299.17 19.13 -0.10
        3116.52
        671.50 -1.36 -5.99 -15.02 310.38 300.22 20.45 0.00
        3409.26
        644.90 -2.76 -7.04 -15.50 312.38 299.36 21.39 0.10
        3730.34
        617.00 -4.66 -8.62 -16.94 313.73 296.57 21.29 0.00
        4079.51
        588.00 -6.86 -10.47 -18.72 314.96 290.75 19.74 -0.10
        4456.67
        558.00 -9.46 -12.54 -20.14 316.27 283.13 17.95 -0.20
        4863.22
        527.40 -12.66 -14.97 -21.12 317.38 279.14 17.11 -0.30
        5296.27
        496.20 -16.16 -17.79 -22.61 318.33 284.04 19.23 -0.30
        5758.48
        464.90 -19.56 -20.97 -26.02 319.13 290.40 25.08 -0.30
        6245.71
        433.60 -23.16 -24.56 -31.12 319.81 293.71 31.39 -0.40
        6759.67
        402.70 -27.16 -28.42 -36.49 320.57 295.56 34.67 -0.40
        7296.51
        372.40 -31.76 -32.70 -40.86 321.22 296.98 35.98 -0.40
        7854.61
        343.00 -36.86 -37.44 -44.23 321.65 301.27 37.04 -0.30
        8429.64
        314.50 -42.16 -42.47 -47.35 322.14 310.10 43.43 -0.20
        9022.97
        287.30 -47.26 -47.43 -51.20 323.08 315.81 57.96 0.00
        9627.89
        261.40 -52.36 -52.47 -56.49 324.23 316.82 73.52 0.10
        10245.55
        236.90 -57.76 -57.83 -61.69 325.20 317.06 80.40 0.10
        10873.83
        213.90 -62.66 -62.71 -67.60 327.13 316.17 74.05 0.00
        11510.42
        192.40 -65.46 -65.49 -68.36 332.70 312.57 64.90 0.00
        12158.76
        172.50 -65.76 -65.82 -73.90 342.70 316.07 51.52 0.00
        12822.03
        154.10 -63.36 -9999.00 -9999.00 -9999.00 304.25 38.31 0.00
        13510.73
        137.10 -60.26 -9999.00 -9999.00 -9999.00 306.68 35.12 0.00
        14233.85
        121.60 -59.36 -9999.00 -9999.00 -9999.00 303.89 30.65 0.00
        14983.05
        107.50 -59.06 -9999.00 -9999.00 -9999.00 311.15 26.57 0.00
        15754.86
        94.70 -59.26 -9999.00 -9999.00 -9999.00 306.31 20.01 0.00
        16548.96
        83.20 -58.96 -9999.00 -9999.00 -9999.00 301.30 16.82 0.00
        17360.10
        72.90 -59.06 -9999.00 -9999.00 -9999.00 301.36 14.55 0.00
        18188.50
        63.70 -59.16 -9999.00 -9999.00 -9999.00 308.21 11.62 0.00
        19033.71
        55.50 -58.56 -59.15 -81.15 490.28 317.12 7.42 0.00
        19898.08
        48.20 -58.66 -9999.00 -9999.00 -9999.00 322.43 3.19 0.00
        20783.70
        41.80 -59.26 -59.98 -82.87 529.91 338.20 2.10 0.00
        21676.89
        36.10 -59.06 -59.90 -83.75 553.10 17.35 3.26 0.00
        22595.19
        31.20 -58.56 -59.58 -84.61 577.99 49.64 5.09 0.00
        23510.42
        26.80 -58.46 -59.64 -85.49 603.93 66.37 6.78 0.00
        24465.51
        22.90 -58.16 -59.56 -86.40 632.56 75.32 8.43 0.00
        25454.50
        19.50 -57.56 -59.27 -87.32 664.14 82.48 10.39 0.00
        26467.36
        16.50 -56.66 -58.81 -88.26 699.51 85.46 12.28 0.00
        27523.78
        13.90 -55.46 -58.22 -89.21 738.71 85.73 13.05 0.00
        28613.41
        11.60 -53.96 -57.55 -90.21 783.25 87.32 12.45 0.00
        29769.99
        9.60 -52.36 -57.01 -91.23 832.80 90.00 10.68 0.00
        30988.61
        7.80 -50.36 -56.47 -92.34 891.71 86.91 7.21 0.00
        32336.63
        6.30 -47.76 -55.89 -93.47 958.88 45.00 4.39 0.00
        33737.56
        4.90 -44.06 -55.29 -94.77 1047.18 357.88 5.24 0.00
        35409.22
        3.70 -39.36 -54.92 -96.20 1157.97 299.90 8.95 0.00
        37312.23
        2.70 -32.26 -54.54 -97.77 1305.53 281.41 22.59 0.00
        39501.20
        1.80 -23.16 -54.94 -99.75 1521.26 277.48 38.79 0.00
        42414.23
        1.00 -18.06 -18.06 -102.52 1836.17 275.34 50.14 0.00
        46759.29
        0.30 -27.66 -27.66 -107.87 2492.58 270.91 49.15 0.00
        55580.06"
    }

    fn get_invalid_test_data1() -> &'static str {
        // Partial last row of first sounding
        "
        SNPARM = PRES;TMPC;TMWC;DWPC;THTE;DRCT;SKNT;OMEG;HGHT
        STNPRM = SHOW;LIFT;SWET;KINX;LCLP;PWAT;TOTL;CAPE;LCLT;CINS;EQLV;LFCT;BRCH

        STID = STNM = 727730 TIME = 170401/0000
        SLAT = 46.92 SLON = -114.08 SELV = 972.0
        STIM = 0

        SHOW = 8.73 LIFT = 8.75 SWET = 35.35 KINX = 10.91
        LCLP = 790.16 PWAT = 9.96 TOTL = 38.90 CAPE = 6.66
        LCLT = 272.76 CINS = 0.00 EQLV = 768.94 LFCT = 789.03
        BRCH = 0.34

        PRES TMPC TMWC DWPC THTE DRCT SKNT OMEG
        HGHT
        906.70 10.54 6.12 1.52 305.69 270.00 2.14 -2.00
        994.01
        901.50 10.04 5.79 1.32 305.54 274.76 2.33 -2.00
        1041.87
        895.70 9.54 5.49 1.20 305.53 274.76 2.33 -2.00
        1095.48
        889.10 8.94 5.14 1.07 305.50 279.46 2.37 -1.90
        1156.78
        881.60 8.24 4.73 0.95 305.48 285.26 2.21 -1.90
        1226.84
        873.30 7.44 4.27 0.82 305.43 291.80 2.10 -1.90
        1304.87
        863.90 6.54 3.75 0.67 305.39 303.69 2.10 -1.80
        1393.86
        853.50 5.54 3.16 0.50 305.35 315.00 2.20 -1.80
        1493.12
        841.80 4.44 2.51 0.31 305.35 329.04 2.27 -1.70
        1605.81
        828.90 3.14 1.75 0.13 305.29 333.43 2.60 -1.60
        1731.35
        814.60 1.64 0.85 -0.11 305.14 338.96 2.70 -1.60
        1872.11
        798.90 0.14 -0.09 -0.41 305.13 340.35 2.89 -1.50
        2028.66
        781.60 -1.56 -1.56 -0.82 305.03 344.05 2.84 -1.50
        2203.74
        762.80 -1.96 -3.42 -5.62 302.99 357.40 4.27 -1.40
        2397.62
        742.40 -1.76 -5.28 -11.99 301.96 351.87 4.12 -1.50
        2613.23
        720.40 -3.06 -6.67 -14.23 302.25 339.44 3.32 -1.60
        2851.90
        696.80 -4.76 -8.23 -16.16 302.59 340.35 2.89 -1.70
        3114.67
        671.70 -6.26 -10.00 -19.99 302.87 10.01 3.36 -1.50
        3402.32
        645.10 -7.16 -11.58 -26.55 303.81 19.86 7.44 -1.20
        3717.63
        617.20 -7.56 -12.67 -35.59 305.94 16.93 14.01 -0.70
        4061.71
        588.20 -8.46 -13.67 -40.54 308.74 12.83 20.12 -0.40
        4435.27
        558.20 -10.56 -15.29 -40.08 310.98 7.72 23.14 -0.30
        4839.31
        527.50 -13.46 -17.38 -37.16 312.88 3.69 24.15 -0.40
        5271.78
        496.30 -17.06 -20.05 -35.14 314.28 0.43 25.84 -0.40
        5732.11
        465.00 -20.86 -22.95 -33.69 315.75 357.06 30.34 -0.30
        6216.94
        433.70 -24.66 -25.99 -33.20 317.45 352.62 37.80 -0.20
        6727.83
        402.80 -28.46 -29.37 -35.15 319.06 347.31 46.00 -0.20
        7261.50
        372.50 -32.66 -33.44 -40.22 320.07 344.46 52.21 -0.10
        7816.95
        343.00 -37.26 -37.90 -46.06 320.96 346.25 57.19 0.00
        8392.29
        314.60 -41.96 -42.43 -51.17 322.16 351.06 66.26 0.20
        8983.17
        287.40 -46.96 -47.22 -53.64 323.36 352.98 79.47 0.40
        9588.53
        261.40 -52.16 -52.28 -56.49 324.53 349.42 89.92 0.40
        10209.16
        237.00 -57.56 -57.63 -61.69 325.46 348.22 92.27 0.30
        10835.31
        213.90 -63.06 -63.10 -67.60 326.51 352.66 85.20 0.30
        11474.23
        192.40 -66.06 -66.11 -73.17 331.69 356.81 59.34 0.10
        12121.03
        172.50 -65.16 -65.22 -73.90 343.69 336.93 45.61 0.10
        12784.30
        154.10 -60.76 -9999.00 -9999.00 -9999.00 337.13 40.48 0.00
        13478.28
        137.10 -58.86 -9999.00 -9999.00 -9999.00 330.91 31.57 0.00
        14208.24
        121.60 -58.36 -9999.00 -9999.00 -9999.00 333.24 25.45 0.00
        14961.66
        107.50 -57.96 -9999.00 -9999.00 -9999.00 328.12 18.76 0.00
        15737.25
        94.70 -57.46 -9999.00 -9999.00 -9999.00 319.01 15.70 0.00
        16536.74
        83.20 -57.76 -9999.00 -9999.00 -9999.00 314.41 13.33 0.00
        17353.56
        72.90 -58.66 -9999.00 -9999.00 -9999.00 315.00 11.81 0.00
        18185.05
        63.70 -58.56 -59.08 -80.30 471.36 322.65 9.29 0.00
        19032.24
        55.50 -58.16 -58.78 -81.15 491.20 330.02 5.83 0.00
        19898.63
        48.20 -58.56 -9999.00 -9999.00 -9999.00 342.35 4.49 0.00
        20785.28
        41.80 -58.76 -59.52 -82.87 531.15 360.00 5.24 0.00
        21679.72
        36.10 -58.56 -59.45 -83.75 554.39 26.57 6.51 0.00
        22600.16
        31.20 -58.06 -59.14 -84.61 579.33 50.39 7.30 0.00
        23517.53
        26.80 -57.86 -59.12 -85.49 605.61 66.04 7.65 0.00
        24475.07
        22.90 -57.56 -59.05 -86.40 634.33 76.29 8.20 0.00
        25466.82
        19.50 -56.96 -58.78 -87.32 665.99 87.56 9.13 0.00
        26482.50
        16.50 -56.06 -58.34 -88.26 701.45 96.58 10.16 0.00
        27541.86
        13.90 -54.66 -57.63 -89.21 741.43 100.20 9.87 0.00
        28635.00
        11.60 -53.16 -57.00 -90.21 786.11 96.34 7.03 0.00
        29795.82
        9.60 -51.46 -56.44 -91.23 836.20 71.57 2.45 0.00
        31019.14
        7.80 -49.26 -55.85 -92.34 896.11 321.34 3.73 0.00
        32373.24
        6.30 -46.36 -55.20 -93.47 964.84 311.88 7.56 0.00
        33781.99
        4.90 -42.66 -54.71 -94.77 1053.58 300.78 10.63 0.00
        35463.95
        3.70 -38.06 -54.47 -96.20 1164.41 282.01 18.67 0.00
        37378.06
        2.70 -30.66 -54.11 -97.77 1314.21 274.52 34.50 0.00
        39580.40
        1.80 -20.26 -54.35 -99.75 1538.91 269.37 53.22 0.00
        42520.13
        1.00 -13.26 -13.26 -102.52 1870.72 263.80 71.91 0.00
        46931.43
        0.30 -23.16 -23.16 -107.87 2538.27 259.15

        STID = STNM = 727730 TIME = 170401/0300
        SLAT = 46.92 SLON = -114.08 SELV = 972.0
        STIM = 3

        SHOW = 9.67 LIFT = 9.84 SWET = 33.41 KINX = 3.88
        LCLP = 822.95 PWAT = 9.52 TOTL = 37.25 CAPE = 0.00
        LCLT = 273.49 CINS = 0.00 EQLV = -9999.00 LFCT = -9999.00
        BRCH = 0.00

        PRES TMPC TMWC DWPC THTE DRCT SKNT OMEG
        HGHT
        906.90 8.04 4.99 1.70 303.11 250.71 4.12 -2.00
        994.72
        901.80 8.24 5.10 1.76 303.96 249.15 4.37 -2.00
        1041.29
        895.90 8.24 5.06 1.67 304.53 250.71 4.12 -2.00
        1095.52
        889.30 8.04 4.87 1.48 304.88 249.78 3.92 -2.00
        1156.58
        881.90 7.54 4.50 1.22 304.93 250.56 3.50 -2.00
        1225.49
        873.50 6.84 4.04 0.97 304.89 249.44 3.32 -1.90
        1304.25
        864.20 6.04 3.49 0.64 304.78 254.05 2.84 -1.90
        1392.10
        853.70 5.14 2.90 0.36 304.75 251.57 2.45 -1.80
        1492.12
        842.00 4.14 2.24 0.05 304.74 251.57 2.45 -1.80
        1604.64
        829.10 2.94 1.39 -0.47 304.48 259.70 2.18 -1.70
        1730.03
        814.80 1.74 0.37 -1.33 304.10 275.19 2.14 -1.60
        1870.70
        799.10 0.54 -0.80 -2.58 303.62 298.30 2.87 -1.60
        2027.30
        781.80 -0.56 -2.25 -4.68 302.87 305.22 4.04 -1.50
        2202.69
        763.00 -0.86 -3.81 -8.76 302.15 306.03 5.28 -1.60
        2397.16
        742.60 -1.36 -5.31 -13.12 301.85 300.96 5.67 -1.70
        2613.25
        720.60 -2.16 -6.67 -16.97 302.14 300.96 5.67 -1.60
        2852.39
        697.00 -3.16 -8.02 -20.81 302.80 316.33 5.91 -1.50
        3116.24
        671.80 -3.96 -9.29 -26.11 303.92 338.46 7.94 -1.20
        3406.98
        645.30 -4.86 -10.45 -31.21 305.62 349.26 11.46 -0.90
        3723.68
        617.40 -6.06 -11.53 -33.19 307.92 350.17 14.78 -0.80
        4070.09
        588.30 -7.86 -12.78 -31.48 310.40 346.91 17.15 -0.70
        4446.37
        558.30 -10.26 -14.37 -29.11 312.74 345.26 19.08 -0.70
        4851.11
        527.60 -13.26 -16.43 -27.67 314.64 346.46 21.58 -0.60
        5284.03
        496.40 -16.46 -18.86 -27.77 316.36 349.62 25.87 -0.40
        5745.11
        465.10 -19.96 -21.83 -29.76 317.61 351.28 32.03 -0.20
        6231.36
        433.80 -23.66 -25.19 -33.20 318.71 350.26 39.02 0.00
        6744.11
        402.90 -27.76 -29.00 -37.48 319.60 349.17 45.49 0.10
        7279.47
        372.60 -32.26 -33.18 -41.54 320.43 349.58 51.55 0.30
        7836.01
        343.10 -37.06 -37.69 -45.57 321.24 350.32 58.92 0.40
        8411.91
        314.60 -41.96 -42.35 -49.07 322.28 349.96 66.88 0.50
        9005.04
        287.40 -47.16 -47.41 -53.64 323.08 348.69 72.30 0.40
        9610.15
        261.50 -52.46 -52.60 -57.74 324.01 347.72 73.95 0.40
        10227.58
        237.00 -57.66 -57.73 -61.69 325.31 347.50 71.83 0.30
        10855.60
        214.00 -62.46 -62.51 -67.60 327.40 346.92 65.21 0.20
        11492.35
        192.50 -65.66 -65.68 -68.36 332.33 343.88 53.19 0.10
        12140.38
        172.50 -65.46 -65.52 -73.90 343.19 331.20 46.78 0.00
        12806.97
        154.10 -62.36 -9999.00 -9999.00 -9999.00 333.80 41.36 0.00
        13497.82
        137.10 -59.76 -9999.00 -9999.00 -9999.00 333.43 30.40 0.00
        14223.50
        121.60 -59.06 -9999.00 -9999.00 -9999.00 333.86 23.37 0.00
        14974.11
        107.50 -58.66 -9999.00 -9999.00 -9999.00 326.82 18.10 0.00
        15747.18
        94.70 -58.06 -9999.00 -9999.00 -9999.00 315.95 16.49 0.00
        16544.25
        83.20 -58.06 -9999.00 -9999.00 -9999.00 312.37 15.00 0.00
        17359.37
        72.90 -58.56 -9999.00 -9999.00 -9999.00 315.62 12.78 0.00
        18190.47
        63.70 -58.56 -59.08 -80.30 471.36 322.03 10.10 0.00
        19037.85
        55.50 -58.36 -58.97 -81.15 490.74 329.30 7.23 0.00
        19903.84
        48.20 -58.66 -9999.00 -9999.00 -9999.00 351.38 6.49 0.00
        20789.87
        41.80 -58.76 -59.52 -82.87 531.15 22.62 7.58 0.00
        21684.10
        36.10 -58.56 -59.45 -83.75 554.39 50.71 8.28 0.00
        22604.55
        31.20 -58.16 -59.23 -84.61 579.06 72.80 8.55 0.00
        23521.70
        26.80 -58.06 -59.29 -85.49 605.05 83.66 8.80 0.00
        24478.57
        22.90 -57.66 -59.14 -86.40 634.03 87.56 9.13 0.00
        25469.64
        19.50 -57.06 -58.86 -87.32 665.68 91.22 9.13 0.00
        26484.84
        16.50 -56.06 -58.34 -88.26 701.45 92.60 8.55 0.00
        27543.95
        13.90 -54.76 -57.70 -89.21 741.09 87.21 7.96 0.00
        28636.84
        11.60 -53.36 -57.14 -90.21 785.40 68.46 7.94 0.00
        29796.87
        9.60 -51.66 -56.57 -91.23 835.44 45.00 8.80 0.00
        31019.09
        7.80 -49.66 -56.08 -92.34 894.51 28.14 9.48 0.00
        32371.37
        6.30 -46.86 -55.44 -93.47 962.71 11.31 7.93 0.00
        33777.30
        4.90 -43.06 -54.87 -94.77 1051.75 313.96 7.56 0.00
        35455.95
        3.70 -38.06 -54.47 -96.20 1164.41 283.45 18.38 0.00
        37368.41
        2.70 -30.26 -54.00 -97.77 1316.37 278.99 34.81 0.00
        39572.59
        1.80 -20.06 -54.31 -99.75 1540.13 275.12 54.41 0.00
        42515.89
        1.00 -13.66 -13.66 -102.52 1867.84 272.13 73.10 0.00
        46925.47
        0.30 -23.66 -23.66 -107.87 2533.19 271.24 80.63 0.00
        55894.26

        STID = STNM = 727730 TIME = 170401/0600
        SLAT = 46.92 SLON = -114.08 SELV = 972.0
        STIM = 6

        SHOW = 11.19 LIFT = 11.70 SWET = 31.69 KINX = -4.95
        LCLP = 840.66 PWAT = 9.51 TOTL = 34.49 CAPE = 0.00
        LCLT = 272.71 CINS = 0.00 EQLV = -9999.00 LFCT = -9999.00
        BRCH = 0.00

        PRES TMPC TMWC DWPC THTE DRCT SKNT OMEG
        HGHT
        907.00 5.54 3.28 0.61 299.37 242.10 3.73 -1.50
        993.61
        901.80 5.94 3.57 0.84 300.58 240.95 4.00 -1.50
        1040.68
        895.90 6.34 3.82 0.96 301.78 243.43 3.90 -1.50
        1094.49
        889.30 6.54 3.92 0.98 302.76 242.10 3.73 -1.50
        1155.17
        881.90 6.34 3.75 0.84 303.25 240.64 3.57 -1.50
        1223.75
        873.50 6.04 3.46 0.55 303.61 236.31 3.50 -1.50
        1302.22
        864.20 5.54 3.05 0.19 303.81 236.31 3.50 -1.50
        1389.86
        853.70 4.84 2.49 -0.25 303.86 236.31 3.50 -1.40
        1489.73
        842.00 3.94 1.78 -0.82 303.73 239.53 3.83 -1.40
        1602.13
        829.10 2.94 0.95 -1.52 303.55 247.17 4.00 -1.40
        1727.45
        814.80 1.84 -0.01 -2.42 303.28 257.20 4.39 -1.40
        1868.12
        799.10 0.84 -1.08 -3.71 303.04 267.80 5.05 -1.40
        2024.80
        781.90 0.04 -2.30 -5.79 302.73 273.58 6.24 -1.50
        2199.42
        763.00 -0.46 -3.74 -9.31 302.27 277.70 7.25 -1.50
        2395.25
        742.60 -0.56 -5.23 -15.01 301.93 283.09 8.57 -1.50
        2611.79
        720.60 -0.76 -6.49 -21.46 302.32 291.80 9.42 -1.40
        2851.85
        697.00 -1.26 -7.43 -26.04 303.69 303.69 10.51 -1.20
        3117.24
        671.80 -2.06 -8.19 -27.30 305.84 316.30 12.08 -1.00
        3409.99
        645.30 -3.26 -8.94 -25.62 308.44 324.09 13.91 -0.80
        3728.77
        617.40 -4.96 -9.92 -23.57 311.02 328.30 15.52 -0.70
        4077.02
        588.30 -7.06 -11.33 -22.96 313.20 333.14 16.76 -0.60
        4454.77
        558.30 -9.56 -13.20 -23.55 314.93 340.02 18.18 -0.40
        4860.78
        527.60 -12.36 -15.51 -25.44 316.30 346.48 20.78 -0.20
        5295.10
        496.40 -15.56 -18.25 -28.05 317.41 350.61 25.00 0.00
        5757.81
        465.10 -19.26 -21.34 -30.07 318.42 351.03 29.90 0.20
        6245.57
        433.80 -23.46 -24.91 -32.11 319.18 347.93 34.36 0.30
        6759.25
        402.90 -27.76 -28.79 -35.15 319.95 344.13 38.36 0.30
        7294.86
        372.60 -32.26 -33.05 -39.62 320.65 340.82 42.58 0.30
        7851.43
        343.10 -36.96 -37.53 -44.22 321.49 338.28 48.29 0.30
        8427.46
        314.60 -42.06 -42.45 -49.07 322.14 337.71 54.80 0.30
        9020.60
        287.40 -47.46 -47.72 -54.62 322.61 337.58 60.10 0.40
        9625.17
        261.50 -52.86 -53.03 -60.93 323.34 336.52 61.42 0.30
        10241.63
        237.00 -57.66 -57.75 -63.87 325.27 334.44 59.42 0.20
        10869.07
        214.00 -61.86 -61.92 -67.60 328.33 331.36 55.11 0.10
        11506.72
        192.50 -65.26 -65.29 -68.36 332.97 327.65 50.82 0.00
        12156.30
        172.50 -66.26 -66.31 -73.90 341.87 325.39 50.27 0.00
        12822.24
        154.10 -63.06 -9999.00 -9999.00 -9999.00 327.76 40.42 0.00
        13510.61
        137.10 -60.36 -9999.00 -9999.00 -9999.00 327.33 30.23 0.00
        14234.07
        121.60 -59.56 -9999.00 -9999.00 -9999.00 324.30 24.63 0.00
        14982.75
        107.50 -59.06 -9999.00 -9999.00 -9999.00 320.75 20.57 0.00
        15754.19
        94.70 -58.46 -9999.00 -9999.00 -9999.00 313.65 17.44 0.00
        16549.78
        83.20 -57.96 -9999.00 -9999.00 -9999.00 310.52 14.04 0.00
        17364.33
        72.90 -58.36 -9999.00 -9999.00 -9999.00 312.77 10.59 0.00
        18196.01
        63.70 -58.66 -59.17 -80.30 471.14 318.95 7.98 0.00
        19043.59
        55.50 -58.56 -59.15 -81.15 490.28 334.98 6.43 0.00
        19908.98
        48.20 -58.66 -9999.00 -9999.00 -9999.00 10.95 6.14 0.00
        20794.59
        41.80 -58.86 -59.61 -82.87 530.91 47.20 7.15 0.00
        21688.62
        36.10 -58.86 -59.72 -83.75 553.62 71.08 7.19 0.00
        22608.21
        31.20 -58.36 -59.40 -84.61 578.53 83.66 7.03 0.00
        23524.29
        26.80 -58.06 -59.29 -85.49 605.05 88.49 7.38 0.00
        24480.72
        22.90 -57.76 -59.22 -86.40 633.74 88.64 8.16 0.00
        25471.55
        19.50 -57.16 -58.94 -87.32 665.37 88.75 8.94 0.00
        26486.29
        16.50 -56.36 -58.57 -88.26 700.48 85.03 8.97 0.00
        27544.42
        13.90 -55.26 -58.07 -89.21 739.39 80.54 9.46 0.00
        28635.30
        11.60 -53.76 -57.41 -90.21 783.97 75.00 11.27 0.00
        29792.95
        9.60 -51.96 -56.75 -91.23 834.31 70.51 13.38 0.00
        31013.22
        7.80 -49.66 -56.08 -92.34 894.51 66.47 13.13 0.00
        32364.59
        6.30 -46.66 -55.34 -93.47 963.56 55.12 7.81 0.00
        33771.15
        4.90 -42.66 -54.71 -94.77 1053.58 319.76 6.62 0.00
        35452.00
        3.70 -37.76 -54.37 -96.20 1165.90 295.20 18.24 0.00
        37367.34
        2.70 -30.06 -53.95 -97.77 1317.46 288.32 31.51 0.00
        39573.84
        1.80 -20.06 -54.31 -99.75 1540.13 283.53 48.15 0.00
        42518.32
        1.00 -14.46 -14.46 -102.52 1862.08 281.80 62.70 0.00
        46921.02
        0.30 -24.86 -24.86 -107.87 2521.01 284.33 65.15 0.00
        55854.56

        STID = STNM = 727730 TIME = 170401/0900
        SLAT = 46.92 SLON = -114.08 SELV = 972.0
        STIM = 9

        SHOW = 11.44 LIFT = 12.26 SWET = 26.49 KINX = -1.93
        LCLP = 850.57 PWAT = 9.92 TOTL = 34.11 CAPE = 0.00
        LCLT = 272.07 CINS = 0.00 EQLV = -9999.00 LFCT = -9999.00
        BRCH = 0.00

        PRES TMPC TMWC DWPC THTE DRCT SKNT OMEG
        HGHT
        906.80 3.94 2.11 -0.20 296.97 237.99 3.67 -0.90
        994.39
        901.70 4.34 2.37 -0.08 298.05 240.95 4.00 -0.90
        1040.28
        895.80 4.84 2.68 0.06 299.36 244.65 4.08 -1.00
        1093.79
        889.20 5.24 2.89 0.08 300.54 247.17 4.00 -1.00
        1154.17
        881.80 5.34 2.89 0.00 301.40 248.75 3.75 -1.00
        1222.46
        873.40 5.74 2.97 -0.29 302.54 251.57 3.69 -1.00
        1300.74
        864.10 5.34 2.63 -0.60 302.89 249.78 3.92 -1.00
        1388.30
        853.60 4.84 2.20 -0.99 303.22 254.05 4.23 -1.00
        1488.13
        841.90 4.04 1.56 -1.51 303.25 258.23 4.76 -1.00
        1600.55
        829.00 3.14 0.77 -2.26 303.16 262.15 5.69 -1.00
        1725.93
        814.70 2.24 -0.11 -3.24 303.08 263.29 6.64 -1.10
        1866.75
        799.00 1.34 -1.12 -4.58 302.95 265.91 8.18 -1.20
        2023.69
        781.80 0.64 -2.32 -6.86 302.69 268.92 10.30 -1.30
        2198.66
        762.90 0.14 -3.72 -10.55 302.28 274.54 12.28 -1.30
        2394.91
        742.50 -0.36 -5.24 -15.71 301.89 282.88 13.95 -1.20
        2611.78
        720.50 -0.76 -6.42 -20.88 302.49 290.98 15.19 -1.00
        2851.96
        696.90 -1.16 -6.92 -21.96 304.75 297.86 15.38 -0.80
        3117.46
        671.80 -1.86 -7.16 -19.69 307.96 304.70 15.37 -0.60
        3409.31
        645.20 -2.96 -7.79 -18.70 310.79 312.37 15.00 -0.40
        3729.73
        617.30 -4.46 -8.94 -19.22 313.00 317.73 14.43 -0.30
        4078.66
        588.20 -6.46 -10.64 -21.04 314.54 319.99 14.20 -0.30
        4457.33
        558.30 -8.96 -12.72 -23.02 315.81 321.71 14.10 -0.30
        4862.96
        527.60 -12.16 -15.18 -24.18 316.91 324.89 15.19 -0.20
        5297.97
        496.40 -15.76 -17.94 -24.97 318.00 325.97 18.05 -0.10
        5760.72
        465.10 -19.56 -21.09 -26.70 318.89 324.12 22.53 0.00
        6248.09
        433.80 -23.46 -24.69 -30.35 319.55 319.35 27.14 -0.10
        6761.52
        402.90 -27.56 -28.62 -35.15 320.21 316.25 31.47 -0.20
        7297.36
        372.50 -31.96 -32.85 -40.54 320.96 318.60 34.97 -0.10
        7856.41
        343.10 -36.76 -37.45 -46.06 321.61 324.80 38.75 0.00
        8431.15
        314.60 -41.86 -42.33 -51.17 322.30 329.55 44.83 0.20
        9024.79
        287.40 -47.26 -47.53 -54.62 322.90 329.79 53.26 0.20
        9629.88
        261.50 -52.56 -52.69 -57.74 323.86 327.53 60.78 0.20
        10247.04
        237.00 -57.66 -57.73 -61.69 325.31 324.29 67.23 0.20
        10874.91
        213.90 -62.46 -62.51 -67.60 327.44 321.85 71.39 0.10
        11514.58
        192.50 -65.36 -65.39 -68.36 332.81 323.03 67.83 0.10
        12160.22
        172.50 -65.76 -65.82 -73.90 342.70 324.17 49.11 0.00
        12826.80
        154.10 -63.86 -63.95 -74.65 357.16 317.65 38.64 0.00
        13514.68
        137.10 -60.46 -9999.00 -9999.00 -9999.00 315.79 29.82 0.00
        14236.60
        121.60 -59.76 -9999.00 -9999.00 -9999.00 314.12 26.79 0.00
        14984.75
        107.50 -59.06 -9999.00 -9999.00 -9999.00 313.59 22.26 0.00
        15755.84
        94.70 -58.66 -9999.00 -9999.00 -9999.00 307.00 17.75 0.00
        16551.05
        83.20 -58.26 -9999.00 -9999.00 -9999.00 298.67 14.16 0.00
        17364.65
        72.90 -58.66 -9999.00 -9999.00 -9999.00 297.00 11.56 0.00
        18195.18
        63.70 -59.06 -59.55 -80.30 470.26 303.69 9.81 0.00
        19041.37
        55.50 -58.66 -59.25 -81.15 490.06 320.60 7.03 0.00
        19905.75
        48.20 -58.56 -9999.00 -9999.00 -9999.00 6.01 3.71 0.00
        20791.37
        41.80 -58.96 -59.70 -82.87 530.66 69.44 3.32 0.00
        21685.39
        36.10 -58.96 -59.81 -83.75 553.36 86.82 3.50 0.00
        22604.55
        31.20 -58.36 -59.40 -84.61 578.53 85.03 4.49 0.00
        23520.42
        26.80 -58.16 -59.38 -85.49 604.77 86.19 5.85 0.00
        24476.63
        22.90 -57.96 -59.39 -86.40 633.15 88.45 7.19 0.00
        25466.77
        19.50 -57.46 -59.19 -87.32 664.45 87.40 8.55 0.00
        26480.33
        16.50 -56.56 -58.73 -88.26 699.84 84.40 9.95 0.00
        27537.24
        13.90 -55.46 -58.22 -89.21 738.71 82.01 11.19 0.00
        28627.12
        11.60 -53.86 -57.48 -90.21 783.61 84.56 12.30 0.00
        29783.97
        9.60 -52.16 -56.88 -91.23 833.56 88.29 13.01 0.00
        31003.41
        7.80 -49.96 -56.25 -92.34 893.31 91.04 10.68 0.00
        32353.26
        6.30 -47.16 -55.59 -93.47 961.43 71.57 4.29 0.00
        33757.32
        4.90 -43.36 -55.00 -94.77 1050.38 329.04 6.80 0.00
        35433.76
        3.70 -38.46 -54.61 -96.20 1162.43 307.18 14.14 0.00
        37343.34
        2.70 -30.96 -54.19 -97.77 1312.58 288.73 24.20 0.00
        39542.45
        1.80 -21.46 -54.59 -99.75 1531.61 279.83 39.82 0.00
        42473.29
        1.00 -16.36 -16.36 -102.52 1848.40 277.59 52.91 0.00
        46847.60
        0.30 -26.46 -26.46 -107.87 2504.76 277.71 52.14 0.00
        55719.47

        STID = STNM = 727730 TIME = 170401/1200
        SLAT = 46.92 SLON = -114.08 SELV = 972.0
        STIM = 12

        SHOW = 11.24 LIFT = 12.58 SWET = 27.14 KINX = 3.00
        LCLP = 861.46 PWAT = 10.48 TOTL = 34.65 CAPE = 0.00
        LCLT = 271.91 CINS = 0.00 EQLV = -9999.00 LFCT = -9999.00
        BRCH = 0.00

        PRES TMPC TMWC DWPC THTE DRCT SKNT OMEG
        HGHT
        906.40 2.74 1.32 -0.54 295.43 214.70 3.07 -0.20
        993.41
        901.20 2.94 1.48 -0.41 296.31 220.24 3.30 -0.30
        1040.00
        895.40 3.04 1.56 -0.34 297.10 225.00 3.85 -0.40
        1092.32
        888.80 3.14 1.63 -0.27 297.99 230.53 4.27 -0.50
        1152.30
        881.30 3.24 1.68 -0.26 298.94 234.16 4.31 -0.50
        1221.02
        873.00 3.94 2.02 -0.33 300.57 240.95 4.00 -0.50
        1297.87
        863.60 4.64 2.27 -0.57 302.20 246.04 3.83 -0.50
        1386.02
        853.20 4.44 2.02 -0.90 302.90 249.78 3.92 -0.50
        1484.75
        841.50 3.74 1.44 -1.41 303.06 250.02 4.55 -0.50
        1597.09
        828.60 2.94 0.67 -2.23 303.01 249.68 5.59 -0.60
        1722.42
        814.30 2.14 -0.27 -3.53 302.79 250.02 6.82 -0.60
        1863.23
        798.60 1.54 -1.45 -5.88 302.31 254.41 8.66 -0.70
        2020.26
        781.40 1.24 -2.82 -9.75 301.68 263.88 10.94 -0.70
        2195.52
        762.60 0.94 -4.17 -14.62 301.32 277.54 13.33 -0.60
        2391.25
        742.20 0.64 -5.11 -18.43 302.09 289.56 15.68 -0.40
        2608.86
        720.20 0.24 -5.43 -18.31 304.42 296.28 17.54 -0.20
        2850.01
        696.60 -0.46 -5.54 -15.94 307.57 299.17 19.13 -0.10
        3116.52
        671.50 -1.36 -5.99 -15.02 310.38 300.22 20.45 0.00
        3409.26
        644.90 -2.76 -7.04 -15.50 312.38 299.36 21.39 0.10
        3730.34
        617.00 -4.66 -8.62 -16.94 313.73 296.57 21.29 0.00
        4079.51
        588.00 -6.86 -10.47 -18.72 314.96 290.75 19.74 -0.10
        4456.67
        558.00 -9.46 -12.54 -20.14 316.27 283.13 17.95 -0.20
        4863.22
        527.40 -12.66 -14.97 -21.12 317.38 279.14 17.11 -0.30
        5296.27
        496.20 -16.16 -17.79 -22.61 318.33 284.04 19.23 -0.30
        5758.48
        464.90 -19.56 -20.97 -26.02 319.13 290.40 25.08 -0.30
        6245.71
        433.60 -23.16 -24.56 -31.12 319.81 293.71 31.39 -0.40
        6759.67
        402.70 -27.16 -28.42 -36.49 320.57 295.56 34.67 -0.40
        7296.51
        372.40 -31.76 -32.70 -40.86 321.22 296.98 35.98 -0.40
        7854.61
        343.00 -36.86 -37.44 -44.23 321.65 301.27 37.04 -0.30
        8429.64
        314.50 -42.16 -42.47 -47.35 322.14 310.10 43.43 -0.20
        9022.97
        287.30 -47.26 -47.43 -51.20 323.08 315.81 57.96 0.00
        9627.89
        261.40 -52.36 -52.47 -56.49 324.23 316.82 73.52 0.10
        10245.55
        236.90 -57.76 -57.83 -61.69 325.20 317.06 80.40 0.10
        10873.83
        213.90 -62.66 -62.71 -67.60 327.13 316.17 74.05 0.00
        11510.42
        192.40 -65.46 -65.49 -68.36 332.70 312.57 64.90 0.00
        12158.76
        172.50 -65.76 -65.82 -73.90 342.70 316.07 51.52 0.00
        12822.03
        154.10 -63.36 -9999.00 -9999.00 -9999.00 304.25 38.31 0.00
        13510.73
        137.10 -60.26 -9999.00 -9999.00 -9999.00 306.68 35.12 0.00
        14233.85
        121.60 -59.36 -9999.00 -9999.00 -9999.00 303.89 30.65 0.00
        14983.05
        107.50 -59.06 -9999.00 -9999.00 -9999.00 311.15 26.57 0.00
        15754.86
        94.70 -59.26 -9999.00 -9999.00 -9999.00 306.31 20.01 0.00
        16548.96
        83.20 -58.96 -9999.00 -9999.00 -9999.00 301.30 16.82 0.00
        17360.10
        72.90 -59.06 -9999.00 -9999.00 -9999.00 301.36 14.55 0.00
        18188.50
        63.70 -59.16 -9999.00 -9999.00 -9999.00 308.21 11.62 0.00
        19033.71
        55.50 -58.56 -59.15 -81.15 490.28 317.12 7.42 0.00
        19898.08
        48.20 -58.66 -9999.00 -9999.00 -9999.00 322.43 3.19 0.00
        20783.70
        41.80 -59.26 -59.98 -82.87 529.91 338.20 2.10 0.00
        21676.89
        36.10 -59.06 -59.90 -83.75 553.10 17.35 3.26 0.00
        22595.19
        31.20 -58.56 -59.58 -84.61 577.99 49.64 5.09 0.00
        23510.42
        26.80 -58.46 -59.64 -85.49 603.93 66.37 6.78 0.00
        24465.51
        22.90 -58.16 -59.56 -86.40 632.56 75.32 8.43 0.00
        25454.50
        19.50 -57.56 -59.27 -87.32 664.14 82.48 10.39 0.00
        26467.36
        16.50 -56.66 -58.81 -88.26 699.51 85.46 12.28 0.00
        27523.78
        13.90 -55.46 -58.22 -89.21 738.71 85.73 13.05 0.00
        28613.41
        11.60 -53.96 -57.55 -90.21 783.25 87.32 12.45 0.00
        29769.99
        9.60 -52.36 -57.01 -91.23 832.80 90.00 10.68 0.00
        30988.61
        7.80 -50.36 -56.47 -92.34 891.71 86.91 7.21 0.00
        32336.63
        6.30 -47.76 -55.89 -93.47 958.88 45.00 4.39 0.00
        33737.56
        4.90 -44.06 -55.29 -94.77 1047.18 357.88 5.24 0.00
        35409.22
        3.70 -39.36 -54.92 -96.20 1157.97 299.90 8.95 0.00
        37312.23
        2.70 -32.26 -54.54 -97.77 1305.53 281.41 22.59 0.00
        39501.20
        1.80 -23.16 -54.94 -99.75 1521.26 277.48 38.79 0.00
        42414.23
        1.00 -18.06 -18.06 -102.52 1836.17 275.34 50.14 0.00
        46759.29
        0.30 -27.66 -27.66 -107.87 2492.58 270.91 49.15 0.00
        55580.06"
    }

    fn get_invalid_test_data2() -> &'static str {
        // Missing required column in first sounding.
        "
        SNPARM = PRES;TMPC;TMWC;DWPC;THTE;DRCT;SKNT;OMEG;HGHT
        STNPRM = SHOW;LIFT;SWET;KINX;LCLP;PWAT;TOTL;CAPE;LCLT;CINS;EQLV;LFCT;BRCH

        STID = STNM = 727730 TIME = 170401/0000
        SLAT = 46.92 SLON = -114.08 SELV = 972.0
        STIM = 0

        SHOW = 8.73 LIFT = 8.75 SWET = 35.35 KINX = 10.91
        LCLP = 790.16 PWAT = 9.96 TOTL = 38.90 CAPE = 6.66
        LCLT = 272.76 CINS = 0.00 EQLV = 768.94 LFCT = 789.03
        BRCH = 0.34

        TMPC TMWC DWPC THTE DRCT SKNT OMEG
        HGHT
        906.70 10.54 6.12 1.52 305.69 270.00 2.14 -2.00
        994.01
        901.50 10.04 5.79 1.32 305.54 274.76 2.33 -2.00
        1041.87
        895.70 9.54 5.49 1.20 305.53 274.76 2.33 -2.00
        1095.48
        889.10 8.94 5.14 1.07 305.50 279.46 2.37 -1.90
        1156.78
        881.60 8.24 4.73 0.95 305.48 285.26 2.21 -1.90
        1226.84
        873.30 7.44 4.27 0.82 305.43 291.80 2.10 -1.90
        1304.87
        863.90 6.54 3.75 0.67 305.39 303.69 2.10 -1.80
        1393.86
        853.50 5.54 3.16 0.50 305.35 315.00 2.20 -1.80
        1493.12
        841.80 4.44 2.51 0.31 305.35 329.04 2.27 -1.70
        1605.81
        828.90 3.14 1.75 0.13 305.29 333.43 2.60 -1.60
        1731.35
        814.60 1.64 0.85 -0.11 305.14 338.96 2.70 -1.60
        1872.11
        798.90 0.14 -0.09 -0.41 305.13 340.35 2.89 -1.50
        2028.66
        781.60 -1.56 -1.56 -0.82 305.03 344.05 2.84 -1.50
        2203.74
        762.80 -1.96 -3.42 -5.62 302.99 357.40 4.27 -1.40
        2397.62
        742.40 -1.76 -5.28 -11.99 301.96 351.87 4.12 -1.50
        2613.23
        720.40 -3.06 -6.67 -14.23 302.25 339.44 3.32 -1.60
        2851.90
        696.80 -4.76 -8.23 -16.16 302.59 340.35 2.89 -1.70
        3114.67
        671.70 -6.26 -10.00 -19.99 302.87 10.01 3.36 -1.50
        3402.32
        645.10 -7.16 -11.58 -26.55 303.81 19.86 7.44 -1.20
        3717.63
        617.20 -7.56 -12.67 -35.59 305.94 16.93 14.01 -0.70
        4061.71
        588.20 -8.46 -13.67 -40.54 308.74 12.83 20.12 -0.40
        4435.27
        558.20 -10.56 -15.29 -40.08 310.98 7.72 23.14 -0.30
        4839.31
        527.50 -13.46 -17.38 -37.16 312.88 3.69 24.15 -0.40
        5271.78
        496.30 -17.06 -20.05 -35.14 314.28 0.43 25.84 -0.40
        5732.11
        465.00 -20.86 -22.95 -33.69 315.75 357.06 30.34 -0.30
        6216.94
        433.70 -24.66 -25.99 -33.20 317.45 352.62 37.80 -0.20
        6727.83
        402.80 -28.46 -29.37 -35.15 319.06 347.31 46.00 -0.20
        7261.50
        372.50 -32.66 -33.44 -40.22 320.07 344.46 52.21 -0.10
        7816.95
        343.00 -37.26 -37.90 -46.06 320.96 346.25 57.19 0.00
        8392.29
        314.60 -41.96 -42.43 -51.17 322.16 351.06 66.26 0.20
        8983.17
        287.40 -46.96 -47.22 -53.64 323.36 352.98 79.47 0.40
        9588.53
        261.40 -52.16 -52.28 -56.49 324.53 349.42 89.92 0.40
        10209.16
        237.00 -57.56 -57.63 -61.69 325.46 348.22 92.27 0.30
        10835.31
        213.90 -63.06 -63.10 -67.60 326.51 352.66 85.20 0.30
        11474.23
        192.40 -66.06 -66.11 -73.17 331.69 356.81 59.34 0.10
        12121.03
        172.50 -65.16 -65.22 -73.90 343.69 336.93 45.61 0.10
        12784.30
        154.10 -60.76 -9999.00 -9999.00 -9999.00 337.13 40.48 0.00
        13478.28
        137.10 -58.86 -9999.00 -9999.00 -9999.00 330.91 31.57 0.00
        14208.24
        121.60 -58.36 -9999.00 -9999.00 -9999.00 333.24 25.45 0.00
        14961.66
        107.50 -57.96 -9999.00 -9999.00 -9999.00 328.12 18.76 0.00
        15737.25
        94.70 -57.46 -9999.00 -9999.00 -9999.00 319.01 15.70 0.00
        16536.74
        83.20 -57.76 -9999.00 -9999.00 -9999.00 314.41 13.33 0.00
        17353.56
        72.90 -58.66 -9999.00 -9999.00 -9999.00 315.00 11.81 0.00
        18185.05
        63.70 -58.56 -59.08 -80.30 471.36 322.65 9.29 0.00
        19032.24
        55.50 -58.16 -58.78 -81.15 491.20 330.02 5.83 0.00
        19898.63
        48.20 -58.56 -9999.00 -9999.00 -9999.00 342.35 4.49 0.00
        20785.28
        41.80 -58.76 -59.52 -82.87 531.15 360.00 5.24 0.00
        21679.72
        36.10 -58.56 -59.45 -83.75 554.39 26.57 6.51 0.00
        22600.16
        31.20 -58.06 -59.14 -84.61 579.33 50.39 7.30 0.00
        23517.53
        26.80 -57.86 -59.12 -85.49 605.61 66.04 7.65 0.00
        24475.07
        22.90 -57.56 -59.05 -86.40 634.33 76.29 8.20 0.00
        25466.82
        19.50 -56.96 -58.78 -87.32 665.99 87.56 9.13 0.00
        26482.50
        16.50 -56.06 -58.34 -88.26 701.45 96.58 10.16 0.00
        27541.86
        13.90 -54.66 -57.63 -89.21 741.43 100.20 9.87 0.00
        28635.00
        11.60 -53.16 -57.00 -90.21 786.11 96.34 7.03 0.00
        29795.82
        9.60 -51.46 -56.44 -91.23 836.20 71.57 2.45 0.00
        31019.14
        7.80 -49.26 -55.85 -92.34 896.11 321.34 3.73 0.00
        32373.24
        6.30 -46.36 -55.20 -93.47 964.84 311.88 7.56 0.00
        33781.99
        4.90 -42.66 -54.71 -94.77 1053.58 300.78 10.63 0.00
        35463.95
        3.70 -38.06 -54.47 -96.20 1164.41 282.01 18.67 0.00
        37378.06
        2.70 -30.66 -54.11 -97.77 1314.21 274.52 34.50 0.00
        39580.40
        1.80 -20.26 -54.35 -99.75 1538.91 269.37 53.22 0.00
        42520.13
        1.00 -13.26 -13.26 -102.52 1870.72 263.80 71.91 0.00
        46931.43
        0.30 -23.16 -23.16 -107.87 2538.27 259.15 80.50 0.00
        55916.08

        STID = STNM = 727730 TIME = 170401/0300
        SLAT = 46.92 SLON = -114.08 SELV = 972.0
        STIM = 3

        SHOW = 9.67 LIFT = 9.84 SWET = 33.41 KINX = 3.88
        LCLP = 822.95 PWAT = 9.52 TOTL = 37.25 CAPE = 0.00
        LCLT = 273.49 CINS = 0.00 EQLV = -9999.00 LFCT = -9999.00
        BRCH = 0.00

        PRES TMPC TMWC DWPC THTE DRCT SKNT OMEG
        HGHT
        906.90 8.04 4.99 1.70 303.11 250.71 4.12 -2.00
        994.72
        901.80 8.24 5.10 1.76 303.96 249.15 4.37 -2.00
        1041.29
        895.90 8.24 5.06 1.67 304.53 250.71 4.12 -2.00
        1095.52
        889.30 8.04 4.87 1.48 304.88 249.78 3.92 -2.00
        1156.58
        881.90 7.54 4.50 1.22 304.93 250.56 3.50 -2.00
        1225.49
        873.50 6.84 4.04 0.97 304.89 249.44 3.32 -1.90
        1304.25
        864.20 6.04 3.49 0.64 304.78 254.05 2.84 -1.90
        1392.10
        853.70 5.14 2.90 0.36 304.75 251.57 2.45 -1.80
        1492.12
        842.00 4.14 2.24 0.05 304.74 251.57 2.45 -1.80
        1604.64
        829.10 2.94 1.39 -0.47 304.48 259.70 2.18 -1.70
        1730.03
        814.80 1.74 0.37 -1.33 304.10 275.19 2.14 -1.60
        1870.70
        799.10 0.54 -0.80 -2.58 303.62 298.30 2.87 -1.60
        2027.30
        781.80 -0.56 -2.25 -4.68 302.87 305.22 4.04 -1.50
        2202.69
        763.00 -0.86 -3.81 -8.76 302.15 306.03 5.28 -1.60
        2397.16
        742.60 -1.36 -5.31 -13.12 301.85 300.96 5.67 -1.70
        2613.25
        720.60 -2.16 -6.67 -16.97 302.14 300.96 5.67 -1.60
        2852.39
        697.00 -3.16 -8.02 -20.81 302.80 316.33 5.91 -1.50
        3116.24
        671.80 -3.96 -9.29 -26.11 303.92 338.46 7.94 -1.20
        3406.98
        645.30 -4.86 -10.45 -31.21 305.62 349.26 11.46 -0.90
        3723.68
        617.40 -6.06 -11.53 -33.19 307.92 350.17 14.78 -0.80
        4070.09
        588.30 -7.86 -12.78 -31.48 310.40 346.91 17.15 -0.70
        4446.37
        558.30 -10.26 -14.37 -29.11 312.74 345.26 19.08 -0.70
        4851.11
        527.60 -13.26 -16.43 -27.67 314.64 346.46 21.58 -0.60
        5284.03
        496.40 -16.46 -18.86 -27.77 316.36 349.62 25.87 -0.40
        5745.11
        465.10 -19.96 -21.83 -29.76 317.61 351.28 32.03 -0.20
        6231.36
        433.80 -23.66 -25.19 -33.20 318.71 350.26 39.02 0.00
        6744.11
        402.90 -27.76 -29.00 -37.48 319.60 349.17 45.49 0.10
        7279.47
        372.60 -32.26 -33.18 -41.54 320.43 349.58 51.55 0.30
        7836.01
        343.10 -37.06 -37.69 -45.57 321.24 350.32 58.92 0.40
        8411.91
        314.60 -41.96 -42.35 -49.07 322.28 349.96 66.88 0.50
        9005.04
        287.40 -47.16 -47.41 -53.64 323.08 348.69 72.30 0.40
        9610.15
        261.50 -52.46 -52.60 -57.74 324.01 347.72 73.95 0.40
        10227.58
        237.00 -57.66 -57.73 -61.69 325.31 347.50 71.83 0.30
        10855.60
        214.00 -62.46 -62.51 -67.60 327.40 346.92 65.21 0.20
        11492.35
        192.50 -65.66 -65.68 -68.36 332.33 343.88 53.19 0.10
        12140.38
        172.50 -65.46 -65.52 -73.90 343.19 331.20 46.78 0.00
        12806.97
        154.10 -62.36 -9999.00 -9999.00 -9999.00 333.80 41.36 0.00
        13497.82
        137.10 -59.76 -9999.00 -9999.00 -9999.00 333.43 30.40 0.00
        14223.50
        121.60 -59.06 -9999.00 -9999.00 -9999.00 333.86 23.37 0.00
        14974.11
        107.50 -58.66 -9999.00 -9999.00 -9999.00 326.82 18.10 0.00
        15747.18
        94.70 -58.06 -9999.00 -9999.00 -9999.00 315.95 16.49 0.00
        16544.25
        83.20 -58.06 -9999.00 -9999.00 -9999.00 312.37 15.00 0.00
        17359.37
        72.90 -58.56 -9999.00 -9999.00 -9999.00 315.62 12.78 0.00
        18190.47
        63.70 -58.56 -59.08 -80.30 471.36 322.03 10.10 0.00
        19037.85
        55.50 -58.36 -58.97 -81.15 490.74 329.30 7.23 0.00
        19903.84
        48.20 -58.66 -9999.00 -9999.00 -9999.00 351.38 6.49 0.00
        20789.87
        41.80 -58.76 -59.52 -82.87 531.15 22.62 7.58 0.00
        21684.10
        36.10 -58.56 -59.45 -83.75 554.39 50.71 8.28 0.00
        22604.55
        31.20 -58.16 -59.23 -84.61 579.06 72.80 8.55 0.00
        23521.70
        26.80 -58.06 -59.29 -85.49 605.05 83.66 8.80 0.00
        24478.57
        22.90 -57.66 -59.14 -86.40 634.03 87.56 9.13 0.00
        25469.64
        19.50 -57.06 -58.86 -87.32 665.68 91.22 9.13 0.00
        26484.84
        16.50 -56.06 -58.34 -88.26 701.45 92.60 8.55 0.00
        27543.95
        13.90 -54.76 -57.70 -89.21 741.09 87.21 7.96 0.00
        28636.84
        11.60 -53.36 -57.14 -90.21 785.40 68.46 7.94 0.00
        29796.87
        9.60 -51.66 -56.57 -91.23 835.44 45.00 8.80 0.00
        31019.09
        7.80 -49.66 -56.08 -92.34 894.51 28.14 9.48 0.00
        32371.37
        6.30 -46.86 -55.44 -93.47 962.71 11.31 7.93 0.00
        33777.30
        4.90 -43.06 -54.87 -94.77 1051.75 313.96 7.56 0.00
        35455.95
        3.70 -38.06 -54.47 -96.20 1164.41 283.45 18.38 0.00
        37368.41
        2.70 -30.26 -54.00 -97.77 1316.37 278.99 34.81 0.00
        39572.59
        1.80 -20.06 -54.31 -99.75 1540.13 275.12 54.41 0.00
        42515.89
        1.00 -13.66 -13.66 -102.52 1867.84 272.13 73.10 0.00
        46925.47
        0.30 -23.66 -23.66 -107.87 2533.19 271.24 80.63 0.00
        55894.26

        STID = STNM = 727730 TIME = 170401/0600
        SLAT = 46.92 SLON = -114.08 SELV = 972.0
        STIM = 6

        SHOW = 11.19 LIFT = 11.70 SWET = 31.69 KINX = -4.95
        LCLP = 840.66 PWAT = 9.51 TOTL = 34.49 CAPE = 0.00
        LCLT = 272.71 CINS = 0.00 EQLV = -9999.00 LFCT = -9999.00
        BRCH = 0.00

        PRES TMPC TMWC DWPC THTE DRCT SKNT OMEG
        HGHT
        907.00 5.54 3.28 0.61 299.37 242.10 3.73 -1.50
        993.61
        901.80 5.94 3.57 0.84 300.58 240.95 4.00 -1.50
        1040.68
        895.90 6.34 3.82 0.96 301.78 243.43 3.90 -1.50
        1094.49
        889.30 6.54 3.92 0.98 302.76 242.10 3.73 -1.50
        1155.17
        881.90 6.34 3.75 0.84 303.25 240.64 3.57 -1.50
        1223.75
        873.50 6.04 3.46 0.55 303.61 236.31 3.50 -1.50
        1302.22
        864.20 5.54 3.05 0.19 303.81 236.31 3.50 -1.50
        1389.86
        853.70 4.84 2.49 -0.25 303.86 236.31 3.50 -1.40
        1489.73
        842.00 3.94 1.78 -0.82 303.73 239.53 3.83 -1.40
        1602.13
        829.10 2.94 0.95 -1.52 303.55 247.17 4.00 -1.40
        1727.45
        814.80 1.84 -0.01 -2.42 303.28 257.20 4.39 -1.40
        1868.12
        799.10 0.84 -1.08 -3.71 303.04 267.80 5.05 -1.40
        2024.80
        781.90 0.04 -2.30 -5.79 302.73 273.58 6.24 -1.50
        2199.42
        763.00 -0.46 -3.74 -9.31 302.27 277.70 7.25 -1.50
        2395.25
        742.60 -0.56 -5.23 -15.01 301.93 283.09 8.57 -1.50
        2611.79
        720.60 -0.76 -6.49 -21.46 302.32 291.80 9.42 -1.40
        2851.85
        697.00 -1.26 -7.43 -26.04 303.69 303.69 10.51 -1.20
        3117.24
        671.80 -2.06 -8.19 -27.30 305.84 316.30 12.08 -1.00
        3409.99
        645.30 -3.26 -8.94 -25.62 308.44 324.09 13.91 -0.80
        3728.77
        617.40 -4.96 -9.92 -23.57 311.02 328.30 15.52 -0.70
        4077.02
        588.30 -7.06 -11.33 -22.96 313.20 333.14 16.76 -0.60
        4454.77
        558.30 -9.56 -13.20 -23.55 314.93 340.02 18.18 -0.40
        4860.78
        527.60 -12.36 -15.51 -25.44 316.30 346.48 20.78 -0.20
        5295.10
        496.40 -15.56 -18.25 -28.05 317.41 350.61 25.00 0.00
        5757.81
        465.10 -19.26 -21.34 -30.07 318.42 351.03 29.90 0.20
        6245.57
        433.80 -23.46 -24.91 -32.11 319.18 347.93 34.36 0.30
        6759.25
        402.90 -27.76 -28.79 -35.15 319.95 344.13 38.36 0.30
        7294.86
        372.60 -32.26 -33.05 -39.62 320.65 340.82 42.58 0.30
        7851.43
        343.10 -36.96 -37.53 -44.22 321.49 338.28 48.29 0.30
        8427.46
        314.60 -42.06 -42.45 -49.07 322.14 337.71 54.80 0.30
        9020.60
        287.40 -47.46 -47.72 -54.62 322.61 337.58 60.10 0.40
        9625.17
        261.50 -52.86 -53.03 -60.93 323.34 336.52 61.42 0.30
        10241.63
        237.00 -57.66 -57.75 -63.87 325.27 334.44 59.42 0.20
        10869.07
        214.00 -61.86 -61.92 -67.60 328.33 331.36 55.11 0.10
        11506.72
        192.50 -65.26 -65.29 -68.36 332.97 327.65 50.82 0.00
        12156.30
        172.50 -66.26 -66.31 -73.90 341.87 325.39 50.27 0.00
        12822.24
        154.10 -63.06 -9999.00 -9999.00 -9999.00 327.76 40.42 0.00
        13510.61
        137.10 -60.36 -9999.00 -9999.00 -9999.00 327.33 30.23 0.00
        14234.07
        121.60 -59.56 -9999.00 -9999.00 -9999.00 324.30 24.63 0.00
        14982.75
        107.50 -59.06 -9999.00 -9999.00 -9999.00 320.75 20.57 0.00
        15754.19
        94.70 -58.46 -9999.00 -9999.00 -9999.00 313.65 17.44 0.00
        16549.78
        83.20 -57.96 -9999.00 -9999.00 -9999.00 310.52 14.04 0.00
        17364.33
        72.90 -58.36 -9999.00 -9999.00 -9999.00 312.77 10.59 0.00
        18196.01
        63.70 -58.66 -59.17 -80.30 471.14 318.95 7.98 0.00
        19043.59
        55.50 -58.56 -59.15 -81.15 490.28 334.98 6.43 0.00
        19908.98
        48.20 -58.66 -9999.00 -9999.00 -9999.00 10.95 6.14 0.00
        20794.59
        41.80 -58.86 -59.61 -82.87 530.91 47.20 7.15 0.00
        21688.62
        36.10 -58.86 -59.72 -83.75 553.62 71.08 7.19 0.00
        22608.21
        31.20 -58.36 -59.40 -84.61 578.53 83.66 7.03 0.00
        23524.29
        26.80 -58.06 -59.29 -85.49 605.05 88.49 7.38 0.00
        24480.72
        22.90 -57.76 -59.22 -86.40 633.74 88.64 8.16 0.00
        25471.55
        19.50 -57.16 -58.94 -87.32 665.37 88.75 8.94 0.00
        26486.29
        16.50 -56.36 -58.57 -88.26 700.48 85.03 8.97 0.00
        27544.42
        13.90 -55.26 -58.07 -89.21 739.39 80.54 9.46 0.00
        28635.30
        11.60 -53.76 -57.41 -90.21 783.97 75.00 11.27 0.00
        29792.95
        9.60 -51.96 -56.75 -91.23 834.31 70.51 13.38 0.00
        31013.22
        7.80 -49.66 -56.08 -92.34 894.51 66.47 13.13 0.00
        32364.59
        6.30 -46.66 -55.34 -93.47 963.56 55.12 7.81 0.00
        33771.15
        4.90 -42.66 -54.71 -94.77 1053.58 319.76 6.62 0.00
        35452.00
        3.70 -37.76 -54.37 -96.20 1165.90 295.20 18.24 0.00
        37367.34
        2.70 -30.06 -53.95 -97.77 1317.46 288.32 31.51 0.00
        39573.84
        1.80 -20.06 -54.31 -99.75 1540.13 283.53 48.15 0.00
        42518.32
        1.00 -14.46 -14.46 -102.52 1862.08 281.80 62.70 0.00
        46921.02
        0.30 -24.86 -24.86 -107.87 2521.01 284.33 65.15 0.00
        55854.56

        STID = STNM = 727730 TIME = 170401/0900
        SLAT = 46.92 SLON = -114.08 SELV = 972.0
        STIM = 9

        SHOW = 11.44 LIFT = 12.26 SWET = 26.49 KINX = -1.93
        LCLP = 850.57 PWAT = 9.92 TOTL = 34.11 CAPE = 0.00
        LCLT = 272.07 CINS = 0.00 EQLV = -9999.00 LFCT = -9999.00
        BRCH = 0.00

        PRES TMPC TMWC DWPC THTE DRCT SKNT OMEG
        HGHT
        906.80 3.94 2.11 -0.20 296.97 237.99 3.67 -0.90
        994.39
        901.70 4.34 2.37 -0.08 298.05 240.95 4.00 -0.90
        1040.28
        895.80 4.84 2.68 0.06 299.36 244.65 4.08 -1.00
        1093.79
        889.20 5.24 2.89 0.08 300.54 247.17 4.00 -1.00
        1154.17
        881.80 5.34 2.89 0.00 301.40 248.75 3.75 -1.00
        1222.46
        873.40 5.74 2.97 -0.29 302.54 251.57 3.69 -1.00
        1300.74
        864.10 5.34 2.63 -0.60 302.89 249.78 3.92 -1.00
        1388.30
        853.60 4.84 2.20 -0.99 303.22 254.05 4.23 -1.00
        1488.13
        841.90 4.04 1.56 -1.51 303.25 258.23 4.76 -1.00
        1600.55
        829.00 3.14 0.77 -2.26 303.16 262.15 5.69 -1.00
        1725.93
        814.70 2.24 -0.11 -3.24 303.08 263.29 6.64 -1.10
        1866.75
        799.00 1.34 -1.12 -4.58 302.95 265.91 8.18 -1.20
        2023.69
        781.80 0.64 -2.32 -6.86 302.69 268.92 10.30 -1.30
        2198.66
        762.90 0.14 -3.72 -10.55 302.28 274.54 12.28 -1.30
        2394.91
        742.50 -0.36 -5.24 -15.71 301.89 282.88 13.95 -1.20
        2611.78
        720.50 -0.76 -6.42 -20.88 302.49 290.98 15.19 -1.00
        2851.96
        696.90 -1.16 -6.92 -21.96 304.75 297.86 15.38 -0.80
        3117.46
        671.80 -1.86 -7.16 -19.69 307.96 304.70 15.37 -0.60
        3409.31
        645.20 -2.96 -7.79 -18.70 310.79 312.37 15.00 -0.40
        3729.73
        617.30 -4.46 -8.94 -19.22 313.00 317.73 14.43 -0.30
        4078.66
        588.20 -6.46 -10.64 -21.04 314.54 319.99 14.20 -0.30
        4457.33
        558.30 -8.96 -12.72 -23.02 315.81 321.71 14.10 -0.30
        4862.96
        527.60 -12.16 -15.18 -24.18 316.91 324.89 15.19 -0.20
        5297.97
        496.40 -15.76 -17.94 -24.97 318.00 325.97 18.05 -0.10
        5760.72
        465.10 -19.56 -21.09 -26.70 318.89 324.12 22.53 0.00
        6248.09
        433.80 -23.46 -24.69 -30.35 319.55 319.35 27.14 -0.10
        6761.52
        402.90 -27.56 -28.62 -35.15 320.21 316.25 31.47 -0.20
        7297.36
        372.50 -31.96 -32.85 -40.54 320.96 318.60 34.97 -0.10
        7856.41
        343.10 -36.76 -37.45 -46.06 321.61 324.80 38.75 0.00
        8431.15
        314.60 -41.86 -42.33 -51.17 322.30 329.55 44.83 0.20
        9024.79
        287.40 -47.26 -47.53 -54.62 322.90 329.79 53.26 0.20
        9629.88
        261.50 -52.56 -52.69 -57.74 323.86 327.53 60.78 0.20
        10247.04
        237.00 -57.66 -57.73 -61.69 325.31 324.29 67.23 0.20
        10874.91
        213.90 -62.46 -62.51 -67.60 327.44 321.85 71.39 0.10
        11514.58
        192.50 -65.36 -65.39 -68.36 332.81 323.03 67.83 0.10
        12160.22
        172.50 -65.76 -65.82 -73.90 342.70 324.17 49.11 0.00
        12826.80
        154.10 -63.86 -63.95 -74.65 357.16 317.65 38.64 0.00
        13514.68
        137.10 -60.46 -9999.00 -9999.00 -9999.00 315.79 29.82 0.00
        14236.60
        121.60 -59.76 -9999.00 -9999.00 -9999.00 314.12 26.79 0.00
        14984.75
        107.50 -59.06 -9999.00 -9999.00 -9999.00 313.59 22.26 0.00
        15755.84
        94.70 -58.66 -9999.00 -9999.00 -9999.00 307.00 17.75 0.00
        16551.05
        83.20 -58.26 -9999.00 -9999.00 -9999.00 298.67 14.16 0.00
        17364.65
        72.90 -58.66 -9999.00 -9999.00 -9999.00 297.00 11.56 0.00
        18195.18
        63.70 -59.06 -59.55 -80.30 470.26 303.69 9.81 0.00
        19041.37
        55.50 -58.66 -59.25 -81.15 490.06 320.60 7.03 0.00
        19905.75
        48.20 -58.56 -9999.00 -9999.00 -9999.00 6.01 3.71 0.00
        20791.37
        41.80 -58.96 -59.70 -82.87 530.66 69.44 3.32 0.00
        21685.39
        36.10 -58.96 -59.81 -83.75 553.36 86.82 3.50 0.00
        22604.55
        31.20 -58.36 -59.40 -84.61 578.53 85.03 4.49 0.00
        23520.42
        26.80 -58.16 -59.38 -85.49 604.77 86.19 5.85 0.00
        24476.63
        22.90 -57.96 -59.39 -86.40 633.15 88.45 7.19 0.00
        25466.77
        19.50 -57.46 -59.19 -87.32 664.45 87.40 8.55 0.00
        26480.33
        16.50 -56.56 -58.73 -88.26 699.84 84.40 9.95 0.00
        27537.24
        13.90 -55.46 -58.22 -89.21 738.71 82.01 11.19 0.00
        28627.12
        11.60 -53.86 -57.48 -90.21 783.61 84.56 12.30 0.00
        29783.97
        9.60 -52.16 -56.88 -91.23 833.56 88.29 13.01 0.00
        31003.41
        7.80 -49.96 -56.25 -92.34 893.31 91.04 10.68 0.00
        32353.26
        6.30 -47.16 -55.59 -93.47 961.43 71.57 4.29 0.00
        33757.32
        4.90 -43.36 -55.00 -94.77 1050.38 329.04 6.80 0.00
        35433.76
        3.70 -38.46 -54.61 -96.20 1162.43 307.18 14.14 0.00
        37343.34
        2.70 -30.96 -54.19 -97.77 1312.58 288.73 24.20 0.00
        39542.45
        1.80 -21.46 -54.59 -99.75 1531.61 279.83 39.82 0.00
        42473.29
        1.00 -16.36 -16.36 -102.52 1848.40 277.59 52.91 0.00
        46847.60
        0.30 -26.46 -26.46 -107.87 2504.76 277.71 52.14 0.00
        55719.47

        STID = STNM = 727730 TIME = 170401/1200
        SLAT = 46.92 SLON = -114.08 SELV = 972.0
        STIM = 12

        SHOW = 11.24 LIFT = 12.58 SWET = 27.14 KINX = 3.00
        LCLP = 861.46 PWAT = 10.48 TOTL = 34.65 CAPE = 0.00
        LCLT = 271.91 CINS = 0.00 EQLV = -9999.00 LFCT = -9999.00
        BRCH = 0.00

        PRES TMPC TMWC DWPC THTE DRCT SKNT OMEG
        HGHT
        906.40 2.74 1.32 -0.54 295.43 214.70 3.07 -0.20
        993.41
        901.20 2.94 1.48 -0.41 296.31 220.24 3.30 -0.30
        1040.00
        895.40 3.04 1.56 -0.34 297.10 225.00 3.85 -0.40
        1092.32
        888.80 3.14 1.63 -0.27 297.99 230.53 4.27 -0.50
        1152.30
        881.30 3.24 1.68 -0.26 298.94 234.16 4.31 -0.50
        1221.02
        873.00 3.94 2.02 -0.33 300.57 240.95 4.00 -0.50
        1297.87
        863.60 4.64 2.27 -0.57 302.20 246.04 3.83 -0.50
        1386.02
        853.20 4.44 2.02 -0.90 302.90 249.78 3.92 -0.50
        1484.75
        841.50 3.74 1.44 -1.41 303.06 250.02 4.55 -0.50
        1597.09
        828.60 2.94 0.67 -2.23 303.01 249.68 5.59 -0.60
        1722.42
        814.30 2.14 -0.27 -3.53 302.79 250.02 6.82 -0.60
        1863.23
        798.60 1.54 -1.45 -5.88 302.31 254.41 8.66 -0.70
        2020.26
        781.40 1.24 -2.82 -9.75 301.68 263.88 10.94 -0.70
        2195.52
        762.60 0.94 -4.17 -14.62 301.32 277.54 13.33 -0.60
        2391.25
        742.20 0.64 -5.11 -18.43 302.09 289.56 15.68 -0.40
        2608.86
        720.20 0.24 -5.43 -18.31 304.42 296.28 17.54 -0.20
        2850.01
        696.60 -0.46 -5.54 -15.94 307.57 299.17 19.13 -0.10
        3116.52
        671.50 -1.36 -5.99 -15.02 310.38 300.22 20.45 0.00
        3409.26
        644.90 -2.76 -7.04 -15.50 312.38 299.36 21.39 0.10
        3730.34
        617.00 -4.66 -8.62 -16.94 313.73 296.57 21.29 0.00
        4079.51
        588.00 -6.86 -10.47 -18.72 314.96 290.75 19.74 -0.10
        4456.67
        558.00 -9.46 -12.54 -20.14 316.27 283.13 17.95 -0.20
        4863.22
        527.40 -12.66 -14.97 -21.12 317.38 279.14 17.11 -0.30
        5296.27
        496.20 -16.16 -17.79 -22.61 318.33 284.04 19.23 -0.30
        5758.48
        464.90 -19.56 -20.97 -26.02 319.13 290.40 25.08 -0.30
        6245.71
        433.60 -23.16 -24.56 -31.12 319.81 293.71 31.39 -0.40
        6759.67
        402.70 -27.16 -28.42 -36.49 320.57 295.56 34.67 -0.40
        7296.51
        372.40 -31.76 -32.70 -40.86 321.22 296.98 35.98 -0.40
        7854.61
        343.00 -36.86 -37.44 -44.23 321.65 301.27 37.04 -0.30
        8429.64
        314.50 -42.16 -42.47 -47.35 322.14 310.10 43.43 -0.20
        9022.97
        287.30 -47.26 -47.43 -51.20 323.08 315.81 57.96 0.00
        9627.89
        261.40 -52.36 -52.47 -56.49 324.23 316.82 73.52 0.10
        10245.55
        236.90 -57.76 -57.83 -61.69 325.20 317.06 80.40 0.10
        10873.83
        213.90 -62.66 -62.71 -67.60 327.13 316.17 74.05 0.00
        11510.42
        192.40 -65.46 -65.49 -68.36 332.70 312.57 64.90 0.00
        12158.76
        172.50 -65.76 -65.82 -73.90 342.70 316.07 51.52 0.00
        12822.03
        154.10 -63.36 -9999.00 -9999.00 -9999.00 304.25 38.31 0.00
        13510.73
        137.10 -60.26 -9999.00 -9999.00 -9999.00 306.68 35.12 0.00
        14233.85
        121.60 -59.36 -9999.00 -9999.00 -9999.00 303.89 30.65 0.00
        14983.05
        107.50 -59.06 -9999.00 -9999.00 -9999.00 311.15 26.57 0.00
        15754.86
        94.70 -59.26 -9999.00 -9999.00 -9999.00 306.31 20.01 0.00
        16548.96
        83.20 -58.96 -9999.00 -9999.00 -9999.00 301.30 16.82 0.00
        17360.10
        72.90 -59.06 -9999.00 -9999.00 -9999.00 301.36 14.55 0.00
        18188.50
        63.70 -59.16 -9999.00 -9999.00 -9999.00 308.21 11.62 0.00
        19033.71
        55.50 -58.56 -59.15 -81.15 490.28 317.12 7.42 0.00
        19898.08
        48.20 -58.66 -9999.00 -9999.00 -9999.00 322.43 3.19 0.00
        20783.70
        41.80 -59.26 -59.98 -82.87 529.91 338.20 2.10 0.00
        21676.89
        36.10 -59.06 -59.90 -83.75 553.10 17.35 3.26 0.00
        22595.19
        31.20 -58.56 -59.58 -84.61 577.99 49.64 5.09 0.00
        23510.42
        26.80 -58.46 -59.64 -85.49 603.93 66.37 6.78 0.00
        24465.51
        22.90 -58.16 -59.56 -86.40 632.56 75.32 8.43 0.00
        25454.50
        19.50 -57.56 -59.27 -87.32 664.14 82.48 10.39 0.00
        26467.36
        16.50 -56.66 -58.81 -88.26 699.51 85.46 12.28 0.00
        27523.78
        13.90 -55.46 -58.22 -89.21 738.71 85.73 13.05 0.00
        28613.41
        11.60 -53.96 -57.55 -90.21 783.25 87.32 12.45 0.00
        29769.99
        9.60 -52.36 -57.01 -91.23 832.80 90.00 10.68 0.00
        30988.61
        7.80 -50.36 -56.47 -92.34 891.71 86.91 7.21 0.00
        32336.63
        6.30 -47.76 -55.89 -93.47 958.88 45.00 4.39 0.00
        33737.56
        4.90 -44.06 -55.29 -94.77 1047.18 357.88 5.24 0.00
        35409.22
        3.70 -39.36 -54.92 -96.20 1157.97 299.90 8.95 0.00
        37312.23
        2.70 -32.26 -54.54 -97.77 1305.53 281.41 22.59 0.00
        39501.20
        1.80 -23.16 -54.94 -99.75 1521.26 277.48 38.79 0.00
        42414.23
        1.00 -18.06 -18.06 -102.52 1836.17 275.34 50.14 0.00
        46759.29
        0.30 -27.66 -27.66 -107.87 2492.58 270.91 49.15 0.00
        55580.06"
    }

    #[test]
    fn test_sounding_through_iterators() {
        let test_data = get_valid_test_data();

        let sounding = UpperAirSection::new(test_data);
        assert_eq!(sounding.into_iter().count(), 5);

        assert_eq!(
            sounding
                .into_iter()
                .map(|snd| snd.lead_time)
                .collect::<Vec<i32>>(),
            vec![0, 3, 6, 9, 12]
        );
    }

    #[test]
    fn test_validation() {
        let sounding = UpperAirSection::new(get_valid_test_data());
        assert!(sounding.validate_section().is_ok());

        let sounding = UpperAirSection::new(get_invalid_test_data1());
        assert!(!sounding.validate_section().is_ok());

        let sounding = UpperAirSection::new(get_invalid_test_data2());
        assert!(!sounding.validate_section().is_ok());
    }
}
