//! Adapter for the Generalized TriG parser from [RIO](https://github.com/Tpt/rio/blob/master/turtle/src/gtrig.rs)

use rio_turtle::{GTriGParser as RioGTriGParser, TurtleError};
use sophia_api::parser::QuadParser;
use sophia_rio::parser::*;
use std::io::BufRead;

/// TriG parser based on RIO.
#[derive(Clone, Debug, Default)]
pub struct GTriGParser {
    /// The base IRI used by this parser to resolve relative IRI-references.
    pub base: Option<String>,
}

impl<B: BufRead> QuadParser<B> for GTriGParser {
    type Source = GeneralizedRioSource<RioGTriGParser<B>, TurtleError>;
    fn parse(&self, data: B) -> Self::Source {
        // TODO issue TurtleError if base can not be parsed
        let base = self.base.clone().and_then(|b| oxiri::Iri::parse(b).ok());
        GeneralizedRioSource::Parser(RioGTriGParser::new(data, base))
    }
}

sophia_api::def_mod_functions_for_bufread_parser!(GTriGParser, QuadParser);

// ---------------------------------------------------------------------------------
//                                      tests
// ---------------------------------------------------------------------------------

#[cfg(test)]
mod test {
    use super::*;
    use sophia_api::dataset::Dataset;
    use sophia_api::ns::rdf;
    use sophia_api::quad::stream::QuadSource;
    use sophia_api::term::matcher::ANY;
    use sophia_inmem::dataset::FastDataset;
    use sophia_term::StaticTerm;

    #[test]
    fn test_simple_gtrig_string() -> std::result::Result<(), Box<dyn std::error::Error>> {
        let gtrig = r#"
            @prefix : <http://example.org/ns/> .

            <#g1> {
                <#me> :knows _:alice.
            }
            <#g2> {
                _:alice a :Person ; :name ?name.
            }
        "#;

        let mut d = FastDataset::new();
        let p = GTriGParser { base: None };
        let c = p.parse_str(&gtrig).add_to_dataset(&mut d)?;
        assert_eq!(c, 3);
        assert!(d
            .quads_matching(
                &StaticTerm::new_iri("#me").unwrap(),
                &StaticTerm::new_iri("http://example.org/ns/knows").unwrap(),
                &ANY,
                &Some(&StaticTerm::new_iri("#g1").unwrap()),
            )
            .next()
            .is_some());
        assert!(d
            .quads_matching(
                &ANY,
                &rdf::type_,
                &StaticTerm::new_iri("http://example.org/ns/Person").unwrap(),
                &Some(&StaticTerm::new_iri("#g2").unwrap()),
            )
            .next()
            .is_some());
        assert!(d
            .quads_matching(
                &ANY,
                &StaticTerm::new_iri("http://example.org/ns/name").unwrap(),
                &StaticTerm::new_variable("name").unwrap(),
                &Some(&StaticTerm::new_iri("#g2").unwrap()),
            )
            .next()
            .is_some());
        Ok(())
    }
}
