//! A Rust implementation of the Stateless OpenPGP Protocol.

use std::{
    fmt,
    io,
    time::SystemTime,
};

#[cfg(feature = "cli")]
pub mod cli;

/// The Stateless OpenPGP Protocol.
pub trait SOP {
    /// Gets version information.
    fn version<'a>(&'a self) -> Result<Box<dyn Version + 'a>>;

    /// Generates a Secret Key.
    ///
    /// Customize the operation using the builder [`GenerateKey`].
    fn generate_key<'a>(&'a self) -> Result<Box<dyn GenerateKey + 'a>>;

    /// Extracts a Certificate from a Secret Key.
    ///
    /// Customize the operation using the builder [`ExtractCert`].
    fn extract_cert<'a>(&'a self) -> Result<Box<dyn ExtractCert + 'a>>;

    /// Creates Detached Signatures.
    ///
    /// Customize the operation using the builder [`Sign`].
    fn sign<'a>(&'a self) -> Result<Box<dyn Sign + 'a>>;

    /// Verifies Detached Signatures.
    ///
    /// Customize the operation using the builder [`Verify`].
    fn verify<'a>(&'a self) -> Result<Box<dyn Verify + 'a>>;

    /// Encrypts a Message.
    ///
    /// Customize the operation using the builder [`Encrypt`].
    fn encrypt<'a>(&'a self) -> Result<Box<dyn Encrypt + 'a>>;

    /// Decrypts a Message.
    ///
    /// Customize the operation using the builder [`Decrypt`].
    fn decrypt<'a>(&'a self) -> Result<Box<dyn Decrypt + 'a>>;

    /// Converts binary OpenPGP data to ASCII.
    ///
    /// Customize the operation using the builder [`Armor`].
    fn armor<'a>(&'a self) -> Result<Box<dyn Armor + 'a>>;

    /// Converts ASCII OpenPGP data to binary.
    ///
    /// Customize the operation using the builder [`Dearmor`].
    fn dearmor<'a>(&'a self) -> Result<Box<dyn Dearmor + 'a>>;
}

pub trait Version<'a> {
    /// Returns the implementation's name.
    fn name(&self) -> String;

    /// Returns the implementation's version string.
    fn version(&self) -> String;
}

/// Builder for [`SOP::generate_key`].
pub trait GenerateKey<'a> {
    /// Disables armor encoding.
    fn no_armor(self: Box<Self>) -> Box<dyn GenerateKey<'a> + 'a>;

    /// Adds a User ID.
    fn userid(self: Box<Self>, userid: &str) -> Box<dyn GenerateKey<'a> + 'a>;

    /// Generates the OpenPGP key.
    fn generate(self: Box<Self>) -> Result<Box<dyn Ready + 'a>>;
}

/// Builder for [`SOP::extract_cert`].
pub trait ExtractCert<'a> {
    /// Disables armor encoding.
    fn no_armor(self: Box<Self>) -> Box<dyn ExtractCert<'a> + 'a>;

    /// Extracts the cert from `key`.
    fn key(self: Box<Self>, key: &mut (dyn io::Read + Send + Sync))
           -> Result<Box<dyn Ready + 'a>>;
}

/// Builder for [`SOP::sign`].
pub trait Sign<'a> {
    /// Disables armor encoding.
    fn no_armor(self: Box<Self>) -> Box<dyn Sign<'a> + 'a>;

    /// Sets signature mode.
    fn mode(self: Box<Self>, mode: SignAs) -> Box<dyn Sign<'a> + 'a>;

    /// Adds the signer key.
    fn key(self: Box<Self>, key: &mut (dyn io::Read + Send + Sync))
           -> Result<Box<dyn Sign<'a> + 'a>>;

    /// Signs data.
    fn data(self: Box<Self>, data: &'a mut (dyn io::Read + Send + Sync))
        -> Result<Box<dyn Ready + 'a>>;
}

/// Builder for [`SOP::verify`].
pub trait Verify<'a> {
    /// Makes SOP consider signatures before this date invalid.
    fn not_before(self: Box<Self>, t: SystemTime) -> Box<dyn Verify<'a> + 'a>;

    /// Makes SOP consider signatures after this date invalid.
    fn not_after(self: Box<Self>, t: SystemTime) -> Box<dyn Verify<'a> + 'a>;

    /// Adds the verification cert.
    fn cert(self: Box<Self>, cert: &mut (dyn io::Read + Send + Sync))
            -> Result<Box<dyn Verify<'a> + 'a>>;

    /// Provides the signatures.
    fn signatures(self: Box<Self>,
                  signatures: &'a mut (dyn io::Read + Send + Sync))
                  -> Result<Box<dyn VerifySignatures + 'a>>;
}

/// Builder for [`SOP::verify`].
pub trait VerifySignatures {
    /// Verifies the authenticity of `data`.
    fn data(self: Box<Self>,
            data: &mut (dyn io::Read + Send + Sync))
            -> Result<Vec<Verification>>;
}

/// Represents a successful signature verification.
#[derive(Debug, PartialEq, Eq, PartialOrd, Ord, Hash)]
pub struct Verification {
    creation_time: SystemTime,
    signing_key_fingerprint: String,
    signing_cert_fingerprint: String,
    message: Option<String>,
}

#[cfg(feature = "cli")]
impl fmt::Display for Verification {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        write!(f,
               "{} {} {}{}",
               chrono::DateTime::<chrono::Utc>::from(self.creation_time())
               .format("%Y-%m-%dT%H:%M:%SZ"),
               self.signing_key_fingerprint(),
               self.signing_cert_fingerprint(),
               if let Some(m) = self.message() {
                   format!(" {}", m)
               } else {
                   "".into()
               })
    }
}

impl Verification {
    /// Creates a `Verification` object.
    pub fn new<'m, T, K, C, M>(creation_time: T,
                               signing_key_fingerprint: K,
                               signing_cert_fingerprint: C,
                               message: M)
                           -> Result<Verification>
    where T: Into<SystemTime>,
          K: ToString,
          C: ToString,
          M: Into<Option<&'m str>>,
    {
        fn normalize(s: String) -> Result<String> {
            // XXX
            Ok(s)
        }
        let signing_key_fingerprint =
            normalize(signing_key_fingerprint.to_string())?;
        let signing_cert_fingerprint =
            normalize(signing_cert_fingerprint.to_string())?;

        Ok(Verification {
            creation_time: creation_time.into(),
            signing_key_fingerprint,
            signing_cert_fingerprint,
            message: message.into().map(Into::into),
        })
    }

    /// Returns the signature's creation time.
    pub fn creation_time(&self) -> SystemTime {
        self.creation_time
    }

    /// Returns the fingerprint of the signing (sub)key.
    pub fn signing_key_fingerprint(&self) -> &str {
        &self.signing_key_fingerprint
    }

    /// Returns the fingerprint of the signing certificate.
    pub fn signing_cert_fingerprint(&self) -> &str {
        &self.signing_cert_fingerprint
    }

    /// Returns a free-form message describing the verification.
    pub fn message(&self) -> Option<&str> {
        self.message.as_ref().map(AsRef::as_ref)
    }
}

/// Builder for [`SOP::encrypt`].
pub trait Encrypt<'a> {
    /// Disables armor encoding.
    fn no_armor(self: Box<Self>) -> Box<dyn Encrypt<'a> + 'a>;

    /// Sets encryption mode.
    fn mode(self: Box<Self>, mode: EncryptAs) -> Box<dyn Encrypt<'a> + 'a>;

    /// Adds the signer key.
    fn sign_with(self: Box<Self>, key: &mut (dyn io::Read + Send + Sync))
                 -> Result<Box<dyn Encrypt<'a> + 'a>>;

    /// Encrypts with the given password.
    fn with_password(self: Box<Self>, password: &str)
                     -> Result<Box<dyn Encrypt<'a> + 'a>>;

    /// Encrypts with the given cert.
    fn with_cert(self: Box<Self>, cert: &mut (dyn io::Read + Send + Sync))
                 -> Result<Box<dyn Encrypt<'a> + 'a>>;

    /// Encrypts the given data yielding the ciphertext.
    fn plaintext(self: Box<Self>,
                 plaintext: &'a mut (dyn io::Read + Send + Sync))
        -> Result<Box<dyn Ready + 'a>>;
}

/// Builder for [`SOP::decrypt`].
pub trait Decrypt<'a> {
    /// Makes SOP consider signatures before this date invalid.
    fn verify_not_before(self: Box<Self>,
                         t: SystemTime)
                         -> Box<dyn Decrypt<'a> + 'a>;

    /// Makes SOP consider signatures after this date invalid.
    fn verify_not_after(self: Box<Self>,
                        t: SystemTime)
                        -> Box<dyn Decrypt<'a> + 'a>;

    /// Adds the verification cert.
    fn verify_with_cert(self: Box<Self>,
                        cert: &mut (dyn io::Read + Send + Sync))
                        -> Result<Box<dyn Decrypt<'a> + 'a>>;

    /// Tries to decrypt with the given session key.
    fn with_session_key(self: Box<Self>, sk: SessionKey)
                        -> Result<Box<dyn Decrypt<'a> + 'a>>;

    /// Tries to decrypt with the given password.
    fn with_password(self: Box<Self>, password: &str)
                     -> Result<Box<dyn Decrypt<'a> + 'a>>;

    /// Adds the decryption key.
    fn with_key(self: Box<Self>, key: &mut (dyn io::Read + Send + Sync))
                -> Result<Box<dyn Decrypt<'a> + 'a>>;

    /// Decrypts `ciphertext`, returning verification results and
    /// plaintext.
    fn ciphertext(self: Box<Self>,
                  ciphertext: &'a mut (dyn io::Read + Send + Sync))
                  -> Result<Box<dyn ReadyWithResult<(Option<SessionKey>,
                                                     Vec<Verification>)> + 'a>>;
}

/// Builder for [`SOP::armor`].
pub trait Armor<'a> {
    /// Overrides automatic detection of label.
    fn label(self: Box<Self>, label: ArmorLabel) -> Box<dyn Armor<'a> + 'a>;

    /// Armors `data`.
    fn data(self: Box<Self>, data: &'a mut (dyn io::Read + Send + Sync))
        -> Result<Box<dyn Ready + 'a>>;
}

/// Builder for [`SOP::dearmor`].
pub trait Dearmor<'a> {
    /// Dearmors `data`.
    fn data(self: Box<Self>, data: &'a mut (dyn io::Read + Send + Sync))
        -> Result<Box<dyn Ready + 'a>>;
}

/// Normalizes the given password.
///
/// See [Passwords are Human-Readable] in the SOP spec.
///
///   [Passwords are Human-Readable]: https://datatracker.ietf.org/doc/html/draft-dkg-openpgp-stateless-cli-02#section-7.8
pub fn normalize_password<P: AsRef<str>>(p: P) -> String {
    // XXX: Maybe do additional checks.
    p.as_ref().trim_end().into()
}

/// An operation ready to be executed.
///
/// To execute the operation, either supply an [`std::io::Write`]r
/// using [`Ready::write_to`] to write the resulting data to, or use
/// [`Ready::to_vec`] to write to a `Vec<u8>`.
pub trait Ready {
    /// Executes the operation writing the result to `sink`.
    fn write_to(self: Box<Self>, sink: &mut (dyn io::Write + Send + Sync))
        -> Result<()>;

    /// Executes the operation writing the result into a `Vec<u8>`.
    fn to_vec(self: Box<Self>) -> Result<Vec<u8>> {
        let mut v = Vec::new();
        self.write_to(&mut v)?;
        Ok(v)
    }
}

/// An operation that returns a value ready to be executed.
///
/// To execute the operation, either supply an [`std::io::Write`]r
/// using [`Ready::write_to`] to write the resulting data to, or use
/// [`Ready::to_vec`] to write to a `Vec<u8>`.
pub trait ReadyWithResult<T> {
    /// Executes the operation writing the result to `sink`.
    fn write_to(self: Box<Self>, sink: &mut (dyn io::Write + Send + Sync))
        -> Result<T>;

    /// Executes the operation writing the result into a `Vec<u8>`.
    fn to_vec(self: Box<Self>) -> Result<(T, Vec<u8>)> {
        let mut v = Vec::new();
        let r = self.write_to(&mut v)?;
        Ok((r, v))
    }
}

/// Represents a session key.
pub struct SessionKey {
    algorithm: u8,
    key: Box<[u8]>,
}

impl SessionKey {
    /// Creates a new session key object.
    pub fn new<A, K>(algorithm: A, key: K) -> Result<Self>
    where A: Into<u8>,
          K: AsRef<[u8]>,
    {
        // XXX: Maybe sanity check key lengths.
        Ok(SessionKey {
            algorithm: algorithm.into(),
            key: key.as_ref().to_vec().into(),
        })
    }

    /// Returns the symmetric algorithm octet.
    pub fn algorithm(&self) -> u8 {
        self.algorithm
    }

    /// Returns the session key.
    pub fn key(&self) -> &[u8] {
        &self.key
    }
}

impl fmt::Display for SessionKey {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        write!(f, "{}:", self.algorithm)?;
        for b in &self.key[..] {
            write!(f, "{:02X}", b)?
        }
        Ok(())
    }
}

impl std::str::FromStr for SessionKey {
    type Err = ParseError;
    fn from_str(sk: &str) -> ParseResult<Self> {
        // The SOP format is:
        //
        //   <decimal-cipher-octet> ":" <hex-session-key>
        let fields = sk.rsplit(':').collect::<Vec<_>>();

        if fields.len() != 2 {
            return Err(ParseError(format!(
                "Expected two colon-separated fields, got {:?}",
                fields)));
        }

        let algo: u8 = fields[0].parse().map_err(
            |e| ParseError(format!("Failed to parse algorithm: {}", e)))?;
        let sk = from_hex(&fields[1], true)?;
        Self::new(algo, sk).map_err(
            |e| ParseError(format!("Bad session key: {}", e)))
    }
}

/// A helpful function for converting a hexadecimal string to binary.
/// This function skips whitespace if `pretty` is set.
fn from_hex(hex: &str, pretty: bool) -> ParseResult<Vec<u8>> {
    const BAD: u8 = 255u8;
    const X: u8 = 'x' as u8;

    let mut nibbles = hex.chars().filter_map(|x| {
        match x {
            '0' => Some(0u8),
            '1' => Some(1u8),
            '2' => Some(2u8),
            '3' => Some(3u8),
            '4' => Some(4u8),
            '5' => Some(5u8),
            '6' => Some(6u8),
            '7' => Some(7u8),
            '8' => Some(8u8),
            '9' => Some(9u8),
            'a' | 'A' => Some(10u8),
            'b' | 'B' => Some(11u8),
            'c' | 'C' => Some(12u8),
            'd' | 'D' => Some(13u8),
            'e' | 'E' => Some(14u8),
            'f' | 'F' => Some(15u8),
            'x' | 'X' if pretty => Some(X),
            _ if pretty && x.is_whitespace() => None,
            _ => Some(BAD),
        }
    }).collect::<Vec<u8>>();

    if pretty && nibbles.len() >= 2 && nibbles[0] == 0 && nibbles[1] == X {
        // Drop '0x' prefix.
        nibbles.remove(0);
        nibbles.remove(0);
    }

    if nibbles.iter().any(|&b| b == BAD || b == X) {
        // Not a hex character.
        return Err(ParseError("Invalid characters".into()));
    }

    // We need an even number of nibbles.
    if nibbles.len() % 2 != 0 {
        return Err(ParseError("Odd number of nibbles".into()));
    }

    let bytes = nibbles.chunks(2).map(|nibbles| {
        (nibbles[0] << 4) | nibbles[1]
    }).collect::<Vec<u8>>();

    Ok(bytes)
}

/// Signature type.
///
/// This is used by [`SOP::sign`] to select the signature type.  See
/// [`sop sign`].
///
///   [`sop sign`]: https://datatracker.ietf.org/doc/html/draft-dkg-openpgp-stateless-cli-02#section-3.4
#[derive(Clone, Copy, Debug)]
pub enum SignAs {
    Binary,
    Text,
}

impl Default for SignAs {
    fn default() -> Self {
        SignAs::Binary
    }
}

impl From<EncryptAs> for SignAs {
    fn from(a: EncryptAs) -> Self {
        match a {
            EncryptAs::Binary => SignAs::Binary,
            EncryptAs::Text => SignAs::Text,
            // XXX: We should inspect the serialized MIME structure
            // and use Text if it is UTF-8, Binary otherwise.  But, we
            // cannot be bothered at this point.
            EncryptAs::MIME => SignAs::Binary,
        }
    }
}

impl std::str::FromStr for SignAs {
    type Err = ParseError;
    fn from_str(s: &str) -> std::result::Result<Self, Self::Err> {
        match s {
            "binary" => Ok(SignAs::Binary),
            "text" => Ok(SignAs::Text),
            _ => Err(ParseError(format!(
                "{:?}, expected one of {{binary|text}}", s))),
        }
    }
}

impl fmt::Display for SignAs {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        match self {
            SignAs::Binary => f.write_str("binary"),
            SignAs::Text => f.write_str("text"),
        }
    }
}

/// Plaintext data format.
///
/// This is used by [`SOP::encrypt`] to select the data format.  See
/// [`sop encrypt`].
///
///   [`sop encrypt`]: https://datatracker.ietf.org/doc/html/draft-dkg-openpgp-stateless-cli-02#section-3.6
#[derive(Clone, Copy, Debug)]
pub enum EncryptAs {
    Binary,
    Text,
    MIME,
}

impl Default for EncryptAs {
    fn default() -> Self {
        EncryptAs::Binary
    }
}

impl std::str::FromStr for EncryptAs {
    type Err = ParseError;
    fn from_str(s: &str) -> std::result::Result<Self, Self::Err> {
        match s {
            "binary" => Ok(EncryptAs::Binary),
            "text" => Ok(EncryptAs::Text),
            "mime" => Ok(EncryptAs::MIME),
            _ => Err(ParseError(format!(
                "{}, expected one of {{binary|text|mime}}", s))),
        }
    }
}

impl fmt::Display for EncryptAs {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        match self {
            EncryptAs::Binary => f.write_str("binary"),
            EncryptAs::Text => f.write_str("text"),
            EncryptAs::MIME => f.write_str("mime"),
        }
    }
}

/// The ASCII Armor Label.
///
/// This is used by [`SOP::armor`] to control the framing that is
/// emitted.  See [`sop armor`].
///
///   [`sop armor`]: https://datatracker.ietf.org/doc/html/draft-dkg-openpgp-stateless-cli-02#section-3.8
#[derive(Clone, Copy, Debug)]
pub enum ArmorLabel {
    Auto,
    Sig,
    Key,
    Cert,
    Message,
}

impl Default for ArmorLabel {
    fn default() -> Self {
        ArmorLabel::Auto
    }
}

impl std::str::FromStr for ArmorLabel {
    type Err = ParseError;
    fn from_str(s: &str) -> std::result::Result<Self, Self::Err> {
        match s {
            "auto" => Ok(ArmorLabel::Auto),
            "sig" => Ok(ArmorLabel::Sig),
            "key" => Ok(ArmorLabel::Key),
            "cert" => Ok(ArmorLabel::Cert),
            "message" => Ok(ArmorLabel::Message),
            _ => Err(ParseError(format!(
                "{:?}, expected one of \
                 {{auto|sig|key|cert|message}}", s))),
        }
    }
}

impl fmt::Display for ArmorLabel {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        match self {
            ArmorLabel::Auto => f.write_str("auto"),
            ArmorLabel::Sig => f.write_str("sig"),
            ArmorLabel::Key => f.write_str("key"),
            ArmorLabel::Cert => f.write_str("cert"),
            ArmorLabel::Message => f.write_str("message"),
        }
    }
}

/// Result specialization.
pub type Result<T> = std::result::Result<T, Error>;

/// SOP errors.
///
/// These are the errors [defined] by the Stateless OpenPGP Protocol.
///
///   [defined]: https://datatracker.ietf.org/doc/html/draft-dkg-openpgp-stateless-cli-02#section-6
#[derive(thiserror::Error, Debug)]
pub enum Error {
    /// No acceptable signatures found ("sop verify").
    #[error("No acceptable signatures found")]
    NoSignature,

    /// Asymmetric algorithm unsupported ("sop encrypt").
    #[error("Asymmetric algorithm unsupported")]
    UnsupportedAsymmetricAlgo,

    /// Certificate not encryption-capable (e.g., expired, revoked,
    /// unacceptable usage flags) ("sop encrypt").
    #[error("Certificate not encryption-capable")]
    CertCannotEncrypt,

    /// Certificate not signing-capable (e.g., expired, revoked,
    /// unacceptable usage flags) ("sop sign").
    #[error("Certificate not signing-capable")]
    CertCannotSign,

    /// Missing required argument.
    #[error("Missing required argument")]
    MissingArg,

    /// Incomplete verification instructions ("sop decrypt").
    #[error("Incomplete verification instructions")]
    IncompleteVerification,

    /// Unable to decrypt ("sop decrypt").
    #[error("Unable to decrypt")]
    CannotDecrypt,

    /// Non-"UTF-8" or otherwise unreliable password ("sop encrypt").
    #[error("Non-UTF-8 or otherwise unreliable password")]
    PasswordNotHumanReadable,

    /// Unsupported option.
    #[error("Unsupported option")]
    UnsupportedOption,

    /// Invalid data type (no secret key where "KEY" expected, etc).
    #[error("Invalid data type")]
    BadData,

    /// Non-text input where text expected.
    #[error("Non-text input where text expected")]
    ExpectedText,

    /// Output file already exists.
    #[error("Output file already exists")]
    OutputExists,

    /// Input file does not exist.
    #[error("Input file does not exist")]
    MissingInput,

    /// A "KEY" input is protected (locked) with a password, and "sop" cannot
    /// unlock it.
    #[error("A KEY input is protected with a password")]
    KeyIsProtected,

    /// A indirect input parameter is a special designator (it starts with
    /// "@"), and a filename matching the designator is actually present.
    #[error("A indirect input parameter is a special designator matches file")]
    AmbiguousInput,

    /// Operation not implemented.
    #[error("Operation not implemented")]
    NotImplemented,

    /// An IO error occurred.
    #[error("IO error")]
    IoError(#[from] std::io::Error),
}

/// Errors during parsing of SOP string representations.
///
/// For types with a defined string representation, we implement
/// [`std::str::FromStr`] for parsing.  This type is used to report
/// errors during parsing.
#[derive(thiserror::Error, Debug)]
#[error("Invalid argument: {}", _0)]
pub struct ParseError(String);

/// Convenience alias.
type ParseResult<T> = std::result::Result<T, ParseError>;
