use solana_program::{
    account_info::{next_account_info, AccountInfo},
    entrypoint::ProgramResult,
    msg,
    program_error::ProgramError,
    program_pack::{IsInitialized},
    pubkey::Pubkey,
    sysvar::{rent::Rent, Sysvar},
};

use crate::{error::TokenWhitelistError, instruction::TokenWhitelistInstruction, state::TokenWhitelist};

pub struct Processor;
impl Processor {
    pub fn process(
        program_id: &Pubkey,
        accounts: &[AccountInfo],
        instruction_data: &[u8],
    ) -> ProgramResult {
        let instruction = TokenWhitelistInstruction::unpack(instruction_data)?;

        match instruction {
            TokenWhitelistInstruction::InitTokenWhitelist { max_whitelist_size } => {
                msg!("Instruction: InitTokenWhitelist");
                Self::process_init_whitelist(
                    accounts,
                    max_whitelist_size,
                    program_id
                )
            }
            TokenWhitelistInstruction::AddToWhitelist {} => {
                msg!("Instruction: AddToWhitelist");
                Self::process_add_whitelist(
                    accounts,
                    program_id
                )
            }
            TokenWhitelistInstruction::RemoveFromWhitelist {} => {
                msg!("Instruction: RemoveFromWhitelist");
                Self::process_remove_whitelist(
                    accounts,
                    program_id
                )
            }
        }
    }

    fn process_init_whitelist(
        accounts: &[AccountInfo],
        max_whitelist_size: u64,
        _program_id: &Pubkey,
    ) -> ProgramResult {
        let account_info_iter = &mut accounts.iter();

        let whitelist_owner = next_account_info(account_info_iter)?;
        if !whitelist_owner.is_signer {
            return Err(ProgramError::MissingRequiredSignature);
        }

        let token_whitelist_account = next_account_info(account_info_iter)?;

        let sysvar_rent_pubkey = &Rent::from_account_info(next_account_info(account_info_iter)?)?;
        if !sysvar_rent_pubkey.is_exempt(token_whitelist_account.lamports(), token_whitelist_account.data_len()) {
            msg!("token whitelist account must be rent exempt");
            return Err(TokenWhitelistError::NotRentExempt.into());
        }

        let mut token_whitelist_state = TokenWhitelist::unpack_from_slice(&token_whitelist_account.data.borrow())?;
        if token_whitelist_state.is_initialized() {
            msg!("token whitelist already initialized");
            return Err(ProgramError::AccountAlreadyInitialized);
        }

        token_whitelist_state.is_initialized = true;
        token_whitelist_state.init_pubkey = *whitelist_owner.key;
        token_whitelist_state.max_whitelist_size = max_whitelist_size;

        token_whitelist_state.pack_into_slice(&mut token_whitelist_account.data.borrow_mut());

        Ok(())
    }

    fn process_add_whitelist(
        accounts: &[AccountInfo],
        _program_id: &Pubkey,
    ) -> ProgramResult {
        let account_info_iter = &mut accounts.iter();

        let whitelist_owner = next_account_info(account_info_iter)?;
        if !whitelist_owner.is_signer {
            return Err(ProgramError::MissingRequiredSignature);
        }

        let token_whitelist_account = next_account_info(account_info_iter)?;
        let account_to_add = next_account_info(account_info_iter)?;

        let mut token_whitelist_state = TokenWhitelist::unpack_from_slice(&token_whitelist_account.data.borrow())?;
        if !token_whitelist_state.is_initialized() {
            msg!("token whitelist needs to be initialized before attempting to add");
            return Err(TokenWhitelistError::TokenWhitelistNotInit.into());
        }
        if whitelist_owner.key != &token_whitelist_state.init_pubkey {
            msg!("signer must be whitelist owner");
            msg!("{}", whitelist_owner.key);
            msg!("{}", token_whitelist_state.init_pubkey);
            return Err(TokenWhitelistError::TokenWhitelistNotOwner.into());
        }

        let whitelist_amount: u64 = 250;
        token_whitelist_state.add_keypair(&account_to_add.key.to_string(), &whitelist_amount);
        token_whitelist_state.pack_into_slice(&mut token_whitelist_account.data.borrow_mut());

        Ok(())
    }

    fn process_remove_whitelist(
        accounts: &[AccountInfo],
        _program_id: &Pubkey,
    ) -> ProgramResult {
        let account_info_iter = &mut accounts.iter();

        let whitelist_owner = next_account_info(account_info_iter)?;
        if !whitelist_owner.is_signer {
            return Err(ProgramError::MissingRequiredSignature);
        }

        let token_whitelist_account = next_account_info(account_info_iter)?;
        let account_to_remove = next_account_info(account_info_iter)?;

        let mut token_whitelist_state = TokenWhitelist::unpack_from_slice(&token_whitelist_account.data.borrow())?;
        if !token_whitelist_state.is_initialized() {
            msg!("token whitelist needs to be initialized before attempting to remove");
            return Err(TokenWhitelistError::TokenWhitelistNotInit.into());
        }
        if whitelist_owner.key != &token_whitelist_state.init_pubkey {
            msg!("signer must be whitelist owner");
            msg!("{}", whitelist_owner.key);
            msg!("{}", token_whitelist_state.init_pubkey);
            return Err(TokenWhitelistError::TokenWhitelistNotOwner.into());
        }

        token_whitelist_state.drop_key(&account_to_remove.key.to_string());
        token_whitelist_state.pack_into_slice(&mut token_whitelist_account.data.borrow_mut());

        Ok(())
    }
}
