#[macro_use]
extern crate log;

mod cli;

use anyhow::anyhow;
use lpc55::bootloader::Bootloader;
use solo2;

fn main() {
    pretty_env_logger::init_custom_env("SOLO2_LOG");
    info!("solo2 CLI startup");

    let args = cli::cli().get_matches();

    if let Err(err) = try_main(args) {
        eprintln!("Error: {}", err);
        std::process::exit(1);
    }
}

fn try_main(args: clap::ArgMatches<'_>) -> anyhow::Result<()> {
    if let Some(args) = args.subcommand_matches("app") {
        use solo2::apps::App;

        if let Some(args) = args.subcommand_matches("mgmt") {
            info!("interacting with management app");
            use solo2::apps::management::App as ManagementApp;
            if args.subcommand_matches("aid").is_some() {
                println!("{}", hex::encode(ManagementApp::aid()).to_uppercase());
                return Ok(());
            }

            let mut app = ManagementApp::new()?;
            let answer_to_select = app.select()?;
            info!("answer to select: {}", &hex::encode(answer_to_select));

            if args.subcommand_matches("boot-to-bootrom").is_some() {
                app.boot_to_bootrom()?;
            }
            if args.subcommand_matches("reboot").is_some() {
                info!("attempting reboot");
                app.reboot()?;
            }
            if args.subcommand_matches("uuid").is_some() {
                let uuid = app.uuid()?;
                println!("{}", hex::encode_upper(uuid.to_be_bytes()));
            }
            if args.subcommand_matches("version").is_some() {
                let version = app.version()?;
                println!("{:?}", version);
            }
        }

        if let Some(args) = args.subcommand_matches("ndef") {
            info!("interacting with NDEF app");
            use solo2::apps::ndef::App as NdefApp;
            if args.subcommand_matches("aid").is_some() {
                println!("{}", hex::encode(NdefApp::aid()).to_uppercase());
                return Ok(());
            }

            let mut app = NdefApp::new()?;
            app.select()?;

            if args.subcommand_matches("capabilities").is_some() {
                let capabilities = app.capabilities()?;
                println!("{}", hex::encode(capabilities));
            }
            if args.subcommand_matches("data").is_some() {
                let data = app.data()?;
                println!("{}", hex::encode(data));
            }
        }

        if let Some(args) = args.subcommand_matches("piv") {
            info!("interacting with PIV app");
            use solo2::apps::piv::App;
            if args.subcommand_matches("aid").is_some() {
                println!("{}", hex::encode(App::aid()).to_uppercase());
                return Ok(());
            }

            let mut app = App::new()?;
            app.select()?;
        }

        if let Some(args) = args.subcommand_matches("provisioner") {
            info!("interacting with Provisioner app");
            use solo2::apps::provisioner::App;
            if args.subcommand_matches("aid").is_some() {
                println!("{}", hex::encode(App::aid()).to_uppercase());
                return Ok(());
            }

            let mut app = App::new()?;
            app.select()?;

            if args.subcommand_matches("generate-ed255-key").is_some() {
                let public_key = app.generate_trussed_ed255_attestation_key()?;
                println!("{}", hex::encode(public_key));
            }
            if args.subcommand_matches("generate-p256-key").is_some() {
                let public_key = app.generate_trussed_p256_attestation_key()?;
                println!("{}", hex::encode(public_key));
            }
            if args.subcommand_matches("reformat-filesystem").is_some() {
                app.reformat_filesystem()?;
            }
            if let Some(args) = args.subcommand_matches("store-ed255-cert") {
                let cert_file = args.value_of("DER").unwrap();
                let certificate = std::fs::read(cert_file)?;
                app.store_trussed_ed255_attestation_certificate(&certificate)?;
            }
            if let Some(args) = args.subcommand_matches("store-p256-cert") {
                let cert_file = args.value_of("DER").unwrap();
                let certificate = std::fs::read(cert_file)?;
                app.store_trussed_p256_attestation_certificate(&certificate)?;
            }
            if args.subcommand_matches("uuid").is_some() {
                let uuid = app.uuid()?;
                println!("{}", hex::encode_upper(uuid.to_be_bytes()));
            }
            if let Some(args) = args.subcommand_matches("write-file") {
                let file = args.value_of("DATA").unwrap();
                let data = std::fs::read(file)?;
                let path = args.value_of("PATH").unwrap();
                app.write_file(&data, &path)?;
            }
        }

        if let Some(args) = args.subcommand_matches("tester") {
            info!("interacting with Tester app");
            use solo2::apps::tester::App;
            if args.subcommand_matches("aid").is_some() {
                println!("{}", hex::encode(App::aid()).to_uppercase());
                return Ok(());
            }

            let mut app = App::new()?;
            app.select()?;
        }
    }

    #[cfg(not(feature = "dev-pki"))]
    if args.subcommand_matches("dev-pki").is_some() {
        return Err(anyhow!(
            "Compile with `--features dev-pki` for dev PKI support!"
        ));
    }
    #[cfg(feature = "dev-pki")]
    if let Some(args) = args.subcommand_matches("dev-pki") {
        if let Some(args) = args.subcommand_matches("fido") {
            let (aaguid, key_trussed, key_pem, cert) = solo2::dev_pki::generate_selfsigned_fido();

            info!("\n{}", key_pem);
            info!("\n{}", cert.serialize_pem()?);

            std::fs::write(args.value_of("KEY").unwrap(), &key_trussed)?;
            std::fs::write(args.value_of("CERT").unwrap(), &cert.serialize_der()?)?;

            println!("{}", hex::encode_upper(aaguid));
        }
    }

    if let Some(args) = args.subcommand_matches("bootloader") {
        let bootloader = || {
            Bootloader::try_find(None, None, None)
                .ok_or(anyhow!("Could not attach to a bootloader"))
        };

        if args.subcommand_matches("reboot").is_some() {
            let bootloader = bootloader()?;
            bootloader.reboot();
        }
        if args.subcommand_matches("ls").is_some() {
            let bootloaders = Bootloader::list();
            for bootloader in bootloaders {
                println!("{:?}", &bootloader);
            }
        }
    }

    // if let Some(command) = args.subcommand_matches("provision") {
    //     let config_filename = command.value_of("CONFIG").unwrap();
    //     let config = lpc55::bootloader::provision::Config::try_from(config_filename)?;

    //     let bootloader = bootloader()?;
    //     for cmd in config.provisions {
    //         info!("cmd: {:?}", cmd);
    //         bootloader.run_command(cmd)?;
    //     }

    //     return Ok(());
    // }

    Ok(())
}
