//! Overly simplistic "PKI" to enable all functionality of Solo 2 apps.
//!
//! In particular, there is no root CA, no use of hardware keys / PKCS #11,
//! no revocation, etc. etc.

use rand_core::{OsRng, RngCore};

pub fn generate_selfsigned_fido(nfc: bool) -> rcgen::Certificate {
    let alg = &rcgen::PKCS_ECDSA_P256_SHA256;

    // 0. generate AAGUID
    let mut aaguid = [0u8; 16];
    OsRng.fill_bytes(&mut aaguid);

    // 1. generate a keypair
    let keypair = rcgen::KeyPair::generate(alg).unwrap();

    // 2. generate self-signed certificate

    let mut tbs = rcgen::CertificateParams::default();
    tbs.alg = alg;
    tbs.serial_number = Some(OsRng.next_u64());

    let now = chrono::Utc::now();
    tbs.not_before = now;
    tbs.not_after = now + chrono::Duration::days(50 * 365);

    let mut subject = rcgen::DistinguishedName::new();
    subject.push(rcgen::DnType::CountryName, "AQ");
    subject.push(rcgen::DnType::OrganizationName, "Example Org");
    subject.push(
        rcgen::DnType::OrganizationalUnitName,
        "Example FIDO Attestation",
    );
    subject.push(rcgen::DnType::CommonName, "example.com");
    tbs.distinguished_name = subject;

    tbs.key_pair = Some(keypair);
    tbs.is_ca = rcgen::IsCa::SelfSignedOnly;
    // TODO: check if `authorityKeyIdentifier=keyid,issuer` is both needed
    tbs.use_authority_key_identifier_extension = true;

    let mut extensions = Vec::new();
    // AAGUID
    extensions.push(rcgen::CustomExtension::from_oid_content(
        // FIDO's PEN + 1.1.4
        &[1, 3, 6, 1, 4, 1, 45724, 1, 1, 4],
        yasna::construct_der(|writer| writer.write_bytes(aaguid.as_ref())),
    ));
    // Transports
    extensions.push(rcgen::CustomExtension::from_oid_content(
        // FIDO's PEN + 2.1.1
        &[1, 3, 6, 1, 4, 1, 45724, 2, 1, 1],
        {
            let mut bits = 0u8;
            // USB
            bits |= 1 << 2;
            if nfc {
                bits |= 1 << 3;
            }
            yasna::construct_der(|writer| writer.write_bitvec_bytes(&[bits], 4))
        },
    ));
    tbs.custom_extensions = extensions;

    let cert = rcgen::Certificate::from_params(tbs).unwrap();

    cert
}
