use std::convert::TryInto;

use headers::HeaderValue;
use http::header::{self, InvalidHeaderValue};
use http::StatusCode;
use hyper::Body;

use crate::{IntoResponse, Response};

#[derive(Debug, Clone)]
pub struct Redirect {
    status: StatusCode,
    location: HeaderValue,
}

impl Redirect {
    pub fn see_other(
        location: impl TryInto<HeaderValue, Error = InvalidHeaderValue>,
    ) -> Result<Self, InvalidHeaderValue> {
        Ok(Redirect {
            status: StatusCode::SEE_OTHER,
            location: location.try_into()?,
        })
    }

    pub fn temporary(
        location: impl TryInto<HeaderValue, Error = InvalidHeaderValue>,
    ) -> Result<Self, InvalidHeaderValue> {
        Ok(Redirect {
            status: StatusCode::TEMPORARY_REDIRECT,
            location: location.try_into()?,
        })
    }

    pub fn permanent(
        location: impl TryInto<HeaderValue, Error = InvalidHeaderValue>,
    ) -> Result<Self, InvalidHeaderValue> {
        Ok(Redirect {
            status: StatusCode::PERMANENT_REDIRECT,
            location: location.try_into()?,
        })
    }
}

impl IntoResponse for Redirect {
    fn into_response(self) -> Response {
        let mut res = Response::new(Body::empty());
        *res.status_mut() = self.status;
        let headers = res.headers_mut();
        headers.insert(header::LOCATION, self.location);
        res
    }
}

impl IntoResponse for InvalidHeaderValue {
    fn into_response(self) -> Response {
        #[cfg(feature = "tracing")]
        tracing::error!(err = %self, "redirect location is not a valid header value");
        StatusCode::INTERNAL_SERVER_ERROR.into_response()
    }
}
