#![forbid(unsafe_code)]
// #![warn(missing_docs)]

pub mod body;
mod redirect;
mod request;
pub mod response;
mod result_ext;
pub mod route;

use std::convert::{Infallible, TryFrom};
use std::future::Future;
use std::pin::Pin;
use std::task::{Context, Poll};

pub use body::Body;
use headers::HeaderMapExt;
pub use http;
pub use hyper;
use hyper::body::HttpBody;
use hyper::service::Service;
pub use redirect::Redirect;
pub use request::{DecodeQueryError, Request, RequestExt};
pub use response::{IntoResponse, Response};
pub use result_ext::ResultExt;

#[derive(Clone)]
pub struct SolarSail<S, H> {
    state: S,
    handler: H,
}

impl<S, H> SolarSail<S, H> {
    pub fn new(state: S, handler: H) -> Self {
        SolarSail { state, handler }
    }

    async fn serve<F>(&self, req: Request) -> http::Response<hyper::Body>
    where
        S: Clone,
        H: Fn(S, Request) -> F,
        F: Future<Output = Response>,
    {
        (self.handler)(self.state.clone(), req)
            .await
            .into_response()
    }

    #[cfg(feature = "server")]
    pub async fn run<F>(self, addr: &std::net::SocketAddr) -> Result<(), hyper::Error>
    where
        S: Clone + Send + Sync + 'static,
        H: Fn(S, Request) -> F + Clone + Send + Sync + 'static,
        F: Future<Output = Response> + Send + 'static,
    {
        let server = hyper::Server::bind(addr).serve(self);
        println!("Listening on http://{}", addr);
        server.await
    }

    #[cfg(feature = "server")]
    pub async fn run_in<F, L, B>(
        self,
        addr: &std::net::SocketAddr,
        layer: L,
    ) -> Result<(), hyper::Error>
    where
        S: Clone + Send + Sync + 'static,
        H: Fn(S, Request) -> F + Clone + Send + Sync + 'static,
        F: Future<Output = Response> + Send + 'static,
        L: tower_layer::Layer<Self>,
        L::Service: Service<http::Request<hyper::Body>, Response = http::Response<B>>
            + Clone
            + Send
            + Sync
            + 'static,
        <L::Service as Service<http::Request<hyper::Body>>>::Future: Send,
        <L::Service as Service<http::Request<hyper::Body>>>::Error: std::error::Error + Send + Sync,
        B: HttpBody + Send + Sync + 'static,
        B::Data: Send,
        B::Error: Into<Box<dyn std::error::Error + Send + Sync>> + Send + Sync,
    {
        let svc = layer.layer(self);
        let svc = hyper::service::make_service_fn(move |_| {
            let svc = svc.clone();
            async move { Ok::<_, std::convert::Infallible>(svc) }
        });
        let server = hyper::Server::bind(addr).serve(svc);
        println!("Listening on http://{}", addr);
        server.await
    }
}

type BoxTrySendFuture<R, E> = Pin<Box<dyn Future<Output = Result<R, E>> + Send>>;

impl<S, H, F> Service<http::Request<hyper::Body>> for SolarSail<S, H>
where
    S: Clone + Send + Sync + 'static,
    H: Fn(S, Request) -> F + Clone + Send + Sync + 'static,
    F: Future<Output = Response> + Send + 'static,
{
    type Response = http::Response<hyper::Body>;
    type Error = Infallible;
    // TODO: get rid of Box?
    type Future = BoxTrySendFuture<Self::Response, Self::Error>;

    fn poll_ready(&mut self, _: &mut Context<'_>) -> Poll<Result<(), Self::Error>> {
        Poll::Ready(Ok(()))
    }

    fn call(&mut self, req: http::Request<hyper::Body>) -> Self::Future {
        let (parts, body) = req.into_parts();
        let body = if body.is_end_stream() {
            Body::empty()
        } else {
            Body::new(
                body,
                parts
                    .headers
                    .typed_get()
                    .map(|h: headers::ContentLength| usize::try_from(h.0).unwrap_or(usize::MAX)),
                parts.headers.get(http::header::CONTENT_TYPE).cloned(),
            )
        };

        let svc = self.clone();
        let req = Request::from_parts(parts, body);
        Box::pin(async move { Ok(svc.serve(req).await) })
    }
}

#[cfg(feature = "server")]
impl<S, H> Service<&hyper::server::conn::AddrStream> for SolarSail<S, H>
where
    S: Clone,
    H: Clone,
{
    type Response = SolarSail<S, H>;
    type Error = hyper::Error;
    type Future = std::future::Ready<Result<Self::Response, Self::Error>>;

    fn poll_ready(&mut self, _: &mut Context<'_>) -> Poll<Result<(), Self::Error>> {
        Poll::Ready(Ok(()))
    }

    fn call(&mut self, _: &hyper::server::conn::AddrStream) -> Self::Future {
        std::future::ready(Ok(self.clone()))
    }
}
