use crate::rejection::Rejection;
use crate::response::Response;
use crate::IntoResponse;

pub trait ResultExt<T, E: std::error::Error + Send + Sync + 'static + IntoResponse> {
    fn map_err_response<F>(self, f: F) -> Result<T, MapErrResponse<E, F>>
    where
        F: FnOnce(Response) -> Response;
}

#[derive(thiserror::Error)]
#[error("{inner}")]
pub struct MapErrResponse<E: std::error::Error + Send + Sync + 'static, F> {
    #[source]
    inner: E,
    map: F,
}

impl<T, E> ResultExt<T, E> for Result<T, E>
where
    E: std::error::Error + Send + Sync + 'static + IntoResponse,
{
    fn map_err_response<F>(self, f: F) -> Result<T, MapErrResponse<E, F>>
    where
        F: FnOnce(Response) -> Response,
    {
        self.map_err(|err| MapErrResponse { inner: err, map: f })
    }
}

impl<E, F> IntoResponse for MapErrResponse<E, F>
where
    E: std::error::Error + Send + Sync + 'static + IntoResponse,
    F: FnOnce(Response) -> Response,
{
    fn into_response(self) -> Result<Response, Rejection> {
        let res = self.inner.into_response()?;
        let res = (self.map)(res);
        Ok(res)
    }
}

impl<E, F> std::fmt::Debug for MapErrResponse<E, F>
where
    E: std::error::Error + Send + Sync + 'static,
{
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        f.debug_struct("MapErrResponse")
            .field("inner", &self.inner)
            // TODO: finish_non_exhaustive
            .finish()
    }
}
