use crate::body::Body;
use crate::IntoResponse;
use http::{Method, StatusCode};
use serde::Deserialize;

pub type Request = http::Request<Body>;

pub trait RequestExt: Sized {
    fn route(&self) -> Route<'_>;
    fn query<'de, T: Deserialize<'de>>(&'de self) -> Result<T, serde_urlencoded::de::Error>;
}

pub struct Route<'a> {
    method: &'a Method,
    segments: Vec<&'a str>,
}

impl<'a> Route<'a> {
    pub fn as_tuple(&'a self) -> (&'a Method, &'a [&'a str]) {
        (self.method, self.segments.as_slice())
    }
}

impl RequestExt for Request {
    fn route(&self) -> Route<'_> {
        let path = cleanup_path(self.uri().path());

        Route {
            method: self.method(),
            segments: if path.is_empty() {
                Vec::new()
            } else {
                path.split('/').collect()
            },
        }
    }

    fn query<'de, T: Deserialize<'de>>(&'de self) -> Result<T, serde_urlencoded::de::Error> {
        let uri = self.uri();
        let query = uri.query();
        serde_urlencoded::from_str(query.unwrap_or(""))
    }
}

fn cleanup_path(segment: &str) -> &str {
    // remove leading `/`
    let segment = segment.strip_prefix('/').unwrap_or(segment);
    // remove trailing `/`
    let segment = segment.strip_suffix('/').unwrap_or(segment);

    segment
}

impl IntoResponse for serde_urlencoded::de::Error {
    fn into_response(self) -> Result<crate::Response, crate::rejection::Rejection> {
        StatusCode::BAD_REQUEST.into_response()
    }
}

#[cfg(test)]
mod tests {
    use super::RequestExt;
    use crate::Body;
    use http::Method;

    #[test]
    fn test_root_path() {
        let req = http::Request::builder()
            .uri("https://example.org/")
            .body(Body::empty())
            .unwrap();
        assert_eq!(req.route().as_tuple(), (&Method::GET, &[] as &[&str]));
    }
}
