use http::StatusCode;

use crate::response::{IntoResponse, Response};

pub struct Rejection {
    cause: Cause,
}

pub enum Cause {
    Err(Box<dyn IntoResponseError + Send + Sync>),
    Status(StatusCode),
}

#[derive(Debug, thiserror::Error)]
#[error("failed with status {status}")]
pub struct StatusError {
    status: StatusCode,
    #[source]
    cause: Option<Box<dyn std::error::Error + Send + Sync>>,
}

pub fn not_found(err: impl Into<Box<dyn std::error::Error + Send + Sync>>) -> StatusError {
    StatusError {
        status: StatusCode::NOT_FOUND,
        cause: Some(err.into()),
    }
}

pub fn internal_server_error(
    err: impl Into<Box<dyn std::error::Error + Send + Sync>>,
) -> StatusError {
    StatusError {
        status: StatusCode::NOT_FOUND,
        cause: Some(err.into()),
    }
}

impl Rejection {
    pub fn into_response(self) -> Result<Response, Rejection> {
        match self.cause {
            Cause::Err(err) => err.into_response_error(),
            Cause::Status(status) => status.into_response(),
        }
    }

    pub fn downcast_ref<T: 'static>(&self) -> Option<&T> {
        if let Cause::Err(err) = &self.cause {
            return err.as_any().downcast_ref();
        }

        None
    }

    pub fn status(status: StatusCode) -> Self {
        Rejection {
            cause: Cause::Status(status),
        }
    }

    pub fn not_found() -> Self {
        Self::status(StatusCode::NOT_FOUND)
    }

    pub fn into_cause(self) -> Cause {
        self.cause
    }
}

impl<E> From<E> for Rejection
where
    E: IntoResponseError + Send + Sync + 'static,
{
    fn from(err: E) -> Self {
        Rejection {
            cause: Cause::Err(Box::new(err)),
        }
    }
}

pub trait IntoResponseError: std::error::Error {
    fn into_response_error(self: Box<Self>) -> Result<Response, Rejection>;
    fn as_any(&self) -> &dyn std::any::Any;
    fn into_err(self: Box<Self>) -> Box<dyn std::error::Error + Send + Sync>;
}

impl<E> IntoResponseError for E
where
    E: IntoResponse + std::error::Error + Send + Sync + 'static,
{
    fn into_response_error(self: Box<Self>) -> Result<Response, Rejection> {
        self.into_response()
    }

    fn as_any(&self) -> &dyn std::any::Any {
        self
    }

    fn into_err(self: Box<Self>) -> Box<dyn std::error::Error + Send + Sync> {
        self
    }
}

impl std::error::Error for Rejection {
    fn source(&self) -> Option<&(dyn std::error::Error + 'static)> {
        if let Cause::Err(err) = &self.cause {
            return err.source();
        }
        None
    }
}

impl std::fmt::Display for Rejection {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &self.cause {
            Cause::Err(err) => err.fmt(f),
            Cause::Status(status) => status.fmt(f),
        }
    }
}

impl std::fmt::Debug for Rejection {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        f.debug_struct("Rejection").finish()
    }
}

impl IntoResponse for StatusError {
    fn into_response(self) -> Result<Response, Rejection> {
        self.status.into_response()
    }
}
