use headers::{ContentType, HeaderMapExt};
use http::StatusCode;
use hyper::Body;
use serde::Serialize;

use crate::rejection::Rejection;

pub type Response = http::Response<Body>;

pub struct Json<T>(pub T);

pub fn json<T: Serialize>(data: T) -> Json<T> {
    Json(data)
}

pub trait IntoResponse {
    fn into_response(self) -> Result<Response, Rejection>;
}

impl IntoResponse for Response {
    fn into_response(self) -> Result<Response, Rejection> {
        Ok(self)
    }
}

impl IntoResponse for &'static str {
    fn into_response(self) -> Result<Response, Rejection> {
        http::Response::builder()
            .body(Body::from(self))
            .map_err(Rejection::from)
    }
}

impl IntoResponse for String {
    fn into_response(self) -> Result<Response, Rejection> {
        http::Response::builder()
            .body(Body::from(self))
            .map_err(Rejection::from)
    }
}

impl IntoResponse for StatusCode {
    fn into_response(self) -> Result<Response, Rejection> {
        http::Response::builder()
            .status(self)
            .body(Body::empty())
            .map_err(Rejection::from)
    }
}

impl IntoResponse for () {
    fn into_response(self) -> Result<Response, Rejection> {
        http::Response::builder()
            .body(Body::empty())
            .map_err(Rejection::from)
    }
}

impl IntoResponse for http::Error {
    fn into_response(self) -> Result<Response, Rejection> {
        StatusCode::INTERNAL_SERVER_ERROR.into_response()
    }
}

impl IntoResponse for serde_json::Error {
    fn into_response(self) -> Result<Response, Rejection> {
        StatusCode::INTERNAL_SERVER_ERROR.into_response()
    }
}

impl<T> IntoResponse for Json<T>
where
    T: Serialize,
{
    fn into_response(self) -> Result<Response, Rejection> {
        let data = serde_json::to_vec(&self.0)?;
        let mut res = http::Response::builder();
        if let Some(headers) = res.headers_mut() {
            headers.typed_insert(ContentType::json())
        }
        Ok(res.body(Body::from(data))?)
    }
}

impl IntoResponse for http::response::Builder {
    fn into_response(self) -> Result<Response, Rejection> {
        self.body(Body::empty()).map_err(Rejection::from)
    }
}
