use anchor_lang::prelude::*;
declare_id!("Fg6PaFpoGXkYsidMpWTK6W2BeZ7FEfcYkg476zPFsLnS");


#[program]
pub mod rando {
    use super::*;

    pub fn initialize(ctx: Context<Initialize>, authority: Pubkey, oracles: [Pubkey; 2], bump: u8) -> ProgramResult {
        msg!("Instruction: Initialize");
        ctx.accounts.state.authority = authority;

        if oracles.len() != ctx.accounts.state.oracles.len() {
            return Err(ErrorCode::IncorrectOracleLength.into())
        }

        for n in 0..ctx.accounts.state.oracles.len() {
            ctx.accounts.state.oracles[n] = oracles[n]
        }
        Ok(())
    }

    pub fn request(ctx: Context<MakeRequest>, bump: u8) -> ProgramResult {
        msg!("Instruction: Request");
        let vault = &mut ctx.accounts.vault;
        vault.nonce = bump;
        vault.request_reference = ctx.accounts.request_reference.key();
        emit!(NewRequestEvent {
            request_reference: *ctx.accounts.request_reference.key
        });

        Ok(())
    }

    pub fn fill_oracle_result(ctx: Context<FillOracleResult>, oracle_result: [u8; 64], bump: u8) -> ProgramResult {
        msg!("Instruction: Fill Oracle Result");
        let mut oracle_index = 17;

        /// fetch oracle's index in the array of oracles.
        for i in 0 .. ctx.accounts.state.oracles.len() {
            if ctx.accounts.state.oracles[i] == *ctx.accounts.oracle.key {
                oracle_index = i;
            }
        }

        /// oracle not found yo, what are you feeding me?
        if oracle_index == 17 {
            return Err(ErrorCode::IncorrectOracleProvided.into())
        }

        /// Already filled yo =)
        if ctx.accounts.vault.done_flag[oracle_index] == 1 {
            return Err(ErrorCode::AlreadyFilled.into())
        }

        /// TODO: Verify signature

        /// fill every n number, n being # of oracles there are.
        /// if n is two, the first oracle will fill index 0,2,4...16
        /// and second oracle will fill index 1,3,5,...15;
        for n in (oracle_index .. 16).step_by(ctx.accounts.state.oracles.len()) {
            ctx.accounts.vault.result[n] = oracle_result[n];
        }

        for n in 0 .. 64 {
            ctx.accounts.vault.oracle_results[oracle_index * 64 + n] = oracle_result[n]
        }

        ctx.accounts.vault.done_flag[oracle_index] = 1;

        /// On finished filling all array, fill the result
        let sum: u8 = ctx.accounts.vault.done_flag.iter().sum();
        if sum == ctx.accounts.vault.done_flag.len() as u8{
            ctx.accounts.vault.numeric_result = u128::from_be_bytes(ctx.accounts.vault.result);
        }

        Ok(())
    }
}

#[derive(Accounts)]
#[instruction(authority: Pubkey, oracles: [Pubkey; 2], bump: u8)]
pub struct Initialize<'info> {
    pub authority: Signer<'info>,
    #[account(init, seeds = [b"state".as_ref()], bump = bump, payer = authority)]
    pub state: ProgramAccount<'info, RandoState>,
    pub system_program: Program<'info, System>,
}

#[derive(Accounts)]
#[instruction(bump: u8)]
pub struct MakeRequest<'info> {
    pub requester: Signer<'info>,
    pub request_reference: AccountInfo<'info>,
    #[account(init, seeds = [b"request-reference-seed", request_reference.key.as_ref()], bump = bump, payer = requester)]
    pub vault: ProgramAccount<'info, RandoResult>,
    pub system_program: Program<'info, System>,
    pub rent: Sysvar<'info, Rent>,
}

#[derive(Accounts)]
#[instruction(oracle_result: [u8; 64], bump: u8)]
pub struct FillOracleResult<'info> {
    pub oracle: Signer<'info>,
    pub request_reference: AccountInfo<'info>,
    #[account(mut, seeds = [b"request-reference-seed", request_reference.key.as_ref()], bump = bump)]
    pub vault: ProgramAccount<'info, RandoResult>,
    pub state: ProgramAccount<'info, RandoState>,
    pub system_program: Program<'info, System>,
    pub rent: Sysvar<'info, Rent>,
}

#[account]
#[derive(Default)]
pub struct RandoState {
    pub authority: Pubkey,
    pub oracles: [Pubkey; 2]
}

#[account]
pub struct RandoResult {
    pub nonce: u8,
    pub request_reference: Pubkey,
    pub oracle_results: [u8;128],
    pub result: [u8; 16],
    pub done_flag: [u8; 2],
    pub numeric_result: u128
}

#[event]
pub struct NewRequestEvent {
    pub request_reference: Pubkey,
}

impl Default for RandoResult {
    fn default() -> Self {
        Self {
            nonce: 0,
            request_reference: Pubkey::new_from_array([0;32]),
            oracle_results: [0;128],
            result: [0; 16],
            done_flag: [0; 2],
            numeric_result: 0
        }
    }
}

#[error]
pub enum ErrorCode {
    #[msg("Incorrect Oracle length between state and provided")]
    IncorrectOracleLength,
    #[msg("Incorrect Oracle Pubkey provided")]
    IncorrectOracleProvided,
    #[msg("Already filled")]
    AlreadyFilled,
}
