use anchor_lang::prelude::*;
use std::mem::size_of;

declare_id!("2LXeKGTxVXwGpxvqLFgHzJyG4CFHXtBCKHXB6LPPv4N4");

#[program]
mod sol_rng {
    use super::*;

    const ORACLE_FEE: u64 = 495000; //  Approximately $0.09 = 0.000000001 * $175 * 495,000

    pub fn initialize(
        ctx: Context<Initialize>,
        request_bump: u8
    ) -> ProgramResult {
        let requester = &mut ctx.accounts.requester.load_init()?;
        let clock: Clock = Clock::get().unwrap();

        requester.authority = *ctx.accounts.authority.key;
        requester.oracle = *ctx.accounts.oracle.key;
        requester.created_at = clock.unix_timestamp;
        requester.count = 0;
        requester.active_request = false;
        requester.last_updated = clock.unix_timestamp;
        requester.bump = request_bump;

        Ok(())
    }

    pub fn request_random(
        ctx: Context<RequestRandom>,
    ) -> ProgramResult {
        let requester_account_info = ctx.accounts.requester.to_account_info();
        let requester = &mut ctx.accounts.requester.load_mut()?;
        let authority = ctx.accounts.authority.key();

        if requester.authority != authority {
            return Err(ErrorCode::Unauthorized.into());
        }

        if requester.oracle != ctx.accounts.oracle.key() {
            return Err(ErrorCode::WrongOracle.into());
        }

        if requester.active_request {
            return Err(ErrorCode::InflightRequest.into());
        }

        let ix = anchor_lang::solana_program::system_instruction::transfer(
            &authority,
            &ctx.accounts.oracle.key(),
            ORACLE_FEE,
        );

        anchor_lang::solana_program::program::invoke(
            &ix,
            &[
                requester_account_info,
                ctx.accounts.authority.to_account_info(),
                ctx.accounts.oracle.clone(),
                ctx.accounts.system_program.to_account_info()
            ]
        )?;

        let clock: Clock = Clock::get().unwrap();

        requester.last_updated = clock.unix_timestamp;
        requester.active_request = true;
        requester.count += 1;
        
        Ok(())
    }

    pub fn publish_random(
        ctx: Context<PublishRandom>,
        random: [u8; 64],
        pkt_id: [u8; 32],
        tls_id: [u8; 32],
    ) -> ProgramResult {
        let loader: Loader<Requester> = Loader::try_from_unchecked(ctx.program_id, &ctx.remaining_accounts[0]).unwrap();
        let mut requester = loader.load_mut()?;

        if requester.oracle != ctx.accounts.oracle.key() {
            return Err(ErrorCode::Unauthorized.into());
        }

        if !requester.active_request {
            return Err(ErrorCode::AlreadyCompleted.into())
        }
        let clock: Clock = Clock::get().unwrap();

        requester.last_updated = clock.unix_timestamp;
        requester.active_request = false;
        requester.random = random;
        requester.pkt_id = pkt_id;
        requester.tls_id = tls_id;

        Ok(())
    }
}


#[derive(Accounts)]
#[instruction(request_bump: u8)]
pub struct Initialize<'info> {
    #[account(
        init, 
        seeds = [b"r-seed".as_ref(), authority.key().as_ref()],
        bump = request_bump,
        payer = authority,
        space = 8 + size_of::<Requester>()
    )]
    pub requester: AccountLoader<'info, Requester>,
    pub authority: Signer<'info>,
    pub oracle: AccountInfo<'info>,
    pub rent: Sysvar<'info, Rent>,
    pub system_program: Program<'info, System>,
}

#[account(zero_copy)]
pub struct Requester {
    pub authority: Pubkey,
    pub oracle: Pubkey,
    pub created_at: i64,
    pub count: u64,
    pub last_updated: i64,
    pub random: [u8; 64],
    pub pkt_id: [u8; 32],
    pub tls_id: [u8; 32],
    pub active_request: bool,
    pub bump: u8,
}

#[derive(Accounts)]
pub struct RequestRandom<'info> {
    #[account(mut)]
    pub requester: AccountLoader<'info, Requester>,
    #[account(signer, mut)]
    pub authority: AccountInfo<'info>,
    #[account(mut)]
    pub oracle: AccountInfo<'info>,
    pub system_program: Program<'info, System>,
}

#[derive(Accounts)]
pub struct PublishRandom<'info> {
    pub oracle: Signer<'info>,
    pub system_program: Program<'info, System>,
}

#[error]
pub enum ErrorCode {
    #[msg("You are not authorized to complete this transaction")]
    Unauthorized,
    #[msg("You have already completed this transaction")]
    AlreadyCompleted,
    #[msg("A request is already in progress. Only one request may be made at a time")]
    InflightRequest,
    #[msg("The Oracle you make the request with must be the same as initialization")]
    WrongOracle,
}