use std::os::unix::net::UnixDatagram;
use std::os::unix::io::AsRawFd;
use std::fs::remove_file;
use std::path::PathBuf;
use std::io::{Write, ErrorKind};
use std::time::Instant;
use nix::sys::select::{select, FdSet};
use nix::sys::time::{TimeVal, TimeValLike};
use structopt::StructOpt;
use tai64::Tai64N;
use chrono::Local;
use problem::prelude::*;
use problem::result::Result as PResult;
use log::*;
use dgrambuf::DatagramBuf;

/// Copy UNIX domain socket datagrams from one socket to two other sockets.
#[derive(Debug, StructOpt)]
struct Cli {
    /// Verbose mode (-v for INFO, -vv for DEBUG)
    #[structopt(short = "v", long, parse(from_occurrences))]
    pub verbose: isize,

    /// Silence mode (-s for no WARN, -ss for no ERROR)
    #[structopt(short = "s", long, parse(from_occurrences))]
    silent: isize,

    /// Maximum size of datagram, in bytes, received from source socket
    #[structopt(short = "m", long, default_value = "196609")]
    max_datagram_size: usize,

    /// Copy datagrams that are of maximum datagram size (possibly truncated) instead of dropping
    #[structopt(short = "c", long)]
    copy_truncated: bool,

    /// Size of buffered socket datagram ring buffer in bytes; must be larger than maximum datagram size
    /// (--max-datagram-size)
    #[structopt(short = "b", long, default_value = "2097152")]
    buffer_size: usize,

    /// Drop datagrams to buffered socket instead of blocking when the datagram buffer is full
    #[structopt(short = "D", long)]
    allow_drop: bool,

    /// Don't fail if buffered socket cannot be connected to on startup
    #[structopt(short = "C", long)]
    buffered_layzy_connect: bool,

    /// Fail if buffered socket disconnects
    #[structopt(short = "F", long)]
    buffered_fail_disconnect: bool,

    /// Prepend datagram with receive timestamp
    /// (-T for Tai64N (binary, 12 bytes), -TT for RFC3339 with nanoseconds followed by single space, 36 bytes)
    #[structopt(short = "T", long, parse(from_occurrences))]
    buffered_timestamp: usize,

    /// Also copy timestamp to blocking socket (see buffered-timestamp)
    #[structopt(short = "t", long)]
    blocking_timestamp: bool,

    /// Source UNIX domain datagram socket: each datagram is copied to blocking and buffered socket
    source_socket: PathBuf,

    /// Blocking destination UNIX domain datagram socket: no datagrams will be processed until this socket starts accepting datagrams
    blocking_socket: PathBuf,

    /// Buffered destination UNIX domain datagram socket: internal buffer is used to store the datagrams until they can be sent to this socket; may block if buffer fills up (see --allow-drop)
    buffered_socket: PathBuf,
}

#[derive(Debug, Clone, Copy)]
enum DatagramTimestamp {
    Tai64N,
    Rfc3339,
}

impl DatagramTimestamp {
    fn from_occurrences(occurrences: usize) -> PResult<Option<DatagramTimestamp>> {
        Ok(match occurrences {
            0 => None,
            1 => Some(DatagramTimestamp::Tai64N),
            2 => Some(DatagramTimestamp::Rfc3339),
            o => problem!("No timestamp format for occurrences: {}", o)?,
        })
    }

    fn len(&self) -> usize {
        match self {
            DatagramTimestamp::Tai64N => 12,
            DatagramTimestamp::Rfc3339 => 36,
        }
    }

    fn write(&self, mut buf: &mut [u8]) -> Result<(), std::io::Error> {
        assert!(buf.len() >= self.len(), "timestamp write buffer to small");
        match self {
            DatagramTimestamp::Tai64N => buf.write_all(&Tai64N::now().to_bytes())?,
            DatagramTimestamp::Rfc3339 => {
                write!(buf, "{} ", Local::now().format("%Y-%m-%dT%H:%M:%S%.9f%:z"))?;
            },
        }
        assert!(buf.is_empty(), "timestamp write buffer not fully overwritten");
        Ok(())
    }
}

fn main() -> FinalResult {
    let args = Cli::from_args();
    let verbosity = args.verbose + 1 - args.silent;
    stderrlog::new()
        .module(module_path!())
        .module("problem")
        .quiet(verbosity < 0)
        .verbosity(verbosity as usize)
        .timestamp(stderrlog::Timestamp::Microsecond)
        .init()
        .unwrap();

    if args.buffer_size < args.max_datagram_size {
        panic!("buffered-buffer-size must be bigger than than max-datagram-size");
    }

    let timestamp_format = DatagramTimestamp::from_occurrences(args.buffered_timestamp)?;

    if let Some(ts) = &timestamp_format {
        if ts.len() >= args.max_datagram_size {
            panic!("max-datagram-size is too small to fit datagram timestamp");
        }
    }

    let blocking = UnixDatagram::unbound().problem_while("creating blocking socket")?;
    blocking.connect(args.blocking_socket).problem_while("connecting to blocking socket")?;

    let buffered = UnixDatagram::unbound().problem_while("creating buffered socket")?;
    let mut buffered_connected = false;

    match buffered.connect(&args.buffered_socket).problem_while("connecting to buffered socket") {
        Ok(_) => {
            buffered.set_nonblocking(true).problem_while("setting buffered socket blocking")?;
            buffered_connected = true;
        }
        Err(err) => if args.buffered_layzy_connect {
            warn!("Startup: failed to connect buffered socket at {:?}: {}", args.buffered_socket, err);
        } else {
            Err(err)?;
        }
    }

    let mut dgram_buf = Vec::new();
    dgram_buf.resize(args.buffer_size, 0);
    let mut dgram_buf = DatagramBuf::from_slice(&mut dgram_buf);

    let mut datagrams_dropped: u64 = 0;
    let mut buffered_ready = false;
    let mut max_backlog: usize = 0;
    let mut last_connection_attempt = Instant::now();

    remove_file(&args.source_socket).ok();
    let source = UnixDatagram::bind(args.source_socket).problem_while("binding source socket")?;

    loop {
        if buffered_connected && buffered_ready && !dgram_buf.is_empty() {
            // flush buffered datagrams if possible; always try in case select returned on error socket
            // error condition
            let mut max_flush = 10;
            while let Some(buf) = dgram_buf.back() {
                debug!("buffered socket: sending {} bytes datagram", buf.len());
                match buffered.send(buf) {
                    Err(err) => match err.kind() {
                        ErrorKind::WouldBlock => {
                            break;
                        }
                        ErrorKind::ConnectionRefused => {
                            buffered_connected = false;
                            if args.buffered_fail_disconnect {
                                problem!("Backlog: buffered socket connection lost: {}", err)?;
                            } else {
                                error!("Backlog: buffered socket connection lost: {}", err);
                            }
                            break;
                        }
                        _ => Err(err)
                    }
                    Ok(ok) => {
                        dgram_buf.pop_back();
                        if datagrams_dropped > 0 {
                            warn!("Dropped {} datagrams due to buffer overflow", datagrams_dropped);
                            datagrams_dropped = 0;
                        }
                        if dgram_buf.is_empty() && max_backlog > 1 {
                            info!("Backlog: flushed; max buffered: {}", max_backlog);
                            max_backlog = 0;
                        }
                        max_flush -= 1;
                        if max_flush == 0 {
                            // let source be processed
                            trace!("flushing: letting go");
                            break;
                        }
                        Ok(ok)
                    }
                }.problem_while("flushing: sending datagram to buffered socket")?;
            }
        }

        let mut read_set = FdSet::new();
        read_set.insert(source.as_raw_fd());

        let mut write_set = FdSet::new();
        write_set.insert(buffered.as_raw_fd());

        let mut error_set = FdSet::new();
        error_set.insert(source.as_raw_fd());

        if buffered_connected {
            error_set.insert(buffered.as_raw_fd());
        }

        // check if we have new datagram ready to pick; or if we have a backlog if it can be flushed
        if dgram_buf.is_empty() {
            trace!("waiting for source");
            // nothing to flush; wait for new datagram
            select(None, Some(&mut read_set), None, Some(&mut error_set), None).problem_while("select: waiting for source and buffered socket")?;
            // try sent the new datagram out, if connected
            buffered_ready = buffered_connected;
        } else if !buffered_connected {
            trace!("waiting for source or reconnect timeout");
            select(None, Some(&mut read_set), None, Some(&mut error_set), Some(&mut TimeVal::seconds(1))).problem_while("select: waiting for source and buffered socket")?;

            // don't reconnect more often than once per second
            if last_connection_attempt.elapsed().as_secs() >= 1 {
                match buffered.connect(&args.buffered_socket).problem_while("connecting to buffered socket") {
                    Ok(_) => {
                        buffered.set_nonblocking(true).problem_while("setting buffered socket blocking")?;
                        buffered_connected = true;
                        info!("Buffered socket reconnected");
                    }
                    Err(err) => warn!("Failed to connect buffered socket at {:?}: {}", args.buffered_socket, err),
                }
                last_connection_attempt = Instant::now();
            }
        } else {
            trace!("waiting for source and buffered socket");
            select(None, Some(&mut read_set), Some(&mut write_set), Some(&mut error_set), None).problem_while("select: waiting for source and buffered socket")?;
            buffered_ready = write_set.contains(buffered.as_raw_fd()) || error_set.contains(buffered.as_raw_fd());
        }


        if !dgram_buf.is_empty() && (!buffered_ready || !buffered_connected) {
            info!("Backlog: {} datagrams, {}/{} bytes", dgram_buf.len(), dgram_buf.used(), dgram_buf.size());
            if max_backlog < dgram_buf.len() {
                max_backlog = dgram_buf.len()
            }
        }

        // get new datagram
        let source_ready = read_set.contains(source.as_raw_fd()) || error_set.contains(source.as_raw_fd());
        if source_ready {
            // allocate new buffer; may need to send out oldest datagrams to make space
            let buf = loop {
                match dgram_buf.alloc_front(args.max_datagram_size) {
                    Some(buf) => break buf,
                    None => {
                        let buf = dgram_buf.back().expect("failed to allocate space in datagram buffer: buffer empty but fails to allocate");

                        // try to send the oldest datagram
                        let send_ok = if !buffered_connected {
                            false
                        } else {
                            match buffered.send(buf) {
                                Err(err) => match err.kind() {
                                    ErrorKind::WouldBlock => Ok(false),
                                    ErrorKind::ConnectionRefused => {
                                        buffered_connected = false;
                                        if args.buffered_fail_disconnect {
                                            problem!("Overflow: buffered socket connection lost: {}", err)?;
                                        } else {
                                            error!("Overflow: buffered socket connection lost: {}", err);
                                        }
                                        Ok(false)
                                    }
                                    _ => Err(err)
                                }
                                Ok(_) => Ok(true),
                            }.problem_while("overflow: sending datagram to buffered socket")?
                        };

                        if send_ok {
                            // free the datagram
                            dgram_buf.pop_back().unwrap();
                            if datagrams_dropped > 0 {
                                warn!("Dropped {} datagrams due to buffer overflow", datagrams_dropped);
                                datagrams_dropped = 0;
                            }
                        } else if args.allow_drop {
                            // drop the datagram
                            dgram_buf.pop_back().unwrap();
                            datagrams_dropped += 1;
                            debug!("drop: need: {}, now has: {} bytes free", args.max_datagram_size, dgram_buf.size() - dgram_buf.used());
                        } else if !buffered_connected {
                            // block until we get connected
                            loop {
                                match buffered.connect(&args.buffered_socket).problem_while("connecting to buffered socket") {
                                    Ok(_) => {
                                        buffered.set_nonblocking(true).problem_while("setting buffered socket blocking")?;
                                        buffered_connected = true;
                                        info!("Overflow: buffered socket reconnected");
                                        break
                                    }
                                    Err(err) => {
                                        warn!("Overflow: failed to connect buffered socket at {:?}: {}", args.buffered_socket, err);
                                        std::thread::sleep(std::time::Duration::from_secs(1));
                                    }
                                }
                            }
                        } else {
                            let mut write_set = FdSet::new();
                            write_set.insert(buffered.as_raw_fd());
                            let mut error_set = FdSet::new();
                            error_set.insert(buffered.as_raw_fd());
                            warn!("Overflow: waiting for buffered socket");
                            select(None, None, Some(&mut write_set), Some(&mut error_set), None).problem_while("overflow: waiting for buffered socket")?;
                        }
                    }
                }
            };

            // try receive new datagram or loop again for a ready socket
            let dgram_len = if let Some(ts) = &timestamp_format {
                let (ts_buf, dgram_buf) = buf.split_at_mut(ts.len());
                source
                    .recv(dgram_buf)
                    .and_then(|count| {
                        ts.write(ts_buf)?;
                        Ok(ts.len() + count)
                    })
            } else {
                source.recv(buf)
            }.problem_while("receiving datagram from source socket")?;
            debug!("source socket: received {} bytes datagram", if let Some(ts) = &timestamp_format { dgram_len - ts.len() } else { dgram_len });

            // detect if datagram was truncated
            if dgram_len == buf.len() {
                if args.copy_truncated {
                    warn!("Copying truncated datagram");
                } else {
                    error!("Dropping truncated datagram");
                    dgram_buf.pop_front();
                    continue;
                }
            }

            // make allocated buffer to fit the actual datagram length
            let buf = dgram_buf.truncate_front(dgram_len);

            // blocking send to blocking socket
            debug!("blocking socket: sending {} bytes datagram", buf.len());
            match (&timestamp_format, args.blocking_timestamp) {
                (Some(ts), false) => blocking.send(&buf[ts.len()..]),
                _ => blocking.send(buf),
            }.problem_while("sending datagram to blocking socket")?;
        }
    }
}
