use serde::{Deserialize, Serialize};
use serde_json::to_string_pretty;
use ureq;

#[derive(Debug, Clone, Deserialize, Serialize)]
pub enum GenericErr {
    PrimeErr,
    QueryErr,
}

#[derive(Debug, Clone, Deserialize, Serialize)]
pub enum Chain {
    Akash,
    Gaia,
    Injective,
    Juno,
    Osmosis,
    Regen,
    Secret,
    Sentinel,
    Stargaze,
    Terra,
}

#[derive(Debug, Clone, Deserialize, Serialize)]
pub struct ChainRPC {
    pub endpoint: String,
    pub last_response: Option<String>,
}

impl ChainRPC {
    pub fn new_and_launch(
        chain: Chain,
        query: &str,
    ) -> Result<ChainRPC, Box<dyn std::error::Error>> {
        match chain {
            Chain::Gaia => Ok(ChainRPC::gaia(query)),
            Chain::Injective => Ok(ChainRPC::injective(query)),
            Chain::Juno => Ok(ChainRPC::juno(query)),
            Chain::Akash => Ok(ChainRPC::akash(query)),
            Chain::Regen => Ok(ChainRPC::regen(query)),
            Chain::Osmosis => Ok(ChainRPC::osmosis(query)),
            Chain::Secret => Ok(ChainRPC::secret(query)),
            Chain::Sentinel => unimplemented!() ,
            Chain::Stargaze => Ok(ChainRPC::stargaze(query)),
            Chain::Terra => Ok(ChainRPC::terra(query)),
        }
    }

    pub fn launch_from_endpoint(endpoint: &str, q: &str) -> ChainRPC {
        let pretty = format_response(endpoint.clone(), q).unwrap();
        ChainRPC {
            endpoint: endpoint.to_string(),
            last_response: Some(pretty),
        }
    }

    // Chains //////////////////////////////////////////////////////////////////
    pub fn akash(q: &str) -> ChainRPC {
        let endpoint = "https://rpc.akash.forbole.com/";
        ChainRPC::launch_from_endpoint(endpoint, q)
    }

    pub fn gaia(q: &str) -> ChainRPC {
        let endpoint = "https://cosmoshub.validator.network/";
        ChainRPC::launch_from_endpoint(endpoint, q)
    }

    pub fn injective(q: &str) -> ChainRPC {
        let endpoint = "https://injective-rpc.api.chainlayer.network/";
        ChainRPC::launch_from_endpoint(endpoint, q)
    }

    pub fn juno(q: &str) -> ChainRPC {
        let endpoint = "https://rpc-juno.itastakers.com/";
        ChainRPC::launch_from_endpoint(endpoint, q)
    }

    pub fn osmosis(q: &str) -> ChainRPC {
        let endpoint = "https://osmosis-1.technofractal.com/";
        ChainRPC::launch_from_endpoint(endpoint, q)
    }

    pub fn regen(q: &str) -> ChainRPC {
        let endpoint = "http://public-rpc.regen.vitwit.com:26657/";
        ChainRPC::launch_from_endpoint(endpoint, q)
    }

    pub fn secret(q: &str) -> ChainRPC {
        let endpoint = "https://rpc-secret.scrtlabs.com/secret-4/rpc/";
        ChainRPC::launch_from_endpoint(endpoint, q)
    }

    pub fn stargaze(q: &str) -> ChainRPC {
        let endpoint = "https://rpc.stargaze-apis.com/";
        ChainRPC::launch_from_endpoint(endpoint, q)
    }

    pub fn terra(q: &str) -> ChainRPC {
        let endpoint = "https://terra-rpc.easy2stake.com/";
        ChainRPC::launch_from_endpoint(endpoint, q)
    }
}

// launch a query with a formatted response ////////////////////////////////////////////////////////////////
pub fn format_response(e: &str, q: &str) -> Result<String, Box<dyn std::error::Error>> {
    let full = format!("{}{}", e, q);
    let res = ureq::get(&full).call()?.into_string()?;
    let j: serde_json::Value = serde_json::from_str(&res)?;
    let pretty = to_string_pretty(&j)?;
    Ok(pretty)
}