use crate::rngs;
use std::convert::TryInto;

pub const DEFAULT_CHARSET: [char; 64] = [
    'M', 'o', 'd', 'u', 'l', 'e', 'S', 'y', 'm', 'b', 'h', 'a', 's', 'O', 'w', 'n', 'P', 'r', '-',
    '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'N',
    'R', 'V', 'f', 'g', 'c', 't', 'i', 'U', 'v', 'z', '_', 'K', 'q', 'Y', 'T', 'J', 'k', 'L', 'x',
    'p', 'Z', 'X', 'I', 'j', 'Q', 'W',
];

/// Basic function that takes all the parameters and generates NanoId string.
pub fn generate(rng: fn(usize) -> Vec<u8>, charset: &[char], length: usize) -> String {
    assert!(
        charset.len() <= u8::max_value() as usize,
        "charset cannot be longer than the max value of `u8`."
    );

    // First, a bitmask is necessary to generate the ID.
    let mask: usize = charset.len().next_power_of_two() - 1;
    // Note: every hardware random generator call is performance expensive,
    // because the system call for entropy collection takes a lot of time.
    // So, to avoid additional system calls, extra bytes are requested in advance.

    // Next, a step determines how many random bytes to generate.
    // The number of random bytes gets decided upon the ID size, mask,
    // alphabet size.
    let step: usize = 8 * length / 5;

    let mut id: String = String::with_capacity(length);

    loop {
        let bytes = rng(step);

        for &byte in &bytes {
            let byte: usize =
                TryInto::<usize>::try_into(byte).expect("cast `u8` -> `usize` failed") & mask;

            if charset.len() > byte {
                id.push(charset[byte]);

                if id.len() == length {
                    return id;
                }
            }
        }
    }
}
/// NanoId struct for builder approach.
pub struct NanoId<const N: usize> {
    length: usize,
    charset: [char; N],
    rng: fn(usize) -> Vec<u8>,
}

impl<const N: usize> NanoId<N> {
    /// Builder Approach
    /// Takes length, charset, rng (Random Number Generator) and returns NanoId instance.
    pub fn new(length: usize, charset: [char; N], rng: fn(usize) -> Vec<u8>) -> NanoId<N> {
        NanoId {
            length,
            charset,
            rng,
        }
    }

    pub fn set_length(mut self, length: usize) -> Self {
        self.length = length;

        self
    }

    pub fn set_charset(mut self, charset: [char; N]) -> Self {
        self.charset = charset;

        self
    }

    pub fn set_rng(mut self, rng: fn(usize) -> Vec<u8>) -> Self {
        self.rng = rng;

        self
    }

    /// Generates NanoId based on builders setting
    pub fn generate(&self) -> String {
        generate(self.rng, &self.charset, self.length)
    }
}

impl Default for NanoId<64> {
    fn default() -> Self {
        NanoId::new(21, DEFAULT_CHARSET, rngs::default)
    }
}

#[cfg(feature = "macros")]
#[macro_export]
macro_rules! nanoid {
    () => {
        $crate::nanoid::generate($crate::rngs::default, &$crate::nanoid::DEFAULT_CHARSET, 21)
    };

    ($length:tt) => {
        $crate::nanoid::generate(
            $crate::rngs::default,
            &$crate::nanoid::DEFAULT_CHARSET,
            $length,
        )
    };

    ($charset:expr) => {
        $crate::nanoid::generate($crate::rngs::default, $charset, 21)
    };

    ($length:tt, $charset: expr) => {
        $crate::nanoid::generate($crate::rngs::default, $charset, $length)
    };

    ($length:tt, $charset: expr, $rng: expr) => {
        $crate::nanoid::generate($rng, $charset, $length)
    };
}
