use chrono::Utc;
use parking_lot::Mutex;
use std::sync::Arc;

pub struct Snowflake {
    epoch: i64,
    worker_id: i32,
    node_id: i32,
    sequence: i32,
    last_timestamp: Arc<Mutex<i64>>,
}

pub struct SnowflakeData {
    pub timestamp: i64,
    pub worker_id: i32,
    pub node_id: i32,
    pub sequence: i32,
}

impl Snowflake {
    /// Generates Snowflake instance, using the defaults. These could be overriden by calling the
    /// relevant set_* functions.
    pub fn new() -> Snowflake {
        Snowflake {
            epoch: 0,
            worker_id: 1,
            node_id: 1,
            sequence: 0,
            last_timestamp: Arc::new(Mutex::new(0)),
        }
    }

    /// Sets the Epoch (milliseconds) to be used.
    /// Defaults to 0, which is 1970/1/1 00:00:00.
    /// for example, 1577836800000 = 2020/1/1 00:00:00.
    pub fn set_epoch(mut self, epoch: i64) -> Self {
        self.epoch = epoch;
        self
    }

    pub fn set_worker_id(mut self, worker_id: i32) -> Self {
        self.worker_id = worker_id;
        self
    }

    pub fn set_node_id(mut self, node_id: i32) -> Self {
        self.node_id = node_id;
        self
    }
    /// Generates u64 snowflake based on current unix timestamp. retrieves other information based
    /// on Snowflake's configurations.
    pub fn generate(mut self) -> i64 {
        let mut last_timestamp = self.last_timestamp.lock();
        let mut timestamp = self.get_timestamp();

        // Check if we're in same ms
        if timestamp == *last_timestamp {
            self.sequence = (self.sequence + 1) & (-1 ^ (-1 << 12));
            if self.sequence == 0 && timestamp <= *last_timestamp {
                timestamp = self.get_timestamp();
            }
        } else {
            self.sequence = 0;
        }
        *last_timestamp = timestamp;

        let data = SnowflakeData {
            timestamp,
            worker_id: self.worker_id,
            node_id: self.node_id,
            sequence: self.sequence,
        };
        Snowflake::assemble(data)
    }

    /// Assembles SnowflakeData to u64 snowflake.
    /// This function only performs Bitwise operations & expects the user to handle collisions, etc
    /// In most cases, you probably don't want to use this.
    pub fn assemble(data: SnowflakeData) -> i64 {
        (data.timestamp << 22) | ((data.worker_id << 17) as i64) | ((data.node_id << 12) as i64) | (data.sequence as i64)
    }

    /// Disassembles u64 snowflake to SnowflakeData.
    pub fn disassemble(&self, id: i64) -> SnowflakeData {
        SnowflakeData {
            timestamp: ((id >> 22) + self.epoch),
            worker_id: ((id & 0x003E_0000) >> 17) as i32,
            node_id: ((id & 0x1F000) >> 12) as i32,
            sequence: (id & 0xFFF) as i32
        }
    }

    /// Gets timestamp to be used, based on Snowflake's epoch configuration.
    fn get_timestamp(&self) -> i64 {
        Utc::now().timestamp_millis() - self.epoch
    }
}
