// Copyright (C) 2019-2021 Aleo Systems Inc.
// This file is part of the snarkVM library.

// The snarkVM library is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.

// The snarkVM library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with the snarkVM library. If not, see <https://www.gnu.org/licenses/>.

#[derive(Error, Debug)]
pub enum SerializationError {
    /// During serialization, we didn't have enough space to write extra info.
    #[error("the last byte does not have enough space to encode the extra info bits")]
    NotEnoughSpace,
    /// During serialization, the data was invalid.
    #[error("the input buffer contained invalid data")]
    InvalidData,
    /// During serialization, non-empty flags were given where none were
    /// expected.
    #[error("the call expects empty flags")]
    UnexpectedFlags,
    /// During serialization, we countered an I/O error.
    #[error("IoError: {0}")]
    IoError(#[from] crate::io::Error),
    /// During serialization with bincode, we encountered a serialization issue
    #[error(transparent)]
    BincodeError(#[from] bincode::Error),
}

impl From<SerializationError> for crate::io::Error {
    fn from(error: SerializationError) -> Self {
        crate::io::Error::new(crate::io::ErrorKind::Other, format!("{}", error))
    }
}
