//! Error type for the crate

use snafu::Snafu;

/// Error type describing faults that can happen during encryption or decryption.
#[derive(Debug, Snafu)]
#[non_exhaustive]
#[snafu(visibility(pub(crate)))]
pub enum Error {
    /// HMAC tag verification failed
    BadHMAC,
    /// An error occurred while deriving a key from a password
    Argon2Failure {
        /// The underlying error
        source: argon2::Error,
    },
    /// Failed to deserialize a key
    KeyDeserialization,
    /// Failed to serialize an item
    ItemSerialization,
    /// Failed to deserialize an item
    ItemDeserialization,
    /// A error occurred while attempting to compress a blob
    Compression {
        /// Underlying zstd error
        source: std::io::Error,
    },
    /// A error occurred while attempting to decompress a blob
    Decompression {
        /// Underlying zstd error
        source: std::io::Error,
    },
    /// An underlying IO error occurred while attempting to read a segment
    SegmentIO {
        /// Underlying error
        source: std::io::Error,
    },
    /// A length mismatch occurred while parsing a segment
    SegmentLength,
    /// No data was provided for a segment, not even a length
    NoData,
    /// Suspected no data provided, please see IO error for details
    NoDataIO {
        /// Underlying error
        source: std::io::Error,
    },
    /// Invalid compression flag in serialized data
    InvalidCompression,
    /// Failed reading/writing an entry
    EntryIO {
        /// Underlying error
        source: std::io::Error,
    },
    /// Indicates an invalid state where an [`LsmFile`](crate::file::LsmFile)'s internal state points to a
    /// value that doesn't exist
    InvalidLsmState,
}
