// Copyright 2021 MaidSafe.net limited.
//
// This SAFE Network Software is licensed to you under The General Public License (GPL), version 3.
// Unless required by applicable law or agreed to in writing, the SAFE Network Software distributed
// under the GPL Licence is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied. Please review the Licences for the specific language governing
// permissions and limitations relating to use of the SAFE Network Software.

use crate::{
    capacity::ChunkHolderDbs,
    error::convert_to_error_message,
    node_ops::{NodeDuties, NodeDuty, OutgoingMsg},
    to_db_key::{from_db_key, ToDbKey},
    Error, Result,
};
use log::{debug, error, info, trace, warn};
use sn_data_types::{Blob, BlobAddress, DataAddress, Error as DtError, PublicKey};
use sn_messaging::{
    client::{
        BlobDataExchange, BlobRead, BlobWrite, ChunkMetadata, CmdError, Error as ErrorMessage,
        HolderMetadata, Message, NodeCmd, NodeQuery, NodeSystemCmd, NodeSystemQuery, QueryResponse,
    },
    Aggregation, DstLocation, EndUser, MessageId,
};

use std::{
    collections::{BTreeMap, BTreeSet},
    fmt::{self, Display, Formatter},
};
use xor_name::XorName;

use super::adult_liveness::AdultLiveness;
use super::adult_reader::AdultReader;

// The number of separate copies of a blob chunk which should be maintained.
const CHUNK_COPY_COUNT: usize = 4;

/// Operations over the data type Blob.
pub(super) struct BlobRecords {
    dbs: ChunkHolderDbs,
    reader: AdultReader,
    adult_liveness: AdultLiveness,
}

impl BlobRecords {
    pub(super) fn new(dbs: ChunkHolderDbs, reader: AdultReader) -> Self {
        Self {
            dbs,
            reader,
            adult_liveness: AdultLiveness::new(),
        }
    }

    pub async fn get_all_data(&self) -> Result<BlobDataExchange> {
        debug!("Getting Blob records");
        // Prepare full_adult details
        let adult_details = &self.dbs.full_adults.lock().await;
        let all_full_adults_keys = adult_details.get_all();
        let mut full_adults = BTreeMap::new();
        for key in all_full_adults_keys {
            let val: String = adult_details
                .get(&key)
                .ok_or_else(|| Error::Logic("Error fetching full Adults".to_string()))?;
            let _ = full_adults.insert(key, val);
        }

        // Prepare older Details
        let holder_details = self.dbs.holders.lock().await;
        let all_holder_keys = holder_details.get_all();
        let mut holders = BTreeMap::new();
        for key in all_holder_keys {
            let val: HolderMetadata = holder_details
                .get(&key)
                .ok_or_else(|| Error::Logic("Error fetching Holder".to_string()))?;
            let _ = holders.insert(key, val);
        }

        // Prepare Metadata Details
        let metadata_details = self.dbs.metadata.lock().await;
        let all_metadata_keys = metadata_details.get_all();
        let mut metadata = BTreeMap::new();
        for key in all_metadata_keys {
            let val: ChunkMetadata = metadata_details
                .get(&key)
                .ok_or_else(|| Error::Logic("Error fetching Metadata".to_string()))?;
            let _ = metadata.insert(key, val);
        }

        Ok(BlobDataExchange {
            full_adults,
            holders,
            metadata,
        })
    }

    pub async fn update(&self, blob_data: BlobDataExchange) -> Result<()> {
        debug!("Updating Blob records");
        let mut orig_full_adults = self.dbs.full_adults.lock().await;
        let mut orig_holders = self.dbs.holders.lock().await;
        let mut orig_meta = self.dbs.metadata.lock().await;

        let BlobDataExchange {
            metadata,
            holders,
            full_adults,
        } = blob_data;

        for (key, value) in full_adults {
            orig_full_adults.set(&key, &value)?;
        }

        for (key, value) in holders {
            orig_holders.set(&key, &value)?;
        }

        for (key, value) in metadata {
            orig_meta.set(&key, &value)?;
        }
        Ok(())
    }

    /// Registered holders not present in provided list of members
    /// will be removed from dbs and no longer tracked for liveness.
    pub async fn retain_members_only(&mut self, members: Vec<XorName>) -> Result<()> {
        let member_names_as_string = members
            .iter()
            .map(|name| name.to_string())
            .collect::<Vec<_>>();

        // full adults
        let mut full_adults_db = self.dbs.full_adults.lock().await;
        let full_adults = full_adults_db.get_all();
        let absent_adults = full_adults
            .into_iter()
            .filter(|key| !member_names_as_string.contains(key))
            .collect::<Vec<_>>();

        for adult in absent_adults {
            let _ = full_adults_db.rem(&adult);
        }

        // holders
        let mut holders_db = self.dbs.holders.lock().await;
        let holders = holders_db.get_all();
        let absent_holders = holders
            .into_iter()
            .filter(|key| !member_names_as_string.contains(key))
            .collect::<Vec<_>>();

        let mut chunks_with_holder_change = BTreeSet::new();

        for key in &absent_holders {
            if let Some(holder) = holders_db.get::<HolderMetadata>(key) {
                chunks_with_holder_change.append(&mut holder.chunks.clone());
            }
            let _ = holders_db.rem(key);
        }

        // chunks
        let mut metadata_db = self.dbs.metadata.lock().await;

        let absent_holders: BTreeSet<XorName> = absent_holders
            .into_iter()
            .map(|holder| from_db_key(&holder).ok())
            .flatten()
            .collect();

        for address in chunks_with_holder_change {
            let chunk_key = &address.to_db_key()?;
            if let Some(mut info) = metadata_db.get::<ChunkMetadata>(chunk_key) {
                let mut any_removed = false;
                for absent_holder in &absent_holders {
                    if info.holders.remove(absent_holder) {
                        any_removed = true;
                    }
                }
                if any_removed {
                    let _ = metadata_db.set(chunk_key, &info)?;
                }
            }
        }

        // stop tracking liveness of absent holders
        self.adult_liveness.stop_tracking(absent_holders);

        Ok(())
    }

    pub(super) async fn write(
        &mut self,
        write: BlobWrite,
        msg_id: MessageId,
        origin: EndUser,
    ) -> Result<NodeDuty> {
        use BlobWrite::*;
        match write {
            New(data) => self.store(data, msg_id, origin).await,
            DeletePrivate(address) => self.delete(address, msg_id, origin).await,
        }
    }

    /// Adds a given node to the list of full nodes.
    pub async fn increase_full_node_count(&mut self, node_id: PublicKey) -> Result<()> {
        info!("No. of Full Nodes: {:?}", self.full_nodes().await);
        info!("Increasing full_node count");
        let _ = self
            .dbs
            .full_adults
            .lock()
            .await
            .lcreate(&XorName::from(node_id).to_string())?
            .ladd(&"Node Full");
        Ok(())
    }

    /// Removes a given node from the list of full nodes.
    async fn decrease_full_node_count_if_present(&mut self, node_name: XorName) -> Result<()> {
        info!("No. of Full Nodes: {:?}", self.full_nodes().await);
        info!("Checking if {:?} is present as full_node", node_name);
        match self
            .dbs
            .full_adults
            .lock()
            .await
            .rem(&node_name.to_string())
        {
            Ok(true) => {
                info!("Node present in DB, remove successful");
                Ok(())
            }
            Ok(false) => {
                info!("Node not found on full_nodes db");
                Ok(())
            }
            Err(e) => {
                error!("Error removing from full_nodes db");
                Err(Error::PickleDb(e))
            }
        }
    }

    /// Number of full chunk storing nodes in the section.
    async fn full_nodes(&self) -> u8 {
        self.dbs.full_adults.lock().await.total_keys() as u8
    }

    async fn store(&mut self, data: Blob, msg_id: MessageId, origin: EndUser) -> Result<NodeDuty> {
        // If the data already exist, check the existing no of copies.
        // If no of copies are less then required, then continue with the put request.
        let target_holders = if let Ok(metadata) = self.get_metadata_for(*data.address()).await {
            if metadata.holders.len() < CHUNK_COPY_COUNT {
                self.get_new_holders_for_chunk(data.address()).await
            } else if data.is_public() {
                trace!("{}: All good, {:?}, chunk already exists.", self, data);
                return Ok(NodeDuty::NoOp);
            } else {
                return Ok(NodeDuty::Send(OutgoingMsg {
                    msg: Message::CmdError {
                        error: CmdError::Data(ErrorMessage::DataExists),
                        id: MessageId::in_response_to(&msg_id),
                        correlation_id: msg_id,
                    },
                    section_source: false, // strictly this is not correct, but we don't expect responses to an error..
                    dst: DstLocation::EndUser(origin),
                    aggregation: Aggregation::AtDestination,
                }));
            }
        } else {
            self.get_holders_for_chunk(data.name())
                .await
                .iter()
                .cloned()
                .collect::<BTreeSet<_>>()
        };

        info!("Storing {} copies of the data", target_holders.len());

        let blob_write = BlobWrite::New(data);

        if self
            .adult_liveness
            .new_write(msg_id, blob_write.clone(), origin, target_holders.clone())
        {
            Ok(NodeDuty::SendToNodes {
                targets: target_holders,
                msg: Message::NodeCmd {
                    cmd: NodeCmd::Chunks {
                        cmd: blob_write,
                        origin,
                    },
                    id: msg_id,
                },
                aggregation: Aggregation::AtDestination,
            })
        } else {
            info!(
                "Operation with MessageId {:?} is already in progress",
                msg_id
            );
            Ok(NodeDuty::NoOp)
        }
    }

    pub async fn record_adult_write_liveness(
        &mut self,
        correlation_id: MessageId,
        result: Result<(), CmdError>,
        src: XorName,
    ) -> Result<NodeDuty> {
        if let Some(blob_write) = self
            .adult_liveness
            .record_adult_write_liveness(correlation_id, src)
        {
            if let Err(err) = result {
                error!("Error at Adult while performing a BlobWrite: {:?}", err);
                // Depending on error, we might have to take action here.
            } else {
                match blob_write {
                    BlobWrite::New(data) => {
                        if let Err(e) = self
                            .set_chunk_holder(*data.address(), src, data.owner().cloned())
                            .await
                        {
                            warn!("Error ({:?}) setting chunk holder ({:?}) of {:?}, sent by origin: {:?}", e, src, *data.address(), data.owner());
                        } else {
                            info!("(Chunk write msg {:?}): Successfully added {:?} to the list of holders for Blob at {:?}", correlation_id, src, data.address());
                        }
                    }
                    BlobWrite::DeletePrivate(_) => (),
                }
            }
        }
        let mut unresponsive_adults = Vec::new();
        for (name, count) in self.adult_liveness.find_unresponsive_adults() {
            warn!(
                "Adult {} has {} pending ops. It might be unresponsive",
                name, count
            );
            unresponsive_adults.push(name);
        }
        Ok(NodeDuty::ProposeOffline(unresponsive_adults))
    }

    pub async fn record_adult_read_liveness(
        &mut self,
        correlation_id: MessageId,
        response: QueryResponse,
        src: XorName,
    ) -> Result<NodeDuty> {
        if !matches!(response, QueryResponse::GetBlob(_)) {
            return Err(Error::Logic(format!(
                "Got {:?}, but only `GetBlob` query responses are supposed to exist in this flow.",
                response
            )));
        }
        if let Some((_address, end_user)) = self
            .adult_liveness
            .record_adult_read_liveness(correlation_id, src)
        {
            return Ok(NodeDuty::Send(OutgoingMsg {
                msg: Message::QueryResponse {
                    response,
                    id: MessageId::in_response_to(&correlation_id),
                    correlation_id,
                },
                dst: DstLocation::EndUser(end_user),
                section_source: false,
                aggregation: Aggregation::AtDestination,
            }));
        }
        let mut unresponsive_adults = Vec::new();
        for (name, count) in self.adult_liveness.find_unresponsive_adults() {
            warn!(
                "Adult {} has {} pending ops. It might be unresponsive",
                name, count
            );
            unresponsive_adults.push(name);
        }
        Ok(NodeDuty::ProposeOffline(unresponsive_adults))
    }

    async fn send_blob_cmd_error(
        &self,
        error: Error,
        msg_id: MessageId,
        origin: EndUser,
    ) -> Result<NodeDuty> {
        let message_error = convert_to_error_message(error)?;
        Ok(NodeDuty::Send(OutgoingMsg {
            msg: Message::CmdError {
                error: CmdError::Data(message_error),
                id: MessageId::in_response_to(&msg_id),
                correlation_id: msg_id,
            },
            section_source: false, // strictly this is not correct, but we don't expect responses to an error..
            dst: DstLocation::EndUser(origin),
            aggregation: Aggregation::AtDestination,
        }))
    }

    async fn delete(
        &mut self,
        address: BlobAddress,
        msg_id: MessageId,
        origin: EndUser,
    ) -> Result<NodeDuty> {
        let metadata = match self.get_metadata_for(address).await {
            Ok(metadata) => metadata,
            Err(error) => return self.send_blob_cmd_error(error, msg_id, origin).await,
        };

        // todo: use signature verification instead
        if let Some(data_owner) = metadata.owner {
            if &data_owner != origin.id() {
                return self
                    .send_blob_cmd_error(
                        Error::NetworkData(DtError::AccessDenied(*origin.id())),
                        msg_id,
                        origin,
                    )
                    .await;
            }
        };

        let mut results = vec![];
        for holder_name in &metadata.holders {
            results.push(self.remove_chunk_holder(address, *holder_name).await)
        }

        if !results.is_empty() {}

        if self.adult_liveness.new_write(
            msg_id,
            BlobWrite::DeletePrivate(address),
            origin,
            metadata.holders.clone(),
        ) {
            let msg = Message::NodeCmd {
                cmd: NodeCmd::Chunks {
                    cmd: BlobWrite::DeletePrivate(address),
                    origin,
                },
                id: msg_id,
            };
            Ok(NodeDuty::SendToNodes {
                msg,
                targets: metadata.holders,
                aggregation: Aggregation::AtDestination,
            })
        } else {
            info!(
                "Operation with MessageId {:?} is already in progress",
                msg_id
            );
            Ok(NodeDuty::NoOp)
        }
    }

    async fn set_chunk_holder(
        &mut self,
        blob_address: BlobAddress,
        holder: XorName,
        owner: Option<PublicKey>,
    ) -> Result<()> {
        // TODO -
        // - if Err, we need to flag this sender as "full" (i.e. add to self.full_adults, try on
        //   next closest non-full adult, or elder if none.  Also update the metadata for this
        //   chunk.  Not known yet where we'll get the chunk from to do that.
        info!("Setting chunk holder");

        let db_key = blob_address.to_db_key()?;
        let mut metadata = self
            .get_metadata_for(blob_address)
            .await
            .unwrap_or_default();

        if metadata.owner.is_some() && owner != metadata.owner {
            return Err(Error::Logic(format!(
                "Failed to set holder: owner({:?}) != metadata.owner({:?})",
                owner, metadata.owner
            )));
        }
        metadata.owner = owner;
        let _ = metadata.holders.insert(holder);

        if let Err(error) = self.dbs.metadata.lock().await.set(&db_key, &metadata) {
            warn!("{}: Failed to write metadata to DB: {:?}", self, error);
            return Err(error.into());
        }

        // We're acting as data handler, received request from client handlers
        let mut holders_metadata = self.get_holder(holder).await.unwrap_or_default();
        let _ = holders_metadata.chunks.insert(blob_address);

        if let Err(error) = self
            .dbs
            .holders
            .lock()
            .await
            .set(&holder.to_db_key()?, &holders_metadata)
        {
            warn!("{}: Failed to write metadata to DB: {:?}", self, error);
            return Err(error.into());
        }
        Ok(())
    }

    async fn remove_chunk_holder(
        &mut self,
        blob_address: BlobAddress,
        holder_name: XorName,
    ) -> Result<()> {
        let db_key = blob_address.to_db_key()?;
        let metadata = self.get_metadata_for(blob_address).await;
        if let Ok(mut metadata) = metadata {
            let holder = self.get_holder(holder_name).await;

            // Remove the chunk from the holder metadata
            if let Ok(mut holder) = holder {
                let _ = holder.chunks.remove(&blob_address);
                if holder.chunks.is_empty() {
                    if let Err(error) = self.dbs.holders.lock().await.rem(&holder_name.to_db_key()?)
                    {
                        warn!(
                            "{}: Failed to delete holder metadata from DB: {:?}",
                            self, error
                        );
                    }
                } else if let Err(error) = self
                    .dbs
                    .holders
                    .lock()
                    .await
                    .set(&holder_name.to_db_key()?, &holder)
                {
                    warn!(
                        "{}: Failed to write holder metadata to DB: {:?}",
                        self, error
                    );
                }
            }

            // Remove the holder from the chunk metadata
            let _ = metadata.holders.remove(&holder_name);
            if metadata.holders.is_empty() {
                if let Err(error) = self.dbs.metadata.lock().await.rem(&db_key) {
                    warn!(
                        "{}: Failed to delete chunk metadata from DB: {:?}",
                        self, error
                    );
                }
            } else if let Err(error) = self.dbs.metadata.lock().await.set(&db_key, &metadata) {
                warn!(
                    "{}: Failed to write chunk metadata to DB: {:?}",
                    self, error
                );
            }
        }
        Ok(())
    }

    pub(super) async fn remove_and_replicate_chunks(
        &mut self,
        holder: XorName,
    ) -> Result<NodeDuties> {
        info!("Replicating chunks");
        let chunks_stored = match self.remove_holder(holder).await {
            Ok(chunks) => chunks,
            _ => return Ok(vec![]),
        };
        let mut cmds = Vec::new();
        for (address, holders) in chunks_stored {
            cmds.extend(self.get_chunk_queries(address, holders).await?);
        }
        Ok(cmds)
    }

    pub(super) async fn replicate_chunk(&mut self, data: Blob) -> Result<NodeDuty> {
        info!("Replicating chunk");
        // If the data already exist, check the existing no of copies.
        // If no of copies are less then required, then continue with the put request.
        let (owner, target_holders) =
            if let Ok(metadata) = self.get_metadata_for(*data.address()).await {
                if metadata.holders.len() < CHUNK_COPY_COUNT {
                    (
                        metadata.owner,
                        self.get_new_holders_for_chunk(data.address()).await,
                    )
                } else {
                    trace!(
                        "{}: All good, {:?}, chunk copy count already satisfied.",
                        self,
                        data
                    );
                    return Ok(NodeDuty::NoOp);
                }
            } else {
                trace!(
                    "{}: Did not find any metadata for the chunk, {:?}. No replication performed.",
                    self,
                    data
                );
                return Ok(NodeDuty::NoOp);
            };

        info!("Storing {} copies of the data", target_holders.len());

        for holder in &target_holders {
            // TODO: This error needs to be handled in some way.
            if let Err(e) = self.set_chunk_holder(*data.address(), *holder, owner).await {
                warn!(
                    "Error ({:?}) when replicating chunk and setting chunk holder ({:?}) of {:?}, owned by: {:?}",
                    e,
                    *holder,
                    *data.address(),
                    owner
                )
            }
        }

        // deterministic msg id for aggregation
        let msg_id = MessageId::from_content(&(*data.name(), owner))?;

        Ok(NodeDuty::SendToNodes {
            targets: target_holders,
            msg: Message::NodeCmd {
                cmd: NodeCmd::System(NodeSystemCmd::ReplicateChunk(data)),
                id: msg_id,
            },
            aggregation: Aggregation::AtDestination,
        })
    }

    async fn get_chunk_queries(
        &mut self,
        address: BlobAddress,
        current_holders: BTreeSet<XorName>,
    ) -> Result<NodeDuties> {
        let mut node_ops = Vec::new();
        let messages = current_holders
            .into_iter()
            .map(|holder| {
                info!("Sending get-chunk query to holder {:?}", holder);
                (
                    Message::NodeQuery {
                        query: NodeQuery::System(NodeSystemQuery::GetChunk(address)),
                        id: MessageId::combine(vec![*address.name(), holder]),
                    },
                    holder,
                )
            })
            .collect::<Vec<_>>();
        for (msg, dst) in messages {
            node_ops.push(NodeDuty::Send(OutgoingMsg {
                msg,
                section_source: true, // i.e. errors go to our section
                dst: DstLocation::Node(dst),
                aggregation: Aggregation::AtDestination,
            }));
        }
        Ok(node_ops)
    }

    pub(super) async fn read(
        &mut self,
        read: &BlobRead,
        msg_id: MessageId,
        origin: EndUser,
    ) -> Result<NodeDuty> {
        match read {
            BlobRead::Get(address) => self.get(*address, msg_id, origin).await,
        }
    }

    async fn get(
        &mut self,
        address: BlobAddress,
        msg_id: MessageId,
        origin: EndUser,
    ) -> Result<NodeDuty> {
        let query_error = |error: Error| {
            Ok(NodeDuty::Send(OutgoingMsg {
                msg: Message::QueryResponse {
                    response: QueryResponse::GetBlob(Err(convert_to_error_message(error)?)),
                    id: MessageId::in_response_to(&msg_id),
                    correlation_id: msg_id,
                },
                section_source: false, // strictly this is not correct, but we don't expect responses to an error..
                dst: DstLocation::EndUser(origin),
                aggregation: Aggregation::AtDestination,
            }))
        };

        let metadata = match self.get_metadata_for(address).await {
            Ok(metadata) => metadata,
            Err(error) => return query_error(error),
        };

        if let Some(data_owner) = metadata.owner {
            if &data_owner != origin.id() {
                return query_error(Error::NetworkData(DtError::AccessDenied(*origin.id())));
            }
        };

        if self
            .adult_liveness
            .new_read(msg_id, address, origin, metadata.holders.clone())
        {
            let msg = Message::NodeQuery {
                query: NodeQuery::Chunks {
                    query: BlobRead::Get(address),
                    origin,
                },
                id: msg_id,
            };

            Ok(NodeDuty::SendToNodes {
                msg,
                targets: metadata.holders,
                aggregation: Aggregation::None,
            })
        } else {
            info!(
                "Operation with MessageId {:?} is already in progress",
                msg_id
            );
            Ok(NodeDuty::NoOp)
        }
    }

    // Updates the metadata of the chunks help by a node that left.
    // Returns the list of chunks that were held along with the remaining holders.
    async fn remove_holder(
        &mut self,
        name: XorName,
    ) -> Result<BTreeMap<BlobAddress, BTreeSet<XorName>>> {
        // stop tracking liveness of removed holder
        self.adult_liveness
            .stop_tracking(vec![name].into_iter().collect());
        // remove from full_nodes if present
        self.decrease_full_node_count_if_present(name).await?;

        let mut blob_addresses: BTreeMap<BlobAddress, BTreeSet<XorName>> = BTreeMap::new();
        let chunk_holder = self.get_holder(name).await;

        if let Ok(holder) = chunk_holder {
            for chunk_address in holder.chunks {
                let db_key = chunk_address.to_db_key()?;
                let chunk_metadata = self.get_metadata_for(chunk_address).await;

                if let Ok(mut metadata) = chunk_metadata {
                    if !metadata.holders.remove(&name) {
                        warn!("doesn't contain the holder",);
                    }

                    let _ = blob_addresses.insert(chunk_address, metadata.holders.clone());

                    if metadata.holders.is_empty() {
                        if let Err(error) = self.dbs.metadata.lock().await.rem(&db_key) {
                            warn!("{}: Failed to write metadata to DB: {:?}", self, error);
                        }
                    } else if let Err(error) =
                        self.dbs.metadata.lock().await.set(&db_key, &metadata)
                    {
                        warn!("{}: Failed to write metadata to DB: {:?}", self, error);
                    }
                }
            }
        }

        // Since the node has left the section, remove it from the holders DB
        if let Err(error) = self.dbs.holders.lock().await.rem(&name.to_db_key()?) {
            warn!("{}: Failed to delete metadata from DB: {:?}", self, error);
        };

        Ok(blob_addresses)
    }

    async fn get_holder(&self, holder: XorName) -> Result<HolderMetadata> {
        match self
            .dbs
            .holders
            .lock()
            .await
            .get::<HolderMetadata>(&holder.to_db_key()?)
        {
            Some(metadata) => {
                if metadata.chunks.is_empty() {
                    warn!("{}: is not responsible for any chunk", holder);
                    Err(Error::NodeDoesNotHoldChunks)
                } else {
                    Ok(metadata)
                }
            }
            None => {
                warn!("{}: is not responsible for any chunk", holder);
                Err(Error::NodeDoesNotHoldChunks)
            }
        }
    }

    async fn get_metadata_for(&self, address: BlobAddress) -> Result<ChunkMetadata> {
        match self
            .dbs
            .metadata
            .lock()
            .await
            .get::<ChunkMetadata>(&address.to_db_key()?)
        {
            Some(metadata) => {
                if metadata.holders.is_empty() {
                    warn!("{}: Metadata holders is empty for: {:?}", self, address);
                    Err(Error::NoHoldersOfChunk)
                } else {
                    Ok(metadata)
                }
            }
            None => {
                warn!(
                    "{}: Did not find metadata in DB for chunk: {:?}",
                    self, address
                );
                Err(Error::NoSuchChunk(DataAddress::Blob(address)))
            }
        }
    }

    // Returns `XorName`s of the target holders for an Blob chunk.
    // Used to fetch the list of holders for a new chunk.
    async fn get_holders_for_chunk(&self, target: &XorName) -> Vec<XorName> {
        self.reader
            .our_adults_sorted_by_distance_to(&target, CHUNK_COPY_COUNT)
            .await
    }

    // Returns `XorName`s of the new target holders for an Blob chunk.
    // Used to fetch the additional list of holders for existing chunks.
    async fn get_new_holders_for_chunk(&self, target: &BlobAddress) -> BTreeSet<XorName> {
        let closest_holders = self
            .get_holders_for_chunk(target.name())
            .await
            .iter()
            .cloned()
            .collect::<BTreeSet<_>>();
        if let Ok(metadata) = self.get_metadata_for(*target).await {
            return closest_holders
                .difference(&metadata.holders)
                .cloned()
                .collect();
        }
        closest_holders
    }
}

impl Display for BlobRecords {
    fn fmt(&self, formatter: &mut Formatter) -> fmt::Result {
        write!(formatter, "BlobRecords")
    }
}
