
/*
 * Database Manager
 */

use std::path::PathBuf;
use sled::Result as SledResult;
use serde::{Serialize, Deserialize};

#[derive(Serialize, Deserialize, PartialEq, Debug)]
struct ReleaseLink {
    name: Option<String>,
    version: Option<String>,
    url: Option<String>,
}

#[derive(Debug)]
pub struct DatabaseManager {
    pub sn_db_path: PathBuf,
    pub db: sled::Db,
}

impl DatabaseManager {
    pub fn new(sn_db_path: PathBuf) -> SledResult<DatabaseManager> {
        let db = sled::open(&sn_db_path)?;
        Ok(DatabaseManager {
            sn_db_path,
            db,
        })
    }

    pub fn init(&self) -> SledResult<()> {

        if self.db.is_empty() {
            self.insert_release_link("releaselink1", "sn_cli", "0.1.0", "http://api.github.com/repos/maidsafe/sn_api/releases/latest")?;
            self.insert_release_link("releaselink2", "sn_node", "0.1.0", "http://api.github.com/repos/maidsafe/sn_node/releases/latest")?;
            self.insert_release_link("releaselink3", "sn_launch_tool", "0.1.0", "http://api.github.com/repos/maidsafe/sn_launch_tool/releases/latest")?;
        }        
        Ok(())
    }

    pub fn insert_release_link(&self, key: &str, name: &str, version: &str, url: &str) -> SledResult<()> {
        let target = ReleaseLink{ 
            version: Some(version.to_string()).filter(|x| !x.is_empty()), 
            name: Some(name.to_string()).filter(|x| !x.is_empty()), 
            url: Some(url.to_string()).filter(|x| !x.is_empty()),
        };
        let encoded: Vec<u8> = bincode::serialize(&target).unwrap();

        let mut releaselink_value = vec![]; 
        releaselink_value.extend_from_slice(&encoded);
        
        self.db.insert(key, releaselink_value)?;
        Ok(())
    }

    pub fn get_repo_api_urls(&self) -> Vec<String> {
        let mut vec = Vec::new();
        
        for kv in self.db.range("releaselink1".."releaselink4") {
            if let Some((key, value)) = kv.ok() {
                let decoded: ReleaseLink = bincode::deserialize(&value).unwrap();
                println!("{:?} {:?} {:?}", decoded.name.unwrap(), decoded.version.unwrap(), decoded.url.to_owned().unwrap());
                vec.push(decoded.url.unwrap());
            }
        }

        return vec;
    }
}
