// Copyright 2020 MaidSafe.net limited.
//
// This SAFE Network Software is licensed to you under The General Public License (GPL), version 3.
// Unless required by applicable law or agreed to in writing, the SAFE Network Software distributed
// under the GPL Licence is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied. Please review the Licences for the specific language governing
// permissions and limitations relating to use of the SAFE Network Software.
#![recursion_limit = "256"]

mod cli;
mod operations;
mod subcommands;

use cli::run;
use color_eyre::{eyre::eyre, Help, Report, Result};
use human_panic::{handle_dump, Metadata};
use std::panic::set_hook;
use tracing::{self, debug};
use tracing_subscriber::filter::EnvFilter;

extern crate comfy_table;

#[allow(dead_code)]
const APP_ID: &str = "net.maidsafe.cli";
#[allow(dead_code)]
const APP_NAME: &str = "Safe CLI";
#[allow(dead_code)]
const APP_VENDOR: &str = "MaidSafe.net Ltd";

#[tokio::main]
async fn main() -> Result<(), Report> {
    color_eyre::install()?;
    if let Ok(filter) = std::env::var("RUST_LOG") {
        let filter = EnvFilter::try_new(filter).map_err(|e| {
            eyre!(
                "Failed to parse the logging filter specified with RUST_LOG: {}",
                e
            )
            .suggestion(
                "Please try another filter.\
                  Example: RUST_LOG=safe=debug,sn_api=debug,sn_client=debug",
            )
        })?;
        tracing_subscriber::fmt()
            .with_env_filter(filter)
            .with_thread_names(true)
            .with_ansi(false);
        tracing_subscriber::fmt::init();
    };

    debug!("Starting Safe CLI...");

    // We register our own panic hook to customise the error message displayed,
    // the default error message generated by 'human_panic' is not exactly what we desire.
    set_hook(Box::new(|panic_info| {
        let metadata = Metadata {
            name: env!("CARGO_PKG_NAME").into(),
            version: env!("CARGO_PKG_VERSION").into(),
            authors: "MaidSafe Developers <dev@maidsafe.net>".into(),
            homepage: "https://maidsafe.net".into(),
        };

        let mut error_msg = String::new();
        error_msg.push_str("The Safe CLI had a problem and crashed. To help us diagnose the problem you can send us a crash report.\n\n");
        match handle_dump(&metadata, panic_info) {
            Some(report_filepath) => error_msg.push_str(&format!("We have generated a report file at \"{}\". Please submit an issue, including the report as an attachment, at {}.\n", report_filepath.display(), env!("CARGO_PKG_REPOSITORY"))),
            None => error_msg.push_str(&format!("Please submit an issue, including details to reproduce it, at {}.\n", env!("CARGO_PKG_REPOSITORY"))),
        }
        error_msg.push_str(
            "Alternatively, you can report it on our dev forum at: https://forum.safedev.org.\n\n",
        );
        error_msg.push_str("In order to improve the software, we rely on people to submit reports.\n\nThank you kindly!\n");
        eprintln!("{}", error_msg);
    }));

    run().await
}
